/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skipstat.c	1.7 95/11/15 Sun Microsystems"

#include <stdio.h>
#include <fcntl.h>
#include <stropts.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <string.h>
#include <netdb.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>

#include <skip_proto.h>
#include <skip_types.h>
#include <skip_ioctl.h>
#include <skip_lib.h>

#define 	UNKNOWN		"<unknown>"
char		*progname;
extern int	errno;

extern char	*optarg;
extern int	optind;

ioctl_if_stat_t		if_stats;
ioctl_key_stat_t	key_stats;
ioctl_hdr_stat_t	hdr_stats;

extern boolean_t 	skip_supported_kp_algs[];
extern char		*skip_kp_algs[];
extern boolean_t 	skip_supported_kp_algs_v1[];
extern char		*skip_kp_algs_v1[];

/*
 * Format error message if invalid options
 */
static void
usage()
{

	fprintf(stderr, "usage:\t %s "
			"[-aCcIkh] [-i <interface>]\n"
			"\t-C:	display cryptographic algorithms supported\n"
			"\t-c:	display cryptographic algorithm statistics\n"
			"\t-I:	display SKIP interface statistics\n"
			"\t-k:	display SKIP key statistics\n"
			"\t-h:	display SKIP header statistics\n"
			"\t-a:	all %s options\n",
			 progname, progname);
			
	fprintf(stderr, "\n");
	exit(1);
}

/*
 * get_if_stats()
 *
 * Get and display per SKIP interface statistics
 */
static int
get_if_stats(char *if_name)
{

	if (skip_get_if_stats(if_name, &if_stats)) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	printf("\nSKIP interface (%s) statistics:\n", if_name);

	printf("skip_if_ipkts:\t\t%u\n", if_stats.skip_if_ipkts);
	printf("skip_if_opkts:\t\t%u\n", if_stats.skip_if_opkts);
	printf("skip_if_encrypts:\t%u\n", if_stats.skip_if_encrypts);
	printf("skip_if_decrypts:\t%u\n", if_stats.skip_if_decrypts);
	printf("skip_if_drops:\t\t%u\n", if_stats.skip_if_drops);
	printf("skip_if_bypasses:\t%u\n", if_stats.skip_if_bypasses);

	return(0);
}

/*
 * get_key_stats()
 *
 * Get and display SKIP key statistics
 */
static int
get_key_stats(char *if_name)
{

	int key_idle, key_bytes;

	/*
	 * Get the key management parameters
	 */
	if (skip_var_get(if_name, "skip_key_max_bytes", &key_bytes) < 0) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	if (skip_var_get(if_name, "skip_key_max_idle", &key_idle) < 0) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	/*
	 * Now, get the keys statistics
	 */
	if (skip_get_key_stats(if_name, &key_stats)) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	printf("\nSKIP key management parameters:\n");
	printf("skip_key_max_idle:\t%d\n", key_idle);
	printf("skip_key_max_bytes:\t%d\n", key_bytes);

	printf("\nSKIP key statistics:\n");

	printf("skip_encrypt_keys_active:\t%u\n",
			key_stats.skip_encrypt_keys_active);
	printf("skip_decrypt_keys_active:\t%u\n",
			key_stats.skip_decrypt_keys_active);
	printf("skip_key_lookups:\t\t%u\n", key_stats.skip_key_lookups);
	printf("skip_keymgr_requests:\t\t%u\n", key_stats.skip_keymgr_requests);
	printf("skip_key_reclaims:\t\t%u\n", key_stats.skip_key_reclaims);
	printf("skip_hash_collisions:\t\t%u\n", key_stats.skip_hash_collisions);

	return(0);
}

/*
 * get_hdr_stats()
 *
 * Get and display SKIP header statistics
 */
static int
get_hdr_stats(char *if_name)
{

	if (skip_get_hdr_stats(if_name, &hdr_stats)) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	printf("\nSKIP header statistics:\n");

	printf("skip_hdr_bad_versions:\t\t%u\n",
			hdr_stats.skip_hdr_bad_versions);
	printf("skip_hdr_short_ekps:\t\t%u\n", hdr_stats.skip_hdr_short_ekps);
	printf("skip_hdr_short_mids:\t\t%u\n", hdr_stats.skip_hdr_short_mids);
	printf("skip_hdr_bad_kp_algs:\t\t%u\n", hdr_stats.skip_hdr_bad_kp_algs);

	printf("V1 skip_hdr_encodes:\t\t%u\n", hdr_stats.skip_hdr_encodes);
	printf("V1 skip_hdr_decodes:\t\t%u\n", hdr_stats.skip_hdr_decodes);
	printf("V1 skip_hdr_runts:\t\t%u\n", hdr_stats.skip_hdr_runts);
	printf("V1 skip_hdr_short_nodeids:\t%u\n",
			hdr_stats.skip_hdr_short_nodeids);

	printf("IPSP skip_ipsp_decodes:\t\t%u\n",
			hdr_stats.skip_ipsp_decodes);
	printf("IPSP skip_ipsp_encodes:\t\t%u\n",
			hdr_stats.skip_ipsp_encodes);
	printf("IPSP skip_hdr_bad_nsid:\t\t%u\n",
			hdr_stats.skip_hdr_bad_nsid);
	printf("IPSP skip_hdr_bad_next:\t\t%u\n",
			hdr_stats.skip_hdr_bad_next);
	printf("IPSP skip_hdr_bad_spi:\t\t%u\n",
			hdr_stats.skip_hdr_bad_spi);
	printf("IPSP skip_hdr_bad_iv:\t\t%u\n",
			hdr_stats.skip_hdr_bad_iv);
	printf("IPSP skip_hdr_short_r_mkeyid:\t%u\n",
			hdr_stats.skip_hdr_short_r_mkeyid);
	printf("IPSP skip_hdr_short_s_mkeyid:\t%u\n",
			hdr_stats.skip_hdr_short_s_mkeyid);
	printf("IPSP skip_hdr_bad_r_mkeyid:\t%u\n",
			hdr_stats.skip_hdr_bad_r_mkeyid);
	return(0);
}

/*
 * get_crypt_list()
 *
 * Get and display crypto modules list
 */
static int
get_crypt_list(char *if_name)
{
	int			idx;

	if (skip_get_kp_algs()) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	printf("\nCryptographic algorithms present (SunScreen Mode):\n");

	for (idx = 0; idx < SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_kp_algs_v1[idx]) {
			printf("Crypto Module Id:\t%u", idx);
			printf("\tCrypto Name:\t%s\n",
				skip_kp_algs_v1[idx] ?
					skip_kp_algs_v1[idx] : UNKNOWN);
		}
        }

	printf("\nCryptographic algorithms present (IPSP Mode):\n");

	for (idx = 0; idx < SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_kp_algs[idx]) {
			printf("Crypto Module Id:\t%u", idx);
			printf("\tCrypto Name:\t%s\n",
				skip_kp_algs[idx] ?
					skip_kp_algs[idx] : UNKNOWN);
		}
        }

	printf("\n");

	return(0);
}

/*
 * get_crypt_stats()
 *
 * Get and display crypto modules statistics
 */
static int
get_crypt_stats(char *if_name)
{
	ioctl_crypt_stat_t	crypt_stats;
	ioctl_alg_list_t	crypt_algs;
	crypt_mod_stat_t	*p_stats = crypt_stats.crypt_stats;
	int			idx;

	if (skip_get_kp_algs()) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}


	crypt_stats.version = SKIP_V1;

	if (skip_get_crypt_stats(if_name, &crypt_stats)) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	printf("\nCryptographic algorithm statistics (SunScreen Mode):\n");

	for (idx = 0; idx < SKIP_MAXCRYPTORS; idx++) {

		if (skip_supported_kp_algs_v1[idx]) {
			printf("\nCrypto Module Name:\t%s\n",
				skip_kp_algs_v1[idx] ?
					skip_kp_algs_v1[idx] : UNKNOWN);
	
			printf("encrypts:\t\t%u\n", p_stats->encrypts);
			printf("encrypterrs:\t\t%u\n", p_stats->encrypterrs);
			printf("decrypts:\t\t%u\n", p_stats->decrypts);
			printf("decrypterrs:\t\t%u\n", p_stats->decrypterrs);

			/*
		  	 * Next cryptor
		 	 */
			p_stats++;
		}
        }
	printf("\n");

	crypt_stats.version = SKIP_V2;

	if (skip_get_crypt_stats(if_name, &crypt_stats)) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(-1);
	}

	printf("\nCryptographic algorithm statistics (IPSP Mode):\n");

	p_stats = crypt_stats.crypt_stats;

	for (idx = 0; idx < SKIP_MAXCRYPTORS; idx++) {

		if (skip_supported_kp_algs[idx]) {
			printf("\nCrypto Module Name:\t%s\n",
				skip_kp_algs[idx] ?
					skip_kp_algs[idx] : UNKNOWN);
	
			printf("encrypts:\t\t%u\n", p_stats->encrypts);
			printf("encrypterrs:\t\t%u\n", p_stats->encrypterrs);
			printf("decrypts:\t\t%u\n", p_stats->decrypts);
			printf("decrypterrs:\t\t%u\n", p_stats->decrypterrs);

			/*
		  	 * Next cryptor
		 	 */
			p_stats++;
		}
        }
	printf("\n");

	return(0);
}

/*
 * skipstat tool
 *
 * Get SKIP driver statistics
 *
 */
main(int argc, char *argv[])
{
	char		*name;
	char		*interface = skip_default_if();

	int		opt, req, rc;
	int		opt_all, opt_if, opt_key, opt_drv, opt_hdr;		
	int		opt_crypt_list, opt_crypt_stats;

	progname = argv[0];
	opt_all = opt_if = opt_key = opt_drv = opt_hdr = 0;
	opt_crypt_list = opt_crypt_stats = 0;

	/*
	 * Command line parsing
	 */
	while ((opt = getopt(argc,argv, "aCchIki:")) != -1) {

		switch (opt) {
		
		case 'a':
			opt_all = 1;
			goto precheck;
			break;

		case 'k':
			opt_key = 1;
			break;

		case 'h':
			opt_hdr = 1;
			break;

		case 'I':
			opt_if = 1;
			break;

		case 'i':
			interface = optarg;
			break;

		case 'C':
			opt_crypt_list = 1;
			break;

		case 'c':
			opt_crypt_stats = 1;
			break;

		default:
			usage();
			

		}

	}

precheck:
	/*
	 * Useful pre-checks
	 */
	if (skip_pre_checks(interface)) {
		switch (errno) {
		case EACCES:
			fprintf(stderr, "%s: you must be root to run this "
				"program\n", progname);
			break;

		case ENODEV:
			fprintf(stderr, "%s\n", skip_errmsg);
			fprintf(stderr, "Check that SKIP is correctly "
				"installed on this system\n", progname);
			break;
		default:
			fprintf(stderr, "%s\n", skip_errmsg);
		}
		exit(1);
	}

	/*
	 * Get all algorithms configured in the kernel
	 */
	if (skip_var_init(progname)) {
		fprintf(stderr, "%s\n", skip_errmsg);
		exit(1);
	}

	/*
	 * Check for default action 
	 */
	req = opt_if+opt_key+opt_drv+opt_hdr+opt_crypt_list+opt_crypt_stats;
	if (!opt_all &&  !req) {
		rc = get_if_stats(interface);
		exit(rc);
	}

	/*
	 * Process each command line option
	 */
	if (opt_all || opt_crypt_list) {
		get_crypt_list(interface);
	}

	if (opt_all || opt_crypt_stats) {
		get_crypt_stats(interface);
	}

	if (opt_all || opt_if) {
		get_if_stats(interface);
	}

	if (opt_all || opt_key) {
		get_key_stats(interface);
	}

	if (opt_all || opt_hdr) {
		get_hdr_stats(interface);
	}

	return (0);
}
