/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)CertInfo.C	1.14 96/04/10"

#include "CertInfo.h"
#include "X509skip.h"
#include "HashCert.h"


CertInfo::CertInfo()
{
	cert = 0;
}

CertInfo::~CertInfo()
{
	if (cert)
		delete cert;	
}

CertInfo::CertInfo(const CertInfo& orig)
{
	nsid = orig.nsid;
	mkid = orig.mkid;

	if (orig.cert)
		cert = orig.cert->dup();
	else
		cert = 0;
}

CertInfo& CertInfo::operator =(const CertInfo& rhs)
{
	if (this != &rhs) {
		nsid = rhs.nsid;
		mkid = rhs.mkid;

		if (cert)
			delete cert;
		if (rhs.cert)
			cert = rhs.cert->dup();
		else
			cert = 0;
	}

	return *this;
}

CertInfo::CertInfo(int ns, const Bstream& name, SkipCert* c)
{
	nsid = ns;
	mkid = name;

	if (c)
		cert = c->dup();
	else
		cert = 0;
}

//  CertInfo to Bstream:
//	nsid | mkid length | mkid | CertType | cert length | cert | expansion

Bstream CertInfo::encode()
{
	Bstream ret;

	Bstream::putbyte(ret, nsid);

	Bstream::putint(ret, mkid.getlength());
	ret += mkid;

	if (cert) {
		Bstream::putint(ret, cert->CertType());
		Bstream x = cert->encode();
		Bstream::putint(ret, x.getlength());
		ret += x;
	} else {
		Bstream::putint(ret, 0);	// CertType == 0
		Bstream::putint(ret, 0);	// cert length == 0
	}

	Bstream::putbyte(ret, 0);

	return ret;
}


//  There are two classes of errors decoding can encounter:
//	damage to the encoding format
//	failure to understand the contents
//
//  Perhaps these should be differentiated

int CertInfo::decode(Bstream& in, CertInfo& ci)
{
	int mkid_len;
	int type;
	int cert_len;
	u_char expansion;

	// first delete contents of existing CertInfo, if any

	if (ci.cert) {
		delete ci.cert;
		ci.cert = NULL;
	}

	if (Bstream::getbyte(in, &ci.nsid))		// nsid
		return 1;

	if (Bstream::getint(in, &mkid_len))		// mkid length
		return 1;

	if (in.getlength() < mkid_len)
		return 1;
	ci.mkid = Bstream(mkid_len, in.getdatap());	// mkid
	in.consume(mkid_len);

	if (Bstream::getint(in, &type))			// CertType
		return 1;

	if (Bstream::getint(in, &cert_len))		// cert length
		return 1;

	if (in.getlength() < cert_len)
		return 1;
	Bstream cert_s(cert_len, in.getdatap());	// cert
	in.consume(cert_len);

	if (SkipCert::decode(type, cert_s, ci.cert))
		return 1;

	if (Bstream::getbyte(in, &expansion))
		return 1;

	if (expansion)		// database newer than this executable
		return 1;

	return 0;
}

void CertInfo::print()
{
      String temp;
      printf("NSID = %03d ",nsid);
      temp = mkid.gethexstr();
      printf("MKID = %s\n", (const char *)temp);
      if (cert == NULL) 
              printf("CERT is NULL\n");
      else
              cert->print();
}

char* CertInfo::get_info_str()
{
	static char buf[256];
	String temp;
	char cert_s[256];

	temp = mkid.gethexstr();
	if (cert == NULL)
		sprintf(cert_s, "cert=NULL");
	else
		sprintf(cert_s, "cert type=%d", cert->CertType());

	sprintf(buf, "nsid=%d mkid=%s %s", nsid, (const char *)temp, cert_s);
	return buf;
}

char* CertInfo::check_valid(AuthPolicy& auth)
{
	char *s;

	if (cert == NULL)
		return "no certificate";

	if (mkid != cert->skip_name())
		return "certificate name mismatch";

	if (cert->skip_notvalidbefore() > NTPNOW)
		return "certificate not valid yet";

	if (cert->skip_notvalidafter() < NTPNOW)
		return "certificate expired";

	if (s = cert->isValid(auth))
		return s;

	return NULL;	// verification succeeded
}

