/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_ah.c	1.11 96/04/02 Sun Microsystems"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_proto.h>
#include <skip_types.h>
#include <skip_keymgrmsgs.h>
#include <skip_crypt.h>
#include <skip_key.h>
#include <skip_ah.h>
#include <skip_ioctl.h>

/*
 * table of MAC Algorithms and associated statistics
 */
const int		skip_max_maclags = SKIP_MAX_MAC;
skip_mac_t		*skip_macalgs[SKIP_MAX_MAC];

/*
 * Imported MAC modules descriptors
 */
extern skip_mac_t	skip_md5_mod;

extern ioctl_hdr_stat_t	skip_hdrs_stats;


/* skip_mac_init()
 * 
 * Returns: none
 */
void
skip_mac_init()
{
	/*
	 * initialise the table of MAC modules to be empty
	 */
	BZERO((caddr_t) skip_macalgs, sizeof skip_macalgs);
	/*
	 * Now, install all supported MAC modules...
	 */
	skip_macalgs[skip_md5_mod.mod_id] = &skip_md5_mod;
}

/* skip_mac_uninit()
 * 
 * Returns: none
 */
void
skip_mac_uninit()
{
	BZERO((caddr_t) skip_macalgs, sizeof skip_macalgs);
}

/*
 * skip_sign()
 *
 * Fill in the AH Header in the SKIP packet
 */
int
skip_sign(
	skip_keycb_t *cp,
	skip_param_t *params,
	unsigned char *ip_start, 
	unsigned char *ah_start,
	int len)
{
	struct ip	*ip = (struct ip *)ip_start;
	skip_mac_t	*mac;
	unsigned short 	ip_sum;
	unsigned short 	ip_off;
	unsigned char 	ip_ttl;
	unsigned char 	ip_tos;
	unsigned char	mac_len;
	unsigned char	next;

	/*
	 * Set the total length in the IP header..
	 */
	ip->ip_len = htons(len);

	/*
	 * Save all IP header fields that with have to reset for AH
	 */
	ip_sum = ip->ip_sum;
	ip_ttl = ip->ip_ttl;
	ip_tos = ip->ip_tos;
	ip_off = ip->ip_off;

	/*
	 * Now, reset them..
	 */
	ip->ip_sum = 0;
	ip->ip_ttl = 0;
	ip->ip_tos = 0;
	ip->ip_off = 0;

	/*
	 * Retrieve the MAC module descriptor
	 */
	mac = GETMACMOD(params->mac_alg);
	if (!mac) {
		return (EINVAL);
	}

	/*
	 * Authenticated  data length is in 32 bits words
	 */
	mac_len = mac->data_len / sizeof(unsigned int);

	/*
	 * Find out the next protocol...
	 */
	if (params->kp_alg) {
		next = SKIP_NEXT_ESP;
	} else {
		next = params->payload_type;
	}

	/*
	 * Fill in the AH header
	 */
	*ah_start++ = next;				/* Protocol	*/
	*ah_start++ = mac_len;				/* Length	*/
	*ah_start++ = 0;				/* Reserved	*/
	*ah_start++ = 0;				/* Reserved	*/
	
	/*
	 * Fill in SPI field...
	 */
	if (params->version == SKIP_V2) {
		/*
		 * Fix SKIP SPI
		 */
		*((unsigned long *)ah_start) = SKIP_SPI;
		ah_start += 4;
	} else {
		/*
		 * Set the SPI with Receiver Master key ID
		 */
		if (params->r_nsid) {
			KEYTOBUF(params->r_mkeyid, ah_start);
			ah_start += params->r_mkeyid.len;
		}
	}

	/*
	 * Cleanup MAC...
	 */
	BZERO(ah_start, mac->data_len);

	/*
	 * Compute the AH Data
	 */
	if ((*mac->sign)(cp, ip_start, ah_start, len) < 0) {
		mac->out_mac_errs++;
		return (EINVAL);
	} else {
		mac->out_mac++;
	}

	/*
	 * Restore IP fields
	 */
	ip->ip_sum = ip_sum;
	ip->ip_ttl = ip_ttl;
	ip->ip_tos = ip_tos;
	ip->ip_off = ip_off;

	return(0);
}

/*
 * skip_auth()
 *
 * Check the AH of a SKIP packet
 */
int
skip_auth(skip_keycb_t *cp,
	skip_param_t *params,
	unsigned char *ip_start, 
	unsigned char *ah_start,
	int len)
{
	struct ip	*ip = (struct ip *)ip_start;
	skip_mac_t	*mac;
	unsigned char	mac_buf[SKIP_MAX_AH_SZ];
	unsigned char	res_buf[SKIP_MAX_AH_SZ];

	unsigned short 	ip_sum;
	unsigned short 	ip_off;
	unsigned char 	ip_ttl;
	unsigned char 	ip_tos;
	unsigned char	mac_len;
	unsigned char 	*p = ah_start;
	unsigned long	skip_spi = 0;

	mac = GETMACMOD(params->mac_alg);

	if (!mac) {
		/*
		 * MAC alg is not supported
		 */
		skip_hdrs_stats.skip_hdr_bad_mac_algs++;
		return (-1);
	}

	/*
	 * Check the packet length...
	 */
	if (len < (int) (SKIP_AH_HDR_SZ + mac->data_len + IPHLEN(ip))) {
		return (-1);
	}

	/*
	 * Get the next protocol
	 */
	params->payload_type = *p;

	p++;
	/*
	 * Authenticated  data length is in 32 bits words
	 */
	mac_len = *p * sizeof(unsigned int);
	p++;

	if (mac_len != mac->data_len) {
		/*
		 * Authenticated  data length is invalid
		 */
		skip_hdrs_stats.skip_hdr_bad_mac_size++;
		return (-1);
	}

	/*
	 * Two reserved bytes...
	 */
	p += 2;

	/*
	 * Get SPI field
	 */
	skip_spi = * (unsigned long *) p;

	/*
	 * Check SKIP SPI field...
	 */
	if (params->version == SKIP_V2) {
		/*
		 * SKIP fix SPI...
		 */
		if (skip_spi != SKIP_SPI) {
			skip_hdrs_stats.skip_hdr_bad_ah_spi++;
			return (-1);
		}
	} else {
		/*
		 * with raw ESP/AH mode, SPI should match the
		 * Sender Master Key ID.
		 */
		if (BCMP((caddr_t) &skip_spi,
				(caddr_t) params->s_mkeyid.buf,
				sizeof(skip_spi))) {
			skip_hdrs_stats.skip_hdr_bad_ah_spi++;
			return (-1);
		}
	}

	p += sizeof (skip_spi);

	/*
	 * We need now to save the authenticated data..
	 */
	BCOPY(p, mac_buf, mac_len);

	/*
	 * And zero them...
	 */
	BZERO(p, mac_len);

	/*
	 * Save all IP header fields that we have to reset for AH
	 */
	ip_sum = ip->ip_sum;
	ip_ttl = ip->ip_ttl;
	ip_tos = ip->ip_tos;
	ip_off = ip->ip_off;

	/*
	 * Now, reset them..
	 */
	ip->ip_sum = 0;
	ip->ip_ttl = 0;
	ip->ip_tos = 0;
	ip->ip_off = 0;

	/*
	 * Compute the AH Data
	 */
	if ((*mac->sign)(cp, ip_start, res_buf, len) < 0) {
		mac->in_mac_errs++;
		return (-1);
	} else {
		mac->in_mac++;
	}

	/*
	 * Restore IP fields
	 */
	ip->ip_sum = ip_sum;
	ip->ip_ttl = ip_ttl;
	ip->ip_tos = ip_tos;
	ip->ip_off = ip_off;

	/*
	 * Now, check if MAC is OK...
	 */
	if (BCMP(res_buf, mac_buf, mac_len)) {
		skip_hdrs_stats.skip_hdr_bad_mac_val++;
		return (-1);
	}

	/*
	 * OK, return the size of the total AH header
	 */
	return(SKIP_AH_HDR_SZ + mac->data_len);

}
