/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)cdp.C	1.13 96/04/25"

#include "Bigint.h"
#include "Bstream.h"

/*
 * System includes
 */
#include <skip_os.h>

#include "my_types.h"
#include "SkipCert.h"
#include "CertInfo.h"
#include "LL.h"
#include "cdp.h"

void
cdp::clear()
{
	action = n_of_recs = reserved = icookie = rcookie = 0;
	version = 1;
	recs.clear();
}	

cdp::cdp() 
{
	cdp::clear();
}

cdp::cdp(u_char a)
{
	cdp::clear();
	action = a;
}

cdp::~cdp()
{
	recs.clear();
}

void cdp::cdp_getok(const CertInfo& c)
{
	cdp_rec rec;

	rec.action = CDP_REC_GETOK;
	rec.ci = c;

	recs.append(rec);
}

void cdp::cdp_getfail(u_char nsid, const Bstream& mkid)
{
	cdp_rec rec;

	rec.action = CDP_REC_GETFAIL;
	rec.ci.nsid = nsid;
	rec.ci.mkid = mkid;

	recs.append(rec);
}

void cdp::cdp_get(u_char nsid, const Bstream& mkid)
{
	cdp_rec rec;

	rec.action = CDP_REC_GET;
	rec.ci.nsid = nsid;
	rec.ci.mkid = mkid;

	recs.append(rec);
}

void cdp::cdp_put(const CertInfo& c)
{
	cdp_rec rec;

	rec.action = CDP_REC_PUT;
	rec.ci = c;

	recs.append(rec);
}

void cdp::cdp_putok(u_char nsid, const Bstream& mkid)
{
	cdp_rec rec;

	rec.action = CDP_REC_PUTOK;
	rec.ci.nsid = nsid;
	rec.ci.mkid = mkid;

	recs.append(rec);
}

void cdp::cdp_putfail(u_char nsid, const Bstream& mkid)
{
	cdp_rec rec;

	rec.action = CDP_REC_PUTFAIL;
	rec.ci.nsid = nsid;
	rec.ci.mkid = mkid;

	recs.append(rec);
}

Bstream cdp_rec_encode(const cdp_rec* rec)
{
	Bstream nullstr;
	Bstream ret;
	u_char res[3];
	int name_len = rec->ci.mkid.getlength() + 1;

	if (name_len > 255)
		return nullstr;		// failure XXX

// encode cdp_rec action

	Bstream::putbyte(ret, rec->action);

// encode CDP_NAME_SKIP + name length + name (nsid + mkid)

	res[0] = CDP_NAME_SKIP;
	res[1] = name_len;
	res[2] = rec->ci.nsid;

	ret += Bstream(3, &res[0]);
	ret += rec->ci.mkid;

// encode cert type + cert length + cert

	if (rec->ci.cert == NULL) {
		Bstream::putbyte(ret, 0);	// type = 0
		Bstream::putshort(ret, 0);	// len = 0
	} else {
		Bstream::putbyte(ret, rec->ci.cert->CertType());

		Bstream cert_s = rec->ci.cert->encode();

		if (cert_s.getlength() > 65535)
			return nullstr;		// failure XXX

		Bstream::putshort(ret, cert_s.getlength());
		ret += cert_s;
	}

	return ret;
}

int cdp_rec_decode(Bstream &str, cdp_rec* rec)
{
	u_char action;
	u_char type;
	u_char name_len;
	u_char nsid;

// decode action + name type + name length + name

	if (str.getlength() < 4 ) 
		return FAILURE;

	str.fetchbyte(action);
	str.fetchbyte(type);
	str.fetchbyte(name_len);
	str.fetchbyte(nsid);

	if (type != CDP_NAME_SKIP)
		return FAILURE;

	rec->action = action;
	rec->ci.nsid = nsid;

	name_len--;		// nsid considered part of name
	if (name_len > str.getlength())
		return FAILURE;	

	rec->ci.mkid = Bstream(name_len, str.getdatap());
	str.consume(name_len);

// decode cert type + cert length + cert

	if (str.getlength() < 3)
		return FAILURE;

	u_char certtype;
	str.fetchbyte(certtype);

	u_short cert_len;
	(void) Bstream::getshort(str, (short*) &cert_len);

	if (cert_len == 0) {
		rec->ci.cert = NULL;
		return SUCCESS;
	}

	if (cert_len > str.getlength())
		return FAILURE;

	Bstream cert_s(cert_len, str.getdatap());
	str.consume(cert_len);

	if (SkipCert::decode(certtype, cert_s, rec->ci.cert))
		return FAILURE;

	return SUCCESS;
}

Bstream
cdp::encode()
{
	Bstream result;
	u_char res[4];

	res[0] = version;
	res[1] = action;
	res[2] = recs.length();
	res[3] = 0;			// reserved

	result = Bstream(4, &res[0]);

	Bstream::putint(result, icookie);
	Bstream::putint(result, rcookie);

	cdp_rec *p;

	for (p = recs.first(); p; p = recs.next())
		result += cdp_rec_encode(p);

	return result;
}

int
cdp::decode(Bstream &str)
{
	int i;

	if (str.getlength() < 4 )
		return FAILURE;

	str.fetchbyte(version);
	str.fetchbyte(action);
	str.fetchbyte(n_of_recs);
	str.fetchbyte(reserved);

	if (version != 1)
		return FAILURE;

	if (Bstream::getint(str, (int*) &icookie))
		return FAILURE;

	if (Bstream::getint(str, (int*) &rcookie))
		return FAILURE;

	for (i = 0; i < n_of_recs; i++) {
		cdp_rec rec;

		if (cdp_rec_decode(str, &rec))
			return FAILURE;

		recs.append(rec);
	}
	
	return SUCCESS;
}

void cdp_rec_print(cdp_rec* rec)
{

	printf("action: %d\n", rec->action);
	printf("nsid:   %d\n", rec->ci.nsid);
	printf("mkid:   ");
	rec->ci.mkid.print();

	if (rec->ci.cert == NULL)
		printf("Certificate NULL\n");
	else {
		printf("Certificate Type: %d\n", rec->ci.cert->CertType());
		rec->ci.cert->print();
	}
}

void
cdp::print()
{
	int i;

	printf("Version = %02d ", version);
	printf("Action = %02d ", action);
	printf("#recs: %d\n", recs.length());
	printf("I-Cookie: %ld R-Cookie: %ld\n", icookie, rcookie);

	cdp_rec *p;
	
	i = 0;
	for (p = recs.first(); p; p = recs.next()) {
		printf("----------\nRecord %d\n",i++);
		cdp_rec_print(p);
	}
}


char* cdp_rec::get_info_str()
{
	static char buf[512];
	char *action_s;

	switch (action) {
	case CDP_REC_GET:
		action_s = "get";
		break;

	case CDP_REC_PUT:
		action_s = "put";
		break;

	case CDP_REC_GETOK:
		action_s = "getok";
		break;

	case CDP_REC_GETFAIL:
		action_s = "getfail";
		break;

	case CDP_REC_PUTOK:
		action_s = "putok";
		break;

	case CDP_REC_PUTFAIL:
		action_s = "putfail";
		break;

	default:
		action_s = "unknown-request";
	}

	sprintf(buf, "action=%s %s", action_s, ci.get_info_str());

	return buf;
}

