/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_ctrl.c	1.28 95/11/15 Sun Microsystems"

#include <stdio.h>
#include <fcntl.h>
#include <stropts.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <string.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <arpa/inet.h>

#include <skip_types.h> 
#include <skip_proto.h>
#include <skip_ioctl.h>
#include <skip_lib.h>

#define	ERRSTRSZ	(256)

char			skip_errmsg[ERRSTRSZ];
extern int		errno;

/* skip_list_hosts()
 *
 * extract list of hosts currently allowed to connect using SKIP.
 *
 * Returns the number of hosts or -1 on error.
 */
int
skip_list_hosts(char *interface, void (*f)())
{
	skip_es_req_t		*reqp;
	int			skip_fd, nhosts, i, len;
	skip_param_t		*hostp;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}
	/*
	 * make a buffer big enough to retrieve all the hosts
	 */
	reqp = (skip_es_req_t *) malloc(sizeof (reqp->if_name) +
				SKIP_MAX_HOSTS * sizeof (skip_param_t));

	if (reqp == NULL) {
		sprintf(skip_errmsg, "%s: error - no more memory", SKIPNAME);
		_skip_close(skip_fd);
		errno = ENOMEM;
		return (-1);
	}
	strcpy(reqp->if_name, interface);

	len = sizeof (reqp->if_name) + SKIP_MAX_HOSTS * sizeof(skip_param_t);

	if ((len = _skip_ioctl(skip_fd, SKIP_ES_LIST_HOSTS, reqp, len)) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot determine "
				"current SKIP hosts", SKIPNAME, errno);
		_skip_close(skip_fd);
		free(reqp);
		return (-1);
	}

	nhosts = (len - sizeof(reqp->if_name))/sizeof (skip_param_t);
	hostp = (skip_param_t *) &reqp->hostinfo;

	for (i = 0; i < nhosts; i++) {
		(*f)(hostp);
		hostp++;
	}

	free(reqp);
	_skip_close(skip_fd);
	return (nhosts);
}

/*
 * allow a host to connect using SKIP
 */
int
skip_allow_host(char *interface, skip_param_t *params)
{
	skip_es_req_t		*reqp;
	skip_param_t		*hostinfo;
	struct hostent		*hp;
	unsigned long		addr;
	int			len, skip_fd;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	reqp = (skip_es_req_t *) malloc(sizeof (reqp->if_name) +
						sizeof (skip_param_t));
	if (reqp == NULL) {
		sprintf(skip_errmsg, "%s: error - no more memory", SKIPNAME);
		_skip_close(skip_fd);
		return (-1);
	}

	/*
	 * set the interface name and the host information
	 */
	strcpy(reqp->if_name, interface);
	hostinfo = (skip_param_t *) &reqp->hostinfo;

	*hostinfo = *params;

	len = sizeof (reqp->if_name) + sizeof (skip_param_t);
	
	if (_skip_ioctl(skip_fd, SKIP_ES_ADD_HOST, reqp, len) < 0) {
		if (errno == EEXIST) {
			sprintf(skip_errmsg, "%s: error - host %s exists", 
				SKIPNAME, inet_ntoa(params->ip_addr));
		} else {
			sprintf(skip_errmsg, "%s: internal error %d - cannot "
				"add SKIP host %s",
				SKIPNAME, errno, inet_ntoa(params->ip_addr));
		}
		free(reqp);
		_skip_close(skip_fd);
		return (-1);
	}
	free(reqp);
	_skip_close(skip_fd);
	return (0);
}

/*
 * disallow a host to connect using SKIP
 */
int
skip_disallow_host(char *interface, skip_param_t *params)
{
	skip_es_req_t		*reqp;
	skip_param_t		*hostinfo;
	int			len, skip_fd;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	reqp = (skip_es_req_t *) malloc(sizeof (reqp->if_name) +
						sizeof (skip_param_t));
	if (reqp == NULL) {
		sprintf(skip_errmsg, "%s: error - no more memory", SKIPNAME);
		_skip_close(skip_fd);
		return (-1);
	}

	/*
	 * set the interface name and the host information
	 */
	strcpy(reqp->if_name, interface);
	hostinfo = (skip_param_t *) &reqp->hostinfo;
	*hostinfo = *params;

	len = sizeof (reqp->if_name) + sizeof (skip_param_t);
	if (_skip_ioctl(skip_fd, SKIP_ES_DEL_HOST, reqp, len) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot remove "
		"SKIP host %s", SKIPNAME, errno, inet_ntoa(params->ip_addr));
		free(reqp);
		_skip_close(skip_fd);
		return (-1);
	}
	free(reqp);
	_skip_close(skip_fd);
	return (0);
}

/*
 * get a host's ACL details
 */
int
skip_get_host(char *interface, skip_param_t *params)
{
	skip_es_req_t		*reqp;
	skip_param_t		*hostinfo;
	int			len, skip_fd;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	reqp = (skip_es_req_t *) malloc(sizeof (reqp->if_name) +
						sizeof (skip_param_t));
	if (reqp == NULL) {
		sprintf(skip_errmsg, "%s: error - no more memory", SKIPNAME);
		_skip_close(skip_fd);
		return (-1);
	}

	/*
	 * set the interface name and the host information
	 */
	strcpy(reqp->if_name, interface);
	hostinfo = (skip_param_t *) &reqp->hostinfo;
	*hostinfo = *params;

	len = sizeof (reqp->if_name) + sizeof (skip_param_t);
	if (_skip_ioctl(skip_fd, SKIP_ES_GET_HOST, reqp, len) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - get details for "
		"host %s", SKIPNAME, errno, inet_ntoa(params->ip_addr));
		free(reqp);
		_skip_close(skip_fd);
		return (-1);
	}

	*params = *hostinfo;
	free(reqp);
	_skip_close(skip_fd);
	return (0);
}

/*
 * returns current access control mode
 */
int
skip_get_mode(char *interface)
{
	skip_es_mode_t		req;
	int			skip_fd;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	/*
	 * set the interface name and the mode information
	 */
	strcpy(req.if_name, interface);

	if (_skip_ioctl(skip_fd, SKIP_ES_GET_MODE, &req, sizeof(req)) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot "
				"get access control mode", SKIPNAME, errno);
		_skip_close(skip_fd);
		return (-1);
	}
	_skip_close(skip_fd);
	return (req.if_mode);
}

/*
 * turn access control on/off
 */
int
skip_set_mode(char *interface, unsigned char if_mode)
{
	skip_es_mode_t		req;
	int			skip_fd;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	/*
	 * set the interface name and the mode information
	 */
	strcpy(req.if_name, interface);
	req.if_mode = if_mode;

	if (_skip_ioctl(skip_fd, SKIP_ES_SET_MODE, &req, sizeof(req)) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot "
				"modify access control", SKIPNAME, errno);
		_skip_close(skip_fd);
		return (-1);
	}
	if (if_mode == SkipInteractive) {
		return (skip_fd);
	}
	_skip_close(skip_fd);
	return (0);
}

int
skip_access_ctrl_on(char *interface)
{
	return (skip_set_mode(interface, SkipAccessControlOn));
}

int
skip_access_ctrl_off(char *interface)
{
	return (skip_set_mode(interface, SkipAccessControlOff));
}
/* skip_get_message()
 *
 * Retrieve a message from the SKIP kernel
 *
 * Returns: -1 on error, message size on success
 */
int 
skip_get_message(int skip_fd, skip_es_notification_t *msgp)
{
#ifdef SYSV
	struct strbuf	ctl;
	int		rc, flags;

	ctl.maxlen = sizeof(*msgp);
	ctl.buf = (char *) msgp;
	flags = 0;

	rc =  getmsg(skip_fd, &ctl, NULL, &flags);
	return (ctl.len);
#else
	return(read(skip_fd, (char *) msgp, sizeof (*msgp)));
#endif
}


/*
 * skip_get_if_stats()
 *
 * Get interface statistics
 */
int 
skip_get_if_stats(char *interface, ioctl_if_stat_t *p_stats)
{

	int			skip_fd;
	skip_ctrl_stats_t 	if_stats;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	memset((char *)&if_stats, 0, sizeof(skip_ctrl_stats_t));

	/*
	 * set the interface name 
	 */
	strcpy(if_stats.if_name, interface);

	if (_skip_ioctl(skip_fd, SKIP_ES_GET_IF_STATS, 
				&if_stats, sizeof(skip_ctrl_stats_t)) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot "
				"get SKIP interface stats", SKIPNAME, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	*p_stats = if_stats.stats.if_stats;

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_get_key_stats()
 *
 * Get key statistics
 */
int 
skip_get_key_stats(char *interface, ioctl_key_stat_t *p_stats)
{

	int			skip_fd;
	skip_ctrl_stats_t 	key_stats;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	memset((char *)&key_stats, 0, sizeof(skip_ctrl_stats_t));
	/*
	 * set the interface name 
	 */
	strcpy(key_stats.if_name, interface);

	if (_skip_ioctl(skip_fd, SKIP_ES_GET_KEY_STATS, 
				&key_stats, sizeof(skip_ctrl_stats_t)) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot "
				"get SKIP interface stats", SKIPNAME, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	*p_stats = key_stats.stats.key_stats;

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_get_hdr_stats()
 *
 * Get header statistics
 */
int 
skip_get_hdr_stats(char *interface, ioctl_hdr_stat_t *p_stats)
{

	int			skip_fd;
	skip_ctrl_stats_t 	hdr_stats;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	memset((char *)&hdr_stats, 0, sizeof(skip_ctrl_stats_t));
	/*
	 * set the interface name 
	 */
	strcpy(hdr_stats.if_name, interface);

	if (_skip_ioctl(skip_fd, SKIP_ES_GET_HDR_STATS, 
				&hdr_stats, sizeof(skip_ctrl_stats_t)) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot "
				"get SKIP interface stats", SKIPNAME, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	*p_stats = hdr_stats.stats.hdr_stats;

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_get_kij_list()
 *
 * Get installed crypto modules list
 */
int 
skip_get_kij_list(char *interface, ioctl_alg_list_t *kij_list)
{

	int			skip_fd;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	if (_skip_ioctl(skip_fd, SKIP_ES_GET_KIJ_ALGS, 
				kij_list, sizeof(ioctl_alg_list_t)) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot "
				"get Kij algs list", SKIPNAME, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}
/*
 * skip_get_crypt_list()
 *
 * Get installed crypto modules list
 */
int 
skip_get_crypt_list(char *interface, ioctl_alg_list_t *p_crypt)
{

	int			skip_fd;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	if (_skip_ioctl(skip_fd, SKIP_ES_GET_CRYPTORS, 
				p_crypt, sizeof(ioctl_alg_list_t)) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot "
				"get crypto modules list", SKIPNAME, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_get_crypt_stat()
 *
 * Get crypto modules statistics
 */
int 
skip_get_crypt_stats(char *interface, ioctl_crypt_stat_t *p_stats)
{

	int			skip_fd;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	if (_skip_ioctl(skip_fd, SKIP_ES_GET_CPT_STATS, 
				p_stats, sizeof(ioctl_crypt_stat_t)) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot "
			"get crypto modules statistics", SKIPNAME, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_var_get()
 *
 * Get named variable value
 */
int 
skip_var_get(char *interface, char  *name, int *value)
{

	int			skip_fd;
	ioctl_skip_sym_t	p_symb = { 0 };

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	/*
	 * Fill in ioctl 
	 */
	p_symb.var_lg = strlen(name);
	strcpy(p_symb.var_name, name);

	if (_skip_ioctl(skip_fd, SKIP_ES_GET_VAR, 
				&p_symb, sizeof(ioctl_skip_sym_t)) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot "
			"get variable %s value", SKIPNAME, errno, name);
		_skip_close(skip_fd);
		return (-1);
	}

	*value = p_symb.var_value;

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_var_set()
 *
 * Set named variable value
 */
int 
skip_var_set(char *interface, char  *name, int value)
{

	int			skip_fd;
	ioctl_skip_sym_t	p_symb = { 0 };

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	/*
	 * Fill in ioctl 
	 */
	p_symb.var_value = value;
	p_symb.var_lg = strlen(name);
	strcpy(p_symb.var_name, name);

	if (_skip_ioctl(skip_fd, SKIP_ES_SET_VAR, 
				&p_symb, sizeof(ioctl_skip_sym_t)) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot "
			"set variable %s value", SKIPNAME, errno, name);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}

/*
 * skip_get_key_status()
 *
 * Get crypt/decrypt keys status
 */
int 
skip_get_key_status(char *interface, ioctl_key_status_t *p_status)
{

	int			skip_fd;

	skip_fd = _skip_open(interface);

	if (skip_fd < 0) {
		return (skip_fd);
	}

	if (_skip_ioctl(skip_fd, SKIP_ES_KEY_STATUS, 
				p_status, sizeof(ioctl_key_status_t)) < 0) {
		sprintf(skip_errmsg, "%s: internal error %d - cannot "
			"get keys status", SKIPNAME, errno);
		_skip_close(skip_fd);
		return (-1);
	}

	_skip_close(skip_fd);
	return (0);
}
