/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_es_bypass.c	1.22 96/10/01 Sun Microsystems"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_proto.h>
#include <skip_types.h>
#include <skip_ioctl.h>
#include <skip_acl.h>
#include <skip_keymgrmsgs.h>
#include <skip_key.h>
#include <skip_es.h>

#define	SKIP_ES_FRAGCACHE_SZ	16	/* # of fragment details to cache */

/*
 * CDP server port for bypass communications
 */
extern int		skip_server_port;
extern int		skip_check_cdp_addr;
static struct ip	skip_es_fragcache[SKIP_ES_FRAGCACHE_SZ];

/*
 * global information
 */

int			skip_es_udp_debug = 0;
static int		skip_es_udp_inited = 0;
static int		skip_es_cdp_max_responses = 1;

/*
 * hash table declarations
 */
#define	UDP_MAX_HASH_ENTRIES	(32)
#define	UDP_HASH_TABLESZ	(16)

/*
 * structure describing a hash table entry
 */
typedef struct skip_es_udp_t {
	struct skip_es_udp_t	*next;		/* hash collision entries */
	struct skip_es_udp_t	*prev;
	unsigned int		hash;		/* hash table position */
	struct in_addr		src;		/* packet information... */
	struct in_addr		dst;
	short			src_port;
	short			dst_port;
	unsigned long		tries;		/* # of requests sent */
	unsigned int		responses;	/* req/rsp # */
	int			ttl;		/* current time-to-live */
	int			max_ttl;	/* max time-to-live */
	long			then;		/* time keeping */
} skip_es_udp_t;

/*
 * function prototypes - static functions
 */
static unsigned int		skip_es_bypass_hash(short, short);

/*
 * hash table information
 */
static skip_es_udp_t	**skip_es_bypass_hashtbl;
static const int	skip_es_bypass_hashtblsz = UDP_HASH_TABLESZ;
static int		skip_es_bypass_ttl = 20;/* entry lifetime (sec) */

/*
 * statistics
 */
static unsigned long	skip_es_udp_num_hash_entries	= 0;
static unsigned long	skip_es_udp_hash_lookups	= 0;
static unsigned long	skip_es_udp_hash_collisions	= 0;
static unsigned long	skip_es_udp_good_requests	= 0;
static unsigned long	skip_es_udp_good_responses	= 0;
static unsigned long	skip_es_udp_bad_requests	= 0;
static unsigned long	skip_es_udp_bad_responses	= 0;
static unsigned long	skip_es_udp_too_many_requests	= 0;
static unsigned long	skip_es_udp_hashtablesz		= UDP_HASH_TABLESZ;
static unsigned long	skip_es_udp_max_entries		= UDP_MAX_HASH_ENTRIES;

/* skip_es_bypass_init()
 *
 * Initialise bypass channel
 * 
 * Returns: none
 */
int 
skip_es_bypass_init()
{
	if (!skip_es_udp_inited) {
		skip_es_udp_inited = 1;
		BZERO((char *) skip_es_fragcache, sizeof(skip_es_fragcache));
		skip_es_bypass_hashtbl = (skip_es_udp_t **)
				SYSMEM_ALLOC(sizeof(skip_es_udp_t *) *
					skip_es_udp_hashtablesz);
		if (skip_es_bypass_hashtbl == NULL) {
			skip_es_udp_inited = 0;
			return(ENOMEM);
		}
		BZERO((char *) skip_es_bypass_hashtbl, 
			sizeof(skip_es_udp_t *) * skip_es_udp_hashtablesz);
	}
	return(0);
}

/* skip_es_bypass_init()
 *
 * Initialise bypass channel
 * 
 * Returns: none
 */
void
skip_es_bypass_uninit()
{
	register int		h;
	register skip_es_udp_t	*sp, *nsp;

	if (skip_es_udp_inited) {
		for (h = 0; h < skip_es_udp_hashtablesz; h++) {
			for (sp = skip_es_bypass_hashtbl[h]; sp;
								sp = nsp) {
				nsp = sp->next;
				SYSMEM_FREE(sp, sizeof(*sp));
			}
		}
		SYSMEM_FREE(skip_es_bypass_hashtbl,
			sizeof(skip_es_udp_t *) * skip_es_udp_hashtablesz);
		skip_es_bypass_hashtbl = NULL;
		skip_es_udp_inited = 0;
	}
}

/* skip_es_fragcache_add()
 *
 * Maintain a circular buffer of fragmentation information.
 * 
 * Returns: none
 */
static void
skip_es_fragcache_add(struct ip *ip)
{
	static int	writer = 0;

	skip_es_fragcache[writer++] = *ip;

	if (writer == SKIP_ES_FRAGCACHE_SZ) {
		writer = 0;
	}
}

/* skip_es_in_fragcache()
 *
 * Returns: B_TRUE if we passed the first fragment, B_FALSE otherwise
 */
static boolean_t
skip_es_in_fragcache(struct ip *ip)
{
	static int		reader = 0;
	register struct ip	*cip;
	register int		r;
	boolean_t		found = B_FALSE;

	r = reader;

	do {
		cip = &skip_es_fragcache[r];

		if ((ip->ip_id == cip->ip_id) &&
			(ip->ip_src.s_addr == cip->ip_src.s_addr) &&
			(ip->ip_dst.s_addr == cip->ip_dst.s_addr)) {

			/*
			 * first fragment is present in cache
			 */

			reader = r;
			found = B_TRUE;
			break;
		}
		r++;
		if (r == SKIP_ES_FRAGCACHE_SZ) {
			r = 0;
		}
	} while (r != reader);
	return (found);
}


/* skip_es_bypass_hash()
 *
 * Compute a hash for the packet
 *
 * Returns: an index (0 .. skip_es_udp_hashtablesz-1) where
 * skip_es_udp_hashtablesz is an even power of 2
 */
static unsigned int
skip_es_bypass_hash(short src_port, short dst_port)
{
	register unsigned char		h;
	
	/*
	 * assumes skip_es_udp_hashtablesz of 256 entries
	 */
	h = src_port ^ dst_port;
	return(h & (UDP_HASH_TABLESZ - 1));
}

/* udp_hash_locate()
 *
 * Find (and optionally create) the hash table entry for the packet
 *
 * Returns: 0 if found or error code 
 */
static skip_es_udp_t *
udp_hash_locate(struct ip *ip, short src_port, short dst_port,
							boolean_t request)
{
	register int			h;
	skip_es_udp_t 			*sp;
	boolean_t			src_addr_ok = B_TRUE;

	skip_es_udp_hash_lookups++;

	h = skip_es_bypass_hash(src_port, dst_port);
	sp = skip_es_bypass_hashtbl[h];

	/*
	 * check collision entries...
	 */

	if (request) {
		while (sp) {
			if ((ip->ip_src.s_addr == sp->src.s_addr) &&
			    (ip->ip_dst.s_addr == sp->dst.s_addr) &&
			    (src_port == sp->src_port) &&
			    (dst_port == sp->dst_port)) {
				break;	/* Found it */
			}
			sp = sp->next;
		}
	} else {
		while (sp) {
			if (skip_check_cdp_addr) {
				/*
				 * enable checking of source IP addresses on
				 * CDP responses. This can cause problems if
				 * the reply comes from a different address
				 * than the request (as can happen with a
				 * multihomed machine).
				 */
				src_addr_ok =
					(ip->ip_src.s_addr == sp->dst.s_addr);
			}
				
			if ((ip->ip_dst.s_addr == sp->src.s_addr) &&
			    src_addr_ok &&
			    (dst_port == sp->src_port) &&
			    (src_port == sp->dst_port)) {
				break;	/* Found it */
			}
			sp = sp->next;
		}
	}

	if (sp)
		return(sp);

	if (request) {

		if (skip_es_udp_num_hash_entries == skip_es_udp_max_entries) {
			skip_es_udp_too_many_requests++;
			return(NULL);
		}
		sp = (skip_es_udp_t *) SYSMEM_ALLOC(sizeof(*sp));
		if (sp == NULL) {
			return(NULL);
		}
		skip_es_udp_num_hash_entries++;

		/*
		 * initialise the hash entry
		 */
		sp->next = sp->prev = NULL;
		sp->ttl = sp->max_ttl = skip_es_bypass_ttl;
		sp->tries = sp->responses = 0;
		sp->then = SKIP_ES_CURRENTTIME;
		sp->src.s_addr = ip->ip_src.s_addr;
		sp->dst.s_addr = ip->ip_dst.s_addr;
		sp->src_port = src_port;
		sp->dst_port = dst_port;
		sp->hash = h;


		if (skip_es_bypass_hashtbl[h] == NULL) {
			skip_es_bypass_hashtbl[h] = sp;
		} else {
			/*
			 * hash collision found - insert in collision chain...
			 */
			skip_es_udp_hash_collisions++;
			sp->next = skip_es_bypass_hashtbl[h];
			skip_es_bypass_hashtbl[h]->prev = sp; 
			skip_es_bypass_hashtbl[h] = sp; 
		}
	}
	return(sp);
}

/* skip_es_bypass_hash_remove()
 *
 * Returns: remove a hash table entry
 */
static void
skip_es_bypass_hash_remove(skip_es_udp_t * sp)
{
	if (sp->next != NULL) {
		sp->next->prev = sp->prev;
	}
	if (sp->prev != NULL) {
		sp->prev->next = sp->next;
	} else {
		skip_es_bypass_hashtbl[sp->hash] = sp->next;
	}
	SYSMEM_FREE(sp, sizeof(*sp));
	skip_es_udp_num_hash_entries--;
}

/* skip_es_time_has_passed()
 *
 * Returns B_TRUE once per second
 *
 * Returns: none.
 */
boolean_t
skip_es_time_has_passed()
{
	register boolean_t	time_has_passed;
	static long		now = 0;

	if ((time_has_passed = SKIP_ES_DELTATIME(now)) > 0) {
		now = SKIP_ES_CURRENTTIME;
	}
	return (time_has_passed);
}

/* skip_es_bypass_maintainance()
 *
 * Perform maintenance on the hash table
 *
 * Returns: none.
 */
static void
skip_es_bypass_maintainance()
{
	register skip_es_udp_t		*sp, *nsp;
	register int			h;

	for (h = 0; h < skip_es_udp_hashtablesz; h++) {
		sp = skip_es_bypass_hashtbl[h];
		while (sp) {
			nsp = sp->next;
			sp->ttl -= SKIP_ES_DELTATIME(sp->then);
			sp->then = SKIP_ES_CURRENTTIME;
			if (sp->ttl <= 0) {
				skip_es_bypass_hash_remove(sp);
			}
			sp = nsp;
		}
	}
}

/* skip_es_udp_req()
 *
 * Process a generic UDP request packet.
 *
 * Returns: 0 if all went well, otherwise -1
 */
static int
skip_es_udp_req(struct ip *ip, short src_port, short dst_port,
	unsigned int max_responses, unsigned int max_time_limit)
{
	skip_es_udp_t		*entry;

	if (max_responses == 0) {
		skip_es_udp_bad_requests++;
		return (EIO);
	}

	if (skip_es_time_has_passed()) {
		skip_es_bypass_maintainance();
	}

	/*
	 * look up request packet in the hash table, creating entry if
	 * not there...
	 */
	if ((entry = udp_hash_locate(ip, src_port, dst_port, B_TRUE))
								== NULL) {
		skip_es_udp_bad_requests++;
		return (-1);
	}

	/*
	 * mark as alive...
	 */
	entry->ttl = max_time_limit ? max_time_limit : entry->max_ttl;
	entry->tries++;
	entry->responses = max_responses;

	/*
	 * could implement a limit to the number of retrans here...
	 */
	skip_es_udp_good_requests++;
	return (0);
}

/* skip_es_udp_res()
 *
 * Process a generic UDP response packet.
 *
 * Returns: 0 if valid, otherwise  -1
 */
static int
skip_es_udp_res(struct ip *ip, short src_port, short dst_port)
{
	skip_es_udp_t		*entry;

	if (skip_es_time_has_passed()) {
		skip_es_bypass_maintainance();
	}

	/*
	 * try to find a corresponding request...
	 */
	if ((entry = udp_hash_locate(ip, src_port, dst_port, B_FALSE))
								== NULL) {
		/*
		 * no request was found... bad
		 */
		skip_es_udp_bad_responses++;
		return (-1);
	}
	/*
	 * corresponding udp_bypass request found, check the response counter
	 */
	entry->responses--;

	if (entry->responses == 0) {
		skip_es_bypass_hash_remove(entry);
	}
	skip_es_udp_good_responses++;
	return (0);
}

/* skip_es_bypass()
 * 
 * Returns: B_TRUE if message can bypass security, B_FALSE otherwise
 */
boolean_t
skip_es_bypass(struct ip *ip, int size)
{
	register int		hdrsz;
	struct udphdr		*udphdr; 

	if (skip_server_port == 0) {
		return (B_FALSE);
	}

	if (ip->ip_off & htons(IP_OFFSET)) {
		/*
		 * not first fragment - check to see if we passed the 1st frag
		 */
		return (skip_es_in_fragcache(ip));
	}

	hdrsz = IPHLEN(ip);

	/*
	 * check IP message size is valid: ip header + udp header
	 */
	if (size < (int) (hdrsz + sizeof (struct udphdr))) {
		return (B_FALSE);
	}

	udphdr = (struct udphdr *) ((char *) ip + hdrsz);

	/*
	 * check for a CDP request
	 */
	if ((udphdr->uh_dport == skip_server_port) &&
		(skip_es_udp_req(ip,
				udphdr->uh_sport,
				udphdr->uh_dport,
				skip_es_cdp_max_responses,
				0) == 0)) {

		/*
		 * allow message, creating fragment cache entry if a fragment
		 */
		if (ip->ip_off & htons(IP_MF)) {
			skip_es_fragcache_add(ip);
		}
		return (B_TRUE);
	}

	/*
	 * check for a CDP response
	 */
	if ((udphdr->uh_sport == skip_server_port) &&
		(skip_es_udp_res(ip,
			udphdr->uh_sport,
			udphdr->uh_dport) == 0)) {

		/*
		 * allow message, creating fragment cache entry if a fragment
		 */
		if (ip->ip_off & htons(IP_MF)) {
			skip_es_fragcache_add(ip);
		}
		return (B_TRUE);
	}

	/*
	 * don't allow bypass
	 */
	return (B_FALSE);
}
