/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/
#pragma ident "@(#)skiplocal.C	1.46 96/10/22"

#include <skip_os.h>

#include <skip_conf.h>

#include "AuthPolicy.h"
#include "X509skip.h"
#include "HashCert.h"
#include "SlotParse.h"
#include "CertInfo.h"
#include "SkipCert.h"
#include "LL.h"
#include "db.h"

#include "Time.h"
#include "Bigint.h"
#include "Bstream.h"
#include "ObjId.h"
#include "X509Cert.h"
#include "HashCert.h"
#include "Passwd.h"
#include "Sig.h"
#include "bn.h"
#include "prime.h"
#include "bn_glue.h"
#include "bnprint.h"
#include "dsa.h"
#include "utils.h"
#include "skip_keymgr.h"
#include "dh_512_params.h"
#include "dh_1024_params.h"
#include "dh_2048_params.h"

extern void randseed_set_skip_paths(char *, char *);


void init_cryptolib();

#define LOCALID_PATH SKIP_DIR"/localid"
AuthPolicy auth;
SlotParse skip_conf;

Bigint bn_exponentiate(const Bigint &, const Bigint &, const Bigint &);

void
usage(char *s){
	fprintf(stderr, "%s [add|export|extract|list|keygen|rm"
		, s);
	exit(1);
}

void
extract_usage(char *s)
{
	fprintf(stderr, "%s extract <slot>\n", s);
	exit(1);
}

void
add_usage(char *s)
{
	fprintf(stderr, "%s add -o slot-type -t cert-type -n nsid -c cert-file"
		       " -s secret-file\n", s);
	exit(1);
}

void
del_usage(char *s)
{
	fprintf(stderr, "%s rm slot\n",s);
	exit(1);
}

void
init_usage(char *s)
{
	fprintf(stderr,"%s init [-qo]\n",s);
	exit(1);
}


void
skip_list_id(int nargc, char *nargv[])
{
	char *slot;
	struct dirent *e;
	DIR *d;
	char buf[1024];
	int c;
	int verbose = 0, machprt = 0;

	while ((c = getopt(nargc-1, &nargv[1],"vV")) != EOF)
		switch (c) {
			case 'v':
				verbose = 1;
				break;
			case 'V':
				machprt = 1;
				break;
			default:
				fprintf(stderr, "Invalid flag %c\n",c);
				break;
		}
	optind++;
        if (optind >= nargc) {
		slot = NULL;
	} else {
		slot = nargv[optind];
	}

	d=opendir(LOCALID_PATH);

	if (d == NULL) {
		fprintf(stderr, "can't open %s", LOCALID_PATH);
		perror("");
		exit(1);
	}

//  Read rc files from the /etc/skip/localid directory

	while ((e = readdir(d))) {
		int len = strlen(e->d_name);
		if (len < 4 || strcmp(e->d_name + len - 3, ".rc") != 0)
			continue;

		SlotParse s;

		sprintf(buf, "%s/%s", LOCALID_PATH, e->d_name);
		if (!s.load(buf))
			continue;

		strncpy(buf, e->d_name, len-3);
		buf[len-3] = '\0';

		if (slot && strcmp(buf, slot) != 0)
			continue;

		if (machprt) 
			printf("name=%s ", buf); 
		else
			printf("Local ID Slot Name: %s", buf);

		s.print(verbose,machprt);
		if (machprt & verbose)
			printf("%%%%%%%%%%");
		printf("\n");

	}

	closedir(d);
}

void
skip_init_id(int argc, char **argv)
{
	struct stat s;
	int overwrite = 0;
	int quiet = 0;
	int c;

	while ((c = getopt(argc-1, &argv[1],"qo")) != EOF)
		switch(c){
			case 'q':
				quiet = 1;
				break;
			case 'o':
				overwrite = 1;
				break;
			default:
				init_usage(argv[0]);
		}
	

	if (stat(LOCALID_PATH, &s)) {
		if (!mkdir(LOCALID_PATH, 0700))
			exit(0);
		else {
			if (!quiet)
				fprintf(stderr,"Couldn't create Directory\n");
			exit(1);
		}
	}
	if (overwrite) {
		char buf[512];
		int ret;
		sprintf(buf,"rm -rf %s", LOCALID_PATH);
		ret = system(buf);
		if (ret) {
			fprintf(stderr,"Unable to Clean old directory %s \n",
					LOCALID_PATH);
			exit(2);
		}
		if (!mkdir(LOCALID_PATH, 0700))
			exit(0);
		else {
			fprintf(stderr,"Unable to Created New Direcotry %s\n",
                                        LOCALID_PATH);
			exit(2);
		}
	} else {
		if (!quiet) 
			fprintf(stderr, "Directory Already exists\n");
		exit(0);
        }
}

void
skip_del_id(int argc, char **argv)
{
	char buf[1024];
	struct stat s;

	
	if (argc <3)
		del_usage(argv[0]);

	argv += 2;
	argc -= 2;
	
	
	sprintf(buf,"%s/%s.rc",LOCALID_PATH, *argv);

	if (stat(buf, &s)) {
		fprintf(stderr,"Local ID slot %s does not exist\n",*argv);
		exit(1);
	}	

	sprintf(buf, "rm %s/%s.*",LOCALID_PATH, *argv);
	if (system(buf)) {
		fprintf(stderr,"Couldn't delete Local ID slot %s\n", *argv);
		exit(1);
	}
	fprintf(stderr,"Successfully deleted LocalId slot %s\n", *argv);
	exit(0);
}


void
skip_extract_id(int argc, char **argv)
{
	char buf[1024];
	Bstream certstr;

	if (argc <3)
		extract_usage(argv[0]);

	argv += 2;
	argc -= 2;
	sprintf(buf,"%s/%s.x509",LOCALID_PATH, *argv);

	certstr = File_to_Bstr(buf);
	if (certstr.getlength() == 0) {
		sprintf(buf, "%s/%s.dhpub",LOCALID_PATH, *argv);
		certstr = File_to_Bstr(buf);
	}
	if (certstr.getlength() == 0) {
		fprintf(stderr,"localid not found or permission denied\n");
		exit(1);
	}
	fwrite(certstr.getdatap(), certstr.getlength(), 1, stdout);
	exit(0);
}


// Return true if two certificates have the same identity (nsid, mkid)
// and overlapping validity periods

int certinfo_match(CertInfo*s, CertInfo *t)
{
    u_long s_from, s_to, t_from, t_to;
    if (s->nsid != t->nsid )
		return 0;

    if( s->mkid != t->mkid )
		return 0;

    s_from = s->cert->skip_notvalidbefore();
    s_to   = s->cert->skip_notvalidafter();
    t_from = t->cert->skip_notvalidbefore();
    t_to   = t->cert->skip_notvalidafter();

    if ((t_from > s_to) || (t_to < s_from))
	return 0;
    return 1;

}

int check_dup_localid(CertInfo& ci)
{
	char buf[1024];
	DIR *d;
	struct dirent *e;
	int ret = 0;

	d = opendir(LOCALID_PATH);

	if (d == NULL)
		return 0;

	while ((e = readdir(d))!=NULL) {
		int len = strlen(e->d_name);

		if (len < 3)
			continue;
		if (strcmp(e->d_name + len - 3, ".rc") != 0)
			continue;

		SlotParse s;
		sprintf(buf, "%s/%s", LOCALID_PATH, e->d_name);

		if (!s.load(buf))
			continue;

		CertInfo d(s.nsid, s.mkid, s.cert);

		if (certinfo_match(&ci, &d)) {
			ret = 1;
			break;
		}
	}
	closedir(d);

	return ret;
}

void check_math(SkipCert *cert, Bstream& secstr)
{
	Bigint g,  p;
	Bigint i(secstr);
	
	cert->skip_params(g, p);
	int n = p.bits();


	if (bn_exponentiate(g, i, p) != cert->skip_pubkey()) {
		fprintf(stderr, "Error: secret does not produce public key\n");
		exit(1);
	}
}

int add_slot(char *slottype, CertInfo& ci, Bstream& secstr, int machprt)
{
	char buf[1024];
	FILE *fp;

	if (ci.cert == NULL) {
		fprintf(stderr, "Error: no certificate\n");
		exit(1);
	}

	char *res = ci.check_valid(auth);
	if (res != NULL)
		fprintf(stderr, "Warning: certificate did not validate - %s\n",
					res);

	if (check_dup_localid(ci)) {
		fprintf(stderr, "Duplicate local identity -- not added\n");
		exit(1);
	}

	if (strcmp(slottype, "soft") == 0)  // verify that g^i mod p == pubkey 
		check_math(ci.cert, secstr);

	// Now actually create the slot

	int filecount = 0;
	do {
		sprintf(buf,"%s/%d.rc", LOCALID_PATH, filecount);
		fp=fopen(buf,"r");
		if (fp == NULL) {
			fp = fopen(buf,"w");
			if (fp == NULL) {
				fprintf(stderr, "can't write %s", buf);
				perror("");
				exit(1);
			}

			fprintf(fp, "nsid %d\n", ci.nsid);
			fprintf(fp, "type %s\n", slottype);
			fclose(fp);
		
			

			if (secstr.getlength() != 0) {
				sprintf(buf,"%s/%d.secret", 
						LOCALID_PATH, filecount);
				secstr.store(buf);
			} else {
				unlink(buf);
				fprintf(stderr, 
					"An error has occured - Exiting\n");
				exit(1);
			}

			sprintf(buf,"%s/%d.%s", LOCALID_PATH, filecount, 
				num_to_Certext(ci.cert->CertType()));

			Bstream certstr = ci.cert->encode();
			certstr.store(buf);

			return filecount;
		}
		fclose(fp);
		filecount++;
	} while(1);
}

void
skip_add_id(int nargc, char *nargv[])
{
	Bstream certstr;
	Bstream secstr;
	SkipCert *cert;
	char *slottype = "soft";
	u_char nsid =0;
	int certtype =0;
	char *secret_file = NULL;
	char *cert_file = NULL;
	int c;
	int seed = 0;
	int slot;

// skipid add -n NSID -o soft -t certtype -s secretfile -c certificate

	if (nargc < 3)
		add_usage(nargv[0]);
	while ((c = getopt(nargc-1, &nargv[1],"n:o:t:s:c:S:")) != EOF)
		switch(c) {
			case 't':
				certtype = Certname_to_num(optarg);
				break;

			case 's':
				secret_file = optarg;
				seed = 0;
				break;

			case 'c':
				cert_file = optarg ;
				break;

			case 'o':
				slottype = optarg;
				break;

			case 'n':
				nsid = atoi(optarg);	
				break;
			default:
				add_usage(nargv[0]);
		}
	
	if (strcmp(slottype,"soft") != 0)
			add_usage(nargv[0]);

	if (secret_file == NULL) 
		add_usage(nargv[0]);
	else  {
		secstr = File_to_Bstr(secret_file);
		if (secstr.getlength() ==0) {
			fprintf(stderr,"Invalid or missing secret\n");
			exit(1);
		}
	}

	if (cert_file == NULL)
		add_usage(nargv[0]);
	
	certstr = File_to_Bstr(cert_file);

	if (certstr.getlength()==0)  {
		fprintf(stderr, "Certificate file %s not found\n", cert_file);
		exit(1);
	}

	if (nsid == 0)
		add_usage(nargv[0]);
	
	if (certtype == 0)
		add_usage(nargv[0]);

	if (SkipCert::decode(certtype, certstr, cert)) {
		fprintf(stderr, "Certificate decode failure: %s\n", cert_file);
		exit(1);
	}

	CertInfo ci(nsid, cert->skip_name(), cert);

	slot = add_slot(slottype, ci, secstr, 0);
	printf("Added local identity slot %d\n", slot);
	exit(0);
}

void
keygen_usage(char *s)
{
	fprintf(stderr, "%s keygen [-m modulus-size] [-l lifetime] [-f] [-V]\n", s);
	exit(1);
}

void
skip_dh_keygen(int nargc, char *nargv[])
{
	int exp_size = 256;		// default exponent size
	int modulus_size ; 		// default mod size in bits
	int lifetime = 1826;		// default 5 years
	u_char nsid = 8;
	int machprt = 0;
	int certtype = SKIP_DH_PUBLIC;
	struct	BigNum sec, pub, gen, mod;
	Bigint	g, p, pubval;
	Bstream secstr;
	int c;
	int prompt = 1;
	int slot;

	randseed_set_skip_paths(SKIP_BIN, SKIP_DIR);

	if (!skip_conf.load(SKIP_CONF, SLOT_SKIP_CONF))
		skip_conf.clear();


	/* Always use large modulus for default */
	modulus_size = 2048;

	while ((c = getopt(nargc-1, &nargv[1],"m:n:t:fV")) != EOF)
		switch(c) {
			case 'f':
				prompt = 0;
				break;

			case 'm':
				modulus_size = atoi(optarg);
				break;

			case 'n':
				nsid = atoi(optarg);	
				break;

			case 't':
				certtype = Certname_to_num(optarg);
				break;

			case 'l':
				lifetime = atoi(optarg);
				break;

			case 'e':
				exp_size = atoi(optarg);
				break;

			case 'V':
				machprt = 1;
				break;

			default:
				keygen_usage(nargv[0]);
		}

	if (nsid != 8)
		keygen_usage(nargv[0]);
	if (certtype != SKIP_DH_PUBLIC)
		keygen_usage(nargv[0]);

	bnInit();
	bnBegin(&gen);
	bnBegin(&mod);
	bnBegin(&pub);
	bnBegin(&sec);



	switch (modulus_size) {
	case 512:
		g = Bigint(base_512, sizeof(base_512));
		p = Bigint(mod_512, sizeof(mod_512));
		break;

	case 1024:
		g = Bigint(base_1024, sizeof(base_1024));
		p = Bigint(mod_1024, sizeof(mod_1024));
		break;

	case 2048:
		g = Bigint(base_2048, sizeof(base_2048));
		p = Bigint(mod_2048, sizeof(mod_2048));
		break;

	default:
		fprintf(stderr, "unsupported modulus size\n");
		exit(1);
	}

	if (!machprt)
		printf("generating local secret with %d modulus size\n",
							modulus_size);

	if (prompt)
		prompt_user_for_randomness();

	// generate secret
	if (genRandBn(&sec, (unsigned)exp_size) < 0) {
		fprintf(stderr, "Error generating random number\n");
		exit(1);
	}

	BigNum_to_Bstream(&sec, &secstr);

	Bigint_to_BigNum(&g, &gen);
	Bigint_to_BigNum(&p, &mod);

	// calculate the public value
	if (bnExpMod(&pub, &gen, &sec, &mod) < 0) {
		fprintf(stderr, "Error calculating public value\n");
		exit(1);
	}
	BigNum_to_Bigint(&pub, &pubval);

	time_t now;

// Round the certificate lifetimes down to a day
// Make sure it starts at least an hour before the current time, to
// allow for N-counter out-of-sync clocks

	now = (NTPNOW - 3600) / 86400;
	now *= 86400;

				// # of days * hours * mins * secs
	unsigned long notafter = now + (lifetime * 24 * 60 * 60);

	HashCert hcert(g, p, pubval, now, notafter);

	SkipCert *cert = &hcert;

	CertInfo ci(nsid, cert->skip_name(), cert);


	slot = add_slot("soft", ci, secstr, machprt);
	if (machprt) {
		printf("slot=%d nsid=%d name=0x%s\n",
				slot, nsid, cert->skip_name().get_info_str());
	}
	else {
		cert->print();
		printf("Added local identity slot %d\n", slot);
	}

	bnEnd(&sec);
	bnEnd(&gen);
	bnEnd(&mod);
	bnEnd(&pub);
	exit(0);
}

void
export_usage(char *s)
{
	fprintf(stderr, "%s export [-s slot|-n nsid]\n", s);
	exit(1);
}

void
print_export(SlotParse& s)
{
	char host[256];
	char *t;
	char algs[256];

	if (t = getenv("SKIPLOCAL_EXPORT_HOST"))
		strcpy(host, t);
	else if (gethostname(host, 256) < 0)
		strcpy(host, "host");
	strcpy(algs, "-k des-ede-k3 -t des-cbc -m md5");

	if (t = getenv("SKIPLOCAL_EXPORT_ARGS"))
		strcpy(algs, t);

	printf("skiphost -a %s -R 0x%s -r %d -s %d %s\n",
		host, s.mkid.get_info_str(), s.nsid, s.nsid, algs);
}

void
skip_export_config(int nargc, char *nargv[])
{
	int c;
	int nsid = -1;
	int slot = -1;
	struct dirent *e;
	DIR *d;
	char buf[1024];
	int found = 0;

	while ((c = getopt(nargc-1, &nargv[1],"n:s:?")) != EOF)
		switch(c) {
			case 'n':
				nsid = atoi(optarg);
				break;

			case 's':
				slot = atoi(optarg);
				break;

			default:
				export_usage(nargv[0]);
		}

	d = opendir(LOCALID_PATH);

	if (d == NULL) {
		fprintf(stderr, "can't open %s", LOCALID_PATH);
		perror("");
		exit(1);
	}

//  Read rc files from the /etc/skip/localid directory

	while ((e = readdir(d))) {
		int len = strlen(e->d_name);
		if (len < 4 || strcmp(e->d_name + len - 3, ".rc") != 0)
			continue;

		SlotParse s;

		sprintf(buf, "%s/%s", LOCALID_PATH, e->d_name);
		if (!s.load(buf))
			continue;

		strncpy(buf, e->d_name, len-3);
		buf[len-3] = '\0';

		if (slot == atoi(buf) ||		// slot matches
		    nsid == s.nsid ||			// nsid matches
		    (slot < 0 && nsid < 0)) {		// take first slot
			print_export(s);
			found = 1;
			break;
		}
	}

	closedir(d);

	if (!found) {
		fprintf(stderr, "no matching slot found\n");
		exit(1);
	}

	exit(0);
}

main(int argc, char *argv[])
{

	init_cryptolib();

	auth.load(SKIP_DIR);

	if (argc < 2) 
		usage(argv[0]);
	if (strcmp(argv[1],"add")==0)
		skip_add_id(argc, argv);

	else if (strcmp(argv[1],"rm")==0 || strcmp(argv[1],"del")==0)
		skip_del_id(argc, argv);

	else if (strcmp(argv[1],"list")==0 || strcmp(argv[1],"ls")==0)
		skip_list_id(argc, argv);

	else if (strcmp(argv[1],"init")==0)
		skip_init_id(argc, argv);
	else if (strcmp(argv[1],"extract")==0)
		skip_extract_id(argc,argv);

	else if (strcmp(argv[1],"keygen")==0)
		skip_dh_keygen(argc, argv);
	else if (strcmp(argv[1],"export")==0)
		skip_export_config(argc, argv);
	else 
		usage(argv[0]);
	exit(1);
}
