/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/
 
#pragma ident "@(#)skiphost.c	1.54 96/10/04 Sun Microsystems"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_proto.h>
#include <skip_types.h>
#include <skip_ioctl.h>
#include <skip_acl.h>
#include <skip_lib.h>
#include <skip_msgs.h>

extern int	errno;

static char	*progname = NULL;
static char	*ifname = NULL;
static char	*hostname = NULL;
static char	*kij_alg = NULL;
static char	*kp_alg = NULL;
static char	*mac_alg = NULL;
static char	*comp_alg = NULL;
static char	*r_nsid = NULL;
static char	*s_nsid = NULL;
static char	*r_mkeyid = NULL;
static char	*s_mkeyid = NULL;
static char	*ip_mask = NULL;
static char	*skip_version = NULL;
static char	*skip_mode = NULL;
static char	*tunnel = NULL;

static int	opt, opt_cpt;
static int	opt_action, opt_more;
static int	opt_if, opt_unpb, opt_plumb;
static int	opt_add, opt_del, opt_exc;
static int	opt_r_nsid, opt_s_nsid;
static int	opt_r_keyid, opt_s_keyid;
static int	opt_kij, opt_crypt;
static int	opt_mac, opt_comp;
static int	opt_mode, opt_vers;
static int	opt_prt, opt_sh, opt_sel;
static int	opt_trs, opt_tunnel;
static int	opt_msk, opt_nomadic;

boolean_t       on_boot = B_FALSE;

#define		SKIP_HOST_MASK	"255.255.255.255"

static void	usage();

/*
 * List of all supported kij 
 */
static 
void kij_list()
{
	int	idx;

	printf("SKIP Kij algorithms (version 1):\n");

	for (idx = 0; idx <SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_kij_alg(idx, SKIP_V1)) {
			printf("\t%s\n", skip_kij_alg_to_name(idx, SKIP_V1));
		}
	}	
	printf("SKIP Kij algorithms:\n");

	for (idx = 0; idx <SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_kij_alg(idx, SKIP_V2)) {
			printf("\t%s\n", skip_kij_alg_to_name(idx, SKIP_V2));
		}
	}	

}

/*
 * List of all supported Kp 
 */
static 
void kp_list()
{
	int	idx;

	printf("SKIP Crypt algorithms (version 1):\n");

	for (idx = 0; idx <SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_kp_alg(idx, SKIP_V1)) {
			printf("\t%s\n", skip_kp_alg_to_name(idx, SKIP_V1));
		}
	}	
	printf("SKIP Crypt algorithms:\n");

	for (idx = 0; idx <SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_kp_alg(idx, SKIP_V2)) {
			printf("\t%s\n", skip_kp_alg_to_name(idx, SKIP_V2));
		}
	}	

}

/*
 * List of all supported MAC 
 */
static 
void mac_list()
{
	int	idx;

	printf("SKIP MAC algorithms:\n");

	for (idx = 0; idx <SKIP_MAXCRYPTORS; idx++) {
		if (skip_supported_mac_alg(idx, SKIP_V2)) {
			printf("\t%s\n", skip_mac_alg_to_name(idx, SKIP_V2));
		}
	}	
}

/*
 * List of all NSID  numbers
 */
static 
void nsid_list()
{
	int idx;
	
	printf("SKIP NSID list:\n");

	for (idx = 1; idx < skip_max_nsids; idx++) {
		printf("\t%d\t(%d bytes):\t%s\n",
			idx, skip_nsids[idx].len, 
			skip_nsids[idx].name ? skip_nsids[idx].name : UNKNOWN);
	}

}


#define MAXHOSTNAME	80
/*
 * Print an ACL entry with "skiphost" command format
 */
static void
skip_cmd_print(skip_param_t *params)
{
	char			mask[MAXHOSTNAME];
	char			keyid[MAXVARSZ*2];
	extern char		*skip_kij_algs[];
	extern char		*skip_kp_algs[];
	extern char		*skip_kij_algs_v1[];
	extern char		*skip_kp_algs_v1[];
	extern char		*skip_mac_algs[];

	if (skip_addr_to_host(&params->mask, mask) < 0) {
		printf("Get IP Mask failed!\n");
		return;
	}

	
	printf("skiphost -i %s -%c %s", ifname, 
		(params->flags & SKIP_EXCL ? 'x' : 'a'),
		(params->flags & SKIP_NOMADIC ? "\""SKIP_NOMAD"\"" :
			(params->ip_addr.s_addr == INADDR_ANY ? 
				SKIP_DEFAULT : inet_ntoa(params->ip_addr))));

	if ((params->mask.s_addr != INADDR_ANY) &&
			(params->mask.s_addr != INADDR_BROADCAST)) {

		printf(" -M %s", mask);
	}

	if (params->ip_addr.s_addr != params->tunnel_addr.s_addr) {
		printf(" -A %s", inet_ntoa(params->tunnel_addr));
	}

	switch (params->version) {

	case SKIP_NONE:
		break;

	case SKIP_V1:
		/*
		 * Version and algorithms
		 */
		printf(" -v 1 -k %s -t %s ",
			(skip_kij_algs_v1[params->kij_alg] ?
				skip_kij_algs_v1[params->kij_alg] : UNKNOWN),
			(skip_kp_algs_v1[params->kp_alg] ?
				skip_kp_algs_v1[params->kp_alg] : UNKNOWN));

		/*
		 * Remote Node ID
		 */
		if (params->r_nsid == SKIP_NSID_IPV4) {
			if (skip_keyid_to_s(&params->r_mkeyid,
						params->r_nsid, keyid) < 0) {
				printf("\t*** Error in key id\n");
			} else {
				printf("-R %s ", keyid);
			}
		} else {
			printf("*** : Invalid NSID/node id\n");
		}

		/*
		 * Local Key ID binding
		 */
		if (params->s_nsid && params->s_mkeyid.len) {
			if (skip_keyid_to_s(&params->s_mkeyid,
						params->s_nsid, keyid) < 0) {
				printf("\t*** Error in key id\n");
			} else {
				printf("-S %s ", keyid);
			}
		}
		break;

	case SKIP_V2 :
	case SKIP_RAW :

		printf(" -v %d ", params->version);

		if (params->kij_alg && (params->version == SKIP_V2)) {
			printf("-k %s ",
				skip_kij_algs[params->kij_alg] ?
				skip_kij_algs[params->kij_alg] : UNKNOWN);
		}

		/*
		 * Crypt alg...
		 */
		if (params->kp_alg) {
			printf("-t %s ",
				skip_kp_algs[params->kp_alg] ?
				skip_kp_algs[params->kp_alg] : UNKNOWN);
		}

		/*
		 *  MAC alg...
		 */
		if (params->mac_alg) {
			printf("-m %s ",
				skip_mac_algs[params->mac_alg] ?
				skip_mac_algs[params->mac_alg] : UNKNOWN);
		}

		/*
		 * Receiver Master Key Id...
		 */
		if (params->r_nsid) {
			printf("-r %d ", params->r_nsid);
		}

		if (params->r_nsid && params->r_mkeyid.len) {
			if (skip_keyid_to_s(&params->r_mkeyid,
						params->r_nsid, keyid) < 0) {
				printf("\t*** Error in key id\n");
			} else {
				printf("-R %s ", keyid);
			}
		}

		/*
		 * Sender Master Key Id...
		 */
		if (params->s_nsid) {
			printf("-s %d ", params->s_nsid);
		}

		if (params->s_nsid && params->s_mkeyid.len) {
			if (skip_keyid_to_s(&params->s_mkeyid,
						params->s_nsid, keyid) < 0) {
				printf("\t*** Error in key id\n");
			} else {
				printf("-S %s ", keyid);
			}
		}
		/*
		 * Transport/Tunneling mode...
		 */
		if (params->ip_mode) {
			printf("-T");
		}

		break;

	default:
		printf("%s:\t\tbad version %d\n",
			inet_ntoa(params->ip_addr), params->version);

	}

	printf("\n");

}

/*
 * Print an ACL entry with "name=value" format
 */
static void
skip_sh_print(skip_param_t *params)
{
	char			mask[MAXHOSTNAME];
	char			keyid[MAXVARSZ*2];
	extern char		*skip_kij_algs[];
	extern char		*skip_kp_algs[];
	extern char		*skip_kij_algs_v1[];
	extern char		*skip_kp_algs_v1[];
	extern char		*skip_mac_algs[];

	if (skip_addr_to_host(&params->mask, mask) < 0) {
		printf("Get IP Mask failed!\n");
		return;
	}

	
	printf("skiphost ifname=%s action=%s address=%s", ifname, 
		(params->flags & SKIP_EXCL ? "exclude" : "add"),
		(params->flags & SKIP_NOMADIC ? "\""SKIP_NOMAD"\"" :
			(params->ip_addr.s_addr == INADDR_ANY ? 
				SKIP_DEFAULT : inet_ntoa(params->ip_addr))));

	if ((params->mask.s_addr != INADDR_ANY) &&
			(params->mask.s_addr != INADDR_BROADCAST)) {

		printf(" mask=%s", mask);
	}

	if (params->ip_addr.s_addr != params->tunnel_addr.s_addr) {
		printf(" tunnel=%s", inet_ntoa(params->tunnel_addr));
	}

	switch (params->version) {

	case SKIP_NONE:
		break;

	case SKIP_V1:
		/*
		 * Version and algorithms
		 */
		printf(" version=1 kij_alg=%s kp_alg=%s ",
			(skip_kij_algs_v1[params->kij_alg] ?
				skip_kij_algs_v1[params->kij_alg] : UNKNOWN),
			(skip_kp_algs_v1[params->kp_alg] ?
				skip_kp_algs_v1[params->kp_alg] : UNKNOWN));

		/*
		 * Remote Node ID
		 */
		if (params->r_nsid == SKIP_NSID_IPV4) {
			if (skip_keyid_to_s(&params->r_mkeyid,
						params->r_nsid, keyid) < 0) {
				printf("\t*** Error in key id\n");
			} else {
				printf("r_mkeyid=%s ", keyid);
			}
		} else {
			printf("*** : Invalid NSID/node id\n");
		}

		/*
		 * Local Key ID binding
		 */
		if (params->s_nsid && params->s_mkeyid.len) {
			if (skip_keyid_to_s(&params->s_mkeyid,
						params->s_nsid, keyid) < 0) {
				printf("\t*** Error in key id\n");
			} else {
				printf("s_mkeyid=%s ", keyid);
			}
		}
		break;

	case SKIP_V2 :
	case SKIP_RAW :

		printf(" version=%d ", params->version);

		if (params->kij_alg && (params->version == SKIP_V2)) {
			printf("kij_alg=%s ",
				skip_kij_algs[params->kij_alg] ?
				skip_kij_algs[params->kij_alg] : UNKNOWN);
		}

		/*
		 * Crypt alg...
		 */
		if (params->kp_alg) {
			printf("kp_alg=%s ",
				skip_kp_algs[params->kp_alg] ?
				skip_kp_algs[params->kp_alg] : UNKNOWN);
		}

		/*
		 *  MAC alg...
		 */
		if (params->mac_alg) {
			printf("mac_alg=%s ",
				skip_mac_algs[params->mac_alg] ?
				skip_mac_algs[params->mac_alg] : UNKNOWN);
		}

		/*
		 * Receiver Master Key Id...
		 */
		if (params->r_nsid) {
			printf("r_nsid=%d ", params->r_nsid);
		}

		if (params->r_nsid && params->r_mkeyid.len) {
			if (skip_keyid_to_s(&params->r_mkeyid,
						params->r_nsid, keyid) < 0) {
				printf("\t*** Error in key id\n");
			} else {
				printf("r_mkeyid=%s ", keyid);
			}
		}

		/*
		 * Sender Master Key Id...
		 */
		if (params->s_nsid) {
			printf("s_nsid=%d ", params->s_nsid);
		}

		if (params->s_nsid && params->s_mkeyid.len) {
			if (skip_keyid_to_s(&params->s_mkeyid,
						params->s_nsid, keyid) < 0) {
				printf("\t*** Error in key id\n");
			} else {
				printf("s_mkeyid=%s ", keyid);
			}
		}
		/*
		 * Transport/Tunneling mode...
		 */
		if (params->ip_mode) {
			printf("transport=yes");
		}

		break;

	default:
		printf("%s:\t\tbad version %d\n",
			inet_ntoa(params->ip_addr), params->version);

	}

	printf("\n");

}
/*
 * Display current status for given interface
 */
static void
summary(char *ifname)
{
	int		rc;

	if (!ifname) {
		printf("summary: bad interface name %s!\n", ifname);
		return;
	}

	if (skip_get_mode(ifname) == SkipAccessControlOff) {
		printf("%s: access control disabled, any host can connect\n",
								ifname);
	} else {
		printf("%s: access control enabled, only authorized SKIP hosts "
						"can connect\n", ifname);
	}
	rc = skip_acl_list(ifname, (void (*)()) skip_host_print);
	if (rc < 0) {
		if (errno == ENODEV) {
			fprintf(stderr,
				"%s: error - %s: not in secure mode.\n",
				SKIPNAME, ifname);
		} else {
			fprintf(stderr, "%s.\n", skip_errmsg);
		}
	}

}

/*
 * Print a "skiphost" like file
 */
static void
acl_list_entries(char *ifname)
{
	int		rc;

	printf("skiphost -i %s -p\n", ifname);

	rc = skip_acl_list(ifname, (void (*)()) skip_cmd_print);
	if (rc < 0) {
		if (errno == ENODEV) {
			fprintf(stderr,
				"%s: error - %s: not in secure mode.\n",
				SKIPNAME, ifname);
		} else {
			fprintf(stderr, "%s.\n", skip_errmsg);
		}
		return;
	}

	if (skip_get_mode(ifname) == SkipAccessControlOff) {
		printf("skiphost -i %s -o off\n", ifname);
	} else {
		printf("skiphost -i %s -o on\n", ifname);
	}
}

/*
 * Print an ACL content in a "name=value" format 
 */
static void
acl_list_sh(char *ifname)
{
	int		rc;

	printf("skiphost ifname=%s action=plumb\n", ifname);

	rc = skip_acl_list(ifname, (void (*)()) skip_sh_print);
	if (rc < 0) {
		if (errno == ENODEV) {
			fprintf(stderr,
				"%s: error - %s: not in secure mode.\n",
				SKIPNAME, ifname);
		} else {
			fprintf(stderr, "%s.\n", skip_errmsg);
		}
		return;
	}

	if (skip_get_mode(ifname) == SkipAccessControlOff) {
		printf("skiphost ifname=%s action=off\n", ifname);
	} else {
		printf("skiphost ifname=%s action=on\n", ifname);
	}
}

/*
 * Plumb or un-plumb the SKIP driver for a give interface
 */
static int
interface_plumb(char *name, int secure)
{
	boolean_t	error = B_FALSE;
	char		*mode;

	mode = secure ? "secure" : "unsecure";

	printf("%s: saving interface state\n", name);
	if (skip_if_state(name, Save) < 0) {
		fprintf(stderr, "%s\n", skip_errmsg);
		return(1);
	}
	printf("%s: bringing interface down\n", name);
	if (skip_if_unplumb(name) < 0) {
		fprintf(stderr,
			"%s: error - %s: bringing interface down FAILED "
			"- unsecure mode\n", SKIPNAME, name);
		return(1);
	}

	printf("%s: starting %s mode\n", name, mode);

	if (skip_if_plumb(name, secure) < 0) {
		error = B_TRUE;
		if (secure) {
			fprintf(stderr,
				"%s: error - %s: secure mode FAILED - "
				"trying to use unsecure mode\n",
				SKIPNAME, name);

			if (skip_if_plumb(name, 0) < 0) {
				fprintf(stderr, "%s: unsecure mode FAILED\n",
					name);
				fprintf(stderr, "%s: replumb FAILED\n", name);
				return(1);
			}
			mode = "unsecure";
			secure = 0;
		} else {
			fprintf(stderr, "%s: unsecure mode FAILED\n", name);
			return(1);
		}
	}
	printf("%s: restoring interface state\n", name);
	skip_if_state(name, Restore);
	printf("%s: now in %s mode\n", name, mode);
	if (error) {
		return(1);
	}

	return(0);
}


/*
 * Change the access control mode On/Off
 */
static int
set_secure(char *ifname)
{

	if (!strcmp(skip_mode, "on")) {
		skip_access_ctrl_on(ifname);
	} else {
		if (!strcmp(skip_mode, "off")) {
			skip_access_ctrl_off(ifname);
		} else {
			fprintf(stderr,
				"%s: invalid access control mode \"%s\".\n",
							progname, skip_mode);
			usage();
			return(1);
		}
	}

	summary(ifname);

	return(0);
}


/*
 * Fill in IP mask
 */
static int
skiphost_mask(struct in_addr *mask)
{

	if (!opt_msk || (strcmp(ip_mask, SKIP_HOST_MASK) == 0)) {
		/*
		 * Host address - use "host" mask
		 */
		mask->s_addr = INADDR_BROADCAST;
		return(0);
	}

	if (skip_host_to_addr(ip_mask, mask) < 0) {
			fprintf(stderr, "%s.\n", skip_errmsg);
			return(-1);

	}

	return(0);
}


/*
 * Check  the Receiver NSID and Master Key Id...
 */
static int
skiphost_rmkeyid(char **keyid, skip_param_t *parms)
{

	if (!keyid) {
		return(-1);
	}

	*keyid = r_mkeyid;

	if (opt_r_nsid) {
		/*
		 * Convert NSID string to ID
		 */
		parms->r_nsid = atoi(r_nsid);
		if ((int) parms->r_nsid >= skip_max_nsids) {
			/*
			 * NSID string is not a valid one
			 */
			fprintf(stderr, "%s: invalid NSID \"%s\".\n",
							progname, r_nsid);
			nsid_list();
			usage();
			return(-1);
		}
	} else {
		/*
		 * Default Receiver NSID is :
		 *			- NONE for IPSP 
		 *			- IPV4 for SunScreen
		 *			- IPV4 for raw ESP/AH mode
		 */
		switch (parms->version) {

		case SKIP_RAW:
			parms->r_nsid = SKIP_NSID_IPV4;
			break;

		case SKIP_V2:
			parms->r_nsid = SKIP_NSID_NONE;
			break;

		case SKIP_V1:
			parms->r_nsid = SKIP_NSID_IPV4;
			break;
		}
	}

	switch (parms->version) {

	case SKIP_RAW:
		/*
		 * For raw ESP/AH mode, only NSID 1 is allowed
		 * and the Receiver Master Key ID defines 
		 * the value of the SPI sent into the ESP/AH headers.
		 */
		if (parms->r_nsid != SKIP_NSID_IPV4) {
			fprintf(stderr, "%s: only Receiver NSID 1 is allowed "
					"for raw ESP/AH mode.\n", progname);
			usage();
			return(-1);
		}

		if (!opt_r_keyid) {
			fprintf(stderr, "%s: -R <SPI> is required "
					"for raw ESP/AH mode.\n", progname);
			usage();
			return(-1);
		}
		break;

	case SKIP_V1:
		/*
		 * Only IPV4 for SunScreen
		 */
		if (parms->r_nsid != SKIP_NSID_IPV4) {
			fprintf(stderr, "%s: only Receiver NSID 1 is allowed "
					"for SKIP version 1.\n", progname);
			usage();
			return(-1);
		}
		break;
	}

	if (!opt_r_keyid) {
		switch (parms->r_nsid) {
	
		case SKIP_NSID_NONE:
			/* 
			 * XXX should we use remote IP address as Key Id ?
			 */
			*keyid = NULL;
			break;
	
		case SKIP_NSID_IPV4:
			/* 
			 * We use remote IP address as Key Id
			 */
			printf("%s: set remote IP address as key id.\n",
								progname);
			*keyid = hostname;
			break;
	
		default:
			/*
			 * The Key Id is missing...
			 */
			fprintf(stderr, "%s: missing receiver NSID/key id.\n",
								progname);
			usage();
			return(-1);
		}
	}

	if (*keyid) {
		if (skip_s_to_keyid(*keyid, parms->r_nsid,
						&parms->r_mkeyid) < 0) {
			fprintf(stderr,
					"%s: Invalid receiver NSID/key id.\n",
					progname);
			usage();
			return(-1);
		}
	}
	
	/*
	 * Check is OK.
	 */
	return(0);
}

/*
 * Add a host/subnet entry into the ACL
 */
static void
acl_add_entry(char *ifname, boolean_t authorized)
{
	char			*keyid;
	int			alg;
	skip_param_t		parms;
	ioctl_keyid_t		keyid_list;
	skip_keyid_list_t	*p_keyid_lst;
	int			key_idx = 0;

	memset((char *) &parms, 0, sizeof(skip_param_t));

	if (!authorized) {
		/*
		 * This is an excluded host...
		 */
		parms.flags = SKIP_EXCL;
	}

	/*
	 * Get the IP address of this entry...
	 */
	if (skip_host_to_addr(hostname, &parms.ip_addr) < 0) {
		fprintf(stderr, "%s.\n", skip_errmsg);
		return;
	}

	/*
	 * Check if a nomadic host
	 */
	opt_nomadic = SKIP_IS_UNKNOWNIP(parms.ip_addr);

	/*
	 * Check if it is a subnet...
	 */
	if (skiphost_mask(&parms.mask) < 0) {
		fprintf(stderr, "Mask to IP address failed\n");
		return;
	}

	if (opt_nomadic && opt_msk) {
		fprintf(stderr, "Cannot have a mask with a nomadic system\n");
		return;
	}

	if (opt_nomadic) {
		parms.flags |= SKIP_NOMADIC;
	}

	if (opt_tunnel) {
		if (opt_trs) {
			fprintf(stderr, "Can't have a Tunnel Address"
					" in Transport mode.\n");
			return;
		}
		if (skip_host_to_addr(tunnel, &parms.tunnel_addr)) {
			fprintf(stderr, "%s.\n", skip_errmsg);
			return;
		}
	} else {
		if (parms.flags & SKIP_NOMADIC) {
			/*
			 * Set the tunnel address as any (use source)
			 */
			parms.tunnel_addr.s_addr = SKIP_UNKNOWNIP;
		}  else {
			/*
			 * Set the tunnel address with the destination address
			 */
			parms.tunnel_addr = parms.ip_addr;
		}
	}

	/*
	 * Check if a cleartext host...
	 */
	opt_sel = opt_r_nsid + opt_s_nsid + opt_r_keyid;
	opt_sel += opt_kij + opt_crypt + opt_mac + opt_comp;
	opt_sel += opt_vers + opt_nomadic + opt_tunnel;

	if (!opt_sel) {
		/*
		 * Adding a cleartext host...
		 */
		printf("Adding ");
		skip_host_print(&parms);
		printf("...");
		fflush(stdout);

		if (skip_acl_add(ifname, &parms) < 0) {
			fprintf(stderr, "\n%s.\n", skip_errmsg);
			return;
		}

		printf("done.\n");
		return;
	}

	/*
	 * Get the SKIP version 
	 */
	if (opt_vers) {
		parms.version = atoi(skip_version);;
		if ((parms.version != SKIP_V1) && 
			(parms.version != SKIP_V2) &&
			(parms.version != SKIP_RAW)) {
			fprintf(stderr, "%s: invalid SKIP version \"%s\".\n",
						progname, skip_version);
			fprintf(stderr, "\tvalid versions : 1 or 2 or 3\n");
			usage();
			return;
		}
	} else {
		/*
		 * Default is IPSP mode 
		 */
		parms.version = SKIP_V2;
	}

	/*
	 * Check for supported options in SunScreen mode...
	 */
	if (parms.version  == SKIP_V1) {
		if (opt_mac) {
			/* 
			 * Forbid AH for SunScreen mode...
			 */
			fprintf(stderr, "%s: "
				"authentication not supported for SKIP "
				"version 1.\n",
				progname);
			usage();
			return;
		}
			
		if (opt_comp) {
			/* 
			 * Forbid Compression for SunScreen mode...
			 */
			fprintf(stderr, "%s: "
				"compression not supported for SKIP "
				"version 1.\n",
				progname);
			usage();
			return;
		}

		if (opt_trs) {
			/*
	  	 	* Forbid Transport mode
	 	 	*/
			fprintf(stderr, "%s: "
				"Transport mode not supported for SKIP "
				"version 1.\n",
				progname);
			usage();
			return;
		}

		/*
	  	 * Check if this host uses encryption ...
	 	 */
		if (!opt_crypt) {
			fprintf(stderr, "%s: "
				"no encryption algorithms specified.\n",
				progname);
			usage();
			return;
		}
	}

	/*
	 * A raw ESP/AH host looks like a SKIP V2 one
	 * but without key encryption.
	 */
	if (parms.version == SKIP_RAW) {

		if (opt_kij) {
			fprintf(stderr, "%s: no key encryption algorithm "
				" for raw ESP/AH mode.\n", progname);
			usage();
			return;
		}
	}

	/*
	 * we must select a key encryption algorithm for traffic encryption
	 */
	if (opt_crypt && !opt_kij && (parms.version != SKIP_RAW)) {
		fprintf(stderr, "%s: no key encryption algorithm specified.\n",
			progname);
		usage();
		return;
	}
 
	/*
	 * we must also select a key encryption algorithm for authentication
	 */
	if (opt_mac && !opt_kij && (parms.version != SKIP_RAW)) {
		fprintf(stderr, "%s: no key encryption algorithm specified.\n",
			progname);
		usage();
		return;
	}

	/*
	 * We must select a key encryption algorithm only for 
	 * traffic encryption and/or authentication
	 */
	if (opt_kij && !(opt_mac + opt_crypt)) {
		fprintf(stderr,
			"%s: no encryption/authentication  "
			"algorithm specified.\n",
			progname);
		usage();
		return;
	}


	/*
	 *  Check for supported Kij algorithms
	 */
	if (opt_kij) {
		alg =  skip_name_to_kij_alg(kij_alg, parms.version);
		if ((alg <0) ||
			(!on_boot &&
			!skip_supported_kij_alg(alg, parms.version))) {
			fprintf(stderr,
				"%s: unsupported kij algorithm \"%s\".\n",
				progname, kij_alg);
			kij_list();
			usage();
			return;
		} else {
			parms.kij_alg = (unsigned char) alg;
		}
	}

	/*
	 *  Check for supported Crypt algorithms
	 */
	if (opt_crypt) {
		alg = skip_name_to_kp_alg(kp_alg, parms.version);
		if ((alg < 0) ||
			(!on_boot && 
			!skip_supported_kp_alg(alg, parms.version))) {
			fprintf(stderr,
				"%s: unsupported Kp algorithm \"%s\".\n",
				progname, kp_alg);
			kp_list();
			usage();
			return;
		} else {
			parms.kp_alg = (unsigned char) alg;
		}
	}

	/*
	 * Check for MAC algorithms
	 */
	if (opt_mac) {
		alg = skip_name_to_mac_alg(mac_alg, parms.version);
		if ((alg < 0) || !skip_supported_mac_alg(alg, parms.version)) {
			fprintf(stderr,
				"%s: unsupported MAC algorithm \"%s\".\n",
				progname, mac_alg);
			mac_list();
			usage();
			return;
		} else {
			parms.mac_alg = (unsigned char) alg;
		}
	}

	/*
	 *  Check for Compression algorithms (Nor supported yet!)
	 */
	if (opt_comp) {
		fprintf(stderr, "%s: unsupported Comp algorithm \"%s\".\n",
						progname, comp_alg);
		usage();
		return;
	}

	/*
	 * Again, we must select a key encryption algorithm only for 
	 * traffic encryption and/or authentication
	 */
	if (parms.kij_alg && !(parms.kp_alg + parms.mac_alg)) {
		fprintf(stderr,
			"%s: no valid encryption/authentication  "
			"algorithm specified.\n",
			progname);
		usage();
		return;
	}

	/*
	 * Check if we want to use transport mode...
	 */
	if (opt_trs) {
		parms.ip_mode = SKIP_TRS_ON;
	}

	/*
	 * Process the Receiver NSID and Master Key Id...
	 */
	if (skiphost_rmkeyid(&keyid, &parms) < 0) {
		return;
	}

	if ((parms.version == SKIP_V1) && (keyid == NULL)) {
		/*
		 * node ID is mandatory for SunScreen mode (nodeid)
		 */
		fprintf(stderr, "%s: missing node ID for SKIP version 1.\n",
								progname);
		usage();
		return;
	}

	if (opt_nomadic && !keyid) {
		fprintf(stderr, "A nomadic system requires a remote key id "
			"(-r/-R options)\n");
		return;
	}

	/*
	 * Process the Sender NSID and Master Key Id...
	 */
	keyid = s_mkeyid;

	/*
	 * First, get the list of local (Sender) Master Key IDs
	 */
	if (skip_get_list_keyids(ifname, &keyid_list) < 0) {
		fprintf(stderr, "%s.\n", skip_errmsg);
		return;
	}

	if (opt_s_nsid) {
		parms.s_nsid = atoi(s_nsid);

		/*
		 * Check NSID range
		 */
		if ((int) parms.s_nsid >= skip_max_nsids) {
			fprintf(stderr, "%s: Invalid sender NSID \"%s\".\n",
							progname, s_nsid);
			nsid_list();
			usage();
			return;
		}

	} else {
		/*
		 * Default Sender NSID is :
		 *			- NONE for IPSP 
		 *			- IPV4 for SunScreen
		 *			- IPV4 for raw ESP/AH mode
		 */
		if (parms.version == SKIP_V2) {
			parms.s_nsid = SKIP_NSID_NONE;
		} else {
			parms.s_nsid = SKIP_NSID_IPV4;
		}
	}

	if ((parms.version == SKIP_V1) && (parms.s_nsid != SKIP_NSID_IPV4)) {
		/*
		 * This NSID is forbiden for SunScreen mode...
		 */
		fprintf(stderr, 
			"%s: sender NSID %d not valid for SKIP version 1.\n",
					progname, parms.s_nsid);
		usage();
		return;
	}

	if (!on_boot && (parms.s_nsid != SKIP_NSID_NONE)) {
		/*
		 * Check if we have, at least, one master key for this NSID
		 */
		p_keyid_lst = &keyid_list.keyid_lst[parms.s_nsid];
		if ((p_keyid_lst == NULL) || (p_keyid_lst->count == 0)) {
			fprintf(stderr, "%s: No Local Key configured for "
					"Sender NSID \"%s\".\n",
					progname, 
					(skip_nsids[parms.s_nsid].name ? 
						skip_nsids[parms.s_nsid].name :
						UNKNOWN))
				;
			return;
		} 
	}
	
	/*
	 * For raw ESP/AH mode, only NSID 1 is allowed
	 * and the Sender Master Key ID defines the
	 * value of the SPI received into the ESP/AH headers.
	 */
	if (parms.version == SKIP_RAW) {
		if (parms.s_nsid != SKIP_NSID_IPV4) {
			fprintf(stderr, "%s: only Sender NSID 1 is allowed "
					"for raw ESP/AH mode.\n", progname);
			usage();
			return;
		}

		if (!opt_s_keyid) {
			fprintf(stderr, "%s: -S <SPI> is required "
					"for raw ESP/AH mode.\n", progname);
			usage();
			return;
		}
	}

	if (opt_s_keyid && 
		(parms.s_nsid != SKIP_NSID_NONE) &&
		((parms.version == SKIP_V2) || (parms.version == SKIP_V1))) {

		if ((int)strncasecmp(s_mkeyid, "0x", 2)) {
			/*
			 * Use a Master Key Index
			 */
			key_idx = atoi(s_mkeyid);

			if (((key_idx - 1) < 0) ||
				((key_idx - 1) >= p_keyid_lst->count)) {
				fprintf(stderr,
					"%s: Invalid Sender Key index.\n",
								progname);
				return;
			}

			/*
			 * Set the Master Key Id value
			 */
			KEYVARSET(parms.s_mkeyid,
					p_keyid_lst->mkeyid[key_idx - 1]);
		}
	}

	if ((key_idx == 0) && (keyid != NULL)) {
		if (skip_s_to_keyid(keyid, parms.s_nsid, &parms.s_mkeyid) < 0) {
			fprintf(stderr, "%s: Invalid Sender NSID/key id.\n",
								progname);
			usage();
			return;
		}
	}

	/*
	 * Local Key ID must be validated now (no longer done in the Kernel)
	 */
	if (!on_boot && parms.s_nsid && opt_s_keyid) {
		for (key_idx = 0; key_idx < p_keyid_lst->count; key_idx++) {
			if (KEYVAREQUAL(parms.s_mkeyid,
				p_keyid_lst->mkeyid[key_idx])) {
				break;
			}
		}
		if (key_idx == p_keyid_lst->count) {
			fprintf(stderr, "%s: Invalid Local key id.\n",
								progname);
			return;
		}
	}

	/*
	 * Keep the user informed on what is going on...
	 */
	printf("Adding ");
	skip_host_print(&parms);
	printf("...");
	fflush(stdout);

	/*
	 * Add this host into the ACL...
	 */
	if (skip_acl_add(ifname, &parms) < 0) {
		fprintf(stderr, "\n%s.\n", skip_errmsg);
		return;
	}
	printf("done.\n");

	return;
}


/*
 * Remove an entry from ACL
 */
static void
acl_del_entry(char *ifname)
{
	skip_param_t	parms, aclentry;
	char		*keyid = NULL;

	memset((char *) &parms, 0, sizeof(skip_param_t));

	/*
	 * Get the IP address of this entry...
	 */
	if (skip_host_to_addr(hostname, &parms.ip_addr) < 0) {
		fprintf(stderr, "%s.\n", skip_errmsg);
		return;
	}


	/*
	 * Check if a nomadic host
	 */
	opt_nomadic = SKIP_IS_UNKNOWNIP(parms.ip_addr);

	/*
	 * Check if it is a subnet...
	 */
	if (skiphost_mask(&parms.mask) < 0) {
		fprintf(stderr, "Mask to IP address failed\n");
		return;
	}

	if (opt_nomadic && opt_msk) {
		fprintf(stderr, "Cannot have a mask with a nomadic system\n");
		return;
	}

	if (!opt_nomadic && (opt_r_nsid || opt_r_keyid)) {
		fprintf(stderr,
			"Cannot have a receiver NSID/Master Key Id "
			"with a non-nomadic system\n");
		return;
	}


	/*
	 * Get the SKIP version 
	 */
	if (opt_vers) {
		parms.version = atoi(skip_version);;
		if ((parms.version != SKIP_V1) && 
			(parms.version != SKIP_V2) &&
			(parms.version != SKIP_RAW)) {
			fprintf(stderr, "%s: invalid SKIP version \"%s\".\n",
						progname, skip_version);
			fprintf(stderr, "\tvalid versions : 1 or 2 or 3\n");
			usage();
			return;
		}
	} else {
		if (opt_nomadic) { 
			/*
		 	 * Default is IPSP mode  for a nomadic entry
		 	 */
			parms.version = SKIP_V2;
		}
	}

	if (opt_nomadic) {

		parms.flags |= SKIP_NOMADIC;

		if (skiphost_rmkeyid(&keyid, &parms) < 0) {
			return;
		}

		if (!keyid) {
			fprintf(stderr,
				"A nomadic system requires a remote key id "
				"(-r/-R options)\n");
			return;
		}
	}

	BCOPY((char *) &parms, (char *) &aclentry, sizeof(skip_param_t));

	/*
	 * Query to the ACL...
	 */
	if (skip_acl_get(ifname, &parms) < 0) {
		if (errno == ENOENT) {
			fprintf(stderr, "%s: ACL entry not found.\n",
								progname);
		} else {
			fprintf(stderr, "\n%s: %s.\n", progname, skip_errmsg);
		}
		return;
	}

	/*
	 * Is this the required entry ?
	 */
	if (!opt_nomadic && 
		(BCMP(&aclentry.ip_addr, &parms.ip_addr,
				sizeof(aclentry.ip_addr)) ||
		BCMP(&aclentry.mask, &parms.mask,
				sizeof(aclentry.mask)))) {
		fprintf(stderr, "%s: invalid ACL entry "
				"(bad IP address/Mask).\n", progname);
		return;
	}

	if (opt_nomadic && 
		!KEYVAREQUAL(aclentry.r_mkeyid, parms.r_mkeyid)) {
		fprintf(stderr, "%s: invalid ACL entry "
				"(bad NSID/Key ID).\n", progname);
		return;
	}

	/*
	 * Keep the user informed on what is going on...
	 */
	printf("%s: remove ", ifname);
	skip_host_print(&parms);
	printf("from SKIP access control list...\n");
	fflush(stdout);

	/*
	 * Remove this entry from ACL...
	 */
	if (skip_acl_del(ifname, &parms) < 0) {
		fprintf(stderr, "%s.\n", skip_errmsg);
		return;
	}
	printf("done.\n");
}

/*
 * Check an ACL entry
 */
static int
acl_check(char *ifname)
{
	skip_param_t parms;

	memset((char *) &parms, 0, sizeof(skip_param_t));

	/*
	 * Get the IP address of this entry...
	 */
	if (skip_host_to_addr(hostname, &parms.ip_addr) < 0) {
		fprintf(stderr, "%s.\n", skip_errmsg);
		return(1);
	}

	/*
	 * Keep the user informed on what is going on...
	 */
	printf("Checking ACL entry %s...", hostname);
	fflush(stdout);

	/*
	 * Query to the ACL...
	 */
	if (skip_acl_get(ifname, &parms) < 0) {
		if (errno == ENOENT) {
			printf("not found.\n");
		} else {
			fprintf(stderr, "\n%s.\n", skip_errmsg);
		}
		return(errno);
	}

	/*
	 * Entry was found...
	 */
	printf("\n");
	skip_host_print(&parms);

	return(0);
}

/*
 * Format error/help message for valid options
 */
static void
usage()
{

	fprintf(stderr, 
		"usage:\n"
		"\t%s [-i <interface>]\n"
		"\t%s [-i <interface>] <hostname/IP address>\n"
		"\t%s [-i <interface>] -o <on/off>\n"
		"\t%s [-i <interface>] [-P|-V]\n"
		"\t%s [-i <interface>] "
			"-d <hostname|IP address|\"*\"> [-M <mask>]...\n"
		"\t\t[-r <receiver NSID>] [-R <receiver key Id>]...\n"
		"\t\t[-v <SKIP version>]\n"
		"\t%s [-i <interface>] "
			"-a <hostname|IP address|\"*\"> [-M <mask>]...\n"
		"\t\t[-k <kij algorithm>] [-t <crypt algorithm>]...\n"
		"\t\t[-m <mac algorithm>] [-c <comp algorithm>]...\n"
		"\t\t[-r <receiver NSID>] [-R <receiver key Id>]...\n"
		"\t\t[-s <sender NSID>] [-S <sender key Id>]...\n"
		"\t\t[-v <SKIP version>] [-A <tunnel address>] [-T]\n"
		"\t%s [-i <interface>] "
			"-x <hostname|IP address|\"*\"> [-M <mask>]...\n"
		"\t\t[-k <kij algorithm>] [-t <crypt algorithm>]...\n"
		"\t\t[-m <mac algorithm>] [-c <comp algorithm>]...\n"
		"\t\t[-r <receiver NSID>] [-R <receiver key Id>]...\n"
		"\t\t[-s <sender NSID>] [-S <sender key Id>]...\n"
		"\t\t[-v <SKIP version>] [-T]\n",
		progname, progname, progname, progname,
		progname, progname, progname);

	fprintf(stderr, "\n");

	fprintf(stderr, "%s: version %s\n", progname, SKIP_RELEASE);
}

/*
 * Check that an options has been used only once
 */
static int
skip_one(int *skip_opt)
{
	if (*skip_opt) {
		usage();
		return(1);
	}
	(*skip_opt)++;
	return(0);
}
#define SKIP_ONE(opt)	if (skip_one(opt)) { return(1); }

/*
 * Check if skiphost must perform strong check on algs
 */
static boolean_t
skip_nocheck(void)
{
	char 	*p_env = NULL;

	p_env = getenv("SKIP_NOCHECK");

	return((p_env ? B_TRUE : B_FALSE));
}

/*
 * skiphost tool
 */
int
SKIPHOST(int argc, char *argv[])
{

	opt = opt_action = opt_cpt = 0;
	opt_if = opt_unpb = opt_plumb = 0;
	opt_add = opt_del = opt_exc = 0;
	opt_r_nsid = opt_s_nsid = 0;
	opt_r_keyid = opt_s_keyid = 0;
	opt_kij = opt_crypt = 0;
	opt_mac = opt_comp = 0;
	opt_mode = opt_vers = 0;
	opt_trs = opt_msk = 0;
	opt_prt = opt_sh = opt_mode = 0;
	opt_nomadic = opt_tunnel = 0;

	ifname = skip_default_if();

	progname = argv[0];

	/*
	 * Check if we are called at boot time...
	 */
	on_boot = skip_nocheck();

	if (on_boot) {
		/*
		 * turn off the use of name service
		 */
		skip_use_nameservice = B_FALSE;
	}

	/*
	 * Command line parsing
	 */
	optind = 1;
	while ((opt = getopt(argc, argv,
			"phuPVTa:d:o:x:i:s:r:S:R:k:t:m:c:v:M:A:")) != -1) {

		switch (opt) {
	
		/*
		 * Allowed skiphost actions
		 */
		case 'u':
			/*
			 * Un-plumb the SKIP driver
			 */
			SKIP_ONE(&opt_unpb);
			opt_action = opt;
			break;

		case 'p':
			/*
			 * Plumb the SKIP driver
			 */
			SKIP_ONE(&opt_plumb);
			opt_action = opt;
			break;

		case 'P':
			/*
			 * Print ACL list in a "skiphost format"
			 */
			SKIP_ONE(&opt_prt);
			opt_action = opt;
			break;

		case 'V':
			/*
			 * Print ACL list in a "name=value" format
			 */
			SKIP_ONE(&opt_sh);
			opt_action = opt;
			break;


		case 'a':
			/*
			 * Add an "authorized" entry into the ACL
			 */
			SKIP_ONE(&opt_add);
			opt_action = opt;
			hostname = optarg;
			break;

		case 'd':
			/*
			 * Remove an entry from the ACL
			 */
			SKIP_ONE(&opt_del);
			opt_action = opt;
			hostname = optarg;
			break;

		case 'x':
			/*
			 * Add an "un-authorized" entry into the ACL
			 */
			SKIP_ONE(&opt_exc);
			opt_action = opt;
			hostname = optarg;
			break;

		case 'o':
			/*
			 * Set SKIP mode on/off
			 */
			SKIP_ONE(&opt_mode);
			opt_action = opt;
			skip_mode = optarg;
			break;

		/*
		 * Algorithms selection
		 */
		case 'k':
			SKIP_ONE(&opt_kij);
			kij_alg = optarg;
			break;

		case 't':
			SKIP_ONE(&opt_crypt);
			kp_alg = optarg;
			break;

		case 'm':
			SKIP_ONE(&opt_mac);
			mac_alg = optarg;
			break;

		case 'c':
			SKIP_ONE(&opt_comp);
			comp_alg = optarg;
			break;

		/*
		 * NSID and Master Key IDs
		 */
		case 's':
			SKIP_ONE(&opt_s_nsid);
			s_nsid = optarg;
			break;

		case 'r':
			SKIP_ONE(&opt_r_nsid);
			r_nsid = optarg;
			break;

		case 'R':
			SKIP_ONE(&opt_r_keyid);
			r_mkeyid = optarg;
			break;

		case 'S':
			SKIP_ONE(&opt_s_keyid);
			s_mkeyid = optarg;
			break;
		/*
		 * Misc...
		 */
		case 'i':
			SKIP_ONE(&opt_if);
			ifname = optarg;
			break;

		case 'v':
			SKIP_ONE(&opt_vers);
			skip_version = optarg;
			break;

		case 'T':
			SKIP_ONE(&opt_trs);
			break;

		case 'M':
			SKIP_ONE(&opt_msk);
			ip_mask = optarg;
			break;

		case 'A':
			SKIP_ONE(&opt_tunnel);
			tunnel = optarg;
			break;

		case 'h':
		default:
			usage();
			return(0);
		}

		if (opt != 'i') {
			opt_cpt++;
		}
	}

	/*
	 * Get the last arguments if any...
	 */
	for ( ; optind < argc; optind++) {
		/*
		 * User may want to check ACL entry...
		 */
		if (!opt_more) {
			opt_more++;
			hostname =  argv[optind];
			opt_action = 'c';
		} else {
			fprintf(stderr, "%s: too many arguments\n",
								progname);
			usage();
			return(1);
		}
	}

	/*
	 * Get the default interface ?
	 */
	if (!strcmp(ifname, SKIP_DEFAULT)) {
		fprintf(stdout,"%s\n", skip_default_if());
		return(0);
	}

	/*
	 * Useful pre-checks
	 */
	if (skip_pre_checks(ifname)) {
		switch (errno) {
		case EACCES:
			fprintf(stderr, "%s: you must be root to run this "
				"program\n", progname);
			break;

		case ENODEV:
			fprintf(stderr, "%s.\n", skip_errmsg);
			fprintf(stderr, "%s: Check that SKIP is correctly "
				"installed on this system\n", progname);
			break;
		default:
			fprintf(stderr, "%s.\n", skip_errmsg);
		}
		return(1);
	}

	/*
	 * Check that no more than one action is selected...
	 */
	if ((opt_unpb + opt_plumb + opt_add + opt_del +
			opt_mode + opt_prt + opt_sh + opt_exc + opt_more) > 1) {
		if (opt_more) {
			fprintf(stderr, "%s: too many arguments\n",
								progname);
		}
		usage();
		return(1);
	}


	/*
	 * Check if SKIP driver is loaded
	 */
	if (!skip_if_module_present(ifname) && !opt_plumb) {

		fprintf(stderr, "%s: ", progname);
		fprintf(stderr, SKIP_MSG_NO_PLUMB "\n", ifname);

		return(1);
	}

	/*
	 * Get SKIP driver info for Kij algs,...
	 */
	if (!(opt_unpb + opt_plumb) && skip_var_init(progname, ifname)) {
		fprintf(stderr, "%s.\n", skip_errmsg);
		return(1);
	}

	/*
	 * Check for default action 
	 */
	if (!(opt_unpb + opt_plumb + opt_add + opt_del +
			opt_mode + opt_prt + opt_sh + opt_exc)) {

		if (opt_cpt) {
			/*
		 	* No option should be specified...
		 	*/	
			usage();
			return(1);
		}

		if (!opt_action) {
			/*
	 	 	* Default action is summarize...
	 	 	*/
			summary(ifname);
			return(0);
		}
	}

	/*
	 * Process command...
	 */

	switch (opt_action)  {

	case 'u':
		/*
		 * Check for invalid options...
		 */
		opt_sel = opt_r_nsid + opt_s_nsid + opt_r_keyid + opt_s_keyid;
		opt_sel += opt_kij + opt_crypt + opt_mac + opt_comp;
		opt_sel += opt_mode + opt_vers + opt_trs + opt_msk;
		opt_sel += opt_tunnel;

		if (opt_sel) {
			fprintf(stderr, "%s -u does not take options\n",
								progname);
			usage();
			return(1);
		}

		/*
		 * un-plumb the SKIP driver
		 */
		interface_plumb(ifname, 0);
		return(0);

	case 'p':
		/*
		 * Check for invalid options...
		 */
		opt_sel = opt_r_nsid + opt_s_nsid + opt_r_keyid + opt_s_keyid;
		opt_sel += opt_kij + opt_crypt + opt_mac + opt_comp;
		opt_sel += opt_mode + opt_vers + opt_trs + opt_msk;
		opt_sel += opt_tunnel;

		if (opt_sel) {
			fprintf(stderr, "%s -p does not take options\n",
								progname);
			usage();
			return(1);
		}

		if (skip_if_module_present(ifname)) {
			fprintf(stderr, "%s: SKIP is already present on %s\n",
							progname, ifname);
			return(1);
		}

		/*
		 * Plumb the SKIP driver and set security mode off
		 */
		interface_plumb(ifname, 1);
		skip_access_ctrl_off(ifname);
		return(0);

	case 'P':
		/*
		 * Check for invalid options...
		 */
		opt_sel = opt_r_nsid + opt_s_nsid + opt_r_keyid + opt_s_keyid;
		opt_sel += opt_kij + opt_crypt + opt_mac + opt_comp;
		opt_sel += opt_mode + opt_vers + opt_trs + opt_msk;
		opt_sel += opt_tunnel;

		if (opt_sel) {
			fprintf(stderr, "%s -P does not take options\n",
								progname);
			usage();
			return(1);
		}

		/*
		 * print each entry in a "skiphost" command format
		 */
		acl_list_entries(ifname);
		return(0);

	case 'V':
		/*
		 * Check for invalid options...
		 */
		opt_sel = opt_r_nsid + opt_s_nsid + opt_r_keyid + opt_s_keyid;
		opt_sel += opt_kij + opt_crypt + opt_mac + opt_comp;
		opt_sel += opt_mode + opt_vers + opt_trs + opt_msk;
		opt_sel += opt_tunnel;

		if (opt_sel) {
			fprintf(stderr, "%s -V does not take options\n",
								progname);
			usage();
			return(1);
		}

		/*
		 * print each entry in a "name=value" command format
		 */
		acl_list_sh(ifname);
		return(0);

	case 'o':
		/*
		 * Check for invalid options...
		 */
		opt_sel = opt_r_nsid + opt_s_nsid + opt_r_keyid + opt_s_keyid;
		opt_sel += opt_kij + opt_crypt + opt_mac + opt_comp;
		opt_sel += opt_vers + opt_trs + opt_msk;
		opt_sel += opt_tunnel;

		if (opt_sel) {
			usage();
			return(1);
		}

		/*
		 * Set secure mode on/off
		 */
		set_secure(ifname);
		return(0);

	case 'a':
		/*
		 * Add an authorized host/subnet into the ACL
		 */
		acl_add_entry(ifname, B_TRUE);
		return(0);

	case 'x':
		/*
		 * Add an excluded host/subnet into the ACL
		 */
		acl_add_entry(ifname, B_FALSE);
		return(0);

	case 'd':
		/*
		 * Check for invalid options...
		 */
		opt_sel = opt_s_nsid + opt_s_keyid;
		opt_sel += opt_kij + opt_crypt + opt_mac + opt_comp;
		opt_sel += opt_mode + opt_trs + opt_tunnel;

		if (opt_sel) {
			usage();
			return(1);
		}
		/*
		 * Remove a host from ACL
		 */
		acl_del_entry(ifname);
		return(0);

	case 'c':
		/*
		 * Check an entry in the ACL
		 */
		return(acl_check(ifname));

	}

	/*
	 * Should never be reached !
	 */
	usage();
	return (1);
}
