/*
 * RFC2367 PF_KEYv2 Key management API message parser
 * Copyright (C) 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: pfkey_v2_parse.c,v 1.10 1999/12/10 17:45:24 rgb Exp $
 */

/*
 *		Template from klips/net/ipsec/ipsec/ipsec_parser.c.
 */

char pfkey_v2_parse_c_version[] = "$Id: pfkey_v2_parse.c,v 1.10 1999/12/10 17:45:24 rgb Exp $";

/*
 * Some ugly stuff to allow consistent debugging code for use in the
 * kernel and in user space
*/

#ifdef __KERNEL__

#include <linux/kernel.h>  /* for printk */

#include <linux/malloc.h> /* kmalloc() */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/interrupt.h> /* mark_bh */

#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */ 
#if defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE)
#include <linux/ipv6.h>        /* struct ipv6hdr */
#endif /* if defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE) */
extern int debug_pfkey;

#define DEBUGGING(format,args...) \
         ((debug_pfkey) ? printk(KERN_INFO "klips_debug:" format , ## args) : 0)


#else /* __KERNEL__ */

#include <sys/types.h>
#include <linux/types.h>
#include <linux/errno.h>

#include "../pluto/constants.h" 
#include "../pluto/defs.h"  /* for PRINTF_LIKE */
#include "../pluto/log.h"  /* for debugging and DBG_log */

extern unsigned int pfkey_lib_debug;  /* bits selecting what to report */

#ifdef PLUTO
#define DEBUGGING(format,args...)  { DBG_log("pfkey_lib_debug:" format, ## args);  }
#else
#define DEBUGGING(format,args...)  if(pfkey_lib_debug) { printf("pfkey_lib_debug:" format, ## args); } else { ; }
#endif

#endif /* __KERNEL__ */


#include <freeswan.h>
#include <pfkeyv2.h>
#include <pfkey.h>

#define SENDERR(_x) do { error = -(_x); goto errlab; } while (0)

uint8_t sadb_satype2proto[] = {
#ifdef __KERNEL__
	0,
	0,
	IPPROTO_AH,
	IPPROTO_ESP,
	0,
	0,
	0,
	0,
	0,
	IPPROTO_IPIP,
	IPPROTO_COMP
#else /* __KERNEL__ */
	0,
	0,
	SA_AH,
	SA_ESP,
	0,
	0,
	0,
	0,
	0,
	SA_IPIP,
	SA_COMP
#endif /* __KERNEL__ */
};

/* Default extension parsers taken from the KLIPS code */

static int
pfkey_sa_parse(struct sadb_ext *pfkey_ext)
{
	int error = 0;
	struct sadb_sa *pfkey_sa = (struct sadb_sa *)pfkey_ext;
	
	DEBUGGING(
		"pfkey_sa_parse:\n");
	/* sanity checks... */
	if(!pfkey_sa) {
		DEBUGGING(
			"pfkey_sa_parse: NULL pointer passed in.\n");
		SENDERR(EINVAL);
	}
	
	if(pfkey_sa->sadb_sa_len != sizeof(struct sadb_sa) / IPSEC_PFKEYv2_ALIGN) {
		DEBUGGING(
			"pfkey_sa_parse: length wrong pfkey_sa->sadb_sa_len=%d sizeof(struct sadb_sa)=%d.\n",
			pfkey_sa->sadb_sa_len,
			sizeof(struct sadb_sa));
		SENDERR(EINVAL);
	}
	
	if(pfkey_sa->sadb_sa_encrypt > SADB_EALG_MAX) {
		DEBUGGING(
			"pfkey_sa_parse: pfkey_sa->sadb_sa_encrypt=%d > SADB_EALG_MAX=%d.\n",
			pfkey_sa->sadb_sa_encrypt,
			SADB_EALG_MAX);
		SENDERR(EINVAL);
	}
	
	if(pfkey_sa->sadb_sa_auth > SADB_AALG_MAX) {
		DEBUGGING(
			"pfkey_sa_parse: pfkey_sa->sadb_sa_auth=%d > SADB_AALG_MAX=%d.\n",
			pfkey_sa->sadb_sa_auth,
			SADB_AALG_MAX);
		SENDERR(EINVAL);
	}
	
	if(pfkey_sa->sadb_sa_state > SADB_SASTATE_MAX) {
		DEBUGGING(
			"pfkey_sa_parse: state=%d exceeds MAX=%d.\n",
			pfkey_sa->sadb_sa_state,
			SADB_SASTATE_MAX);
		SENDERR(EINVAL);
	}
	
	if(pfkey_sa->sadb_sa_state == SADB_SASTATE_DEAD) {
		DEBUGGING(
			"pfkey_sa_parse: state=%d is DEAD=%d.\n",
			pfkey_sa->sadb_sa_state,
			SADB_SASTATE_DEAD);
		SENDERR(EINVAL);
	}
	
	if(pfkey_sa->sadb_sa_replay > 64) {
		DEBUGGING(
			"pfkey_sa_parse: replay window size: %d"
			" -- must be 0 <= size <= 64\n",
			pfkey_sa->sadb_sa_replay);
		SENDERR(EINVAL);
	}
	
	
	if(! ((pfkey_sa->sadb_sa_exttype ==  SADB_EXT_SA) ||
	      (pfkey_sa->sadb_sa_exttype ==  SADB_X_EXT_SA2)))
	{
		DEBUGGING(
			"pfkey_sa_parse: unknown exttype=%d, expecting SADB_EXT_SA=%d or SADB_X_EXT_SA2=%d.\n",
			pfkey_sa->sadb_sa_exttype,
			SADB_EXT_SA,
			SADB_X_EXT_SA2);
		SENDERR(EINVAL);
	}
	DEBUGGING(
		"pfkey_sa_parse: successfully found len=%d exttype=%d spi=%08lx replay=%d state=%d flags=%d.\n",
		pfkey_sa->sadb_sa_len,
		pfkey_sa->sadb_sa_exttype,
		ntohl(pfkey_sa->sadb_sa_spi),
		pfkey_sa->sadb_sa_replay,
		pfkey_sa->sadb_sa_state,
		pfkey_sa->sadb_sa_flags);

 errlab:
	return error;
}	

static int
pfkey_lifetime_parse(struct sadb_ext  *pfkey_ext)
{
	int error = 0;
	struct sadb_lifetime *pfkey_lifetime = (struct sadb_lifetime *)pfkey_ext;

	DEBUGGING(
		    "pfkey_lifetime_parse:\n");
	/* sanity checks... */
	if(!pfkey_lifetime) {
		DEBUGGING(
			"pfkey_lifetime_parse: NULL pointer passed in.\n");
		SENDERR(EINVAL);
	}

	if(pfkey_lifetime->sadb_lifetime_len !=
	   sizeof(struct sadb_lifetime) / IPSEC_PFKEYv2_ALIGN) {
		DEBUGGING(
			"pfkey_lifetime_parse: length wrong pfkey_lifetime->sadb_lifetime_len=%d sizeof(struct sadb_lifetime)=%d.\n",
			pfkey_lifetime->sadb_lifetime_len,
			sizeof(struct sadb_lifetime));
		SENDERR(EINVAL);
	}

	if((pfkey_lifetime->sadb_lifetime_exttype != SADB_EXT_LIFETIME_HARD) &&
	   (pfkey_lifetime->sadb_lifetime_exttype != SADB_EXT_LIFETIME_SOFT) &&
	   (pfkey_lifetime->sadb_lifetime_exttype != SADB_EXT_LIFETIME_CURRENT)) {
		DEBUGGING( 
			"pfkey_lifetime_parse: unexpected ext_type=%d.\n", 
			pfkey_lifetime->sadb_lifetime_exttype); 
		SENDERR(EINVAL);
	}

errlab:
	return error;
}

static int
pfkey_address_parse(struct sadb_ext *pfkey_ext)
{
	int error = 0;
	int saddr_len = 0;
	struct sadb_address *pfkey_address = (struct sadb_address *)pfkey_ext;
	struct sockaddr* s = (struct sockaddr*)((char*)pfkey_address + sizeof(*pfkey_address));
	char ipaddr_txt[ADDRTOA_BUF];
	
	DEBUGGING(
		"pfkey_address_parse:\n");
	/* sanity checks... */
	if(!pfkey_address) {
		DEBUGGING(
			"pfkey_address_parse: NULL pointer passed in.\n");
		SENDERR(EINVAL);
	}
	
	if(pfkey_address->sadb_address_len <
	   (sizeof(struct sadb_address) + sizeof(struct sockaddr))/
	   IPSEC_PFKEYv2_ALIGN) {
		DEBUGGING(
			"pfkey_address_parse: size wrong 1 ext_len=%d, adr_ext_len=%d, saddr_len=%d.\n",
			pfkey_address->sadb_address_len,
			sizeof(struct sadb_address),
			sizeof(struct sockaddr));
		SENDERR(EINVAL);
	}
	
	if(pfkey_address->sadb_address_reserved) {
		DEBUGGING(
			"pfkey_address_parse: res=%d, must be zero.\n",
			pfkey_address->sadb_address_reserved);
		SENDERR(EINVAL);
	}
	
	switch(s->sa_family) {
	case AF_INET:
		DEBUGGING(
			"pfkey_address_parse: found address family=%d, AF_INET.\n",
			s->sa_family);
		saddr_len = sizeof(struct sockaddr_in);
		addrtoa(((struct sockaddr_in*)s)->sin_addr, 0, ipaddr_txt, sizeof(ipaddr_txt));
		DEBUGGING(
			"pfkey_address_parse: found address=%s.\n",
			ipaddr_txt);
		break;
#if defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE)
	case AF_INET6:
		saddr_len = sizeof(struct sockaddr_in6);
		break;
#endif /* defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE) */
	default:
		DEBUGGING(
			"pfkey_address_parse: s->sa_family=%d not supported.\n",
			s->sa_family);
		SENDERR(EPFNOSUPPORT);
	}
	
	if(pfkey_address->sadb_address_len !=
	   DIVUP(sizeof(struct sadb_address) + saddr_len, IPSEC_PFKEYv2_ALIGN)) {
		DEBUGGING(
			"pfkey_address_parse: size wrong 2 ext_len=%d, adr_ext_len=%d, saddr_len=%d.\n",
			pfkey_address->sadb_address_len,
			sizeof(struct sadb_address),
			saddr_len);
		SENDERR(EINVAL);
	}
	
	if(pfkey_address->sadb_address_prefixlen != 0) {
		DEBUGGING(
			"pfkey_address_parse: address prefixes not supported yet.\n");
		SENDERR(EAFNOSUPPORT); /* not supported yet */
	}
	
	
	if( ! ( (pfkey_address->sadb_address_exttype == SADB_EXT_ADDRESS_SRC) ||
		(pfkey_address->sadb_address_exttype == SADB_EXT_ADDRESS_DST) ||
		(pfkey_address->sadb_address_exttype == SADB_EXT_ADDRESS_PROXY) ||
		(pfkey_address->sadb_address_exttype == SADB_X_EXT_ADDRESS_DST2) ))
	{
		DEBUGGING( 
			"pfkey_address_parse: unexpected ext_type=%d.\n", 
			pfkey_address->sadb_address_exttype); 
		SENDERR(EINVAL); 
	}
	
	/* XXX check if port!=0 */
	
	DEBUGGING(
		"pfkey_address_parse: successful.\n");
 errlab:
	return error;
}

static int
pfkey_key_parse(struct sadb_ext *pfkey_ext)
{
	int error = 0;
	struct sadb_key *pfkey_key = (struct sadb_key *)pfkey_ext;

	DEBUGGING(
		"pfkey_key_parse:\n");
	/* sanity checks... */

	if(!pfkey_key) {
		DEBUGGING(
			"pfkey_key_parse: NULL pointer passed in.\n");
		SENDERR(EINVAL);
	}

	if(pfkey_key->sadb_key_len < sizeof(struct sadb_key) / IPSEC_PFKEYv2_ALIGN) {
		DEBUGGING(
			"pfkey_key_parse: size wrong ext_len=%d, key_ext_len=%d.\n",
			pfkey_key->sadb_key_len,
			sizeof(struct sadb_key));
		SENDERR(EINVAL);
	}

	if(!pfkey_key->sadb_key_bits) {
		DEBUGGING(
			"pfkey_key_parse: key length set to zero, must be non-zero.\n");
		SENDERR(EINVAL);
	}

	if(pfkey_key->sadb_key_len !=
	   DIVUP(sizeof(struct sadb_key) * 8 + pfkey_key->sadb_key_bits,
		 64)) {
		DEBUGGING(
			"pfkey_key_parse: key length=%d does not agree with extension length=%d.\n",
			pfkey_key->sadb_key_bits,
			pfkey_key->sadb_key_len);
		SENDERR(EINVAL);
	}
	
	if(pfkey_key->sadb_key_reserved) {
		DEBUGGING(
			"pfkey_key_parse: res=%d, must be zero.\n",
			pfkey_key->sadb_key_reserved);
		SENDERR(EINVAL);
	}

	if(! ( (pfkey_key->sadb_key_exttype == SADB_EXT_KEY_AUTH) ||
	       (pfkey_key->sadb_key_exttype == SADB_EXT_KEY_ENCRYPT))) {
		DEBUGGING(
			"pfkey_key_parse: expecting extension type AUTH or ENCRYPT, got %d.\n",
			pfkey_key->sadb_key_exttype);
		SENDERR(EINVAL);
	}

	DEBUGGING(
		"pfkey_key_parse: success, found len=%d exttype=%d bits=%d reserved=%d.\n",
		pfkey_key->sadb_key_len,
		pfkey_key->sadb_key_exttype,
		pfkey_key->sadb_key_bits,
		pfkey_key->sadb_key_reserved);

errlab:
	return error;
}

static int
pfkey_ident_parse(struct sadb_ext *pfkey_ext)
{
	int error = 0;
	struct sadb_ident *pfkey_ident = (struct sadb_ident *)pfkey_ext;

	/* sanity checks... */
	if(pfkey_ident->sadb_ident_len < sizeof(struct sadb_ident) / IPSEC_PFKEYv2_ALIGN) {
		DEBUGGING(
			"pfkey_ident_parse: size wrong ext_len=%d, key_ext_len=%d.\n",
			pfkey_ident->sadb_ident_len,
			sizeof(struct sadb_ident));
		SENDERR(EINVAL);
	}

	if(pfkey_ident->sadb_ident_type > SADB_IDENTTYPE_MAX) {
		DEBUGGING(
			"pfkey_ident_parse: ident_type=%d out of range, must be less than %d.\n",
			pfkey_ident->sadb_ident_reserved,
			SADB_IDENTTYPE_MAX);
		SENDERR(EINVAL);
	}

	if(pfkey_ident->sadb_ident_reserved) {
		DEBUGGING(
			"pfkey_ident_parse: res=%d, must be zero.\n",
			pfkey_ident->sadb_ident_reserved);
		SENDERR(EINVAL);
	}

	/* string terminator/padding must be zero */
	if(pfkey_ident->sadb_ident_len > sizeof(struct sadb_ident) / IPSEC_PFKEYv2_ALIGN) {
		if(*((char*)pfkey_ident + pfkey_ident->sadb_ident_len * IPSEC_PFKEYv2_ALIGN - 1)) {
			DEBUGGING(
				"pfkey_ident_parse: string padding must be zero, last is 0x%02x.\n",
				*((char*)pfkey_ident +
				  pfkey_ident->sadb_ident_len * IPSEC_PFKEYv2_ALIGN - 1));
			SENDERR(EINVAL);
		}
	}
	
	if( ! ((pfkey_ident->sadb_ident_exttype == SADB_EXT_IDENTITY_SRC) ||
	       (pfkey_ident->sadb_ident_exttype == SADB_EXT_IDENTITY_DST))) {
		DEBUGGING(
			"pfkey_key_parse: expecting extension type IDENTITY_SRC or IDENTITY_DST, got %d.\n",
			pfkey_ident->sadb_ident_exttype);
		SENDERR(EINVAL);
	}

errlab:
	return error;
}

static int
pfkey_sens_parse(struct sadb_ext *pfkey_ext)
{
	int error = 0;
	struct sadb_sens *pfkey_sens = (struct sadb_sens *)pfkey_ext;

	/* sanity checks... */
	if(pfkey_sens->sadb_sens_len < sizeof(struct sadb_sens) / IPSEC_PFKEYv2_ALIGN) {
		DEBUGGING(
			"pfkey_sens_parse: size wrong ext_len=%d, key_ext_len=%d.\n",
			pfkey_sens->sadb_sens_len,
			sizeof(struct sadb_sens));
		SENDERR(EINVAL);
	}

	DEBUGGING(
		"pfkey_sens_parse: Sorry, I can't parse exttype=%d yet.\n",
		pfkey_ext->sadb_ext_type);
	SENDERR(EINVAL); /* don't process these yet */

errlab:
	return error;
}

static int
pfkey_prop_parse(struct sadb_ext *pfkey_ext)
{
	int error = 0;
	int i;
	struct sadb_prop *pfkey_prop = (struct sadb_prop *)pfkey_ext;

	/* sanity checks... */
	if(pfkey_prop->sadb_prop_len < sizeof(struct sadb_prop) / IPSEC_PFKEYv2_ALIGN) {
		DEBUGGING(
			"pfkey_prop_parse: size wrong ext_len=%d, key_ext_len=%d.\n",
			pfkey_prop->sadb_prop_len,
			sizeof(struct sadb_prop));
		SENDERR(EINVAL);
	}

	for(i=0; i<3; i++) {
		if(pfkey_prop->sadb_prop_reserved[i]) {
			DEBUGGING(
				"pfkey_prop_parse: res[%d]=%d, must be zero.\n",
				i, pfkey_prop->sadb_prop_reserved[i]);
			SENDERR(EINVAL);
		}
	}

	DEBUGGING(
		"pfkey_prop_parse: Sorry, I can't parse exttype=%d yet.\n",
		pfkey_ext->sadb_ext_type);
	SENDERR(EINVAL); /* don't process these yet */

errlab:
	return error;
}

static int
pfkey_supported_parse(struct sadb_ext *pfkey_ext)
{
	int error = 0;
	unsigned int i;
	struct sadb_supported *pfkey_supported = (struct sadb_supported *)pfkey_ext;
	struct sadb_alg *pfkey_alg;

	/* sanity checks... */
	if(pfkey_supported->sadb_supported_len <
	   sizeof(struct sadb_supported) / IPSEC_PFKEYv2_ALIGN) {
		DEBUGGING(
			"pfkey_supported_parse: size wrong ext_len=%d, key_ext_len=%d.\n",
			pfkey_supported->sadb_supported_len,
			sizeof(struct sadb_supported));
		SENDERR(EINVAL);
	}

	if(pfkey_supported->sadb_supported_reserved) {
		DEBUGGING(
			"pfkey_supported_parse: res=%d, must be zero.\n",
			pfkey_supported->sadb_supported_reserved);
		SENDERR(EINVAL);
	}

	pfkey_alg = (struct sadb_alg*)((char*)pfkey_ext + sizeof(struct sadb_supported));
	for(i = 0;
	    i < pfkey_supported->sadb_supported_len -
		    (sizeof(struct sadb_supported) / IPSEC_PFKEYv2_ALIGN);
	    i++) {
		/* process algo description */
		if(!pfkey_alg->sadb_alg_reserved) {
			DEBUGGING(
				"pfkey_supported_parse: alg#%d, alg_id=%d, res=%d, must be zero.\n",
				i,
				pfkey_alg->sadb_alg_id,
				pfkey_alg->sadb_alg_reserved);
			SENDERR(EINVAL);
		}
		pfkey_alg += sizeof(struct sadb_alg);
	}
	
	DEBUGGING(
		"pfkey_supported_parse: Sorry, I can't parse exttype=%d yet.\n",
		pfkey_ext->sadb_ext_type);
	SENDERR(EINVAL); /* don't process these yet */
	
 errlab:
	return error;
}

static int
pfkey_spirange_parse(struct sadb_ext *pfkey_ext)
{
	int error = 0;
	struct sadb_spirange *pfkey_spirange = (struct sadb_spirange *)pfkey_ext;
	
	/* sanity checks... */
        if(pfkey_spirange->sadb_spirange_len !=
	   sizeof(struct sadb_spirange) / IPSEC_PFKEYv2_ALIGN) {
		DEBUGGING(
			"pfkey_spirange_parse: size wrong ext_len=%d, key_ext_len=%d.\n",
			pfkey_spirange->sadb_spirange_len,
			sizeof(struct sadb_spirange));
                SENDERR(EINVAL);
        }
	
        if(pfkey_spirange->sadb_spirange_reserved) {
		DEBUGGING(
			" pfkey_spirange_parse: reserved=%d must be set to zero.\n",
			pfkey_spirange->sadb_spirange_reserved);
                SENDERR(EINVAL);
        }
	
        if(pfkey_spirange->sadb_spirange_max < pfkey_spirange->sadb_spirange_min) {
		DEBUGGING(
			" pfkey_spirange_parse: minspi=%d must be < maxspi=%d.\n",
			pfkey_spirange->sadb_spirange_min,
			pfkey_spirange->sadb_spirange_max);
                SENDERR(EINVAL);
        }
	
	if(pfkey_spirange->sadb_spirange_min <= 255) {
		DEBUGGING(
			" pfkey_spirange_parse: minspi=%d must be > 255.\n",
			pfkey_spirange->sadb_spirange_min);
		SENDERR(EEXIST);
	}
	
 errlab:
	return error;
}

static int
pfkey_x_kmprivate_parse(struct sadb_ext *pfkey_ext)
{
	int error = 0;
	struct sadb_x_kmprivate *pfkey_x_kmprivate = (struct sadb_x_kmprivate *)pfkey_ext;

	/* sanity checks... */
	if(pfkey_x_kmprivate->sadb_x_kmprivate_len <
	   sizeof(struct sadb_x_kmprivate) / IPSEC_PFKEYv2_ALIGN) {
		DEBUGGING(
			"pfkey_x_kmprivate_parse: size wrong ext_len=%d, key_ext_len=%d.\n",
			pfkey_x_kmprivate->sadb_x_kmprivate_len,
			sizeof(struct sadb_x_kmprivate));
		SENDERR(EINVAL);
	}

	if(pfkey_x_kmprivate->sadb_x_kmprivate_reserved) {
		DEBUGGING(
			" pfkey_x_kmprivate_parse: reserved=%d must be set to zero.\n",
			pfkey_x_kmprivate->sadb_x_kmprivate_reserved);
		SENDERR(EINVAL);
	}

	DEBUGGING(
		"pfkey_x_kmprivate_parse: Sorry, I can't parse exttype=%d yet.\n",
		pfkey_ext->sadb_ext_type);
	SENDERR(EINVAL); /* don't process these yet */

errlab:
	return error;
}

static int
pfkey_x_satype_parse(struct sadb_ext *pfkey_ext)
{
	int error = 0;
	int i;
	struct sadb_x_satype *pfkey_x_satype = (struct sadb_x_satype *)pfkey_ext;

	DEBUGGING(
		"pfkey_x_satype_parse:\n");
	/* sanity checks... */
	if(pfkey_x_satype->sadb_x_satype_len !=
	   sizeof(struct sadb_x_satype) / IPSEC_PFKEYv2_ALIGN) {
		DEBUGGING(
			"pfkey_x_satype_parse: size wrong ext_len=%d, key_ext_len=%d.\n",
			pfkey_x_satype->sadb_x_satype_len,
			sizeof(struct sadb_x_satype));
		SENDERR(EINVAL);
	}
	
	if(!pfkey_x_satype->sadb_x_satype_satype) {
		DEBUGGING(
			"pfkey_x_satype_parse: satype is zero, must be non-zero.\n");
		SENDERR(EINVAL);
	}

	if(pfkey_x_satype->sadb_x_satype_satype > SADB_SATYPE_MAX) {
		DEBUGGING(
			"pfkey_x_satype_parse: satype %d > max %d\n", 
			pfkey_x_satype->sadb_x_satype_satype, SADB_SATYPE_MAX);
		SENDERR(EINVAL);
	}

	if(!(sadb_satype2proto[pfkey_x_satype->sadb_x_satype_satype])) {
		DEBUGGING(
			"pfkey_x_satype_parse: proto lookup from satype=%d failed.\n",
			pfkey_x_satype->sadb_x_satype_satype);
		SENDERR(EINVAL);
	}

	for(i = 0; i < 3; i++) {
		if(pfkey_x_satype->sadb_x_satype_reserved[i]) {
			DEBUGGING(
				" pfkey_x_satype_parse: reserved[%d]=%d must be set to zero.\n",
				i, pfkey_x_satype->sadb_x_satype_reserved[i]);
			SENDERR(EINVAL);
		}
	}
	
errlab:
	return error;
}

int (*ext_default_parsers[SADB_EXT_MAX +1])(struct sadb_ext*) =
{
	NULL, /* pfkey_msg_parse, */
	pfkey_sa_parse,
	pfkey_lifetime_parse,
	pfkey_lifetime_parse,
	pfkey_lifetime_parse,
	pfkey_address_parse,
	pfkey_address_parse,
	pfkey_address_parse,
	pfkey_key_parse,
	pfkey_key_parse,
	pfkey_ident_parse,
	pfkey_ident_parse,
	pfkey_sens_parse,
	pfkey_prop_parse,
	pfkey_supported_parse,
	pfkey_supported_parse,
	pfkey_spirange_parse,
	pfkey_x_kmprivate_parse,
	pfkey_x_satype_parse,
	pfkey_sa_parse,
	pfkey_address_parse
};

int
pfkey_msg_parse(struct sadb_msg *pfkey_msg,
		int (*ext_parsers[])(struct sadb_ext*),
		struct sadb_ext *extensions[],
		int dir)
{
	int error = 0;
	int remain;
	struct sadb_ext *pfkey_ext;
	int extensions_seen = 0;
	
	DEBUGGING(
		"pfkey_msg_parse: parsing message "
		"ver=%d, type=%d, errno=%d, satype=%d, len=%d, res=%d, seq=%d, pid=%d.\n", 
		pfkey_msg->sadb_msg_version,
		pfkey_msg->sadb_msg_type,
		pfkey_msg->sadb_msg_errno,
		pfkey_msg->sadb_msg_satype,
		pfkey_msg->sadb_msg_len,
		pfkey_msg->sadb_msg_reserved,
		pfkey_msg->sadb_msg_seq,
		pfkey_msg->sadb_msg_pid);
	
	if(ext_parsers == NULL) ext_parsers = ext_default_parsers;
	
	pfkey_extensions_init(extensions);
	
	remain = pfkey_msg->sadb_msg_len;
	remain -= sizeof(struct sadb_msg) / IPSEC_PFKEYv2_ALIGN;
	
	pfkey_ext = (struct sadb_ext*)((char*)pfkey_msg +
				       sizeof(struct sadb_msg));
	
	extensions[0] = (struct sadb_ext *) pfkey_msg;
	
	
	if(pfkey_msg->sadb_msg_version != PF_KEY_V2) {
		DEBUGGING(
			"pfkey_msg_parse: "
			"not PF_KEY_V2 msg, found %d, should be %d.\n",
			pfkey_msg->sadb_msg_version,
			PF_KEY_V2);
		SENDERR(EINVAL);
	}

	if(!pfkey_msg->sadb_msg_type) {
		DEBUGGING(
			"pfkey_msg_parse: msg type not set, must be non-zero..\n");
		SENDERR(EINVAL);
	}

	if(pfkey_msg->sadb_msg_type > SADB_MAX) {
		DEBUGGING(
			"pfkey_msg_parse: msg type=%d > max=%d.\n",
			pfkey_msg->sadb_msg_type,
			SADB_MAX);
		SENDERR(EINVAL);
	}

	switch(pfkey_msg->sadb_msg_type) {
	case SADB_GETSPI:
	case SADB_UPDATE:
	case SADB_ADD:
	case SADB_DELETE:
	case SADB_GET:
	case SADB_ACQUIRE:
	case SADB_REGISTER:
	case SADB_EXPIRE:
#if 0
	case SADB_X_PROMISC:
	case SADB_X_PCHANGE:
#endif
	case SADB_X_GRPSA:
		if(!pfkey_msg->sadb_msg_satype) {
			DEBUGGING(
				"pfkey_msg_parse: satype is zero, must be non-zero for msg_type %d.\n", pfkey_msg->sadb_msg_type);
			SENDERR(EINVAL);
		}
		switch(pfkey_msg->sadb_msg_satype) {
		case SADB_SATYPE_ESP:
		case SADB_SATYPE_AH:
		case SADB_X_SATYPE_IPIP:
			break;
		default:
			DEBUGGING(
				"pfkey_msg_parse: satype=%d is not supported yet.\n", pfkey_msg->sadb_msg_satype);
			SENDERR(EINVAL);
		}
		break;
	default:
	}
	
	DEBUGGING(
		"pfkey_msg_parse: remain=%d, ext_type=%d, ext_len=%d.\n", 
		remain, pfkey_ext->sadb_ext_type, pfkey_ext->sadb_ext_len);
	
	DEBUGGING(
		"pfkey_msg_parse: extensions "
		"permitted=%08x, required=%08x.\n",
		extensions_bitmaps[dir][EXT_BITS_PERM][pfkey_msg->sadb_msg_type],
		extensions_bitmaps[dir][EXT_BITS_REQ][pfkey_msg->sadb_msg_type]);
	
	extensions_seen =  1 ;
	
	while( (remain * IPSEC_PFKEYv2_ALIGN) >= sizeof(struct sadb_ext) ) {
		/* Is there enough message left to support another extension header? */
		if(remain < pfkey_ext->sadb_ext_len) {
			DEBUGGING(
				"pfkey_msg_parse: remain %d less than ext len %d.\n", 
				remain, pfkey_ext->sadb_ext_len);
			SENDERR(EINVAL);
		}
		
		DEBUGGING(
			"pfkey_msg_parse: parsing ext type=%d remain=%d.\n",
			pfkey_ext->sadb_ext_type,
			remain);
		
		/* Is the extension header type valid? */
		if((pfkey_ext->sadb_ext_type > SADB_EXT_MAX) || (!pfkey_ext->sadb_ext_type)) {
			DEBUGGING(
				"pfkey_msg_parse: ext type %d invalid, SADB_EXT_MAX=%d.\n", 
				pfkey_ext->sadb_ext_type, SADB_EXT_MAX);
			SENDERR(EINVAL);
		}
		
		/* Have we already seen this type of extension? */
		if((extensions_seen & ( 1 << pfkey_ext->sadb_ext_type )) != 0)
		{
			DEBUGGING(
				"pfkey_msg_parse: ext type %d already seen.\n", 
				pfkey_ext->sadb_ext_type);
			SENDERR(EINVAL);
		}
		
		/* Is this type of extension permitted for this type of message? */
		if(!(extensions_bitmaps[dir][EXT_BITS_PERM][pfkey_msg->sadb_msg_type] &
		     1<<pfkey_ext->sadb_ext_type)) {
			DEBUGGING(
				"pfkey_msg_parse: ext type %d not permitted, exts_perm_in=%08x, 1<<type=%08x\n", 
				pfkey_ext->sadb_ext_type, 
				extensions_bitmaps[dir][EXT_BITS_PERM][pfkey_msg->sadb_msg_type],
				1<<pfkey_ext->sadb_ext_type);
			SENDERR(EINVAL);
		}

		DEBUGGING(
			"pfkey_msg_parse: About to parse extension %d %p with parser %p.\n",
			pfkey_ext->sadb_ext_type,
			pfkey_ext,
			ext_parsers[pfkey_ext->sadb_ext_type]);
		/* Parse the extension */
		if((error = ext_parsers[pfkey_ext->sadb_ext_type](pfkey_ext))) {
			DEBUGGING(
				"pfkey_msg_parse: extension parsing for type %d failed with error %d.\n",
				pfkey_ext->sadb_ext_type, error); 
			SENDERR(-error);
		}
		DEBUGGING(
			"pfkey_msg_parse: Extension %d parsed.\n",
			pfkey_ext->sadb_ext_type);
		
		/* Mark that we have seen this extension and remember the header location */
		extensions_seen |= ( 1 << pfkey_ext->sadb_ext_type );
		extensions[pfkey_ext->sadb_ext_type] = pfkey_ext;
		
		/* Calculate how much message remains */
		remain -= pfkey_ext->sadb_ext_len;

		if(!remain) {
			break;
		}
		/* Find the next extension header */
		pfkey_ext = (struct sadb_ext*)((char*)pfkey_ext +
			pfkey_ext->sadb_ext_len * IPSEC_PFKEYv2_ALIGN);
	}

	if(remain) {
		DEBUGGING(
			"pfkey_msg_parse: unexpected remainder of %d.\n", 
			remain);
		/* why is there still something remaining? */
		SENDERR(EINVAL);
	}

	/* check required extensions */
	DEBUGGING(
		"pfkey_msg_parse: extensions "
		"permitted=%08x, seen=%08x, required=%08x.\n",
		extensions_bitmaps[dir][EXT_BITS_PERM][pfkey_msg->sadb_msg_type],
		extensions_seen,
		extensions_bitmaps[dir][EXT_BITS_REQ][pfkey_msg->sadb_msg_type]);
	
	if((extensions_seen &
	    extensions_bitmaps[dir][EXT_BITS_REQ][pfkey_msg->sadb_msg_type]) !=
	   extensions_bitmaps[dir][EXT_BITS_REQ][pfkey_msg->sadb_msg_type]) {
		DEBUGGING(
			"pfkey_msg_parse: required extensions missing:%08x.\n",
			extensions_bitmaps[dir][EXT_BITS_REQ][pfkey_msg->sadb_msg_type] -
			(extensions_seen &
			 extensions_bitmaps[dir][EXT_BITS_REQ][pfkey_msg->sadb_msg_type]));
		SENDERR(EINVAL);
	}
	
	if((((struct sadb_msg*)extensions[SADB_EXT_RESERVED])->sadb_msg_type == SADB_ADD) ||
	   (((struct sadb_msg*)extensions[SADB_EXT_RESERVED])->sadb_msg_type == SADB_UPDATE)) {

		/* check maturity */
		if(((struct sadb_sa*)extensions[SADB_EXT_SA])->sadb_sa_state !=
		   SADB_SASTATE_MATURE) {
			DEBUGGING(
				"pfkey_msg_parse: "
				"state=%d for add or update should be MATURE=%d.\n",
				((struct sadb_sa*)extensions[SADB_EXT_SA])->sadb_sa_state,
				SADB_SASTATE_MATURE);
			SENDERR(EINVAL);
		}
		
		/* check AH and ESP */
		if(((struct sadb_msg*)extensions[SADB_EXT_RESERVED])->sadb_msg_satype ==
		   SADB_SATYPE_AH) {
			if(!(((struct sadb_sa*)extensions[SADB_EXT_SA]) &&
			     ((struct sadb_sa*)extensions[SADB_EXT_SA])->sadb_sa_auth !=
			     SADB_AALG_NONE)) {
				DEBUGGING(
					"pfkey_msg_parse: "
					"auth alg is zero, must be non-zero for AH SAs.\n");
				SENDERR(EINVAL);
			}
			if(((struct sadb_sa*)(extensions[SADB_EXT_SA]))->sadb_sa_encrypt !=
			   SADB_EALG_NONE) {
				DEBUGGING(
					"pfkey_msg_parse: "
					"AH handed encalg=%d, must be zero.\n",
					((struct sadb_sa*)(extensions[SADB_EXT_SA]))->sadb_sa_encrypt);
				SENDERR(EINVAL);
			}
		}
		
		if(((struct sadb_msg*)extensions[SADB_EXT_RESERVED])->sadb_msg_satype ==
		   SADB_SATYPE_ESP) {
			if(!(((struct sadb_sa*)extensions[SADB_EXT_SA]) &&
			     ((struct sadb_sa*)extensions[SADB_EXT_SA])->sadb_sa_encrypt !=
			     SADB_EALG_NONE)) {
				DEBUGGING(
					"pfkey_msg_build: "
					"encrypt alg=%d is zero, must be non-zero for ESP=%d SAs.\n",
					((struct sadb_sa*)extensions[SADB_EXT_SA])->sadb_sa_encrypt,
					((struct sadb_msg*)extensions[SADB_EXT_RESERVED])->sadb_msg_satype);
				SENDERR(EINVAL);
			}
			if((((struct sadb_sa*)(extensions[SADB_EXT_SA]))->sadb_sa_encrypt ==
			    SADB_EALG_NULL) &&
			   (((struct sadb_sa*)(extensions[SADB_EXT_SA]))->sadb_sa_auth ==
			    SADB_AALG_NONE) ) {
				DEBUGGING(
					"pfkey_msg_parse: "
					"ESP handed encNULL+authNONE, illegal combination.\n");
				SENDERR(EINVAL);
			}
		}
	}
		
errlab:

	return error;
}

/*
 * $Log: pfkey_v2_parse.c,v $
 * Revision 1.10  1999/12/10 17:45:24  rgb
 * Added address debugging.
 *
 * Revision 1.9  1999/12/09 23:11:42  rgb
 * Ditched <string.h> include since we no longer use memset().
 * Use new pfkey_extensions_init() instead of memset().
 * Added check for SATYPE in pfkey_msg_build().
 * Tidy up comments and debugging comments.
 *
 * Revision 1.8  1999/12/07 19:55:26  rgb
 * Removed unused first argument from extension parsers.
 * Removed static pluto debug flag.
 * Moved message type and state checking to pfkey_msg_parse().
 * Changed print[fk] type from lx to x to quiet compiler.
 * Removed redundant remain check.
 * Changed __u* types to uint* to avoid use of asm/types.h and
 * sys/types.h in userspace code.
 *
 * Revision 1.7  1999/12/01 22:20:51  rgb
 * Moved pfkey_lib_debug variable into the library.
 * Added pfkey version check into header parsing.
 * Added check for SATYPE only for those extensions that require a
 * non-zero value.
 *
 * Revision 1.6  1999/11/27 11:58:05  rgb
 * Added ipv6 headers.
 * Moved sadb_satype2proto protocol lookup table from
 * klips/net/ipsec/pfkey_v2_parser.c.
 * Enable lifetime_current checking.
 * Debugging error messages added.
 * Add argument to pfkey_msg_parse() for direction.
 * Consolidated the 4 1-d extension bitmap arrays into one 4-d array.
 * Add CVS log entry to bottom of file.
 * Moved auth and enc alg check to pfkey_msg_parse().
 * Enable accidentally disabled spirange parsing.
 * Moved protocol/algorithm checks from klips/net/ipsec/pfkey_v2_parser.c
 *
 */
