/**
*  Class Safer - The SAFER cypher in all its variants.
*  <P>
*  Coded Mr. Tines &lt;tines@windsong.demon.co.uk&gt; 1998
*  and released into the public domain
*  <P>
* THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
* BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
* WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*  <P>
* @author Mr. Tines
* @version 1.0 23-Dec-1998
*/

package uk.co.demon.windsong.crypt.cea;

public class Safer
{
    /**
    * maximum number of rounds (this value by default)
    */
    private static final int MAXROUNDS  = 13;
    /**
    * block size in bytes
    */
    private static final int SAFERBLOCKSIZE = 8;
    /**
    * log table look-up array size
    */
    private static final int LOGTABSIZE = 256;

    /**
    * Slices the low byte value out of an int
    * @param v value to slice
    * @return the low byte
    */
    private static final int BYTE(int v)
    {
        return v & 0xFF;
    }

    /**
    * Rotates the low byte
    * @param v value to rotate
    * @param n number of bits of rotation
    * @return the rotated byte
    */
    private static final byte ROTL(int v, int n)
    {
        return (byte)BYTE( ((v)<<(n)) | ( ((int)BYTE(v)) >>> (8-(n))) );
    }

    /**
    * anti-log table
    */
    private static byte[] exp = new byte[LOGTABSIZE];
    /**
    * log table
    */
    private static byte[] log = new byte[LOGTABSIZE];
    /**
    * initialisation tracking
    */
    private static boolean logTabSet = false;

    /**
    * set up log and anti-log tables
    */
    private static void setLogTable()
    {
	    int l, e;
        for(l=0, e=1; !logTabSet && l < LOGTABSIZE; ++l)
        {
   	        exp[l] = (byte)BYTE(e);
            log[exp[l]&0xFF] = (byte)BYTE(l);
            e = (e*45) % 257;
        }
        logTabSet = true;
    }

    /**
    * round functions
    */
    private static final void ADD(byte[] w, int a, int b)
        {w[b]+=w[a]; w[a]+=w[b];}
    private static final void SUB(byte[] w, int a, int b)
        {w[a]-=w[b]; w[b]-=w[a];}

    /**
    * size of full key schedule
    */
    private static final int EXPANDEDKEY =
        (1 + SAFERBLOCKSIZE * ( 2*MAXROUNDS + 1));

    /**
    * the key schedule
    */
    private class Keyschedule
    {
        /**
        * the full key schedule bytes
        */
	    byte[] k = new byte[EXPANDEDKEY];

        /**
        * wipes the whole keyschedule
        */
        void destroy()
        {
            if(k != null)
            {
                for(int i=0; i<k.length; ++i) k[i] = 0;
            }
            k = null;
        }

        /**
        * performs forward operation
        * @param in the data to encrypt
        * @param offin the start of the input
        * @param out the result
        * @param offout the start of the output
        */
        void SAFERencrypt(byte[] in, int offin, byte[] out, int offout)
        {
            byte tmp;
            byte[] work = new byte[SAFERBLOCKSIZE];
            int round = Math.min(MAXROUNDS, k[0]);
            int ko = 0;
            System.arraycopy(in, offin, work, 0, work.length);

            while(round-- != 0)
            {
                /* part 1, simple mixing with key */
		        work[0] ^= k[++ko]; work[1] += k[++ko];
		        work[2] += k[++ko]; work[3] ^= k[++ko];
		        work[4] ^= k[++ko]; work[5] += k[++ko];
		        work[6] += k[++ko]; work[7] ^= k[++ko];

                /* part 2, logarithms in the group */
                work[0] = (byte)(exp[work[0]&0xFF] + k[++ko]);
                work[1] = (byte)(log[work[1]&0xFF] ^ k[++ko]);
		        work[2] = (byte)(log[work[2]&0xFF] ^ k[++ko]);
                work[3] = (byte)(exp[work[3]&0xFF] + k[++ko]);
		        work[4] = (byte)(exp[work[4]&0xFF] + k[++ko]);
                work[5] = (byte)(log[work[5]&0xFF] ^ k[++ko]);
		        work[6] = (byte)(log[work[6]&0xFF] ^ k[++ko]);
                work[7] = (byte)(exp[work[7]&0xFF] + k[++ko]);

                /* part 3, mutual increment */
                ADD(work, 0, 1);
                ADD(work, 2, 3);
                ADD(work, 4, 5);
                ADD(work, 6, 7);

                ADD(work, 0, 2);
                ADD(work, 1, 3);
                ADD(work, 4, 6);
                ADD(work, 5, 7);

                ADD(work, 0, 4);
                ADD(work, 1, 5);
                ADD(work, 2, 6);
                ADD(work, 3, 7);

                /* part 4, permute */
                tmp     = work[1];
                work[1] = work[4];
                work[4] = work[2];
                work[2] = tmp;
                tmp     = work[3];
                work[3] = work[5];
                work[5] = work[6];
                work[6] = tmp;
            } /* end of round */

            /* mix to output */
	        out[0+offout] = (byte)(work[0] ^ k[++ko]);
            out[1+offout] = (byte)(work[1] + k[++ko]);
	        out[2+offout] = (byte)(work[2] + k[++ko]);
            out[3+offout] = (byte)(work[3] ^ k[++ko]);
            out[4+offout] = (byte)(work[4] ^ k[++ko]);
            out[5+offout] = (byte)(work[5] + k[++ko]);
	        out[6+offout] = (byte)(work[6] + k[++ko]);
            out[7+offout] = (byte)(work[7] ^ k[++ko]);
            tmp = 0;
            for(int w=0; w<SAFERBLOCKSIZE; ++w) work[w] = 0;
        }

        /**
        * performs return operation
        * @param in the data to decrypt
        * @param offin the start of the input
        * @param out the result
        * @param offout the start of the output
        */
        void SAFERdecrypt(byte[] in, int offin, byte[] out, int offout)
        {
            byte[] work = new byte[SAFERBLOCKSIZE];
            byte tmp;
            int round = Math.min(MAXROUNDS,  k[0]);
            int ko = (2*round+1)*SAFERBLOCKSIZE;

            /* undo mix to output */
	        work[7] = (byte)(in[7+offin] ^ k[ko--]);
            work[6] = (byte)(in[6+offin] - k[ko--]);
	        work[5] = (byte)(in[5+offin] - k[ko--]);
            work[4] = (byte)(in[4+offin] ^ k[ko--]);
	        work[3] = (byte)(in[3+offin] ^ k[ko--]);
            work[2] = (byte)(in[2+offin] - k[ko--]);
	        work[1] = (byte)(in[1+offin] - k[ko--]);
            work[0] = (byte)(in[0+offin] ^ k[ko--]);

            while(round-- != 0)
            {
                /* undo part 4, permute */
                tmp     = work[2];
                work[2] = work[4];
                work[4] = work[1];
                work[1] = tmp;
                tmp     = work[6];
                work[6] = work[5];
                work[5] = work[3];
                work[3] = tmp;

                /* undo part 3, mutual increment */
                SUB(work, 0, 4);
                SUB(work, 1, 5);
                SUB(work, 2, 6);
                SUB(work, 3, 7);

                SUB(work, 0, 2);
                SUB(work, 1, 3);
                SUB(work, 4, 6);
                SUB(work, 5, 7);

                SUB(work, 0, 1);
                SUB(work, 2, 3);
                SUB(work, 4, 5);
                SUB(work, 6, 7);

                /* undo parts 1&2, mixing and logarithms in the group */
                work[7] -= k[ko--]; work[6] ^= k[ko--];
                work[5] ^= k[ko--]; work[4] -= k[ko--];
                work[3] -= k[ko--]; work[2] ^= k[ko--];
                work[1] ^= k[ko--]; work[0] -= k[ko--];

		        work[7] = (byte)(log[work[7]&0xFF] ^ k[ko--]);
                work[6] = (byte)(exp[work[6]&0xFF] - k[ko--]);
		        work[5] = (byte)(exp[work[5]&0xFF] - k[ko--]);
                work[4] = (byte)(log[work[4]&0xFF] ^ k[ko--]);
		        work[3] = (byte)(log[work[3]&0xFF] ^ k[ko--]);
                work[2] = (byte)(exp[work[2]&0xFF] - k[ko--]);
		        work[1] = (byte)(exp[work[1]&0xFF] - k[ko--]);
                work[0] = (byte)(log[work[0]&0xFF] ^ k[ko--]);

            } /* end of round */

            /* copy to output */
            System.arraycopy(work, 0, out, offout, work.length);
            for(int w=0; w<SAFERBLOCKSIZE; ++w) work[w] = 0;
        }

        /**
        * Initialises the key schedult
        * @param key1 the first 64 bits of key
        * @param key1off the start of the key
        * @param key2 the second 64 bits of key
        * @param key2off the start of the key
        * @param rounds the number of rounds to use
        * @param skVariant signals to use the strenthened version
        */
        Keyschedule(byte[] key1, int key1off, byte[] key2, int key2off,
            int rounds, boolean skVariant)
        {
	        int i;
	        byte[] worka = new byte[1+SAFERBLOCKSIZE];
            byte[] workb = new byte[1+SAFERBLOCKSIZE];
            int ko = 0;

            k[ko++] = (byte)Math.min(MAXROUNDS, rounds);
            worka[SAFERBLOCKSIZE] = workb[SAFERBLOCKSIZE] = 0;
            for(i=0; i<SAFERBLOCKSIZE; ++i)
            {
                worka[SAFERBLOCKSIZE] ^= worka[i] = ROTL(key1[i+key1off], 5);
   	            workb[SAFERBLOCKSIZE] ^= workb[i] = k[ko++] = key2[i+key2off];
            }

            if(!logTabSet) setLogTable();

            for (i=1; i<=k[0]; ++i)
            {
   	            int j;
                for(j=0; j<SAFERBLOCKSIZE+1; ++j)
                {
      	            worka[j] = ROTL(worka[j], 6);
      	            workb[j] = ROTL(workb[j], 6);
                }
                for(j=0; j<SAFERBLOCKSIZE; ++j)
                {
      	            if(skVariant)
                    {
         	            k[ko++] = (byte)BYTE(
                            worka[ (0xFF&(j + 2*i -1)) % (1+SAFERBLOCKSIZE)]
                                +exp[exp[0xFF&BYTE(18 * i + j + 1)]&0xFF]
                        );
                    }
                    else
                    {
         	            k[ko++] = (byte)BYTE( worka[j]
                            +exp[exp[0xFF&BYTE(18 * i + j + 1)]&0xFF] );
			        }
                }
                for(j=0; j<SAFERBLOCKSIZE; ++j)
                {
      	            if(skVariant)
                    {
         	            k[ko++] = (byte)BYTE(
                            workb[ (0xFF&(j + 2*i)) % (1+SAFERBLOCKSIZE)]
                                +exp[exp[0xFF&BYTE(18 * i + j + 10)]&0xFF]
                        );
                    }
                    else
                    {
         	            k[ko++] = (byte)BYTE( workb[j]
                            +exp[exp[0xFF&BYTE(18 * i + j + 10)]&0xFF] );
			        }
                }
            } /* next i */
            for(i=0; i<= SAFERBLOCKSIZE; ++i)
            {
                worka[i] = 0;
                workb[i] = 0;
            }
        }

    }// end inner class

    /**
    * defined default round counts
    */
    private static final int[] defaultRounds = {6, 10, 8, 10};

    /**
    * the key size in bytes
    */
    private final int keysize;

    /**
    * the variant to use
    */
    private final boolean skVariant;

    /**
    * number of rounds chosen
    */
    private final int rounds;

    /**
    * default constructor SAFER-SK128 w/13 rounds
    */
    public Safer()
    {
        this(true, true, MAXROUNDS);
    }

    /**
    * Complex constructor
    * @param fullkeylen true for 128 bit key, false for 64 bit
    * @param strong true for SK variant
    * @param roundCount number of rounds to use
    */
    public Safer(boolean fullkeylen, boolean strong, int roundCount)
    {
        keysize = fullkeylen ? 16 : 8;
        skVariant = strong;

        int minr = 6;
        //if(strong) minr = 8;
        if(fullkeylen) minr = 10;

        if(roundCount < minr) roundCount = minr;
        if(roundCount > MAXROUNDS) roundCount = MAXROUNDS;
        rounds =  roundCount;
    }

    /**
    * Is the jacket doing triple encryption?
    */
    boolean triple = false;
    /**
    * The key schedule data
    */
    Keyschedule[] ks = null;

    /**
    * Initialise the object with one or three key blocks
    * @param key array of key bytes, 1 or 3 key block lengths
    * @param triple true if three keys for triple application
    */
    public void init(byte[] key, int keyoffset, boolean triple)
    {
        this.triple = triple;
	    int keys = triple ? 3 : 1;
	    int i;
        int delta = (getKeysize() > 8) ? 8 : 0;

        ks = new Keyschedule[keys];
	    for(i=0; i < keys; i++)
	    {
            ks[i] = new Keyschedule(key, keyoffset+i*getKeysize(),
                key, keyoffset+delta+i*getKeysize(), rounds, skVariant);
	    }
    }

    /**
    * Transform one block in ecb mode
    * @param encrypt true if forwards transformation
    * @param in input block
    * @param offin offset into block of input data
    * @param out output block
    * @param offout offset into block of output data
    */
    public final void ecb(boolean encrypt, byte[] in, int offin,
        byte[] out, int offout)
    {
        if(!triple)
        {
   	        if(encrypt) ks[0].SAFERencrypt(in, offin, out, offout);
            else ks[0].SAFERdecrypt(in, offin, out, offout);
        }
        else
        {
   	        byte[] tmp = new byte[SAFERBLOCKSIZE];
            byte[] tmp2 = new byte[SAFERBLOCKSIZE];
            if(encrypt)
            {
      	        ks[0].SAFERencrypt(in, offin, tmp, 0);
			    ks[1].SAFERencrypt(tmp, 0, tmp2, 0);
			    ks[0].SAFERencrypt(tmp2, 0, out, offout);
            }
            else
            {
      	        ks[0].SAFERdecrypt(in, offin, tmp, 0);
			    ks[1].SAFERdecrypt(tmp, 0, tmp2, 0);
			    ks[0].SAFERdecrypt(tmp2, 0, out, offout);
            }
            for(int i=0; i<SAFERBLOCKSIZE;++i)
            {
                tmp[i] = 0;
                tmp2[i] = 0;
            }
        }
    }

    /**
    * Wipe key schedule information
    */
    public final void destroy()
    {
        if(ks != null)
        {
            for(int i=0; i<ks.length; ++i)
            {
                ks[i].destroy();
                ks[i] = null;
            }
            ks = null;
        }
        triple = false;
    }

    /**
    * Provide infomation of desired key size
    * @return byte length of key
    */
    public int getKeysize()
    {
        return keysize;
    }

    /**
    * Provide infomation of algorithm block size
    * @return byte length of block
    */
    public int getBlocksize()
    {
        return SAFERBLOCKSIZE;
    }


    /**
    * drives the test vectors
    */
    public static void main(String[] args)
    {
        test();
    }

    /**
    * drives the test vectors
    */
    static void test()
    {
        int[] round = {6,6,10,10,10};
        // some standard test vectors
        byte[][] keys = {
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
              0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 },
            { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	          0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 },
            { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
	          0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
            { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	          0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 }};
        byte[][] plain = {
            { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 },
            { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 },
            { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 },
            { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 },
            { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 }};
        byte[][] cipher = {
            { 0x15, 0x1B, (byte)0xFF, 0x02, (byte)0xAD, 0x11, (byte)0xBF, 0x2D },
            { 0x5F, (byte)0xCE, (byte)0x9B, (byte)0xA2, 0x05, (byte)0x84, 0x38, (byte)0xC7 },
            { 0x41, 0x4C, 0x54, 0x5A, (byte)0xB6, (byte)0x99, 0x4A, (byte)0xF7 },
            { (byte)0xFF, 0x78, 0x11, (byte)0xE4, (byte)0xB3, (byte)0xA7, 0x2E, 0x71 },
            { 0x49, (byte)0xC9, (byte)0x9D, (byte)0x98, (byte)0xA5, (byte)0xBC, 0x59, 0x08 }};


        byte[] work = new byte[SAFERBLOCKSIZE];
        byte[] work2 = new byte[SAFERBLOCKSIZE];
        int i,j;
        setLogTable();


        for(j=0; j<5; ++j)
        {
            Safer s = new Safer(j>1, true, round[j]);
            s.init(keys[j], 0, false);
            s.ecb(true, plain[j], 0, work, 0);

	        for(i=0; i<8; ++i)
	        {
	            System.out.print(" "+
                    Integer.toHexString(0xFF&cipher[j][i]));
	        }
	        System.out.println(" is expected");
	        for(i=0; i<8; ++i)
	        {
	            System.out.print(" "+
                    Integer.toHexString(0xFF&work[i]));
	        }
	        System.out.println(" is generated");

	        //s.ecb(false, cipher[j], 0, work, 0);
	        s.ecb(false, work, 0, work2, 0);
	        for(i=0; i<8; ++i)
	        {
	            System.out.print(" "+
                    Integer.toHexString(0xFF&plain[j][i]));
	        }
	        System.out.println(" is expected");
	        for(i=0; i<8; ++i)
	        {
	            System.out.print(" "+
                    Integer.toHexString(0xFF&work2[i]));
	        }
	        System.out.println(" is generated");
            System.out.println("");
        }
    }

}
 