/*
 *	desgen.c - compiles DES permutation and substitution tables into 
 *		functions which may be invoked by the DES library routines for
 *		efficiency.  Each permutation table is character array 
 *		specifying which des input bit is permuted to that output bit.
 *		The generated file is used in the des64.c library source
 *
 *
 *  Author:	Lawrence Brown <lpb@csadfa.oz.au>		Dec 1987
 *		Computer Science, UC UNSW, Australian Defence Force Academy,
 *			Canberra, ACT 2600, Australia.
 *
 *		The user interface and some code has been inspired by the PD
 *		programs of:	Phil Karn, KA9Q <karn@flash.bellcore.com> 1986
 *		  and		Jim Gillogly 1977.
 *		and by 		Michael Barr-David's honours DES program 1987
 *
 *  Modifications:
 *		v2.0 - compile permutation matrices into functions, lpb 12/87
 *		v2.2 - compile S-boxes into table with natural index, lpb 12/87
 *		v2.3 - compile permutations to macros for efficiency, lpb 12/87
 *		v3.0 - compile combined SP operation into a table, lpb 1/88
 *		v3.2 - cast param types explicitly from descriptors, lpb 7/89
 *		v4.0 - parse flags on sp_desc, compile in E with perms, lpb 7/89
 *
 *  Copyright 1989 by Lawrence Brown and CITRAD. All rights reserved.
 *      This program may not be sold or used as inducement to buy a
 *      product without the written permission of the author.
 *
 */

#ifndef lint
static char rcsid[]  = "$Header: desgen.c,v 2.1 90/07/18 10:56:23 lpb Exp $";
#endif
#ifndef DES_TABLES
#define DES_TABLES "des64.i"	/* file containing DES tables, may be overridden on cc command line */
#endif
#ifndef GEN_TABLES
#define GEN_TABLES "gen64.i"	/* file containing generated DES tables, may be overridden on cc command line */
#endif

#include	<stdio.h>
#include	"des.h"		/* generic DES typedefs                      */
#include	"despriv.h"	/* private DES typedefs                      */
#include	"desgen.h"	/* desgen specific typedefs                  */

#include	DES_TABLES	/* DES Permutation & Substitution Tables     */


char	*Name = "";		/* name this program was called by */
FILE	*fileout;		/* output file to be processed                */
char	*genfile = GEN_TABLES;	/* name of generated output file             */
char	*genvers = "desgen v4.0";	/* version number of this program    */

main(argc, argv)
int	argc;
char	**argv;
{
	int	i;
	char	*t, *ctime();
	long	it, time();

	Name = argv[0];
	if (argc > 1) {			/* output file name supplied */
		genfile = argv[1];
		if ((fileout = fopen(genfile, "w")) == NULL) {
			fprintf(stderr,"%s: unable to open output file %s\n",
				Name, genfile);
			perror(Name);
			exit(1);
		}
	} else {
		fileout = stdout;	/* otherwise use stdout for output */
	}
	
	time(&it);
	t = ctime(&it);
	fprintf(fileout, "/*\n");
	fprintf(fileout, " *\t%s - compiled DES permutations and substitution tables\n",
			genfile);
	fprintf(fileout, " *\t\tAutomatically generated by %s from %s\n", 
			genvers, DES_TABLES);
	fprintf(fileout, " *\t\t  of form %s\n", des64_vers);
	fprintf(fileout, " *\t\t  on %s", t);
	fprintf(fileout, " *\t\t  for use in the DES library routines\n");
	fprintf(fileout, " *\t\tCopyright (C) 1989 by Lawrence Brown and CITRAD\n");
	fprintf(fileout, " *\n */\n\n");
	fprintf(fileout, "#include\t\"bits.h\"\n\n");	/* include bit posn defns    */

	for (i = 0; perms[i].fn != 0; i++) {	/* Step through perm tables  */
		build_perm(&(perms[i]));	/* compile next perm         */
	}
	build_spboxes(spboxes);			/* build SP-boxes table      */
	build_keysched();			/* output key schedule       */
	exit(0);
}

/*
 *	build_perm(d) - compiles the permutation specified by d
 *
 *		nb: all bit references start from 1, and must be offset
 *			to 0 when indexing arrays
 */
build_perm(d)
sp_desc	*d;	/* descriptor for a permutation                      */
{
	char	*perm = d->perm; /* pointer to next slot in perm table       */
	int	i, j, k;	/* current output word, offset, and DES bit  */
	int	b, w, o;	/* current input DES bit, word, bit offset   */
	int	s;		/* shift reqd from input to output offset    */

	fprintf(fileout, "#define %s(", d->fn);	/* declare macro     */
	if (d->flags & O_REG)		/* output var either set of vars     */
	    for (i=0; i<d->owords; i++)
		fprintf(fileout, "out%d,", i);
	else				/* or an array                       */
		fprintf(fileout, "out,");
	if (d->flags & I_REG) {		/*  input var either set of vars     */
	    for (i=0; i<d->iwords-1; i++)
		fprintf(fileout, "in%d,", i);
	    fprintf(fileout, "in%d)\t{\\\n", d->iwords-1);
	} else				/* or an array                       */
		fprintf(fileout, "in)\t{\\\n");

	/* if 2 or less input words, and input not reg vars, use registers   */
	if (!(d->flags & I_REG) && (d->iwords <= 2)) {
	    fprintf(fileout, "\tregister %s\t", d->itype);
	    for (i=0; i<d->iwords; i++)	/* save input words in register vars */
		fprintf(fileout, "in%d = ((%s *)in)[%d]%s",
			i, d->itype, i, (i < d->iwords - 1 ? ", " : ";\\\n"));
	}

	/* build expressions for each output word    */
	k = 1;			/* start with first DES output bit           */
	for (i=1; i<=d->owords; i++) {	/* for word in the output array      */
		if (d->flags & O_REG)
			fprintf(fileout, "\tout%d = \\\n", i-1);
		else
			fprintf(fileout, "\t((%s *)out)[%d] = \\\n", d->otype, i-1);
		for (j=1; j<=d->obits; j++) {	/* for each DES output bit   */
				/* find which input word & off set in word   */
				/*   bit permuted to this output loc is      */
			b = perm[k-1] - 1;	/* get input DES bit by perm */
			w = b / d->ibits;	/* find input word b is in   */
			o = d->imsb - (b % d->ibits);	/* find offset to b  */
				/* & find left shift reqd to this output bit */
			s = (d->omsb - (j - 1)) - o;
				/* write this component of expression        */
			/* if in uses register vars  */
			if ((d->flags & I_REG) || (d->iwords <= 2))
			    fprintf(fileout, "\t\t(in%d & BIT%d) %s %d",
				w, o, (s<0 ? ">>" : "<<"), (s<0 ? -s : s));
			else		/* else explicitly cast in array ref */
			    fprintf(fileout, "\t\t(((%s *)in)[%d] & BIT%d) %s %d",
				d->itype, w, o,
				(s<0 ? ">>" : "<<"), (s<0 ? -s : s));
			if (j != d->obits)	/* end line with | or ;      */
				fprintf(fileout, " | \\\n");
			else
				fprintf(fileout, " ; \\\n");
			k++;			/* increment output bit no  */
		}
	}
	fprintf(fileout, "}\n\f\n\n");
}


/*
 *	build_spboxes(d) - compiles the S-boxes & permutation P described by d
 */
build_spboxes(d)
sp_desc	*d;	/* descriptors for S-boxes & permutation P                   */
{
	sp_desc	*sd = &(d[0]);		/* descriptor for S-boxes & SP output*/
	sp_desc	*pd = &(d[1]);		/* descriptor for permutation P      */
	sp_desc	*ed = &(d[2]);		/* descriptor for permutation E      */
	int	bpw = (sd->obits / 4);	/* no of S-box outputs per word      */
	int	i, j;
	int	w, o;			/* which word, bit offset            */
	Long	rc, s, sp, spe[2];	/* S-box row-col, S-box, SP val, SPE */
	Long	ssize = 1 << sd->ibits;	/* size of array for each S-box      */

	if (pd->flags & O_PE)		/* if have merged S.P.E double size  */
		ssize *= 2;
	
	fprintf(fileout, "%s %s[%d][%d] = {\n", sd->otype, sd->fn, sd->iwords, ssize);

	for (i=0; i<sd->iwords; i++) {	/* for each S-box                    */
		fprintf(fileout, "\t\t\t\t /* S[%d] \t\t */\n", (i+1));
		w = i / bpw;		/* which output word & offset        */
		o = sd->obits - ((i % bpw) * 4) - 4;
		for (j=0; j<64; j++) {	/* for each S-Box input value        */
				/* build this component of expression         */
			rc =	(j & 0x20)  |		/* reorder S-box index*/
				((j << 4) & 0x10)  |	/*   bits 1,6 are row */
				((j >> 1) & 0x0F);	/*   bits 2-5 are col */
			s = S[i][rc] << o;		/* get S-box output   */
			perm(&sp, &s, pd);		/* and permutate by P */
			if (pd->flags & O_PE) {		/* if have S.P.E      */
				perm(spe, &sp, ed);	/* permute sp val by E*/
				fprintf(fileout, " 0x%08x, 0x%08x,", spe[0], spe[1]);
			} else {
				fprintf(fileout, " 0x%08x,", sp);	/* write output expr  */
			}
			if (((j + 1) % 4) == 0)
				fprintf(fileout, "\n");
		}
	}
	fprintf(fileout, "};\n\n\n");
}



/*
 *	build_keysched() - outputs the key schedule tables - simply copies defn
 */
build_keysched()
{
	register int i;

	fprintf(fileout, "char keyrot[ROUNDS] = {");
	for (i=0; i< ROUNDS; i++) 
		fprintf(fileout, "%d%s",	(int)keyrot[i],
				((i < ROUNDS-1) ? ", " : " };\n\n"));
}


/*
 *	perm(out, in, pd) is the general permutation of an input
 *		block to an output block, under the control of a permutation 
 *		described by pd. Each element of array pd->perm specifies 
 *		which input bit is to be permuted to the output bit with 
 *		the same index as the array element.
 *
 *	nb: to set bits in the output word, a mask with a single 1 in it is
 *		used. On each step, the 1 is shifted into the next location
 */

perm(out, in , pd)
Long	*out;		/* Output block to be permuted                       */
Long	*in;		/* Input  block after permutation                    */
sp_desc	*pd;		/* Permutation descriptor structure                  */
{
	Long	mask;			/* mask used to set bit in output    */
	char	*p = pd->perm;		/* permutation array pointer         */
	register int	i, j;

	for (j=0; j < pd->owords; j++) {
		out[j] = 0L;			/* Clear output word         */
		mask = 1 << pd->omsb;		/* mask starts with MSB set  */
		for (i=0; i<pd->obits; i++) {	/* For each bit position     */	
			if (bit(in, (int)*p++, pd) == 1) /* If input bit set */	
				out[j] |= mask;		/*  set output bit   */
			mask >>= 1;			/* Shift mask        */
		}
	}
}


/*
 *	bit(n, pos, pd) return bit at position pos in n (1 <= pos )
 *		of input word(s) used by permutation descriptor pd
 */
int
bit(n, pos, pd)
Long	*n;
int	pos;
sp_desc	*pd;		/* Permutation descriptor structure                  */
{
	register int	b, d, o;

	pos--;
	d = pos / pd->ibits;		/* calculate which half the bit is in */
	o = pd->imsb - (pos % pd->ibits); /* calculate offset within the half */
	b = ((n[d] >> o) & 01);		/* shift wanted bit to LSB, mask it   */
	return (b);			/* and return it                      */
}

