//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : bigfloat_lattice.h 
// Author      : Werner Backes (WB), Thorsten Lauer (TL) 
// Last change : WB/TL, Feb 12 1995, initial version
//

#ifndef _bigfloat_lattice_h
#define _bigfloat_lattice_h

#include <LiDIA/bigfloat.h>
#include <LiDIA/timer.h>

class lattice_gensys;
class lattice_basis;

class bigfloat_lattice 
{
  private :
    friend      class lattice_gensys;
    friend      class lattice_basis;
    long 	rows;				// Number of Zeilen
    long 	cols;				// Number of Spalten
    bigfloat**	value;				// Values column by column

    long 	cols_rank;			// rank of the lattice
						// for longernal use

//
// static variables for temporary use to increase speed
//
    static bigfloat tempvz;
    static bigfloat tempmz0;
    static bigfloat tempmz1;
    static bigfloat tempmz2;
    static bigfloat tempmz3;
    static bigfloat tempmz4;
    static bigfloat ergmz;

    static long vectsize;

    static long tempvectsize;
    static bigfloat* tempvect0;    
    static bigfloat* tempvect1;    

    static long last;

  public :

//
// Constructor / Destructors 
//
    bigfloat_lattice(long, long);
    bigfloat_lattice(long, long, bigfloat**);
    bigfloat_lattice(const bigfloat_lattice&);
    ~bigfloat_lattice();    

//  
// Input / Output
//

#ifndef __GNUC__
    friend istream& operator >> (istream&, lattice_gensys&);
#endif

    friend istream& operator >> (istream&, bigfloat_lattice&);
    friend ostream& operator << (ostream&, const bigfloat_lattice&);

//
// Assignments
//

    bigfloat_lattice operator = (const bigfloat_lattice&);
    void assign(const bigfloat_lattice&);
    void assign_zero();

//
// Algorithms
//

    void lll(bigfloat_lattice&, const bigfloat&, 
             double, unsigned long&);   
    void lll_trans(bigfloat_lattice&, bigfloat_lattice&, 
                   const bigfloat&, double, unsigned long&);   
    void mlll(bigfloat*, unsigned long&, timer&);
    void lin_gen_system(bigfloat_lattice&, long, double,
                        unsigned long&, timer&, long&);
    void compute_basis(const bigfloat_lattice&, const bigfloat_lattice&);
    long gram_schmidt_orth(bigfloat_lattice&, bigfloat_lattice&, long);
    int lll_check(double, long);
    double lll_check(long);
    int is_reduced();

  protected :
    
//
// "Vector" - Operations
//

#ifdef __GNUC__

    friend void assign(bigfloat* a, bigfloat* b)
    {
      debug_handler("bigfloat_lattice","friend void assign(a, b)");
      long i;
      for (i=bigfloat_lattice::vectsize-1;i>=0;i--)
        a[i].assign(b[i]);
    }

    friend void assign_zero(bigfloat* a)
    {
      debug_handler("bigfloat_lattice","friend void assign_zero(a)");
      long i;
      for (i=bigfloat_lattice::vectsize-1;i>=0;i--)
        a[i].assign_zero();
    } 

    friend void add(bigfloat* c, bigfloat* a, bigfloat* b)
    {
      debug_handler("bigfloat_lattice","friend void add(c, a, b)");
      long i;
      for (i=bigfloat_lattice::vectsize-1;i>=0;i--)
        add(c[i],a[i],b[i]);
    }

    friend void subtract(bigfloat* c, bigfloat* a, bigfloat* b)
    {
      debug_handler("bigfloat_lattice","friend void subtract(c, a, b)");
      long i;
      for (i=bigfloat_lattice::vectsize-1;i>=0;i--)
        subtract(c[i],a[i],b[i]);
    }

   friend void scalmul(bigfloat* b, const bigfloat& d, bigfloat* a)
    {
      debug_handler("bigfloat_lattice","friend scalmul(b, d, a)");
      long i;
      for (i=bigfloat_lattice::vectsize-1;i>=0;i--)
        multiply(b[i], a[i], d);
    }

    friend void scalprod(bigfloat& res, bigfloat* a, bigfloat* b)
    {
      debug_handler("bigfloat_lattice","friend scalprod(res, a, b)");
      long i;
      res.assign_zero();
      for (i=bigfloat_lattice::vectsize-1;i>=0;i--) 
        {
          multiply(bigfloat_lattice::tempvz, a[i], b[i]);
          add(res, res, bigfloat_lattice::tempvz);
        }
    }

    friend void l2_norm(bigfloat& norm, bigfloat* v)
    {
      debug_handler("bigfloat_lattice","l2_norm(norm, v)");
      long i;
      norm.assign_zero();		// Initialisation
      for (i=bigfloat_lattice::vectsize-1;i>=0;i--) 
        {
          square(bigfloat_lattice::tempvz,v[i]);
          add(norm,norm,bigfloat_lattice::tempvz);
        }
      sqrt(norm, norm);
    }

    friend void l1_norm(bigfloat& norm, bigfloat* v)
    {
      debug_handler("bigfloat_lattice","l1_norm(norm, v)");
      long i;
      norm.assign_zero();	 // Initialisation
      for (i=bigfloat_lattice::vectsize-1;i>=0;i--) 
        {
          bigfloat_lattice::tempvz.assign(abs(v[i]));
          add(norm,norm,bigfloat_lattice::tempvz);
        }
    }

#else

    friend void assign(bigfloat*, bigfloat*); 
    friend void assign_zero(bigfloat*);
    friend void add(bigfloat*, bigfloat*, bigfloat*);
    friend void subtract(bigfloat*, bigfloat*, bigfloat*);

    friend void scalmul(bigfloat*, const bigfloat&, bigfloat*);
    friend void scalprod(bigfloat&, bigfloat*, bigfloat*);
    
    friend void l1_norm(bigfloat&, bigfloat*);
    friend void l2_norm(bigfloat&, bigfloat*);

#endif

    friend void swap(bigfloat*& a, bigfloat*& b)
    {
      register bigfloat* temp;
      temp=a;
      a=b;
      b=temp;
    }


//
//  Algorithm - Subroutines
//
    
    void alpha_compute(bigfloat&);
    void gamma_compute(bigfloat&, long);
    void zwei_pot_q_compute(bigfloat&, long&, bigfloat&);
    void cholesky(bigfloat&);

};

#endif


