//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : double_lattice.h 
// Author      : Werner Backes (WB), Thorsten Lauer (TL) 
// Last change : WB/TL, Feb 12 1995, initial version
//

#ifndef _double_lattice_h
#define _double_lattice_h

#include <LiDIA/timer.h>

#ifndef rint
#define rint(x) (((x)-floor(x))<0.5?floor(x):ceil(x)) 
#endif

class lattice_gensys;
class lattice_basis;

class double_lattice 
{
  private :
    friend      class lattice_gensys;
    friend      class lattice_basis;
    long 	rows;				// Number of Zeilen
    long 	cols;				// Number of Spalten
    double**	value;				// Values column by column

    long 	cols_rank;			// rank of the lattice
						// for longernal use

//
// static variables for temporary use to increase speed
//
    static double tempmz0;
    static double tempmz1;

    static long vectsize;

    static long tempvectsize;
    static double* tempvect0;
    static double* tempvect1;    

    static long last;

  public :

//
// Constructor / Destructors 
//
    double_lattice(long, long);
    double_lattice(long, long, double**);
    double_lattice(const double_lattice&);
    ~double_lattice();    

//  
// Input / Output
//

    friend istream& operator >> (istream&, double_lattice&);
    friend ostream& operator << (ostream&, const double_lattice&);

//
// Assignments
//

    double_lattice operator = (const double_lattice&);
    void assign(const double_lattice&);
    void assign_zero();

//
// Algorithms
//

    void lll(double_lattice&, const double&, 
             double, unsigned long&);   
    void lll_trans(double_lattice&, double_lattice&, 
                   const double&, double, unsigned long&);   
    void mlll(double*, unsigned long&, timer&);
    void lin_gen_system(double_lattice&, double,
                        unsigned long&, timer&, long&);
    void compute_basis(const double_lattice&, const double_lattice&);
    long gram_schmidt_orth(double_lattice&, double_lattice&);
    int lll_check(double);
    double lll_check();
    int is_reduced();

  protected :
    
//
// "Vector" - Operations
//

#ifdef __GNUC__

    friend void assign(double* a, double* b)
    {
      debug_handler("double_lattice","friend void assign(a, b)");
      long i;
      for (i=double_lattice::vectsize-1;i>=0;i--)
        a[i]=b[i];
    }

    friend void assign_zero(double* a)
    {
      debug_handler("double_lattice","friend void assign_zero(a)");
      long i;
      for (i=double_lattice::vectsize-1;i>=0;i--)
        a[i]=0;
    }

    friend void add(double* c, double* a, double* b)
    {
      debug_handler("double_lattice","friend void add(c, a, b)");
      long i;
      for (i=double_lattice::vectsize-1;i>=0;i--)
        c[i]=a[i]+b[i];
    }

    friend void subtract(double* c, double* a, double* b)
    {
      debug_handler("double_lattice","friend void subtract(c, a, b)");
      long i;
      for (i=double_lattice::vectsize-1;i>=0;i--)
        c[i]=a[i]-b[i];
    }

    friend void scalmul(double* b, const double& d, double* a)
    {
      debug_handler("double_lattice","friend scalmul(b, d, a)");
      long i;
      for (i=double_lattice::vectsize-1;i>=0;i--)
        b[i]=a[i]*d;
    }


    friend void scalprod(double& res, double* a, double* b)
    {
      debug_handler("double_lattice","friend scalprod(res, a, b)");
      long i;
      res=0;
      for (i=double_lattice::vectsize-1;i>=0;i--)
        res+=a[i]*b[i];
    }

    friend void l2_norm(double& norm, double* v)
    {
      debug_handler("double_lattice","l2_norm(norm, v)");
      long i;
      norm=0;		// Initialisation
      for (i=double_lattice::vectsize-1;i>=0;i--)
        norm+=v[i]*v[i];
      norm=sqrt(norm);
    }

    friend void l1_norm(double& norm, double* v)
    {
      debug_handler("double_lattice","l1_norm(norm, v)");
      long i;
      norm=0;	 // Initialisation
      for (i=double_lattice::vectsize-1;i>=0;i--)
        norm+=fabs(v[i]);
    }

#else

    friend void assign(double*, double*); 
    friend void assign_zero(double*);
    friend void add(double*, double*, double*);
    friend void subtract(double*, double*, double*);

    friend void scalmul(double*, const double&, double*);
    friend void scalprod(double&, double*, double*);
    
    friend void l1_norm(double&, double*);
    friend void l2_norm(double&, double*);

#endif

    friend void swap(double*& a, double*& b)
    {
      register double* temp;
      temp=a;
      a=b;
      b=temp;
    }
    
//
//  Algorithm - Subroutines
//
    
    void alpha_compute(double&);
    void gamma_compute(double&, long);
    void zwei_pot_q_compute(double&, long&, double&);
    void cholesky(double&);

};

#endif
