//
// LiDIA - a library for computational number theory
//   Copyright (c) 1994, 1995 by the LiDIA Group
//
// File        : bigmod.c 
// Author      : Thomas Papanikolaou (TP)
// Last change : TP, Jan 29 1995, initial version 
//               Volker M"uller (VM), Feb 7 1995, modified
//               invert() to invert(int) and the operators
//               so that they call the procedures. Added
//               bigmod randomize(void).
//

#include <LiDIA/bigmod.h>

bigint bigmod::M;

void bigmod::set_modulus(const bigint &m)
{
  if (m.is_zero())
    lidia_error_handler("bigmod", "set_modulus::zero modulus.");
  bigmod::M.assign(m);
  if (m.is_negative())
     bigmod::M.negate();
}

const bigint & bigmod::modulus()
{
  return bigmod::M;
}

void bigmod::normalize()
{
  bigint q, r;
  div_rem(q, r, I, bigmod::M);
  I.assign(r);
  if (I.is_negative())
    add(I, I, bigmod::M);
}

/**
** constructors and destructor; we could leave out some of these
**/

bigmod::bigmod()
{
}

bigmod::bigmod(int i)
{
    I.assign(i);
    this->normalize();
}

bigmod::bigmod(long l)
{
    I = l;
    this->normalize();
}

bigmod::bigmod(unsigned long ul)
{
    I = ul;
    this->normalize();
}

bigmod::bigmod(double d)
{
    I = d;
    this->normalize();
}

bigmod::bigmod(const bigint & i)
{
    I.assign(i);
    this->normalize();
}

bigmod::bigmod(const bigmod & a)
{
    I.assign(a.I);
    this->normalize();
}

bigmod::~bigmod()
{
}

/**
** inline member functions
**/

int bigmod::
length() const
{
    return I.length();
}

int bigmod::
bit_length() const
{
    return I.bit_length();
}

int bigmod::
is_zero() const
{
    return I.is_zero();
}

int bigmod::
is_one() const
{
    return I.is_one();
}

int bigmod::
intify(int &i) const
{
    return I.intify(i);
}

int bigmod::
longify(long &i) const
{
    return I.longify(i);
}

void bigmod::
negate()
{
    if (!I.is_zero())
    {
        I.negate();
        add(I, I, bigmod::M);
    }
}

bigint bigmod::
invert(int i)
{
    bigint d, u, v;
    d = xgcd(u, v, I, bigmod::M);
    if (d.is_one())
    {
        I.assign(u);
	this->normalize();
    }
    else
    {
	if (i)
	{
          warning_handler("bigmod", "invert::inverse does not exist.");
        }
	else
	  lidia_error_handler("bigmod", "invert::inverse does not exist.");
    }
    return d;
}

const bigint & bigmod::mantissa() const
{
  return I;
}

void bigmod::
assign_zero()
{
    I.assign_zero();
}

void bigmod::
assign_one()
{
    I.assign_one();
}

void bigmod::
assign(int i)
{
    I.assign(i);
    this->normalize();
}

void bigmod::
assign(long i)
{
    I.assign(i);
    this->normalize();
}

void bigmod::
assign(unsigned long ui)
{
    I.assign(ui);
    this->normalize();
}

void bigmod::
assign(const bigint & a)
{
    I.assign(a);
    this->normalize();
}

void bigmod::
assign(const bigmod & a)
{
    I.assign(a.I);
}

void bigmod::
multiply_by_2()
{
    I.multiply_by_2();
    if (I.compare(bigmod::M) >= 0)
        subtract(I, I, bigmod::M);
}

void bigmod::
divide_by_2()
{
    if (bigmod::M.is_even() && !I.is_even())
	lidia_error_handler("bigmod", "divide_by_2()::even modulus");
    if (!I.is_even())
        add(I, I, bigmod::M);
    I.divide_by_2();
}

/**
** Type checking
**/

int 
is_char(const bigmod & a)
{
    return is_char(a.I);
}

int 
is_uchar(const bigmod & a)
{
    return is_uchar(a.I);
}

int 
is_short(const bigmod & a)
{
    return is_short(a.I);
}

int 
is_ushort(const bigmod & a)
{
    return is_ushort(a.I);
}

int 
is_int(const bigmod & a)
{
    return is_int(a.I);
}

int 
is_uint(const bigmod & a)
{
    return is_uint(a.I);
}

int 
is_long(const bigmod & a)
{
    return is_long(a.I);
}

int 
is_ulong(const bigmod & a)
{
    return is_ulong(a.I);
}

/**
** assignments
**/

int bigmod::operator = (int i)
{
    I.assign(i);
    this->normalize();
    return i;
}

long bigmod::operator = (long l)
{
    I = l;
    this->normalize();
    return l;
}

unsigned long bigmod::operator = (unsigned long ul)
{
    I = ul;
    this->normalize();
    return ul;
}

double bigmod::operator = (double d)
{    
    I = d;
    this->normalize();
    return d;
}

bigint bigmod::operator = (const bigint & a)
{
    I.assign(a);
    this->normalize();
    return a;
}

bigmod & bigmod::operator = (const bigmod & a)
{
    I.assign(a.I);
    return *this;
}

/**
** comparisons
**/

int operator == (const bigmod & a, const bigmod & b)
{
    return (a.I.compare(b.I) == 0);
}

int operator != (const bigmod & a, const bigmod & b)
{
    return (a.I.compare(b.I) != 0);
}

/**
** operator overloading
**/

bigmod operator - (const bigmod & a)
{
    bigmod c(a);
    if (!c.I.is_zero())
    {
        c.I.negate();
        add(c.I, c.I, bigmod::M);
    }
    return c;
}

bigmod operator + (const bigmod & a, const bigmod & b)
{
    bigmod c;
    add(c, a, b);
    return c;
}

bigmod operator - (const bigmod & a, const bigmod & b)
{
    bigmod c;
    subtract(c, a, b);
    return c;
}

bigmod operator * (const bigmod & a, const bigmod & b)
{
    bigmod c;
    multiply(c, a, b);
    return c;
}

bigmod operator / (const bigmod & a, const bigmod & b)
{
    bigmod c;
    divide(c, a, b);
    return c;
}

bigmod & bigmod::operator += (const bigmod & a)
{
    add(*this, *this, a);
    return *this;
}

bigmod & bigmod::operator -= (const bigmod & a)
{
    subtract(*this, *this, a);
    return *this;
}

bigmod & bigmod::operator *= (const bigmod & a)
{
    multiply(*this, *this, a);
    return *this;
}

bigmod & bigmod::operator /= (const bigmod & a)
{
    divide(*this, *this, a);
    return *this;
}

bigmod & bigmod::operator++ ()
{
    inc(*this);
    return *this;
}

bigmod & bigmod::operator-- ()
{
    dec(*this);
    return *this;
}

int bigmod::operator ! ()
{
    return I.is_zero();
}

/**
** Procedural versions
**/

void 
invert(bigmod & a, const bigmod & b)
{
  a.assign(b);
  a.invert();
}

void 
negate(bigmod & a, const bigmod & b)
{
  a.assign(b);
  a.negate();
}

void
add(bigmod & c, const bigmod & a, const bigmod & b)
{
    add(c.I, a.I, b.I);
    if (c.I.compare(bigmod::M) >= 0)
      subtract(c.I, c.I, bigmod::M);
}

void
subtract(bigmod & c, const bigmod & a, const bigmod & b)
{
    subtract(c.I, a.I, b.I);
    if (c.I.is_negative())
      add(c.I, c.I, bigmod::M);
}

void
multiply(bigmod & c, const bigmod & a, const bigmod & b)
{
    multiply(c.I, a.I, b.I);
    remainder(c.I, c.I, bigmod::M);
}

void
divide(bigmod & c, const bigmod & a, const bigmod & b)
{
    bigint d, u, v;
    d = xgcd(u, v, b.I, bigmod::M);
    if (!d.is_one())
        lidia_error_handler("bigmod", "operator/::inverse undefined");
    if (u.is_negative())
        add(u, u, bigmod::M);
    multiply(c.I, a.I, u);
    remainder(c.I, c.I, bigmod::M);
}

void
power(bigmod & c, const bigmod & a, const bigint & b)
{
    bigmod multiplier;
    bigint exponent;

    if (b.is_zero() || a.is_one())
        c.assign_one();
    else
    {
        exponent.assign(b);
        if (exponent.is_negative())
        {
            bigint d, u, v;
            d = xgcd(u, v, a.I, bigmod::M);
            if (!d.is_one())
                lidia_error_handler("bigmod", "operator^::inverse undefined");
            if (u.is_negative())
                add(u, u, bigmod::M);
            multiplier.I.assign(u);
            exponent.negate();
        }
        else
            multiplier.I.assign(a.I);
        c.assign_one();
        while (exponent.is_gt_zero())
        {
            if (!exponent.is_even())
		multiply(c, c, multiplier);
            square(multiplier, multiplier);
            exponent.divide_by_2();
        }
    }
}

void
power(bigmod & c, const bigmod & a, long i)
{
    bigmod multiplier;
    long exponent;

    if ((i == 0) || a.is_one())
        c.assign_one();
    else
    {
        exponent = i;
        if (exponent < 0)
        {
            bigint d, u, v;
            d = xgcd(u, v, a.I, bigmod::M);
            if (!d.is_one())
                lidia_error_handler("bigmod", "operator^::inverse undefined");
            if (u.is_negative())
                add(u, u, bigmod::M);
            multiplier.I.assign(u);
            exponent = -exponent;
        }
        else
            multiplier.I = a.I;
        c.assign_one();
        while (exponent > 0)
        {
            if (exponent & 1)
		multiply(c, c, multiplier);
            square(multiplier, multiplier);
            exponent >>= 1;
        }
    }
}

void
inc(bigmod & c)
{
    inc(c.I);
    if (c.I.abs_compare(bigmod::M) == 0)
        c.I.assign_zero();
}

void
dec(bigmod & c)
{
    if (c.I.is_zero())
        c.I.assign(bigmod::M);
    dec(c.I);
}

void
add(bigmod & c, const bigmod & a, long i)
{
    add(c.I, a.I, i);
    c.normalize();
}

void
add(bigmod & c, long i, const bigmod & a)
{
    add(c.I, a.I, i);
    c.normalize();
}

void
subtract(bigmod & c, const bigmod & a, long i)
{
    subtract(c.I, a.I, i);
    c.normalize();
}

void
subtract(bigmod & c, long i, const bigmod & a)
{
    subtract(c.I, a.I, i);
    c.I.negate();
    c.normalize();
}

void
multiply(bigmod & c, const bigmod & a, long i)
{
    multiply(c.I, a.I, i);
    c.normalize();
}

void
multiply(bigmod & c, long i, const bigmod & a)
{
    multiply(c.I, a.I, i);
    c.normalize();
}

void
divide(bigmod & c, const bigmod & a, long i)
{
    bigint d, u, v, b = i;
    d = xgcd(u, v, b, bigmod::M);
    if (!d.is_one())
        lidia_error_handler("bigmod", "operator/::inverse undefined");
    if (u.is_negative())
        add(u, u, bigmod::M);
    multiply(c.I, a.I, u);
    remainder(c.I, c.I, bigmod::M);
}

void
divide(bigmod & c, long i, const bigmod & a)
{
    invert(c, a);
    multiply(c, c, i);
}

/**
** functions
**/

bigmod
inverse(const bigmod & a)
{
    bigmod c;
    c.assign(a);
    c.invert();
    return c;
}

void
seed(const bigmod & a)
{
  seed(a.I);
}

bigmod
randomize()
{
    bigmod c;
    c.I.assign(randomize(bigmod::M));
    if (c.I.compare(bigmod::M) == 0) 
                c.I.assign_zero();
    return c;
}

bigint
mantissa(const bigmod & a)
{
    return (a.I);
}

bigmod
randomize(const bigmod & a)
{
    bigmod c;
    c.I.assign(randomize(a.I));
    return c;
}

double
dbl(const bigmod & a)
{
    return dbl(a.I);
}

void
normalize(bigmod & a, const bigmod & b)
{
  bigint q, r;
  div_rem(q, r, b.I, bigmod::M);
  a.I.assign(r);
}

void
square(bigmod & a, const bigmod & b)
{
    square(a.I, b.I);
    remainder(a.I, a.I, bigmod::M);

}

void
swap(bigmod & a, bigmod & b)
{
    swap(b.I, a.I);
}

/**
** input / output
**/

istream & operator >> (istream & in, bigmod & a)
{
    in >> a.I;
    a.normalize();
    return in;
}

ostream & operator << (ostream & out, const bigmod & a)
{
    out << a.I << " mod (" << bigmod::M << ")";
    return out;
}

int
string_to_bigmod(char *s, bigmod & a)
{
    int i = string_to_bigint(s, a.I);
    a.normalize();
    return i;
}

int
bigmod_to_string(const bigmod & a, char *s)
{
    return bigint_to_string(a.I, s);
}

/**
** using fread/fwrite
**/

void bigmod::
read_from_file(FILE * fp)
{
    I.read_from_file(fp);
    this->normalize();
}

void bigmod::
write_to_file(FILE * fp)
{
    I.write_to_file(fp);
}

/**
** using fscanf/fprintf
**/

void bigmod::
scan_from_file(FILE * fp)
{
    I.scan_from_file(fp);
    this->normalize();
}

void bigmod::
print_to_file(FILE * fp)
{
    I.print_to_file(fp);
}
