/*********************************************************************
**
**     File name:               ssh_debug.c
**                                  
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 handle debugging messages
**
**     Author/Date:             Tadayoshi Kohno, 28 November 1997
**				Modified, 21 February 1998
**
**     Notes:
**	Reduced this down from a mondo header file to this.
**	Some of the main advantages with this is that I can turn debugging
**	on and off.  Some disadvantages of this is that these functions
**	are still called even if I don't want debugging.
**
**	Modified in February to allow for the application to choose
**	which functions to use for displaying the debug messages.
**	Also, through ssh_struct and ssh_debug_struct, there can
**	debug levels (on, off) and display functions per simultanious
**	open-connections.
**
**     Functions:
**	ssh_debug_activate_new	turn on/off debugging
**	ssh_debug_setfunc_new	specify output function
**
**	ssh_debugger_new	display debugging message
**	ssh_debug_int_new	display debug message with int param
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/cia/kohno/libssh/libssh/RCS/ssh_debug.c,v 3.0 1998/02/24 00:10:44 kohno Exp $";
#endif


#include "ssh_debug.h"


/*********************************************************************
**
**     Function:                ssh_debug_activate_new
**
**     Purpose:                 turn on/off debugging for ssh_info
**				connection
**
**     Parameters:              ssh_info	info on current connection
**				choice		do we want debugging?
**
**     Side effects:            if choice is SSH_YES_DEBUG, turn
**				on debugging, otherwise turn it off
**
**     Author/Date:             Tadayoshi Kohno, 28 November 1997
**     Modified:		Tadayoshi Kohno, 21 February 1998
**					for ssh_struct * ssh_info
**
*********************************************************************/

void ssh_debug_activate_new
(
	struct ssh_struct * ssh_info,		/* info on debugging */
	int choice				/* should we debug */
)
{
	if (choice == SSH_YES_DEBUG)
	{
		ssh_info->debug_info.debug_flag = 1;
	} else
	{
		ssh_info->debug_info.debug_flag = 0;
	}
}

/******************************************************************************
**
**     File/Function name:	ssh_debug_setfunc_new
**
**     Purpose:			set the function to display debug messages
**				for the debug_info debug set
**
**     Preconditions:		debug_func valid
**
**     Parameters:		debug_info	attributes to change
**				debug_func	function to use for above
**
**     Exit (post) conditions:	none
**
**     Side effects:		debug_info->{debug_func,print_debug} set
**
**     Author/Date:		Tadayoshi Kohno, 21 Feb 1998
**
**     Notes:
**	A NULL debug_func means that we will not do debugging
**
******************************************************************************/

void ssh_debug_setfunc_new
(
	struct ssh_debug_struct * debug_info,
	SSHDebugPrinter debug_func
)
{
	if (debug_func == (SSHDebugPrinter) NULL)
	{
		debug_info->debug_flag = 0;
		debug_info->print_debug = (SSHDebugPrinter) NULL;

		return;
	}

	debug_info->print_debug = debug_func;
}


/*********************************************************************
**
**     Function:                ssh_debugger_new
**
**     Purpose:                 show a string of debugging information
**				via debug_info->print_debug
**
**     Parameters:              debug_info	info on how to debug
**				msg		debugging message
**				header		debugging message header
**
**     Side effects:            debugging message displayed if
**				ssh_debug_flag is set
**
**     Author/Date:             Tadayoshi Kohno, 28 November 1997
**     Modified:		Tadayoshi Kohno, 21 February 1998
**					for ssh_struct * ssh_info
**
*********************************************************************/

void ssh_debugger_new
(
	struct ssh_debug_struct * debug_info,	/* info on debugging */
	char * msg,				/* debugging message */
	char * header				/* debugging header */
)
{
	if (debug_info->print_debug == (SSHDebugPrinter) NULL)
	{
		return;
	}

	if (debug_info->debug_flag)
	{
		(*(debug_info->print_debug))(header, msg);
	}
}



/*********************************************************************
**
**     Function:                ssh_debug_int_new
**
**     Purpose:                 display debugging info with int param
**
**     Preconditions:		msg is of printf format with %d for
**				param1
**
**     Side effects:            debugging info displayed
**
**     Author/Date:             Tadayoshi Kohno, 28 November 1997
**     Modified:		Tadayoshi Kohno, 21 February 1998
**					for ssh_struct * ssh_info
**
*********************************************************************/

void ssh_debug_int_new
(
	struct ssh_debug_struct * debug_info,	/* info on debugging */
	char * msg,				/* debugging message */
	int param1,				/* first parameter */
	char * header				/* message header */

)
{
	/* xxx don't use any constants!!! */
	char debug_str[1000];	/* final debugging message */

	if (debug_info->print_debug == (SSHDebugPrinter) NULL)
	{
		return;
	}

	if (debug_info->debug_flag)
	{
		sprintf(debug_str, msg, param1);

		(*(debug_info->print_debug))(header, debug_str);
	}
}

