/*********************************************************************
**
**     File name:               ssh_errno.c
**                                  
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 Handle ssh_errno error information
**				(setting, getting, understanding).
**
**     Author/Date:             Tadayoshi Kohno, 26 November 1997
**
**     Notes:
**	Because we hope to not use these functions much, it is not
**	much overhead to call them to set/get the static ssh_errno variable.
**	There are a lot of better places to spend optimization time.
**
**	Whenever possible, I try to pass the errno up to the application
**	so that the application can tell the user what went wrong.  Meaning
**	instead of saying that authentication went wrong (which is really
**	general) I like to say encoding (or something) went wrong.
**
**     Functions:
**	ssh_errno_set		set the ssh_errno variable
**	ssh_errno_get		get value of ssh_errno
**
**	ssh_errno_to_str	get a string that represents the errno.
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/libssh/RCS/ssh_errno.c,v 3.13 1998/04/18 15:42:44 kohno Exp $";
#endif

#include <string.h>
#include <stdio.h>

#include "ssh_errno.h"

/*
**	ssh_errno contains the errno for ssh.  I know it's not capitalized
**	as specified by some style manuals.  However, because of its limited
**	use and its specific purpose, not capitalizing it here is "OK."
*/
static int ssh_errno = SSH_ERRNO_NONE;

/*********************************************************************
**
**     Function:                ssh_errno_set
**
**     Purpose:                 set the ssh error information
**
**     Parameters:              error_value	error value to save
**
**     Side effects:            ssh_errno set to error_value
**
**     Author/Date:             Tadayoshi Kohno, 26 November 1997
**
*********************************************************************/

void ssh_errno_set
(
	int error_value		/* error value to save */
)
{
	ssh_errno = error_value;
}

/*********************************************************************
**
**     Function:                ssh_errno_get
**
**     Purpose:                 return ssh_errno value
**
**     Entry (pre) conditions:  ssh_errno set
**
**     Return value:            value of ssh_errno
**
**     Author/Date:             Tadayoshi Kohno, 27 November 1997
**
*********************************************************************/

int ssh_errno_get()
{
	return(ssh_errno);
}

/*********************************************************************
**
**     Function:                ssh_errno_to_str
**
**     Purpose:                 convert ssh_errno to a string
**
**     Entry (pre) conditions:  ssh_errno set
**				*msg allocated (space of SSH_ERRNO_LEN)
**
**     Parameters:              msg		message for ssh_errno
**
**     Side effects:            msg set to string representing errno
**
**     Author/Date:             Tadayoshi Kohno, 27 November 1997
**     Modified:		Tadayoshi Kohno, 14 March 1998
**					(SSH_ERRNO_RANDOM_SEED_MSG)
**					(SSH_ERRNO_NULL_POINTER_MSG)
**
*********************************************************************/

void ssh_errno_to_str
(
	char * msg		/* string for error message */
)
{
	switch (ssh_errno)
	{
	case SSH_ERRNO_NONE:
		strcpy(msg, SSH_ERRNO_NONE_MSG);
		break;

	case SSH_ERRNO_LOW_MEM:
		strcpy(msg, SSH_ERRNO_LOW_MEM_MSG);
		break;

	case SSH_ERRNO_MSG_PARSE:
		strcpy(msg, SSH_ERRNO_MSG_PARSE_MSG);
		break;

	case SSH_ERRNO_RECV:
		strcpy(msg, SSH_ERRNO_RECV_MSG);
		break;

	case SSH_ERRNO_SEND:
		strcpy(msg, SSH_ERRNO_SEND_MSG);
		break;

	case SSH_ERRNO_PROTO_PARSE:
		strcpy(msg, SSH_ERRNO_PROTO_PARSE_MSG);
		break;

	case SSH_ERRNO_BAD_CRC:
		strcpy(msg, SSH_ERRNO_BAD_CRC_MSG);
		break;

	case SSH_ERRNO_PASSWD_FAILED:
		strcpy(msg, SSH_ERRNO_PASSWD_FAILED_MSG);
		break;

	case SSH_ERRNO_AUTH:
		strcpy(msg, SSH_ERRNO_AUTH_MSG);
		break;

	case SSH_ERRNO_USER:
		strcpy(msg, SSH_ERRNO_USER_MSG);
		break;

	case SSH_ERRNO_UNKNOWN_CIPHER:
		strcpy(msg, SSH_ERRNO_UNKNOWN_CIPHER_MSG);
		break;

	case SSH_ERRNO_VERSIONS:
		strcpy(msg, SSH_ERRNO_VERSIONS_MSG);
		break;

	case SSH_ERRNO_TYPE:
		strcpy(msg, SSH_ERRNO_TYPE_MSG);
		break;

	case SSH_ERRNO_MALLOC:
		strcpy(msg, SSH_ERRNO_MALLOC_MSG);
		break;

	case SSH_ERRNO_TODO:
		strcpy(msg, SSH_ERRNO_TODO_MSG);
		break;

	case SSH_ERRNO_FINDCIPHER:
		strcpy(msg, SSH_ERRNO_FINDCIPHER_MSG);
		break;

	case SSH_ERRNO_INIT_CIPHER:
		strcpy(msg, SSH_ERRNO_INIT_CIPHER_MSG);
		break;

	case SSH_ERRNO_DISCONNECTED:
		strcpy(msg, SSH_ERRNO_DISCONNECTED_MSG);
		break;

	case SSH_ERRNO_AUTHMODE:
		strcpy(msg, SSH_ERRNO_AUTHMODE_MSG);
		break;

	case SSH_ERRNO_IDENTITY_FILE:
		strcpy(msg, SSH_ERRNO_IDENTITY_FILE_MSG);
		break;

	case SSH_ERRNO_IDENTITY_MISC:
		strcpy(msg, SSH_ERRNO_IDENTITY_MISC_MSG);
		break;

	case SSH_ERRNO_ID_STRING:
		strcpy(msg, SSH_ERRNO_ID_STRING_MSG);
		break;

	case SSH_ERRNO_IDENTITY_DECRYPT:
		strcpy(msg, SSH_ERRNO_IDENTITY_DECRYPT_MSG);
		break;

	case SSH_ERRNO_RSA_FAILED:
		strcpy(msg, SSH_ERRNO_RSA_FAILED_MSG);
		break;

	case SSH_ERRNO_CHALLENGE_DECRYPT:
		strcpy(msg, SSH_ERRNO_CHALLENGE_DECRYPT_MSG);
		break;

	case SSH_ERRNO_IDENTITY_SAVE:
		strcpy(msg, SSH_ERRNO_IDENTITY_SAVE_MSG);
		break;

	case SSH_ERRNO_IDENTITY_LOAD:
		strcpy(msg, SSH_ERRNO_IDENTITY_LOAD_MSG);
		break;

	case SSH_ERRNO_PHRASE_SHORT:
		strcpy(msg, SSH_ERRNO_PHRASE_SHORT_MSG);
		break;

	case SSH_ERRNO_REQUEST_PTY:
		strcpy(msg, SSH_ERRNO_REQUEST_PTY_MSG);
		break;

	case SSH_ERRNO_RANDOM_SEED:
		strcpy(msg, SSH_ERRNO_RANDOM_SEED_MSG);
		break;

	case SSH_ERRNO_NULL_POINTER:
		strcpy(msg, SSH_ERRNO_NULL_POINTER_MSG);
		break;

	case SSH_ERRNO_PACKET_BIG:
		strcpy(msg, SSH_ERRNO_PACKET_BIG_MSG);
		break;

	case SSH_ERRNO_COMPRESS_LEVEL:
		strcpy(msg, SSH_ERRNO_COMPRESS_LEVEL_MSG);
		break;

	case SSH_ERRNO_REQUEST_COMPRESS:
		strcpy(msg, SSH_ERRNO_REQUEST_COMPRESS_MSG);
		break;

	case SSH_ERRNO_ENCODE_MESSAGE:
		strcpy(msg, SSH_ERRNO_ENCODE_MESSAGE_MSG);
		break;

	case SSH_ERRNO_ZLIB_MEM_ERROR:
		strcpy(msg, SSH_ERRNO_ZLIB_MEM_ERROR_MSG);
		break;

	case SSH_ERRNO_ZLIB_STREAM_ERROR:
		strcpy(msg, SSH_ERRNO_ZLIB_STREAM_ERROR_MSG);
		break;

	case SSH_ERRNO_ZLIB_VERSION_ERROR:
		strcpy(msg, SSH_ERRNO_ZLIB_VERSION_ERROR_MSG);
		break;

	case SSH_ERRNO_ZLIB_MISC:
		strcpy(msg, SSH_ERRNO_ZLIB_MISC_MSG);
		break;

	case SSH_ERRNO_COMPRESSION_WRONG:
		strcpy(msg, SSH_ERRNO_COMPRESSION_WRONG_MSG);
		break;

	case SSH_ERRNO_ZLIB_STALL_ERROR:
		strcpy(msg, SSH_ERRNO_ZLIB_STALL_ERROR_MSG);
		break;

	case SSH_ERRNO_ZLIB_STATE_ERROR:
		strcpy(msg, SSH_ERRNO_ZLIB_STATE_ERROR_MSG);
		break;

	case SSH_ERRNO_ZLIB_MORE_OUT_ERROR:
		strcpy(msg, SSH_ERRNO_ZLIB_MORE_OUT_ERROR_MSG);
		break;

	case SSH_ERRNO_ZLIB_DEALLOC:
		strcpy(msg, SSH_ERRNO_ZLIB_DEALLOC_MSG);
		break;

	default:
		sprintf(msg, "%s: %d", SSH_ERRNO_UNKNOWN_MSG, ssh_errno);
		break;
	}
}

