/*********************************************************************
**
**     File name:               ssh_packet.c
**
**                              Copyright 1997, 1998 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 pack and unpack a message
**
**     Author/Date:             Tadayoshi Kohno, 21 November 1997
**
**     References:              draft-ylonen-ssh-protocol-00.txt
**
**     Notes:
**	The functions in this file pack and unpack a packet.  The
**	packet protocol is described in draft-ylonen-ssh-protocol-00.txt
**	for SSHv1.5.  Basically, each packet contains a
**		packet length (length of packet - [padding + this field])
**		padding	(8 - length % 8)
**		packet type/message type
**		data
**		crc
**	So the functions in this file take the data and put it into packet
**	form and visa versa.
**
**     Functions:
**	ssh_packet_unpack			unpack a packet
**	ssh_packet_pack				pack a packet
**
**	ssh_packet_unpack_block			unpack a packet into a type+data
**						block
**	ssh_packet_pack_block			pack a packet given a type+data
**						block
**
**	ssh_packet_block_glom			create a type+data block
**	ssh_packet_block_extract		extract a type+data block
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/libssh/RCS/ssh_packet.c,v 3.12 1998/05/09 17:19:00 kohno Exp $";
#endif

#include <string.h>
#include <stdlib.h>

#include <rand.h>

#include "ssh.h"
#include "ssh_debug.h"
#include "ssh_packet.h"
#include "ssh_crypt.h"

#include "ssh_util.h"

/*********************************************************************
**
**     Function:                ssh_packet_unpack
**
**     Purpose:                 extract the data from a packet.
**
**     Entry (pre) conditions:  memory for *data allocate
**				pointers valid
**
**     Parameters:              ssh_info	info about current connection
**
**				packet		packet to unpack
**
**				*data_len	length of data
**				*type		type of message
**				*data		data in packet
**
**				out_size	available mem for data
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**
**				ssh_errno set for
**					SSH_ERRNO_LOW_MEM
**					SSH_ERRNO_BAD_CRC
**				if not enough memory for *data, or
**				there's a bad CRC.
**
**     Side effects:            *type set to packet type
**				*data_len set to actual length of
**				data (just data, not including type, crc)
**				*data holds data in packet (not including
**				type, crc)
**
**     Author/Date:             Tadayoshi Kohno, 21 November 1997
**
**     Notes:
**	xxx we probably don't need to check for the null pointers
**	xxx because calling functions check -- maybe make a SSH_PARANOID
**	xxx compile option
**
*********************************************************************/

int ssh_packet_unpack
(
	struct ssh_struct * ssh_info,	/* info about current connection */

	const uint8_t * packet,	/* packet to unpack */

	uint32_t * data_len,	/* length of data */
	uint8_t * type,		/* type of message */
	uint8_t * data,		/* data in packet */

	uint32_t out_size	/* size malloc'ed for * data */
)
{
	uint8_t * packet_ptr;	/* pointer into packet */
	uint32_t packet_length;	/* length of packet */
	uint32_t crc;		/* crc of packet */
	int padding;		/* padding length */

	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	/*
	**	First lets computer the packet length and the padding
	*/
	my_bcopy((void *) packet, (void *) &packet_length,
		SSH_PACKET_LENGTH_LEN);
	packet_length = ntohl(packet_length);

	padding = SSH_UNITS - packet_length % SSH_UNITS;

	/*
	**	Now we know that the actual data is
	**	SSH_PACKET_LENGTH_LEN + padding bytes into packet
	*/
	*type = packet[SSH_PACKET_LENGTH_LEN + padding];

	/*
	**	Now make sure we can have enough output space
	*/
	*data_len = packet_length - SSH_DATA_LEN_DIFF;

	if (*data_len > out_size)
	{
		ssh_errno_set(SSH_ERRNO_LOW_MEM);
		return(S_BAD);
	}

	/*
	**	copy data to user space
	*/
	packet_ptr = ((uint8_t *) packet) + SSH_PACKET_LENGTH_LEN
		+ padding + SSH_TYPE_LEN;
	my_bcopy((void *) packet_ptr, (void *) data, *data_len);

	/*
	**	Finally, lets check the CRC
	*/
	packet_ptr = ((uint8_t *) packet) + SSH_PACKET_LENGTH_LEN
		+ padding + SSH_TYPE_LEN + *data_len;
	my_bcopy((void *) packet_ptr, (void *) &crc, SSH_PACKET_CRC_LEN);
	crc = ntohl(crc);

	if (crc !=
		ssh_crc32(packet + SSH_PACKET_LENGTH_LEN,
		*data_len + SSH_TYPE_LEN + padding))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"bad crc", "ssh_packet_unpack");

		ssh_errno_set(SSH_ERRNO_BAD_CRC);
		return(S_BAD);
	}

	return(S_GOOD);
}


/*********************************************************************
**
**     Function:                ssh_packet_pack
**
**     Purpose:                 make a packet given the type and data
**
**     Entry (pre) conditions:  memory for packet allocated
**				pointers valid
**
**     Parameters:              ssh_info	ssh connection info
**
**				*packet		output packet
**				*total_len	total length of *packet
**
**				type		message type to pack
**				data		data for the packet
**				data_len	length of "data"
**
**				out_size	available mem for *packet
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**
**				ssh_errno set for SSH_ERRNO_LOW_MEM
**				if not enought memory for packet
**
**     Side effects:            an SSHv1.5 packet is formed from the type
**				and data given.  the resulting packet
**				is stored to *packet, and *total_len
**				contains the length of *packet.
**
**     Author/Date:             Tadayoshi Kohno, 21 November 1997
**     Modified:		Tadayoshi Kohno, 14 March 1998
**					(just a call to RAND_bytes)
**
**     References:                  
**
**     Notes:
**	buffer overflow checking should be everywhere -- packet formation
**	especially xxx
**
*********************************************************************/
int ssh_packet_pack
(
	struct ssh_struct * ssh_info,	/* ssh connection information */
	uint8_t * packet,	/* packet to pack */
	uint32_t * total_len,	/* total length of output */
	uint8_t type,		/* type of message */
	const uint8_t * data,	/* data for message */
	uint32_t data_len,	/* length of data */
	uint32_t out_size	/* valid packet size max */
)	
{
	uint8_t * packet_ptr;	/* pointer to packet */
	uint32_t packet_length;	/* length of packet (type, data, crc) */
	uint32_t nbo_packet_length;	/* packet length in network order */
	uint32_t padding;		/* bytes for padding */
	uint32_t crc;			/* packet's crc */

	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	packet_length = data_len + SSH_DATA_LEN_DIFF;
	padding = SSH_UNITS - packet_length % SSH_UNITS;

	*total_len = packet_length + padding + SSH_PACKET_LENGTH_LEN;

	if (*total_len > out_size)
	{
		ssh_errno_set(SSH_ERRNO_LOW_MEM);
		return(S_BAD);
	}


	/*
	**	now lets fill the packet
	*/
	nbo_packet_length = htonl(packet_length);
	my_bcopy((void *) &nbo_packet_length, (void *) packet,
		SSH_PACKET_LENGTH_LEN);

	/*
	**	fill padding with random bytes
	*/

	RAND_bytes(packet + SSH_PACKET_LENGTH_LEN, padding);

	/*
	**	set type
	*/
	packet[SSH_PACKET_LENGTH_LEN + padding] = type;

	/*
	**	copy in the data
	*/
	packet_ptr = packet + SSH_PACKET_LENGTH_LEN + padding + SSH_TYPE_LEN;
	my_bcopy((void *) data, (void *) packet_ptr, data_len);

	/*
	**	set the crc
	*/
	packet_ptr = packet + SSH_PACKET_LENGTH_LEN + padding
		+ SSH_TYPE_LEN + data_len;

	crc = ssh_crc32(packet + SSH_PACKET_LENGTH_LEN, data_len
		+ SSH_TYPE_LEN + padding);
	crc = htonl(crc);

	my_bcopy((void *) &crc, (void *) packet_ptr, SSH_PACKET_CRC_LEN);
	
	return(S_GOOD);
}


/*********************************************************************
**
**     Function:                ssh_packet_unpack_block
**
**     Purpose:                 extract the block (type + data) from
**				a packet.
**
**     Entry (pre) conditions:  memory for *block allocate
**				pointers valid
**
**     Parameters:              ssh_info	info about current connection
**
**				packet		packet to unpack
**
**				*block_len	length of block (type + content)
**				*block		block in packet (type + content)
**
**				out_size	available mem for block
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**
**				ssh_errno set for
**					SSH_ERRNO_LOW_MEM
**					SSH_ERRNO_BAD_CRC
**				if not enough memory for *block, or
**				there's a bad CRC.
**
**     Side effects:          	*block_len set to actual length of
**				block (including type and data)
**				*block holds type+data in packet (not including
**				crc)
**
**     Author/Date:             Tadayoshi Kohno, 21 November 1997
**				Tadayoshi Kohno, 11 April 1998
**					(to allow *block to point to
**					type + content).
**
**     Notes:
**
*********************************************************************/

int ssh_packet_unpack_block
(
	struct ssh_struct * ssh_info,	/* info about current connection */

	const uint8_t * packet,	/* packet to unpack */

	uint32_t * block_len,	/* length of block */
	uint8_t * block,	/* block in packet */

	uint32_t out_size	/* size malloc'ed for * block */
)
{
	uint8_t * packet_ptr;	/* pointer into packet */
	uint32_t packet_length;	/* length of packet */
	uint32_t crc;		/* crc of packet */
	int padding;		/* padding length */

	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	/*
	**	First lets compute the packet length and the padding
	*/
	my_bcopy((void *) packet, (void *) &packet_length,
		SSH_PACKET_LENGTH_LEN);
	packet_length = ntohl(packet_length);

	padding = SSH_UNITS - packet_length % SSH_UNITS;

	/*
	**	Now make sure we can have enough output space for the
	**	data + type
	*/
	*block_len = packet_length - SSH_DATA_BLOCK_LEN_DIFF;

	if (*block_len > out_size)
	{
		ssh_errno_set(SSH_ERRNO_LOW_MEM);
		return(S_BAD);
	}

	/*
	**	copy data to user space
	*/
	packet_ptr = ((uint8_t *) packet) + SSH_PACKET_LENGTH_LEN + padding;
	my_bcopy((void *) packet_ptr, (void *) block, *block_len);

	/*
	**	Finally, lets check the CRC
	*/
	packet_ptr = ((uint8_t *) packet) + SSH_PACKET_LENGTH_LEN + padding
		+ *block_len;
	my_bcopy((void *) packet_ptr, (void *) &crc, SSH_PACKET_CRC_LEN);
	crc = ntohl(crc);

	if (crc !=
		ssh_crc32(packet + SSH_PACKET_LENGTH_LEN,
		*block_len + padding))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"bad crc", "ssh_packet_unpack_block");

		ssh_errno_set(SSH_ERRNO_BAD_CRC);
		return(S_BAD);
	}

	return(S_GOOD);
}


/*********************************************************************
**
**     Function:                ssh_packet_pack_block
**
**     Purpose:                 make a packet given the type and data
**
**     Entry (pre) conditions:  memory for packet allocated
**				pointers valid
**
**     Parameters:              ssh_info	ssh connection info
**
**				*packet		output packet
**				*total_len	total length of *packet
**
**				block		block (data+type) for packet
**				block_len	length of block (data + type)
**
**				out_size	available mem for *packet
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**
**				ssh_errno set for SSH_ERRNO_LOW_MEM
**				if not enought memory for packet
**
**     Side effects:            an SSHv1.5 packet is formed from the type
**				and data given.  the resulting packet
**				is stored to *packet, and *total_len
**				contains the length of *packet.
**
**     Author/Date:             Tadayoshi Kohno, 21 November 1997
**     Modified:		Tadayoshi Kohno, 14 March 1998
**					(just a call to RAND_bytes)
**				Tadayoshi Kohno, 11 April 1998
**					(block now includes type (type not
**					a seperate parameter))
**
**     References:                  
**
**     Notes:
**	buffer overflow checking should be everywhere -- packet formation
**	especially xxx
**
*********************************************************************/

int ssh_packet_pack_block
(
	struct ssh_struct * ssh_info,	/* ssh connection information */
	uint8_t * packet,	/* packet to pack */
	uint32_t * total_len,	/* total length of output */
	const uint8_t * block,	/* block for message (data + type) */
	uint32_t block_len,	/* length of block (data + type) */
	uint32_t out_size	/* valid packet size max */
)	
{
	uint8_t * packet_ptr;	/* pointer to packet */
	uint32_t packet_length;	/* length of packet (type, data, crc) */
	uint32_t nbo_packet_length;	/* packet length in network order */
	uint32_t padding;		/* bytes for padding */
	uint32_t crc;			/* packet's crc */

	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

	packet_length = block_len + SSH_DATA_BLOCK_LEN_DIFF;
	padding = SSH_UNITS - packet_length % SSH_UNITS;

	*total_len = packet_length + padding + SSH_PACKET_LENGTH_LEN;

	if (*total_len > out_size)
	{
		ssh_errno_set(SSH_ERRNO_LOW_MEM);
		return(S_BAD);
	}


	/*
	**	now lets fill the packet
	*/
	nbo_packet_length = htonl(packet_length);
	my_bcopy((void *) &nbo_packet_length, (void *) packet,
		SSH_PACKET_LENGTH_LEN);

	/*
	**	fill padding with random bytes
	*/

	RAND_bytes(packet + SSH_PACKET_LENGTH_LEN, padding);

	/*
	**	copy in the data
	*/
	packet_ptr = packet + SSH_PACKET_LENGTH_LEN + padding;
	my_bcopy((void *) block, (void *) packet_ptr, block_len);

	/*
	**	set the crc
	*/
	packet_ptr = packet + SSH_PACKET_LENGTH_LEN + padding + block_len;

	crc = ssh_crc32(packet + SSH_PACKET_LENGTH_LEN, block_len + padding);
	crc = htonl(crc);

	my_bcopy((void *) &crc, (void *) packet_ptr, SSH_PACKET_CRC_LEN);
	
	return(S_GOOD);
}

/******************************************************************************
**
**     File/Function name:	ssh_packet_block_glom
**
**     Purpose:			form a string containing the concatenation
**				of the type and data
**
**     Preconditions:		pointers valid
**				memory allocated for *ouput = data_len + 
**					SSH_TYPE_LEN
**
**     Parameters:		*output		concatenated type, data
**				*output_len	length of *output
**
**				type		type to glom
**				data		data to glom
**				data_len	length of data
**
**				out_size	available bytes for *output
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:	S_BAD		out_size too small
**
**     Side effects:		type, data concatenated into *output
**				*output_len set to data_len + SSH_TYPE_LEN
**
**     Author/Date:		Tadayoshi Kohno, 11 April 1998
**
**     Notes:
**	The use of this is probably pretty inefficient and this should
**	be rolled into the ssh_cmsg_xxx_encode() functions.  Nevertheless,
**	it is here for testing compression.
**
******************************************************************************/

int ssh_packet_block_glom
(
	uint8_t * output,		/* glomed type and data */
	uint32_t * output_len,		/* length of glommed output */

	uint8_t type,			/* type for glomming */

	const uint8_t * data,		/* data to glom */
	uint32_t data_len,		/* input length */

	uint32_t out_size		/* output size */
)
{
/*
**	make sure we have enough memory
*/
	*output_len = data_len + SSH_TYPE_LEN;

	if (out_size < *output_len)
	{
		ssh_errno_set(SSH_ERRNO_LOW_MEM);
		return(S_BAD);
	}

/*
**	form the glommed output
*/
	output[0] = type;

	my_bcopy((void *) data, (void *) (output + SSH_TYPE_LEN),
		data_len);

	return(S_GOOD);
}

/******************************************************************************
**
**     File/Function name:	ssh_packet_block_extract
**
**     Purpose:			extract the type, data from a block
**
**     Preconditions:		pointers valid
**				memory for data allocated (SSH_TYPE_LEN bytes
**					less than input_len
**
**     Parameters:		*type		type stored in block
**				*data		data stored in block
**				*data_len	length of data
**
**				input		input block of type + data
**				input_lne	length of input
**
**				data_size	available memory for *data
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:	S_BAD		data_size too small
**
**     Side effects:		*type contains type contained in input
**				*data contains data contained in input
**				*data_len set to length (in bytes) of *data
**
**     Author/Date:		Tadayoshi Kohno, 11 April 1998
**
**     Notes:
**
******************************************************************************/

int ssh_packet_block_extract
(
	uint8_t * type,			/* type from glommed block */

	uint8_t * data,			/* data from glommed block */
	uint32_t * data_len,		/* length of data */

	const uint8_t * input,		/* glommed type + data to extract */
	uint32_t input_len,		/* length of input */

	uint32_t data_size		/* available length of *data */
)
{
/*
**	make sure we have enough memory
*/
	*data_len = input_len - SSH_TYPE_LEN;

	if (data_size < *data_len)
	{
		ssh_errno_set(SSH_ERRNO_LOW_MEM);
		return(S_BAD);
	}
/*
**	extract the data
*/
	*type = input[0];

	my_bcopy((void *) (input + SSH_TYPE_LEN), (void *) data,
		*data_len);

	return(S_GOOD);
}


