/* This is one of the cipher files for the cipher interface written
** by wart@ugcs.caltech.edu
**
** Please don't steal my code without my permission.
**
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "term.h"
#include "types.h"
#include "ctypes.h"

vigenere::vigenere(){
  period = 0;
  period_set = FALSE;
  clen = 0;
  length = 0;
}

int vigenere::execute_option(char option){
  switch(option){
    case SUBSTITUTE:
      substitute();
      break; 
    case UNDO:
      undo();
      break;
    case ALPHFIT:
      alphfit();
      break;
    default:
      base_exec_option(option);
      break;
  }

  return TRUE;
}
    
void vigenere::setup_key(){
  int i;

  delete key;
  key = new char[period+1];
  for(i = 0; i < period; i++)
    key[i] = UNCHANGED;
  key[period] = '\0';
}

void vigenere::init_cipher(){
  length = strlen(cipher);
  clen = length / period;
}

void vigenere::substitute(){
  int column;
  char tmp_str[STRINGLENGTH];
  char ct_letter, pt_letter;

  prompt("What column is the letter in? ");
  column = (int) (get_char() - '0');
  if(column < 1 || column > period)
    msgerror("Bad position.");
  else{
    prompt("What is the new letter? (ct,pt) ");
    read_line(tmp_str);
    if(sscanf(tmp_str, "%c,%c", &ct_letter, &pt_letter) != 2){
      msgerror("Bad format.");
    }
    else if(!isalpha(ct_letter) || !isalpha(pt_letter)){
      msgerror("Bad letter.");
    }
    else{
      sub_letter(ct_letter | ' ', pt_letter | ' ', column-1);
    }
  }
}

void vigenere::alphfit(){
  char tmp_str[STRINGLENGTH];
  int i, column;

  prompt("Fit which column? (* for all) ");
  read_line(tmp_str);
  if(*tmp_str == '*'){
    for(i = 0; i < period; i++){
      fit_column(i);
    }
  }
  else if(sscanf(tmp_str, "%d", &column) != 1){
    msgerror("Bad column.");
  }
  else if(column < 1 || column > period){
    msgerror("Column out of range.");
  }
  else{
    fit_column(column-1);
  }
}

void vigenere::undo(){
  int column, i;

  prompt("Which column? (* for all) ");
  column = get_char();
  if(column == '*'){
    for(i = 0; i < period; i++)
      key[i] = UNCHANGED;
  }
  else{
    key[column - '1'] = UNCHANGED;
  }
}

void vigenere::show_menu(){
  menu(1, "Options: (S)ubstitute  (A)pply frequency fit  (U)ndo column  (W)rite  (Q)uit");
}

void vigenere::show_cipher(){
  int i, line_length;
  char tmp_str[20];

  line_length = 80 / (period+1);
  i = 0;
  while(i < length){
    strncpy(tmp_str, cipher+i, period);
    tmp_str[period] = '\0';
    msgprint((i*(period+1)/period)%(line_length*(period+1)), (i/(line_length*period))*3 + 1, tmp_str);
    decode(tmp_str);
    msgprint((i*(period+1)/period)%(line_length*(period+1)), (i/(line_length*period))*3 + 0, tmp_str);
    i += period; 
  }
}

void vigenere::decode(char *message){
  int i;

  for(i = 0; i < strlen(message); i++){
    if(key[i] == UNCHANGED)
      message[i] = BLANK;
    else
      message[i] = get_pt(message[i], key[i]);
  }
}

void vigenere::show_key(){
  int i;
  char temp_str[3];

  for(i = 0; i < period; i++){
    if(key[i] != UNCHANGED){
      sprintf(temp_str, "%2d", key[i] - 'a');
      msgprint(3*i+4, 15, temp_str);
      put_char(key[i], 3*i + 5, 16);
    }
    else{
      msgprint(3*i+4, 15, " .");
      put_char(BLANK, 3*i + 5, 16);
    }
  }
}

char beaufort::get_key_letter(char ct_letter, char pt_letter){
  char key_letter='\0';

  ct_letter |= ' ';
  pt_letter |= ' ';

  if(isalpha(ct_letter) && isalpha(pt_letter))
    key_letter = (ct_letter + pt_letter - 'a' - 'a' + 26)%26+'a';
  else
    key_letter = '\0';
  
  return key_letter;
}

char variant::get_key_letter(char ct_letter, char pt_letter){
  char key_letter='\0';

  ct_letter |= ' ';
  pt_letter |= ' ';

  if(isalpha(ct_letter) && isalpha(pt_letter))
    key_letter = (pt_letter - ct_letter + 26)%26+'a';
  else
    key_letter = '\0';
  
  return key_letter;
}

char vigenere::get_key_letter(char ct_letter, char pt_letter){
  char key_letter='\0';

  ct_letter |= ' ';
  pt_letter |= ' ';

  if(isalpha(ct_letter) && isalpha(pt_letter))
    key_letter = (ct_letter - pt_letter + 26)%26+'a';
  else
    key_letter = '\0';

  return key_letter;
}

char porta::get_key_letter(char ct_letter, char pt_letter){
  char key_letter='\0';

  ct_letter |= ' ';
  pt_letter |= ' ';

  if(isalpha(ct_letter) && isalpha(pt_letter) && ((ct_letter - pt_letter > 12 || pt_letter - ct_letter > 12))){
    if(ct_letter < pt_letter){
      pt_letter -= 13;
      key_letter = (pt_letter - ct_letter)%13*2 + 'a';
    }
    else{
      ct_letter -= 13;
      key_letter = (ct_letter - pt_letter)%13*2 + 'a';
    }
  }
  else
    key_letter = '\0';

  return key_letter;
}

char variant::get_pt(char ct, char key_letter){
  char pt='\0';

  ct |= ' ';
  if(isalpha(key_letter)){
    key_letter |= ' ';
    pt = (ct + key_letter - 'a' - 'a'+26)%26 + 'a';
  }
  else
    pt = '\0';

  return pt;
}

char beaufort::get_pt(char ct, char key_letter){
  char pt='\0';

  ct |= ' ';
  if(isalpha(key_letter)){
    key_letter |= ' ';
    pt = (key_letter - ct + 26)%26 + 'a';
  }
  else
    pt = '\0';

  return pt;
}

char vigenere::get_pt(char ct, char key_letter){
  char pt='\0';

  ct |= ' ';

  if(isalpha(key_letter)){
    key_letter |= ' ';
    pt = (ct - key_letter + 26)%26 + 'a';
  }
  else
    pt = '\0';

  return pt;
}

char porta::get_pt(char ct, char key_letter){
  char pt='\0';

  if(isalpha(key_letter) && isalpha(ct)){
    key_letter |= ' ';
    key_letter -= 'a';
    key_letter /= 2;

    ct |= ' ';
    ct -= 'a';

    if(ct < 13){
      pt = (ct + key_letter)%13+'n';
    }
    else{
      ct -= 13;
      pt = (ct - key_letter + 13)%13+'a';
    }
  }
  else
    pt = '\0';

  return pt;
}

char variant::get_ct(char pt, char key_letter){
  char ct='\0';

  pt |= ' ';

  if(isalpha(key_letter)){
    key_letter |= ' ';
    ct = (pt - key_letter + 26)%26 + 'a';
  }
  else
    ct = '\0';

  return ct;
}

char beaufort::get_ct(char pt, char key_letter){
  char ct='\0';

  pt |= ' ';

  if(isalpha(key_letter)){
    key_letter |= ' ';
    ct = (key_letter - pt + 26)%26 + 'a';
  }
  else
    ct = '\0';

  return ct;
}

char vigenere::get_ct(char pt, char key_letter){
  char ct='\0';

  pt |= ' ';

  if(isalpha(key_letter)){
    key_letter |= ' ';
    ct = (pt + key_letter - 'a' - 'a'+26)%26 + 'a';
  }
  else
    ct = '\0';

  return ct;
}

char porta::get_ct(char pt, char key_letter){
  return get_pt(pt, key_letter);
}

void vigenere::sub_letter(char ct_letter, char pt_letter, int column){
  key[column] = get_key_letter(ct_letter, pt_letter);
  if(!key[column])
    key[column] = BLANK;
}

void vigenere::fit_column(int column){
  int i, k, fweights[26];
  static int weights[] = { 1863, 954, 1477, 1644, 2114, 1447, 1204, 1544,
   1869, 301, 477, 1544, 1398, 1892, 1869, 1431, 477, 1887, 1799, 1969,
   1431, 1114, 1204, 699, 1279, 0};
  int maxweight=0, maxpos=0;

  for(i = 0; i < 26; i++)
    fweights[i] = 0;

  /* Rotate through the 26 possible keys
  */
  for(k = 0; k < 26; k++){
    /* Rotate each letter in the column
    */
    for(i = column; i < length; i+=period){
      fweights[k] += weights[get_pt(cipher[i], k+'a') - 'a'];
    }
  }

  /* Find the largest weight and make that substitution.
  */

  for(i = 0; i < 26; i++){
    if(maxweight < fweights[i]){
      maxweight = fweights[i];
      maxpos = i;
    }
  }

  sub_letter(get_ct('a', maxpos+'a'), 'a', column);
}

void vigenere::copy_key(char *string){
  strcpy(string, key);
  while(strchr(string, UNCHANGED))
    *strchr(string, UNCHANGED) = BLANK;
}

void vigenere::decipher(char *string){
  int i=0;

  while(i < length){
    *string++ = get_pt(cipher[i], key[i%period]);
    i++;
  }
  *string = '\0';
}

void vigenere::read_key(FILE *fptr){
  fgets(key, period+2, fptr);
  key[period] = '\0';
  while(strchr(key, BLANK))
    *strchr(key, BLANK) = UNCHANGED;
}
