/* Auxiliary program for working around problems with Tcl/Tk on Windows */
#include <stdio.h>
#include <string.h>
#define USE_KADM5_API_VERSION 2
#include <kadm5/admin.h>
#include <com_err.h>
#include <k5-int.h>
#include <errno.h>
#include <stdlib.h>
#include <pwd.h>

/*
 * default keytab -- why is this here?
 */
#define DEFAULT_KEYTAB "WRFILE:/etc/v5srvtab"

time_t get_date();

/* Emulation procedures for Tcl */

#define TCL_OK 0
#define TCL_ERROR 1
#define TCL_STATIC 0

typedef struct _tcl_fake_interp {
    char *result;
    char *errorCode;
} Tcl_Interp;

typedef void* ClientData;

void Tcl_ResetResult(Tcl_Interp *interp)
{
    if (interp->result)
	free(interp->result);
    interp->result = malloc(128);
    interp->result[0] = '\0';

    if (interp->errorCode)
	free(interp->errorCode);
    interp->errorCode = malloc(128);
    interp->errorCode[0] = '\0';
}

void Tcl_SetResult(Tcl_Interp *interp, char *str, int type)
{
    if (str) {
	if (interp->result) free(interp->result);
	interp->result = strdup(str);
    }
}

void Tcl_AppendResult(Tcl_Interp *interp, ...)
{
    va_list argList;
    char *string, *p;
    int ilen, len;

    ilen = strlen(interp->result);
    len = ilen;

    va_start(argList, interp);
    while (1) {
	string = va_arg(argList, char *);
	if (!string) break;
	len += strlen(string);
    }
    va_end(argList);

    interp->result = realloc(interp->result, len);
    
    p = &(interp->result[ilen]);
    va_start(argList, interp);
    while (1) {
	string = va_arg(argList, char *);
	if (!string) break;
	strcpy(p, string);
	p += strlen(string);
    }
    va_end(argList);
}

void Tcl_AppendElement(Tcl_Interp *interp, char *string)
{
    int ilen, len;
    char *p;
    
    len = strlen(string);
    ilen = strlen(interp->result);
    interp->result = realloc(interp->result, ilen + len + 3);
    p = &interp->result[ilen];

    if (ilen != 0) 
	*p++ = ' ';
    
    if (len == 0 || strpbrk(string, " \f\n\t\r$;[{\"\\") != NULL) {
	*p++ = '{';
	strcpy(p, string);
	p[len] = '}';
	p[len+1] = '\0';
    } else {
	strcpy(p, string);
    }	
}

void Tcl_SetErrorCode(Tcl_Interp *interp, ...)
{
    va_list argList;
    char *string, *p;
    int ilen, len;

    ilen = strlen(interp->errorCode);
    len = ilen;

    va_start(argList, interp);
    while (1) {
	string = va_arg(argList, char *);
	if (!string) break;
	len += strlen(string) + 1;
    }
    va_end(argList);

    interp->errorCode = realloc(interp->errorCode, len);
    
    p = &interp->errorCode[ilen];
    va_start(argList, interp);
    while (1) {
	string = va_arg(argList, char *);
	if (!string) break;
	if (p != interp->errorCode) *p++ = ' ';
	strcpy(p, string);
	p += strlen(string);
    }
    va_end(argList);
}

int Tcl_GetInt(Tcl_Interp *interp, char *str, int *out)
{
    int i;
    char *end;
    errno = 0;
    i = (int)strtol(str, &end, 0);
    if (end == str) {
        if (interp != (Tcl_Interp *) NULL) {
	    Tcl_AppendResult(interp, "expected integer but got \"",
			     str, "\"", (char *) NULL);
        }
	return TCL_ERROR;
    }
    if (errno == ERANGE) {
        if (interp != (Tcl_Interp *) NULL) {
            interp->result = "integer value too large to represent";
            Tcl_SetErrorCode(interp, "ARITH", "IOVERFLOW",
                    interp->result, (char *) NULL);
        }
	return TCL_ERROR;
    }
    *out = i;
    return TCL_OK;
}

typedef int (Tcl_CmdProc)(ClientData c, Tcl_Interp* i, int ac, char *av[]);
char *procnames[255];
Tcl_CmdProc *procedures[255];
int nprocs = 0;

void Tcl_CreateCommand(Tcl_Interp *interp, char *procname, 
		       Tcl_CmdProc *procedure,
		       ClientData c1, ClientData c2)
{
    procnames[nprocs] = procname;
    procedures[nprocs] = procedure;
    nprocs++;
}

void Tcl_kadm5_init(Tcl_Interp *interp);

int main() 
{
    char buf[4096];
    char *argv[128];
    int i;
    Tcl_Interp interp;

    for(i=0;i<128;i++) argv[i] = malloc(1024);

    Tcl_kadm5_init(&interp);
    
    while(1) {
	int i, argc, ret;
	int nested_braces=0;
	char *p;
	char *argp;
	int done = 0;

	/* read in a line */
	if (fgets(buf, 4096, stdin) == NULL)
	    /* EOF */
	    exit(0);
	    
	/* break into pieces */
	p = buf;
	argc = 0;
	argp = &argv[0][0];
	while(!done) {
	    switch(*p) {
	    case ' ':
	    case '\n':
	    case '\r':
		if (!nested_braces) {
		    /* advance to next non space */
		    while(*p == ' ' || *p == '\n' || *p == '\r') {
			p++;
		    }
		    *argp = '\0';
		    if (*p != '\0') {
			argc++;
			argp = &argv[argc][0];
		    }
		    break;
		}
	    case '{':
		if (!nested_braces) {
		    p++;
		    nested_braces++;
		    break;
		}
	    case '}':
		if (nested_braces) {
		    p++;
		    nested_braces--;
		    break;
		}
	    case '\\':
		p++;
		*argp++ = *p;
		break;
	    case '\0':
		done = 1;
		*argp = '\0';
		break;
	    default:
		*argp++ = *p++;
	    }
	}
	/* look up procedure and run it */
	for(i=0; i<nprocs; i++) {
	    if (!strcmp(argv[0],procnames[i])) {
		Tcl_ResetResult(&interp);
		ret = (*procedures[i])(NULL, &interp, argc+1, argv);
		if (ret == TCL_OK) {
		    printf("ERR 0\n");
		} else {
		    printf("ERR 1 %s\n", interp.errorCode);
		}
		printf("%s\n", interp.result);
		printf("EOF\n");
		fflush(NULL);
		break;
	    }
	}
    }
}

/* end of Tcl procedures */

/* special struct to convert flag names for principals
   to actual krb5_flags for a principal */
struct pflag {
    char *flagname;		/* name of flag as typed to CLI */
    int flaglen;		/* length of string (not counting -,+) */
    krb5_flags theflag;		/* actual principal flag to set/clear */
    int set;			/* 0 means clear, 1 means set (on '-') */
};

static struct pflag flags[] = {
{"disallow_postdated",		18,	KRB5_KDB_DISALLOW_POSTDATED,	0},
{"disallow_forwardable",	20,	KRB5_KDB_DISALLOW_FORWARDABLE,	0},
{"disallow_tgs_req",		16,	KRB5_KDB_DISALLOW_TGT_BASED,	0},
{"disallow_renewable",		18,	KRB5_KDB_DISALLOW_RENEWABLE,	0},
{"disallow_proxiable",		18,	KRB5_KDB_DISALLOW_PROXIABLE,	0},
{"disallow_dup_skey",		17,	KRB5_KDB_DISALLOW_DUP_SKEY,	0},
{"disallow_tix",		12,	KRB5_KDB_DISALLOW_ALL_TIX,	0},
{"requires_preauth",		16,	KRB5_KDB_REQUIRES_PRE_AUTH,	0},
{"requires_hwauth",		15,	KRB5_KDB_REQUIRES_HW_AUTH,	0},
{"needchange",			10,	KRB5_KDB_REQUIRES_PWCHANGE,	0},
{"disallow_svr",		12,	KRB5_KDB_DISALLOW_SVR,		0},
{"password_changing_service",	25,	KRB5_KDB_PWCHANGE_SERVICE,	0}
};

static char *prflags[] = {
    "DISALLOW_POSTDATED",	/* 0x00000001 */
    "DISALLOW_FORWARDABLE",	/* 0x00000002 */
    "DISALLOW_TGT_BASED",	/* 0x00000004 */
    "DISALLOW_RENEWABLE",	/* 0x00000008 */
    "DISALLOW_PROXIABLE",	/* 0x00000010 */
    "DISALLOW_DUP_SKEY",	/* 0x00000020 */
    "DISALLOW_ALL_TIX",		/* 0x00000040 */
    "REQUIRES_PRE_AUTH",	/* 0x00000080 */
    "REQUIRES_HW_AUTH",		/* 0x00000100 */
    "REQUIRES_PWCHANGE",	/* 0x00000200 */
    "UNKNOWN_0x00000400",	/* 0x00000400 */
    "UNKNOWN_0x00000800",	/* 0x00000800 */
    "DISALLOW_SVR",		/* 0x00001000 */
    "PWCHANGE_SERVICE"		/* 0x00002000 */
};

/* This hash entry contains handle-specific information to keep around
   for each kadmin connection */
typedef struct _tcl_kadm5_server_info {
    krb5_context context;
    void *server_handle;
    char *default_realm;
} tcl_kadm5_server_info;

tcl_kadm5_server_info server_info;

static int put_server_info(Tcl_Interp *interp, 
			   tcl_kadm5_server_info *info, char **name)
{
    server_info.context = info->context;
    server_info.server_handle = info->server_handle;
    server_info.default_realm = info->default_realm;
    return TCL_OK;
}

static int remove_server_info(Tcl_Interp *interp, char *name) 
{
    return TCL_OK;
}

/* get handle from global variable */
#define GET_HANDLE() \
    void *server_handle; \
    krb5_context context; \
    char *def_realm; \
    server_handle = server_info.server_handle; \
    context = server_info.context; \
    def_realm = server_info.default_realm; 

/* Helper procedure for loading integer values from Tcl into longs */
int Tcl_GetLong(Tcl_Interp *interp, char *str, long *dest)
{
    int tmp, ret;
    ret = Tcl_GetInt(interp, str, &tmp);
    if (ret == TCL_OK) 
	*dest = tmp;
    return ret;
}

int tcl_kadm5_set_error(Tcl_Interp *interp, kadm5_ret_t code)
{
    char *code_string, *error_string;
  
    switch (code) {
    case KADM5_FAILURE: code_string = "KADM5_FAILURE"; break;
    case KADM5_AUTH_GET: code_string = "KADM5_AUTH_GET"; break;
    case KADM5_AUTH_ADD: code_string = "KADM5_AUTH_ADD"; break;
    case KADM5_AUTH_MODIFY: code_string = "KADM5_AUTH_MODIFY"; break;
    case KADM5_AUTH_DELETE: code_string = "KADM5_AUTH_DELETE"; break;
    case KADM5_AUTH_INSUFFICIENT:
	code_string = "KADM5_AUTH_INSUFFICIENT"; break;
    case KADM5_BAD_DB: code_string = "KADM5_BAD_DB"; break;
    case KADM5_DUP: code_string = "KADM5_DUP"; break;
    case KADM5_RPC_ERROR: code_string = "KADM5_RPC_ERROR"; break;
    case KADM5_NO_SRV: code_string = "KADM5_NO_SRV"; break;
    case KADM5_BAD_HIST_KEY: code_string = "KADM5_BAD_HIST_KEY"; break;
    case KADM5_NOT_INIT: code_string = "KADM5_NOT_INIT"; break;
    case KADM5_INIT: code_string = "KADM5_INIT"; break;
    case KADM5_BAD_PASSWORD: code_string = "KADM5_BAD_PASSWORD"; break;
    case KADM5_UNK_PRINC: code_string = "KADM5_UNK_PRINC"; break;
    case KADM5_UNK_POLICY: code_string = "KADM5_UNK_POLICY"; break;
    case KADM5_BAD_MASK: code_string = "KADM5_BAD_MASK"; break;
    case KADM5_BAD_CLASS: code_string = "KADM5_BAD_CLASS"; break;
    case KADM5_BAD_LENGTH: code_string = "KADM5_BAD_LENGTH"; break;
    case KADM5_BAD_POLICY: code_string = "KADM5_BAD_POLICY"; break;
    case KADM5_BAD_HISTORY: code_string = "KADM5_BAD_HISTORY"; break;
    case KADM5_BAD_PRINCIPAL: code_string = "KADM5_BAD_PRINCIPAL"; break;
    case KADM5_BAD_AUX_ATTR: code_string = "KADM5_BAD_AUX_ATTR"; break;
    case KADM5_PASS_Q_TOOSHORT: code_string = "KADM5_PASS_Q_TOOSHORT"; break;
    case KADM5_PASS_Q_CLASS: code_string = "KADM5_PASS_Q_CLASS"; break;
    case KADM5_PASS_Q_DICT: code_string = "KADM5_PASS_Q_DICT"; break;
    case KADM5_PASS_REUSE: code_string = "KADM5_PASS_REUSE"; break;
    case KADM5_PASS_TOOSOON: code_string = "KADM5_PASS_TOOSOON"; break;
    case KADM5_POLICY_REF: code_string = "KADM5_POLICY_REF"; break;
    case KADM5_PROTECT_PRINCIPAL: 
	code_string = "KADM5_PROTECT_PRINCIPAL"; break;
    case KADM5_BAD_SERVER_HANDLE: 
	code_string = "KADM5_BAD_SERVER_HANDLE"; break;
    case KADM5_BAD_STRUCT_VERSION:
	code_string = "KADM5_BAD_STRUCT_VERSION"; break;
    case KADM5_OLD_STRUCT_VERSION:
	code_string = "KADM5_OLD_STRUCT_VERSION"; break;
    case KADM5_NEW_STRUCT_VERSION:
	code_string = "KADM5_NEW_STRUCT_VERSION"; break;
    case KADM5_BAD_API_VERSION: code_string = "KADM5_BAD_API_VERSION"; break;
    case KADM5_OLD_LIB_API_VERSION:
	code_string = "KADM5_OLD_LIB_API_VERSION"; break;
    case KADM5_OLD_SERVER_API_VERSION:
	code_string = "KADM5_OLD_SERVER_API_VERSION"; break;
    case KADM5_NEW_LIB_API_VERSION:
	code_string = "KADM5_NEW_LIB_API_VERSION"; break;
    case KADM5_NEW_SERVER_API_VERSION:
	code_string = "KADM5_NEW_SERVER_API_VERSION"; break;
    case KADM5_SECURE_PRINC_MISSING:
	code_string = "KADM5_SECURE_PRINC_MISSING"; break;
    case KADM5_NO_RENAME_SALT: code_string = "KADM5_NO_RENAME_SALT"; break;
    case KADM5_BAD_CLIENT_PARAMS:
	code_string = "KADM5_BAD_CLIENT_PARAMS"; break;
    case KADM5_BAD_SERVER_PARAMS:
	code_string = "KADM5_BAD_SERVER_PARAMS"; break;
    case KADM5_AUTH_LIST: code_string = "KADM5_AUTH_LIST"; break;
    case KADM5_AUTH_CHANGEPW: code_string = "KADM5_AUTH_CHANGEPW"; break;
    case KADM5_GSS_ERROR: code_string = "KADM5_GSS_ERROR"; break;
    case OSA_ADB_DUP: code_string = "OSA_ADB_DUP"; break;
    case OSA_ADB_NOENT: code_string = "ENOENT"; break;
    case OSA_ADB_DBINIT: code_string = "OSA_ADB_DBINIT"; break;
    case OSA_ADB_BAD_POLICY: code_string = "Bad policy name"; break;
    case OSA_ADB_BAD_PRINC: code_string = "Bad principal name"; break;
    case OSA_ADB_BAD_DB: code_string = "Invalid database."; break;
    case OSA_ADB_XDR_FAILURE: code_string = "OSA_ADB_XDR_FAILURE"; break;
    case OSA_ADB_BADLOCKMODE: code_string = "OSA_ADB_BADLOCKMODE"; break;
    case OSA_ADB_CANTLOCK_DB: code_string = "OSA_ADB_CANTLOCK_DB"; break;
    case OSA_ADB_NOTLOCKED: code_string = "OSA_ADB_NOTLOCKED"; break;
    case OSA_ADB_NOLOCKFILE: code_string = "OSA_ADB_NOLOCKFILE"; break;
    case OSA_ADB_NOEXCL_PERM: code_string = "OSA_ADB_NOEXCL_PERM"; break;
    case KRB5_KDB_INUSE: code_string = "KRB5_KDB_INUSE"; break;
    case KRB5_KDB_UK_SERROR: code_string = "KRB5_KDB_UK_SERROR"; break;
    case KRB5_KDB_UK_RERROR: code_string = "KRB5_KDB_UK_RERROR"; break;
    case KRB5_KDB_UNAUTH: code_string = "KRB5_KDB_UNAUTH"; break;
    case KRB5_KDB_NOENTRY: code_string = "KRB5_KDB_NOENTRY"; break;
    case KRB5_KDB_ILL_WILDCARD: code_string = "KRB5_KDB_ILL_WILDCARD"; break;
    case KRB5_KDB_DB_INUSE: code_string = "KRB5_KDB_DB_INUSE"; break;
    case KRB5_KDB_DB_CHANGED: code_string = "KRB5_KDB_DB_CHANGED"; break;
    case KRB5_KDB_TRUNCATED_RECORD:
	code_string = "KRB5_KDB_TRUNCATED_RECORD"; break;
    case KRB5_KDB_RECURSIVELOCK:
	code_string = "KRB5_KDB_RECURSIVELOCK"; break;
    case KRB5_KDB_NOTLOCKED: code_string = "KRB5_KDB_NOTLOCKED"; break;
    case KRB5_KDB_BADLOCKMODE: code_string = "KRB5_KDB_BADLOCKMODE"; break;
    case KRB5_KDB_DBNOTINITED: code_string = "KRB5_KDB_DBNOTINITED"; break;
    case KRB5_KDB_DBINITED: code_string = "KRB5_KDB_DBINITED"; break;
    case KRB5_KDB_ILLDIRECTION: code_string = "KRB5_KDB_ILLDIRECTION"; break;
    case KRB5_KDB_NOMASTERKEY: code_string = "KRB5_KDB_NOMASTERKEY"; break;
    case KRB5_KDB_BADMASTERKEY: code_string = "KRB5_KDB_BADMASTERKEY"; break;
    case KRB5_KDB_INVALIDKEYSIZE:
	code_string = "KRB5_KDB_INVALIDKEYSIZE"; break;
    case KRB5_KDB_CANTREAD_STORED:
	code_string = "KRB5_KDB_CANTREAD_STORED"; break;
    case KRB5_KDB_BADSTORED_MKEY:
	code_string = "KRB5_KDB_BADSTORED_MKEY"; break;
    case KRB5_KDB_CANTLOCK_DB: code_string = "KRB5_KDB_CANTLOCK_DB"; break;
    case KRB5_KDB_DB_CORRUPT: code_string = "KRB5_KDB_DB_CORRUPT"; break;
    case KRB5_PARSE_ILLCHAR: code_string = "KRB5_PARSE_ILLCHAR"; break;
    case KRB5_PARSE_MALFORMED: code_string = "KRB5_PARSE_MALFORMED"; break;
    case KRB5KDC_ERR_S_PRINCIPAL_UNKNOWN: 
	code_string = "KRB5KDC_ERR_S_PRINCIPAL_UNKNOWN"; break;
    case KRB5_REALM_UNKNOWN: code_string = "KRB5_REALM_UNKNOWN"; break;
    case KRB5_KDC_UNREACH: code_string = "KRB5_KDC_UNREACH"; break;
    case KRB5_KDCREP_MODIFIED: code_string = "KRB5_KDCREP_MODIFIED"; break;
    case KRB5KRB_AP_ERR_BAD_INTEGRITY: 
	code_string  = "KRB5KRB_AP_ERR_BAD_INTEGRITY"; break;
    case KRB5KDC_ERR_C_PRINCIPAL_UNKNOWN: 
	code_string = "KRB5KDC_ERR_C_PRINCIPAL_UNKNOWN"; break;
    case KRB5_CONFIG_BADFORMAT: code_string = "KRB5_CONFIG_BADFORMAT"; break;
    case EINVAL: code_string = "EINVAL"; break;
    case ENOENT: code_string = "ENOENT"; break;
    default: code_string = "UNKNOWN"; break;
    }
    
    error_string = (char *) error_message(code);
    
    /* Set the result to the descriptive string, and the error code
       to a three element list describing the code type, the code
       string, and the description. */
    Tcl_ResetResult(interp);
    Tcl_AppendResult(interp, error_string, NULL);
    Tcl_SetErrorCode(interp, "KADM5", code_string, error_string, 0);
    return TCL_ERROR;
}     

/* this is a wrapper to go around krb5_parse_principal so we can set
   the default realm up properly */
krb5_error_code kadmin_parse_name(char *name, krb5_principal *principal,
				  char *def_realm, krb5_context context)
{
    char *cp, *fullname;
    krb5_error_code retval;
    
    fullname = (char *)malloc(strlen(name) + 1 + strlen(def_realm) + 1);
    if (fullname == NULL)
	return ENOMEM;
    strcpy(fullname, name);
    cp = strchr(fullname, '@');
    while (cp) {
	if (cp - fullname && *(cp - 1) != '\\')
	    break;
	else
	    cp = strchr(cp, '@');
    }
    if (cp == NULL) {
	strcat(fullname, "@");
	strcat(fullname, def_realm);
    }
    retval = krb5_parse_name(context, fullname, principal);
    free(fullname);
    return retval;
}

int tcl_kadm5_init(ClientData clientData,
		   Tcl_Interp *interp, int argc, char *argv[])
{
     kadm5_ret_t ret;
     int tcl_ret;
     void *handle;
     char *handle_name;
     char *whoami = argv[0];
     kadm5_config_params params;
     krb5_ccache cc;
     krb5_principal princ;
     char *luser, *canon, *cp;
     struct passwd *pw;
     char *ccache_name = NULL;
     char *keytab_name = NULL;
     char *def_realm = NULL;
     char *princstr = NULL;
     char *password = NULL;
     int freeprinc = 0;
     int use_keytab = 0;
     tcl_kadm5_server_info *info = malloc(sizeof(tcl_kadm5_server_info));

     if (info == NULL) {
	 Tcl_AppendResult(interp, "out of memory", NULL);
	 return TCL_ERROR;
     }

     /* Zero out the parameters */
     memset((char *) &params, 0, sizeof(params));

     if (krb5_init_context(&info->context)) {
       Tcl_AppendResult(interp, whoami, 
			": could not initialize krb5 library", NULL);
       return TCL_ERROR;
     }

     argv++, argc--;
     /* Loop through arguments and get parameters */
     while(argc) {
	 if (argv[0][0] == '-') {
	     switch (argv[0][1]) {
	     case 'r':
		 if (argc == 1) goto missingarg;
		 def_realm = argv[1];
		 break;
	     case 'c':
		 if (argc == 1) goto missingarg;
		 ccache_name = argv[1];
		 break;
	     case 'p':
		 if (strcmp(argv[0],"-princ") == 0) {
		     if (argc == 1) goto missingarg;
		     princstr = argv[1];
		     break;
		 } else if (strcmp(argv[0], "-password") == 0) {
		     if (argc == 1) goto missingarg;
		     password = argv[1];
		     break;
		 } else 
		     goto argerror;
	     case 'k':
		 if (argc == 1) goto missingarg;
		 use_keytab = 1;
		 if (argv[1][0] != (char)0) 
		     keytab_name = argv[1];
		 break;
	     default:
		 goto argerror;
	     }
	     argv += 2; argc -= 2;
	 } else 
	     goto argerror;
     }

     if (ccache_name && use_keytab) {
	 Tcl_AppendResult (interp, 
			   "cannot use both credentials cache and keytab",
			   NULL);
	 return TCL_ERROR;
     }

     /* find default realm */
     if (def_realm == NULL && 
	 krb5_get_default_realm(info->context, &def_realm)) {
       Tcl_AppendResult(interp, whoami, ": unable to get default realm", NULL);
       return TCL_ERROR;
     }
     params.mask |= KADM5_CONFIG_REALM;
     params.realm = def_realm;

     /* Find credentials cache */
     if (ccache_name == NULL) {
	 if (krb5_cc_default(info->context, &cc)) {
	     Tcl_AppendResult(interp, whoami,
			      ": unable to open default credentials cache",
			      NULL);
	     return TCL_ERROR;
	 }
     } else {
	 if (krb5_cc_resolve(info->context, ccache_name, &cc)) {
	     Tcl_AppendResult(interp, whoami, 
			      ": unable to open credentials cache ", 
			      ccache_name, NULL);
	     return TCL_ERROR;
	 }
     }

     /* Figure out the principal name.  This is taken in large part
        from the code in kadmin.c  */
     if (princstr == NULL) {
	 if (ccache_name != NULL &&
	     !krb5_cc_get_principal(info->context,cc,&princ)) {
	     if (krb5_unparse_name(info->context,princ,&princstr)) {
		 Tcl_AppendResult(interp, whoami, 
				  ": unable to get principal name", NULL);
		 krb5_free_principal(info->context, princ);
		 return TCL_ERROR;
	     }
	     krb5_free_principal(info->context,princ);
	     freeprinc++;
	 } else if (!use_keytab) {
	     if (krb5_sname_to_principal(info->context, NULL, "host", 
					 KRB5_NT_SRV_HST, &princ)) {
		 Tcl_AppendResult(interp, whoami, 
				  ": could not create host service principal",
				  NULL);
		 return TCL_ERROR;
	     }
	     if (krb5_unparse_name(info->context, princ, &princstr)) {
		 Tcl_AppendResult(interp, whoami,
				  ": unable to get principal name", NULL);
		 return TCL_ERROR;
	     }
	     krb5_free_principal(info->context,princ);
	     freeprinc++;
	 } else if (!krb5_cc_get_principal(info->context, cc, &princ)) {
	     char *realm = NULL;
	     if (krb5_unparse_name(info->context, princ, &canon)) {
		 Tcl_AppendResult(interp, whoami,
				  ": unable to get principal name", NULL);
		 krb5_free_principal(info->context, princ);
		 return TCL_ERROR;
	     }
	     /* strip out realm of principal if it's there */
	     realm = strchr(canon, '@');
	     while (realm) {
		 if (realm - canon && *(realm - 1) != '\\')
		     break;
		 else
		     realm = strchr(realm, '@');
	     }
	     if (realm)
		 *realm++ = '\0';
	     cp = strchr(canon, '/');
	     while (cp) {
		 if (cp - canon && *(cp - 1) != '\\')
		     break;
		 else
		     cp = strchr(cp, '/');
	     }
	     if (cp != NULL)
		 *cp = '\0';
	     princstr = (char*)malloc(strlen(canon) + 6 /* "/admin" */ +
				      (realm ? 1 + strlen(realm) : 0) + 1);
	     if (princstr == NULL) {
		 Tcl_AppendResult(interp, "out of memory", NULL);
		 return TCL_ERROR;
	    }
	    strcpy(princstr, canon);
	    strcat(princstr, "/admin");
	    if (realm) {
		strcat(princstr, "@");
		strcat(princstr, realm);
	    }
	    free(canon);
	    krb5_free_principal(info->context, princ);
	    freeprinc++;
	 } else if (luser = getenv("USER")) {
	     princstr = (char *) malloc(strlen(luser) + 7 /* "/admin@" */ +
					strlen(def_realm) + 1);
	     if (princstr == NULL) {
		 Tcl_AppendResult(interp, "out of memory", NULL);
		 return TCL_ERROR;
	     }
	     strcpy(princstr, luser);
	     strcat(princstr, "/admin");
	     strcat(princstr, "@");
	     strcat(princstr, def_realm);
	     freeprinc++;
	 } else if (pw = getpwuid(getuid())) {
	     princstr = (char *)malloc(strlen(pw->pw_name) +
				       7 + /* "/admin@" */
				       strlen(def_realm) + 1);
	     if (princstr == NULL) {
		 Tcl_AppendResult(interp, "out of memory", NULL);
		 return TCL_ERROR;
	     }
	     strcpy(princstr, pw->pw_name);
	     strcat(princstr, "/admin@");
	     strcat(princstr, def_realm);
	     freeprinc++;
	 } else {
	     Tcl_AppendResult(interp, 
			      "unable to figure out a principal name", NULL);
	     return TCL_ERROR;
	 }

     }
     
     /* Initialize the kadm5 connection.  If given a ccache, use it. */
     if (ccache_name)
	 ret = kadm5_init_with_creds(princstr, cc, KADM5_ADMIN_SERVICE,
				     &params, KADM5_STRUCT_VERSION,
				     KADM5_API_VERSION_2, 
				     &info->server_handle);
     else if (use_keytab)
	 ret = kadm5_init_with_skey(princstr, keytab_name,
				    KADM5_ADMIN_SERVICE, &params,
				    KADM5_STRUCT_VERSION,
				    KADM5_API_VERSION_2, 
				    &info->server_handle);
     else 
	 ret = kadm5_init_with_password(princstr, password, 
					KADM5_ADMIN_SERVICE, &params, 
					KADM5_STRUCT_VERSION,
					KADM5_API_VERSION_2, 
					&info->server_handle);

     if (ret != KADM5_OK)
	 return tcl_kadm5_set_error(interp, ret);

     if (freeprinc) free(princstr);
     
     info->default_realm = def_realm;

     /* register the WRFILE keytab type and set it as the default */
     {
       extern char *krb5_defkeyname;
       extern krb5_kt_ops krb5_ktf_writable_ops;
       
       krb5_defkeyname = DEFAULT_KEYTAB;

       ret = krb5_kt_register(info->context, &krb5_ktf_writable_ops);
       if (ret)
	 tcl_kadm5_set_error(interp, ret);
     }
     
     if ((tcl_ret = put_server_info(interp, info, &handle_name))
	 != TCL_OK) {
	 return tcl_ret;
     }
     
     Tcl_SetResult(interp, "initialized", TCL_STATIC);
     return TCL_OK;

 missingarg:
     Tcl_AppendResult(interp, "missing value for ", argv[0], NULL);
     return TCL_ERROR;
 argerror:
     Tcl_AppendResult(interp, "bad option \"", argv[0], "\": ",
		      "should be -princ, -password, -keytab, -realm, -ccache",
		      NULL);
     return TCL_ERROR;
}

int tcl_kadm5_destroy(ClientData clientData, Tcl_Interp *interp,
			   int argc, char *argv[])
{
    kadm5_ret_t ret;

    GET_HANDLE();

    if (argc != 1) {
	Tcl_AppendResult(interp, "wrong # args: \"", 
			 argv[0], "\" does not accept arguments", NULL);
	return TCL_ERROR;
    }

    ret = kadm5_destroy(server_handle);

    if (ret != KADM5_OK) 
	return tcl_kadm5_set_error(interp, ret);
 
    return remove_server_info(interp, "");
}

int tcl_kadm5_durtostr(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[])
{
    static char out[128];
    char *cp;
    int days, hours, minutes, seconds;
    int duration, ret;
    
    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " duration\"", NULL);
	return TCL_ERROR;
    }

    if ((ret = Tcl_GetInt(interp, argv[1], &duration)) != TCL_OK)
	return ret;

    days = duration / (24 * 3600);
    duration %= 24 * 3600;
    hours = duration / 3600;
    duration %= 3600;
    minutes = duration / 60;
    duration %= 60;
    seconds = duration;

    cp = out;

    if (days != 0)
      cp += sprintf(cp, "%d day%s", days, (days == 1 ? "" : "s"));
    
    if (hours != 0) {
      if (cp != out)
	*cp++ = ' ';
      cp += sprintf(cp, "%d hour%s", hours, (hours == 1 ? "" : "s"));
    }

    if (minutes != 0) {
      if (cp != out)
	*cp++ = ' ';
      cp += sprintf(cp, "%d minute%s", minutes, (minutes == 1 ? "" : "s"));
    }

    if (seconds != 0) {
      if (cp != out)
	*cp++ = ' ';
      cp += sprintf(cp, "%d second%s", seconds, (seconds == 1 ? "" : "s"));
    }

    Tcl_AppendResult(interp, out, NULL);
    return TCL_OK;
}
    
int tcl_kadm5_datetostr(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[])
{
    static char out[30];
    struct tm *tm;
    time_t lcltim;
    int when;
    int ret;
    
    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " date\"", NULL);
	return TCL_ERROR;
    }
    
    if ((ret = Tcl_GetInt(interp, argv[1], &when)) != TCL_OK)
	return ret;

    if (when == 0) {
	Tcl_AppendResult(interp, "never", NULL);
    } else {
	lcltim = when;
	tm = localtime(&lcltim);
	strftime(out, 30, "%a %b %d %H:%M:%S %Z %Y", tm);
	Tcl_AppendResult(interp, out, NULL);
    }

    return TCL_OK;
}

int tcl_kadm5_strtodate(ClientData clientData, Tcl_Interp *interp,
			int argc, char *argv[])
{
    time_t date;
    char buf[20];

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], "string\"", NULL);
	return TCL_ERROR;
    }

    date = get_date(argv[1], NULL);
    if (date == (time_t)-1) {
	Tcl_AppendResult(interp, "Invalid date spec \"", argv[1], "\"", NULL);
	return TCL_ERROR;
    }

    sprintf(buf,"%d",date);
    Tcl_AppendResult(interp, buf, NULL);
    return TCL_OK;
}

int tcl_kadm5_strtodur(ClientData clientData, Tcl_Interp *interp,
			int argc, char *argv[])
{
    time_t date, now;
    char buf[20];

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], "string\"", NULL);
	return TCL_ERROR;
    }

    time(&now);
    date = get_date(argv[1], NULL);
    if (date == (time_t)(-1)) {
	Tcl_AppendResult(interp, "Invalid time spec \"", argv[1], "\"", NULL);
	return TCL_ERROR;
    }
    
    sprintf(buf, "%d", date - now);
    Tcl_AppendResult(interp, buf, NULL);
    return TCL_OK;
}

int tcl_kadm5_getprinc(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[])
{
    kadm5_principal_ent_rec dprinc;
    krb5_principal princ;
    krb5_error_code ret;
    char *canon, *modcanon;
    char buf[BUFSIZ];
    int i;

    GET_HANDLE();

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], "principal\"", NULL);
	return TCL_ERROR;
    }

    memset(&dprinc, 0, sizeof(dprinc));
    memset(&princ, 0, sizeof(princ));

    if ((ret = kadmin_parse_name(argv[1], &princ, def_realm, context)) ||
	(ret = krb5_unparse_name(context, princ, &canon)))
	return tcl_kadm5_set_error(interp, ret);

    ret = kadm5_get_principal(server_handle, princ, &dprinc, 
			      KADM5_PRINCIPAL_NORMAL_MASK | KADM5_KEY_DATA);
    krb5_free_principal(context, princ);
    if (ret) {
	free(canon);
	return tcl_kadm5_set_error(interp, ret);
    }
    if ((ret = krb5_unparse_name(context, dprinc.mod_name, &modcanon))) {
	kadm5_free_principal_ent(server_handle, &dprinc);
	free(canon);
	return tcl_kadm5_set_error(interp, ret);
    }
    
    /* Building the result */
    Tcl_AppendResult(interp, canon, NULL);
    sprintf(buf, " %d %d %d %d", dprinc.princ_expire_time,
	    dprinc.last_pwd_change, dprinc.pw_expiration, dprinc.max_life);
    Tcl_AppendResult(interp, buf, NULL);
    Tcl_AppendElement(interp, modcanon);
    sprintf(buf, " %d", dprinc.mod_date, 0);
    Tcl_AppendResult(interp, buf, NULL);

    /* build list of attributes */
    Tcl_AppendResult(interp, " {", 0);
    for(i = 0; i < sizeof(prflags) / sizeof(char *); i++) {
	if (dprinc.attributes & (krb5_flags) 1 << i)
	    Tcl_AppendElement(interp, prflags[i]);
    }
    Tcl_AppendResult(interp, "}", 0);

    sprintf(buf, " %d %d", dprinc.kvno, dprinc.mkvno);
    Tcl_AppendResult(interp, buf, 0);
    Tcl_AppendElement(interp, (dprinc.policy?dprinc.policy:""));
    sprintf(buf, " %d %d %d %d", dprinc.max_renewable_life, 
	    dprinc.last_success, dprinc.last_failed, dprinc.fail_auth_count);
    Tcl_AppendResult(interp, buf, 0);

    /* build list of keys */
    Tcl_AppendResult(interp, " {", 0);
    for(i = 0; i < dprinc.n_key_data; i++) {
	char enctype[BUFSIZ], salttype[BUFSIZ];
	if (krb5_enctype_to_string(dprinc.key_data[i].key_data_type[0],
				   enctype, sizeof(enctype)))
	    sprintf(enctype, "%d", dprinc.key_data[i].key_data_type[0]);
	if (dprinc.key_data[i].key_data_ver > 1) {
	    if (krb5_salttype_to_string(dprinc.key_data[i].key_data_type[1],
					salttype, sizeof(salttype)))
		sprintf(salttype, "%d", dprinc.key_data[i].key_data_type[1]);
	} else {
	    sprintf(salttype,"");
	}
	sprintf(buf, "%d %d {%s} {%s}", 
		dprinc.key_data[i].key_data_ver,
		dprinc.key_data[i].key_data_kvno,
		enctype, salttype);
	Tcl_AppendElement(interp, buf);
    }
    Tcl_AppendResult(interp, "}", NULL);

    return TCL_OK;
}

int tcl_kadm5_getprincs(ClientData clientData, Tcl_Interp *interp,
			int argc, char *argv[])
{
    krb5_error_code ret;
    char *exp, **names;
    int i, count;

    GET_HANDLE();

    exp = NULL;
    if (argc > 2) {
	Tcl_AppendResult(interp, "wrong # options: should be \"", argv[0],
			 " [searchexp]\"", NULL);
	return TCL_ERROR;
    }
    if (argc == 2) exp = argv[1];

    ret = kadm5_get_principals(server_handle, exp, &names, &count);
    if (ret)
	return tcl_kadm5_set_error(interp, ret);
    
    for (i = 0; i < count; i++)
	Tcl_AppendElement(interp, names[i]);
    kadm5_free_name_list(server_handle, names, count);
    return TCL_OK;
}

int tcl_kadm5_getnextprincs(ClientData clientData, Tcl_Interp *interp,
			    int argc, char *argv[])
{
    krb5_error_code ret;
    char *start, **names;
    int i, count, tcl_ret;

    GET_HANDLE();

    start = NULL;
    if (argc > 3 || argc < 2) {
	Tcl_AppendResult(interp, "wrong # options: should be \"", argv[0],
			 " n [startprinc]\"", NULL);
	return TCL_ERROR;
    }
    if (argc == 3) start = argv[2];
    if ((tcl_ret = Tcl_GetInt(interp, argv[1], &count)) != TCL_OK)
	return tcl_ret;

    ret = kadm5_get_next_principals(server_handle, start, &names, &count);
    if (ret)
	return tcl_kadm5_set_error(interp, ret);
    
    for (i = 0; i < count; i++)
	Tcl_AppendElement(interp, names[i]);
    kadm5_free_name_list(server_handle, names, count);
    return TCL_OK;
}

int tcl_kadm5_parse_princ_args(Tcl_Interp *interp, int argc, char *argv[],
			       kadm5_principal_ent_t princ, long *mask,
			       char **pass, int *randkey,
			       char *def_realm, krb5_context context)
{
    int i, j;
    krb5_error_code ret;
    int tcl_ret;

    *mask = 0;
    *pass = NULL;
    *randkey = 0;
    for (i=1; i < argc - 1; i++) {
	switch (strlen(argv[i])) {
	case 5:
	    if (!strcmp("-kvno", argv[i])) {
		if (++i > argc - 2)
		    goto missingarg;
		else {
		    if ((tcl_ret = Tcl_GetInt(interp, argv[i], 
					      &princ->kvno)) 
			!= TCL_OK) return tcl_ret;
		    *mask |= KADM5_KVNO;
		    continue;
		}
	    }
	case 7:
	    if (!strcmp("-expire", argv[i])) {
		if (++i > argc - 2)
		    goto missingarg;
		else {
		    if ((tcl_ret = Tcl_GetInt(interp, argv[i], 
					      &princ->princ_expire_time)) 
			!= TCL_OK) return tcl_ret;
		    *mask |= KADM5_PRINC_EXPIRE_TIME;
		    continue;
		}
	    } else if (!strcmp("-policy", argv[i])) {
		if (++i > argc - 2)
		    goto missingarg;
		else {
		    princ->policy = argv[i];
		    *mask |= KADM5_POLICY;
		    continue;
		}
	    }
	case 8: 
	    if (!strcmp("-maxlife", argv[i])) {
		if (++i > argc - 2)
		    goto missingarg;
		else {
		    if ((tcl_ret = Tcl_GetInt(interp, argv[i],
					      &princ->max_life)) != TCL_OK)
			return tcl_ret;
		    *mask |= KADM5_MAX_LIFE;
		    continue;
		}
	    } else if (!strcmp("-randkey", argv[i])) {
		++*randkey;
		continue;
	    }
	case 9:
	    if (!strcmp("-pwexpire", argv[i])) {
		if (++i > argc - 2)
		    goto missingarg;
		else {
		    if ((tcl_ret = Tcl_GetInt(interp, argv[i],
					      &princ->pw_expiration))
			!= TCL_OK) return tcl_ret;
		    *mask |= KADM5_PW_EXPIRATION;
		    continue;
		}
	    } else if (!strcmp("-password", argv[i])) {
		if (++i > argc - 2)
		    goto missingarg;
		else {
		    *pass = argv[i];
		    continue;
		}
	    }
	case 12:
	    if (!strcmp("-clearpolicy", argv[i])) {
		princ->policy = NULL;
		*mask |= KADM5_POLICY_CLR;
		continue;
	    }
	case 13:
	    if (!strcmp("-maxrenewlife", argv[i])) {
		if (++i > argc - 2)
		    goto missingarg;
		else {
		    if ((tcl_ret = Tcl_GetInt(interp, argv[i],
					      &princ->max_renewable_life)) != TCL_OK)
			return tcl_ret;
		    *mask |= KADM5_MAX_RLIFE;
		    continue;
		}
	    }
	}
	/* test the flags */
	for (j = 0; j < sizeof (flags) / sizeof (struct pflag); j++) {
	    if (strlen(argv[i]) == flags[j].flaglen + 1 &&
		!strcmp(flags[j].flagname,
			&argv[i][1] /* strip off leading + or - */)) {
		if (flags[j].set && argv[i][0] == '-' ||
		    !flags[j].set && argv[i][0] == '+') {
		    princ->attributes |= flags[j].theflag;
		    *mask |= KADM5_ATTRIBUTES;
		    break;
		} else if (flags[j].set && argv[i][0] == '+' ||
			   !flags[j].set && argv[i][0] == '-') {
		    princ->attributes &= ~flags[j].theflag;
		    *mask |= KADM5_ATTRIBUTES;
		    break;
		} else {
		    Tcl_AppendResult(interp, "bad option \"",argv[i],"\"",
				     NULL);
		    return TCL_ERROR;
		}
	    }
	}
    }
    if (i != argc - 1) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " [-options ...] principal\"", NULL);
	return TCL_ERROR;
    }

    ret = kadmin_parse_name(argv[i], &princ->principal, def_realm, context);
    if (ret)
	return tcl_kadm5_set_error(interp, ret);
    
    return TCL_OK;

 missingarg:
    Tcl_AppendResult(interp, "missing value for ", argv[i-1], NULL);
    return TCL_ERROR;
}
      
int tcl_kadm5_addprinc(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[])
{
    kadm5_principal_ent_rec princ;
    long mask;
    int randkey = 0;
    char *pass;
    krb5_error_code ret;
    int tcl_ret;

    GET_HANDLE();

    /* Zero files in principal */
    memset(&princ, 0, sizeof(princ));

    if (argc == 1) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], "[options ...] principal\"", NULL);
	return TCL_ERROR;
    }

    princ.attributes = 0;
    if ((tcl_ret = tcl_kadm5_parse_princ_args(interp, argc, argv, &princ, 
					      &mask, &pass, &randkey, 
					      def_realm, context)) != TCL_OK)
	return tcl_ret;

    if (randkey) {
	princ.attributes |= KRB5_KDB_DISALLOW_ALL_TIX;
	mask |= KADM5_ATTRIBUTES;
	pass = "dummy";
    } else if (pass == NULL) {
	Tcl_AppendResult(interp, 
			 "bad option: no -password or -randkey given for \"",
			 argv[0], "\"", NULL);
	return TCL_ERROR;
    }

    mask |= KADM5_PRINCIPAL;
    ret = kadm5_create_principal(server_handle, &princ, mask, pass);
    if (ret) {
	krb5_free_principal(context, princ.principal);
	return tcl_kadm5_set_error(interp, ret);
    }
    if (randkey) {
	ret = kadm5_randkey_principal(server_handle, princ.principal, 
				      NULL, NULL);
	if (ret) {
	    krb5_free_principal(context, princ.principal);
	    return tcl_kadm5_set_error(interp, ret);
	}
	princ.attributes &= ~KRB5_KDB_DISALLOW_ALL_TIX;
	mask = KADM5_ATTRIBUTES;
	ret = kadm5_modify_principal(server_handle, &princ, mask);
	if (ret) {
	    krb5_free_principal(context, princ.principal);
	    return tcl_kadm5_set_error(interp, ret);
	}
    }	
    krb5_free_principal(context, princ.principal);

    return TCL_OK;
}

int tcl_kadm5_modprinc(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[])
{
    kadm5_principal_ent_rec princ, oldprinc;
    krb5_principal kprinc;
    long mask;
    int randkey = 0;
    char *pass;
    krb5_error_code ret;
    int tcl_ret;

    GET_HANDLE();

    /* Zero files in principal */
    memset(&princ, 0, sizeof(princ));

    if (argc == 1) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], "[options ...] principal\"", NULL);
	return TCL_ERROR;
    }

    if ((ret = kadmin_parse_name(argv[argc-1], &kprinc, def_realm, context)))
	return tcl_kadm5_set_error(interp,ret);

    ret = kadm5_get_principal(server_handle, kprinc, &oldprinc, 
			      KADM5_PRINCIPAL_NORMAL_MASK);
    krb5_free_principal(context,kprinc);
    if (ret) 
	return tcl_kadm5_set_error(interp, ret);

    princ.attributes = oldprinc.attributes;
    kadm5_free_principal_ent(server_handle, &oldprinc);
    if ((tcl_ret = tcl_kadm5_parse_princ_args(interp, argc, argv, &princ, 
					      &mask, &pass, &randkey, 
					      def_realm, context)) != TCL_OK)
	return tcl_ret;

    if (randkey) {
	Tcl_AppendResult(interp, "bad option: -randkey not allowed for \"",
			 argv[0], "\"", NULL);
	return TCL_ERROR;
    }

    ret = kadm5_modify_principal(server_handle, &princ, mask);
    krb5_free_principal(context, princ.principal);
    if (ret) {
	return tcl_kadm5_set_error(interp, ret);
    }

    return TCL_OK;
}

int tcl_kadm5_delprinc(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[])
{
    kadm5_ret_t ret;
    krb5_principal princ;
   
    GET_HANDLE();

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " principal\"", NULL);
	return TCL_ERROR;
    }

    if ((ret = kadmin_parse_name(argv[1], &princ, def_realm, context)))
	return tcl_kadm5_set_error(interp, ret);

    ret = kadm5_delete_principal(server_handle, princ);
    krb5_free_principal(context, princ);
    if (ret)
	return tcl_kadm5_set_error(interp, ret);

    return TCL_OK;
}

int tcl_kadm5_cpw(ClientData clientData, Tcl_Interp *interp,
		  int argc, char *argv[])
{
    kadm5_ret_t ret;
    krb5_principal princ;
    int randkey = 0;
    
    GET_HANDLE();

    if (argc < 3 || argc > 4) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], 
			 " [-randkey|-password pw] principal\"", NULL);
	return TCL_ERROR;
    }

    if ((ret = kadmin_parse_name(argv[argc-1], &princ, 
				 def_realm, context)))
	return tcl_kadm5_set_error(interp, ret);

    if (!strcmp("-randkey", argv[1])) 
	ret = kadm5_randkey_principal(server_handle, princ, NULL, NULL);
    else if (!strcmp("-password", argv[1]))
	ret = kadm5_chpass_principal(server_handle, princ, argv[2]);
    else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
			 "\": should be -randkey or -password", NULL);
	return TCL_ERROR;
    }
    
    krb5_free_principal(context, princ);
    if (ret)
	return tcl_kadm5_set_error(interp, ret);
    else 
	return TCL_OK;
}

int tcl_kadm5_parse_policy_args(Tcl_Interp *interp, int argc, char *argv[],
				kadm5_policy_ent_t policy, long *mask)
{
    int i, tcl_ret;

    *mask = 0;
    for (i=1; i < argc - 1; i++) {
	switch (strlen(argv[i])) {
	case 8:
	    if (!strcmp(argv[i], "-maxlife")) {
		if (++i > argc - 2)
		    goto missingarg;
		else {
		    if ((tcl_ret = Tcl_GetLong(interp, argv[i], 
					       &policy->pw_max_life))
			!= TCL_OK)
			return tcl_ret;
		    *mask |= KADM5_PW_MAX_LIFE;
		    continue;
		}
	    } else if (!strcmp(argv[i], "-minlife")) {
		if (++i > argc - 2)
		    goto missingarg;
		else {
		    if ((tcl_ret = Tcl_GetLong(interp, argv[i], 
					       &policy->pw_min_life))
			!= TCL_OK)
			return tcl_ret;
		    *mask |= KADM5_PW_MIN_LIFE;
		    continue;
		}
	    } else if (!strcmp(argv[i], "-history")) {
		if (++i > argc - 2)
		    goto missingarg;
		else {
		    if ((tcl_ret = Tcl_GetLong(interp, argv[i], 
					       &policy->pw_history_num))
			!= TCL_OK)
			return tcl_ret;
		    *mask |= KADM5_PW_HISTORY_NUM;
		    continue;
		}
	    }
	case 10:
	    if (!strcmp(argv[i], "-minlength")) {
		if (++i > argc - 2)
		    goto missingarg;
		else {
		    if ((tcl_ret = Tcl_GetLong(interp, argv[i], 
					       &policy->pw_min_length))
			!= TCL_OK)
			return tcl_ret;
		    *mask |= KADM5_PW_MIN_LENGTH;
		    continue;
		}
	    }
	case 11:
	    if (!strcmp(argv[i], "-minclasses")) {
		if (++i > argc - 2)
		    goto missingarg;
		else {
		    if ((tcl_ret = Tcl_GetLong(interp, argv[i], 
					       &policy->pw_min_classes))
			!= TCL_OK)
			return tcl_ret;
		    *mask |= KADM5_PW_MIN_CLASSES;
		    continue;
		}
	    }
	default:
	    Tcl_AppendResult(interp, "bad option \"",argv[i],"\"", NULL);
	    return TCL_ERROR;
	}
    }
    if (i != argc - 1) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " [-options ...] policy\"", NULL);
	return TCL_ERROR;
    } else
	return TCL_OK;

 missingarg:
    Tcl_AppendResult(interp, "missing value for ", argv[i-1], NULL);
    return TCL_ERROR;
}

int tcl_kadm5_addpol(ClientData clientData, Tcl_Interp *interp,
		     int argc, char *argv[])
{
    krb5_error_code ret;
    int tcl_ret;
    long mask;
    kadm5_policy_ent_rec policy;

    GET_HANDLE();

    memset(&policy, 0, sizeof(policy));
    if ((tcl_ret = tcl_kadm5_parse_policy_args(interp, argc, argv, 
					       &policy, &mask)) != TCL_OK)
	return tcl_ret;
	
    policy.policy = argv[argc - 1];
    mask |= KADM5_POLICY;
    if ((ret = kadm5_create_policy(server_handle, &policy, mask)))
	return tcl_kadm5_set_error(interp, ret);
    else return TCL_OK;
}

int tcl_kadm5_modpol(ClientData clientData, Tcl_Interp *interp,
		     int argc, char *argv[])
{
    krb5_error_code ret;
    int tcl_ret;
    long mask;
    kadm5_policy_ent_rec policy;

    GET_HANDLE();

    memset(&policy, 0, sizeof(policy));
    if ((tcl_ret = tcl_kadm5_parse_policy_args(interp, argc, argv, 
					       &policy, &mask)) != TCL_OK)
	return tcl_ret;
	
    policy.policy = argv[argc - 1];
    if ((ret = kadm5_modify_policy(server_handle, &policy, mask)))
	return tcl_kadm5_set_error(interp, ret);
    else 
	return TCL_OK;
}

int tcl_kadm5_delpol(ClientData clientData, Tcl_Interp *interp,
		     int argc, char *argv[])
{
    krb5_error_code ret;
    
    GET_HANDLE();

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], 
			 " policy\"", NULL);
	return TCL_ERROR;
    }

    if ((ret = kadm5_delete_policy(server_handle, argv[1])))
	return tcl_kadm5_set_error(interp, ret);
    else 
	return TCL_OK;
}
    
int tcl_kadm5_getpol(ClientData clientData, Tcl_Interp *interp,
		     int argc, char *argv[])
{
    krb5_error_code ret;
    kadm5_policy_ent_rec policy;
    char buf[50];

    GET_HANDLE();

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], 
			 " policy\"", NULL);
	return TCL_ERROR;
    }

    if ((ret = kadm5_get_policy(server_handle, argv[1], &policy))) 
	return tcl_kadm5_set_error(interp, ret);
 
    Tcl_AppendResult(interp, policy.policy, NULL);
    sprintf(buf, " %d %d %d %d %d %d", policy.pw_max_life, 
	    policy.pw_min_life, policy.pw_min_length, policy.pw_min_classes,
	    policy.pw_history_num, policy.policy_refcnt);
    Tcl_AppendResult(interp, buf, NULL);

    kadm5_free_policy_ent(server_handle, &policy);
    return TCL_OK;
}

int tcl_kadm5_getpols(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[])
{
    krb5_error_code ret;
    char *exp, **names;
    int i, count;

    GET_HANDLE();

    exp = NULL;
    if (argc > 2) {
	Tcl_AppendResult(interp, "wrong # options: should be \"", argv[0],
			 " [searchexp]\"", NULL);
	return TCL_ERROR;
    }
    if (argc == 2) exp = argv[1];

    ret = kadm5_get_policies(server_handle, exp, &names, &count);
    if (ret)
	return tcl_kadm5_set_error(interp, ret);
    
    for (i = 0; i < count; i++)
	Tcl_AppendElement(interp, names[i]);
    kadm5_free_name_list(server_handle, names, count);
    return TCL_OK;
}

int
process_keytab(Tcl_Interp *interp, krb5_context context, 
	       char **keytab_str, krb5_keytab *keytab) 
{
    int code;
    
    if (*keytab_str == NULL) {
	if (! (*keytab_str = strdup(DEFAULT_KEYTAB))) {
	    Tcl_AppendResult(interp, "out of memory", NULL);
	    return TCL_ERROR;
	}
	code = krb5_kt_default(context, keytab);
	if (code != 0) {
	    free(*keytab_str);
	    return tcl_kadm5_set_error(interp, code);
	}
    } else {
	if (strchr(*keytab_str, ':') != NULL) {
	    if (! (*keytab_str = strdup(*keytab_str))) {
		Tcl_AppendResult(interp, "out of memory", NULL);
		return TCL_ERROR;
	    }
	} else {
	    char *tmp = *keytab_str;
	    
	    if (! (*keytab_str = (char *) malloc(strlen("WRFILE:")+
						 strlen(tmp)+1))) {
		Tcl_AppendResult(interp, "out of memory", NULL);
		return TCL_ERROR;
	    }
	    sprintf(*keytab_str, "WRFILE:%s", tmp);
	}

	code = krb5_kt_resolve(context, *keytab_str, keytab);
	if (code != 0) {
	    free(keytab_str);
	    return tcl_kadm5_set_error(interp, code);
	}
    }
    
    return TCL_OK;
}

int add_keytab_principal(Tcl_Interp *interp, char *keytab_str, 
			 krb5_keytab keytab, char *princ_str)
{
  krb5_principal princ;
  kadm5_principal_ent_rec princ_rec;
  krb5_keytab_entry new_entry;
  krb5_keyblock *keys;
  int ret, nkeys, i;
  
  GET_HANDLE();
  
  ret = krb5_parse_name(context, princ_str, &princ);
  if (ret)
    goto handle_err;
  
  ret = kadm5_randkey_principal(server_handle, princ, &keys, &nkeys);
  if (ret)
    goto handle_err;
  
  if ((ret = kadm5_get_principal(server_handle, princ, &princ_rec,
				 KADM5_PRINCIPAL_NORMAL_MASK)))
    goto handle_err;
  
  for(i = 0; i < nkeys; i++) {
    memset((char *) &new_entry, 0, sizeof(new_entry));
    new_entry.principal = princ;
    new_entry.key = keys[i];
    new_entry.vno = princ_rec.kvno;
    
    ret = krb5_kt_add_entry(context, keytab, &new_entry);
    if (ret != 0) {
      (void) kadm5_free_principal_ent(server_handle, &princ_rec);
      goto handle_err;
    }
  }
  
  ret = kadm5_free_principal_ent(server_handle, &princ_rec);
  
handle_err:
  if (nkeys) {
    for (i = 0; i < nkeys; i++)
      krb5_free_keyblock_contents(context, &keys[i]);
    free(keys);
  }
  if (princ)
    krb5_free_principal(context, princ);
  if (ret)
    return tcl_kadm5_set_error(interp, ret);
  else return TCL_OK;
}

int remove_keytab_principal(Tcl_Interp *interp, char *keytab_str, 
			    krb5_keytab keytab, char *princ_str, 
			    char *kvno_str)
{
    krb5_principal princ;
    krb5_keytab_entry entry;
    krb5_kt_cursor cursor;
    enum { UNDEF, SPEC, HIGH, ALL, OLD } mode;
    int ret, kvno, flag;

    GET_HANDLE();

    if ((ret = krb5_parse_name(context, princ_str, &princ)))
	return tcl_kadm5_set_error(interp, ret);

    mode = UNDEF;
    kvno = 0;
    if (kvno_str == NULL)
	mode = HIGH; 
    else if (!strcmp(kvno_str, "all"))
	mode = ALL;
    else if (!strcmp(kvno_str, "old"))
	mode = OLD;
    else {
	mode = SPEC;
	if ((ret = Tcl_GetInt(interp, kvno_str, &kvno)) != TCL_OK)
	    return ret;
    }

    ret = krb5_kt_get_entry(context, keytab, princ, kvno, 0, &entry);
    if (ret) {
	if (ret == ENOENT) {
	    Tcl_AppendResult(interp, "Keytab \"", keytab_str, 
			     "\" does not exist.", NULL);
	} else if (ret == KRB5_KT_NOTFOUND) {
	    if (mode != SPEC) 
		Tcl_AppendResult(interp, "No entry for principal \"", 
				 princ_str, "\" in keytab \"", keytab_str, 
				 "\".", NULL);
	    else
		Tcl_AppendResult(interp, "No entry for principal \"",
				 princ_str, "\" with kvno \"", kvno_str,
				 "\" in keytab \"", keytab_str, 
				 "\".", NULL);
	} else
	    return tcl_kadm5_set_error(interp, ret);
	return TCL_ERROR;
    }

    /* set kvno to specified value, or highest kvno */
    kvno = entry.vno;
    krb5_kt_free_entry(context, &entry);

    if ((ret = krb5_kt_start_seq_get(context, keytab, &cursor))) 
	return tcl_kadm5_set_error(interp, ret);

    flag = 0;
    while ((ret = krb5_kt_next_entry(context, keytab, 
				     &entry, &cursor)) == 0) {
	if (krb5_principal_compare(context, princ, entry.principal) &&
	    ((mode == ALL) ||
	     (mode == SPEC && entry.vno == kvno) ||
	     (mode == OLD && entry.vno != kvno) ||
	     (mode == HIGH && entry.vno == kvno))) {
	    
	    if ((ret = krb5_kt_end_seq_get(context, keytab, &cursor)) ||
		(ret = krb5_kt_remove_entry(context, keytab, &entry)) ||
		(ret = krb5_kt_start_seq_get(context, keytab, &cursor)))
		return tcl_kadm5_set_error(interp, ret);
	    flag++;
	}
	
	krb5_kt_free_entry(context, &entry);
    }
    if (ret && ret != KRB5_KT_END)
	return tcl_kadm5_set_error(interp, ret);

    if ((ret = krb5_kt_end_seq_get(context, keytab, &cursor))) 
	return tcl_kadm5_set_error(interp, ret);
	
    /* return an error */
    if (flag == 0 && mode == OLD) {
	Tcl_AppendResult(interp, "Principal \"", princ_str, 
			 "\" has only one entry in keytab \"", keytab_str,
			 "\".", NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}    

int tcl_kadm5_ktadd(ClientData clientData, Tcl_Interp *interp,
                       int argc, char *argv[])
{
    krb5_keytab keytab = 0;
    char *princ_str, *keytab_str = NULL, **princs;
    int tcl_ret, code, num, i;
    char *func_name;
 
    GET_HANDLE();

    func_name = argv[0];
 
    if (argc < 2) {
        Tcl_AppendResult(interp, "wrong # options: should be \"", func_name,
                         " [-keytab ktname] principal ...\"", NULL);
        return TCL_ERROR;
    }
 
    if (!strcmp(argv[1], "-keytab")) {
        keytab_str = argv[2];
        argc -= 2; argv += 2;
    
	if (argc < 2) {
	    Tcl_AppendResult(interp, "wrong # options: should be \"",
			     func_name,
			     " [-keytab ktname] principal ...\"", NULL);
	    return TCL_ERROR;
	}
    }
 
    if ((tcl_ret = process_keytab(interp, context, &keytab_str, &keytab)))
        return tcl_ret;

    argv++;  /* point to the next argument, not the filename! */
    
    while (*argv) {
        add_keytab_principal(interp, keytab_str, keytab, *argv);
        argv++;
    }
 
    code = krb5_kt_close(context, keytab);
    if (code != 0)
        return tcl_kadm5_set_error(interp, code);
 
    free(keytab_str);
    return TCL_OK;
}
 
int tcl_kadm5_ktremove(ClientData clientData, Tcl_Interp *interp,
                       int argc, char *argv[])
{
    krb5_keytab keytab = 0;
    char *princ_str, *keytab_str = NULL, **princs;
    int ret, num, i;
 
    GET_HANDLE();
    
    if (argc < 2) {
        Tcl_AppendResult(interp, "wrong # options: should be \"", argv[0],
                         " [-keytab ktname] principal ...\"", NULL);
        return TCL_ERROR;
    }
 
    if (!strcmp(argv[1], "-keytab")) {
        keytab_str = argv[2];
        argc -= 2; argv += 2;
    }
    
    if (argc != 2 && argc != 3) {
	Tcl_AppendResult(interp, "wrong # options: should be \"", argv[0],
			 " [-keytab ktname] principal [kvno|all|old]\"", 
			 NULL);
	return TCL_ERROR;
    }

    if ((ret = process_keytab(interp, context, &keytab_str, &keytab)) 
	!= TCL_OK)
	return ret;

    if (argc == 2)
	ret = remove_keytab_principal(interp, keytab_str, keytab, 
				       argv[1], NULL);
    else
	ret = remove_keytab_principal(interp, keytab_str, keytab, 
				       argv[1], argv[2]);
    if (ret != TCL_OK) return ret;
 
    ret = krb5_kt_close(context, keytab);
    if (ret != 0)
        return tcl_kadm5_set_error(interp, ret);
 
    free(keytab_str);
    return TCL_OK;
}

int tcl_kadm5_ktname(ClientData clientData, Tcl_Interp *interp,
		     int argc, char *argv[])
{
    /* returns the default keytab name */
    char buf[BUFSIZ];
    int ret;
    krb5_keytab kt;

    GET_HANDLE();

    if (argc != 1 || argc > 2) {
 	Tcl_AppendResult(interp, "wrong # options: should be \"", argv[0],
			 "\"", NULL);
	return TCL_ERROR;
    }

    if ((ret = krb5_kt_default(context, &kt)) ||
	(ret = krb5_kt_get_name(context, kt, buf, BUFSIZ)))
	return tcl_kadm5_set_error(interp, ret);

    Tcl_AppendResult(interp, buf, NULL);

    return TCL_OK;
}
   
int tcl_kadm5_ktlist(ClientData clientData, Tcl_Interp *interp,
		     int argc, char *argv[])
{
    krb5_keytab kt;
    krb5_keytab_entry entry;
    krb5_kt_cursor cursor;
    char buf[512];
    char *pname;
    int ret;

    GET_HANDLE();

    if (argc < 1 || argc > 2) {
 	Tcl_AppendResult(interp, "wrong # options: should be \"", argv[0],
			 " [keytab]\"", NULL);
	return TCL_ERROR;
    }

    if (argc == 1) {
	if ((ret = krb5_kt_default(context, &kt)))
	    return tcl_kadm5_set_error(interp, ret);
    } else {
	if ((ret = krb5_kt_resolve(context, argv[1], &kt)))
	    return tcl_kadm5_set_error(interp, ret);
    }

    if ((ret = krb5_kt_start_seq_get(context, kt, &cursor)))
	return tcl_kadm5_set_error(interp, ret);
	
    while ((ret = krb5_kt_next_entry(context, kt, &entry, &cursor)) == 0) {
	if ((ret = krb5_unparse_name(context, entry.principal, &pname)))
	    return tcl_kadm5_set_error(interp, ret);
	
	sprintf(buf, "%d %d %s", entry.vno, entry.timestamp, pname);
	Tcl_AppendElement(interp, buf);
	free(pname);
    }

    if (ret && ret != KRB5_KT_END)
	return tcl_kadm5_set_error(interp, ret);
    if ((ret = krb5_kt_end_seq_get(context, kt, &cursor)))
	return tcl_kadm5_set_error(interp, ret);
    
    return TCL_OK;
}

int tcl_krb5_get_default_realm(ClientData clientData, Tcl_Interp *interp,
			       int argc, char *argv[])
{
  /* returns the default realm name */
    char *realm;
    int ret;
    krb5_keytab kt;
    krb5_context tmpcontext;
    
    if (argc != 1) {
 	Tcl_AppendResult(interp, "wrong # options: should be \"", argv[0],
			 "\"", NULL);
	return TCL_ERROR;
    }
    
    if (krb5_init_context(&tmpcontext)) {
	Tcl_AppendResult(interp, "could not initialize krb5 context", NULL);
	return TCL_ERROR;
    }
    
    ret = krb5_get_default_realm(tmpcontext, &realm);
    
    if (ret)
	return tcl_kadm5_set_error(interp, ret);

    Tcl_AppendResult(interp, realm, NULL);
    free(realm);

    return TCL_OK;
}
   
void Tcl_kadm5_init(Tcl_Interp *interp)
{
     Tcl_CreateCommand(interp, "kadm5_init", tcl_kadm5_init, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_destroy", tcl_kadm5_destroy, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_durtostr", tcl_kadm5_durtostr, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_datetostr", tcl_kadm5_datetostr, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_strtodate", tcl_kadm5_strtodate, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_strtodur", tcl_kadm5_strtodur, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_getprinc", tcl_kadm5_getprinc, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_addprinc", tcl_kadm5_addprinc, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_modprinc", tcl_kadm5_modprinc, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_delprinc", tcl_kadm5_delprinc, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_cpw", tcl_kadm5_cpw, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_change_password", tcl_kadm5_cpw, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_getprincs", tcl_kadm5_getprincs, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_getpol", tcl_kadm5_getpol, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_getpols", tcl_kadm5_getpols, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_addpol", tcl_kadm5_addpol, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_modpol", tcl_kadm5_modpol, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_delpol", tcl_kadm5_delpol, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_ktadd", tcl_kadm5_ktadd, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_ktremove", tcl_kadm5_ktremove, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_ktlist", tcl_kadm5_ktlist, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_ktname", tcl_kadm5_ktname, 0, 0);
     Tcl_CreateCommand(interp, "kadm5_getnextprincs", 
		       tcl_kadm5_getnextprincs, 0, 0);
     Tcl_CreateCommand(interp, "krb5_get_default_realm",
		       tcl_krb5_get_default_realm, 0, 0);
}
