/*
 * Main procedure body for the PasswordSync server process.
 * This process reads password change notification sent from
 * a registered .DLL in NT4.x and then calls a list of .DLL's
 * in turn to allow them to synchronise as they wish with
 * external password databases.
 */

#include <stdio.h>
#include <syslog.h>
#include <stdlib.h>
#include <signal.h>
#include <stdarg.h>
#include <errno.h>
#include <windows.h>
#include "win32service.h"

typedef struct {
  HINSTANCE lib_hinstance;
  char *lib_name;
  int lib_argc;
  char **lib_argv;
  int WINAPI (*initialize)(int argc, char **argv);
  int WINAPI (*password_changed)(const wchar_t *username, const wchar_t *password, int uid);
} dll_instance;

int global_do_exit = 0;

#define PWS_DLL_TOKEN_KEY_REG_NAME \
"SOFTWARE\\Cygnus Solutions\\PasswordSync\\DLLToken"

#define PWS_DLL_TOKEN_REG_VALUE_NAME "Token"

#define READ_BUF_SIZE 2048
#define PASSWD_SYNC_PIPE_NAME "\\\\.\\PIPE\\PASSWORDSYNC"

#define DLL_PASSWORDSYNC_ENTRY_NAME "PasswordSync"
#define DLL_INIT_ENTRY_NAME "Initialize"

/*
 * Request main to exit.
 */

int request_exit(int dummy)
{
  global_do_exit = 1;
  return 0;
}

/*
 * Split a REG_MULTI_SZ pointer in argv_block into the
 * argc, argv pair needed for the dll call.
 * Returns 0 on success, -1 on fail.
 */

int setup_args( dll_instance *dllp, char *argv_block)
{
  char *p;
  int argc;

  for(argc = 0, p = argv_block; *p != 0; p += strlen(p)+1, argc++)
    ;

  if((dllp->lib_argv = (char **)malloc( (argc + 1)*sizeof(char *))) == 0) {
    log_message(LOG_ERR, "malloc fail in setup_args.");
    return -1;
  }
  dllp->lib_argc = argc;
  for( argc = 0, p = argv_block; *p != 0; p += strlen(p)+1, argc++) {
    dllp->lib_argv[argc] = p;
  }
  dllp->lib_argv[argc] = 0; /* null terminate arg list */
  return 0;
}

/*
 * Function that scans the value entries in the key
 * PWS_DDL_REG_NAME. Each value name is the base
 * name of a .DLL (without the terminating .DLL). The
 * value of the entry is a REG_MULTI_SZ containing a list
 * of argv entries needed for that .dll. This function loads
 * the dll_instance array and sets number to the number 
 * loaded (not not all loads may have been successful). 
 * Returns 0 on success, -1 on fail.
 */

int get_dll_list( dll_instance **list, int *number)
{
  int i, j;
  int successful_load = 0;
  char *argv_block = 0;
  DWORD num_vals = 0;
  DWORD max_valname = 0;
  DWORD max_valuelen = 0;
  DWORD datasize;
  DWORD namesize;
  DWORD type;
  DWORD err;
  dll_instance *dlls = 0;
  dll_instance *dllp = 0;
  HKEY hkey = INVALID_HANDLE_VALUE;
  
  *number = 0;
  
  if((err = RegOpenKeyEx(HKEY_LOCAL_MACHINE, PWS_DLL_REG_NAME, 0,
                KEY_READ, &hkey)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "Failed to open registry key %s. Error was : %s", 
		PWS_DLL_REG_NAME, str_oserr(err));
    goto cleanup;
  }

  /* Find out how many sub keys there are */
  if((err = RegQueryInfoKey(hkey, 0, 0, 0,
                     0, 0, 0,
                     &num_vals, &max_valname, &max_valuelen,
                     0, 0)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "Failed to get information on registry key %s. Error was : %s",
                PWS_DLL_REG_NAME, str_oserr(err));
    goto cleanup;
  }

  if(num_vals == 0) {
    log_message(LOG_WARNING, 
	"The registry value %s has no values.", PWS_DLL_REG_NAME);
    goto cleanup;
  }

  /* Allocate memory for that number of entries. */
  if((dlls = calloc( num_vals * sizeof(dll_instance), 1)) == 0) {
    log_message(LOG_ERR, "calloc fail in allocating %d dll_instances.", num_vals);
    goto cleanup;
  }

  for(i = 0; i < num_vals; i++) {

    dllp = &dlls[i];
    dllp->lib_hinstance = INVALID_HANDLE_VALUE;

    if((dllp->lib_name = (char *)malloc(max_valname + 5)) == 0) {
      log_message(LOG_ERR, "malloc fail in allocating space for the dll name.");
      goto cleanup;
    }

    if((argv_block = (char *)malloc( max_valuelen )) == 0) {
      log_message(LOG_ERR, "malloc fail in allocating argv block.");
      goto cleanup;
    }

    /*
     * Read the valuename and the value.
     */
    namesize = max_valname + 5;
    datasize = max_valuelen;
    if((err = RegEnumValue( hkey, (DWORD)i, dllp->lib_name, &namesize, 0, &type, 
                     argv_block, &datasize)) != ERROR_SUCCESS) {
      log_message(LOG_ERR, "Failed to get the registry value for value %d, key %s. Error was : %s",
             i, PWS_DLL_REG_NAME, str_oserr(err));
    }
    /* The type must be REG_MULTI_SZ */
    if(type != REG_MULTI_SZ) {
      log_message(LOG_ERR, "Incorrect type for registry value %d, key %s.",
             i, PWS_DLL_REG_NAME);
      goto cleanup;
    }

    /* Append .DLL onto the dllp->lib_name */
    strcat( dllp->lib_name, ".DLL");
    if(setup_args( dllp, argv_block ) != 0 )
      goto cleanup;

    /* Now try the LoadLibrary on the DLL name */
    if((dllp->lib_hinstance = LoadLibrary( dllp->lib_name )) == 0) {
      /* Log a warning. */
      log_message(LOG_WARNING, "LoadLibrary failed for Library %s", dllp->lib_name);
      continue;
    }

    /* Now try and get the function addresses. */
    if((dllp->initialize= GetProcAddress( dllp->lib_hinstance, 
                                        DLL_INIT_ENTRY_NAME )) == 0 ) {
      /* Log an error. */
      log_message(LOG_ERR, "GetProcAddress for %s failed for Library %s",
             DLL_INIT_ENTRY_NAME, dllp->lib_name);
      FreeLibrary(dllp->lib_hinstance);
      dllp->lib_hinstance = INVALID_HANDLE_VALUE;
      continue;
    }
    if((dllp->password_changed = GetProcAddress( dllp->lib_hinstance, 
                                        DLL_PASSWORDSYNC_ENTRY_NAME )) == 0 ) {
      /* Log an error. */
      log_message(LOG_ERR, "GetProcAddress for %s failed for Library %s",
             DLL_PASSWORDSYNC_ENTRY_NAME, dllp->lib_name);
      FreeLibrary(dllp->lib_hinstance);
      dllp->lib_hinstance = INVALID_HANDLE_VALUE;
      continue;
    }
    /* Call initialize. */
    if((*dllp->initialize)( dllp->lib_argc, dllp->lib_argv) != 0) {
      log_message(LOG_ERR, "%s failed for Library %s",
             DLL_INIT_ENTRY_NAME, dllp->lib_name);
      FreeLibrary(dllp->lib_hinstance);
      dllp->lib_hinstance = INVALID_HANDLE_VALUE;
      continue;
    }
    successful_load++;
  }

  if(successful_load == 0) {
    log_message(LOG_ERR, "All attempts to get a PasswordSync library failed.");
    goto cleanup;
  }

  RegCloseKey(hkey);

  *list = dlls;
  *number = num_vals;
  return 0;

cleanup:

  if(hkey != INVALID_HANDLE_VALUE)
    RegCloseKey(hkey);

  for(i = 0; i < num_vals; i++) {
    dllp = &dlls[i];
    if(dllp->lib_hinstance != INVALID_HANDLE_VALUE)
      FreeLibrary(dllp->lib_hinstance);
    if(dllp->lib_name != 0)
      free(dllp->lib_name);
    if(dllp->lib_argv) {
      /* The argv block base is held in argv[0] */
      free((char *)dllp->lib_argv[0]);
      free((char *)dllp->lib_argv);
    }
  }
  if(argv_block != 0)
    free(argv_block);
  if(dlls != 0)
    free(dlls);

  return -1;
}

/*
 * Function to get the token that the Password Sync DLL has
 * created for us in the registry. Used to prove we are
 * an Administrator or SYSTEM user.
 */

int get_dll_token( char **tokenp, unsigned int *token_size)
{
  DWORD err;
  DWORD type;
  HKEY hkey;
  
  *tokenp = 0;
  
  if((err = RegOpenKeyEx(HKEY_LOCAL_MACHINE, PWS_DLL_TOKEN_KEY_REG_NAME, 0,
                KEY_READ, &hkey)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "Failed to open registry key %s. Error was : %s", 
		PWS_DLL_TOKEN_KEY_REG_NAME, str_oserr(err));
    return -1;
  }
  /* Try and get the value size. */
  *token_size = 0;
  if((err = RegQueryValueEx(hkey, PWS_DLL_TOKEN_REG_VALUE_NAME, 0, &type, 0, token_size)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "Failed to get the size for registry value %s. Error was : %s",
                PWS_DLL_TOKEN_REG_VALUE_NAME, str_oserr(err));
    RegCloseKey(hkey);
    return -1;
  }
  /* Allocate the memory for the value */
  if((*tokenp = (char *)malloc(*token_size)) == 0) {
    log_message(LOG_ERR, "malloc failed getting the value for the registry value %s.",
                PWS_DLL_TOKEN_REG_VALUE_NAME);
    RegCloseKey(hkey);
    return -1;
  }
  /* Now get the value */
  if((err = RegQueryValueEx(hkey,PWS_DLL_TOKEN_REG_VALUE_NAME, 0, &type, (LPBYTE)*tokenp, 
							(DWORD *)token_size)) != ERROR_SUCCESS) {
    log_message(LOG_ERR,"Failed to retrieve Registry value %s. Error was : %s",
                PWS_DLL_TOKEN_REG_VALUE_NAME, str_oserr(err));
    free(*tokenp);
    *tokenp = 0;
    RegCloseKey(hkey);
    return -1;
  }
  return 0;  
}

/*
 * Function to send token back down the named pipe to the
 * password changing dll.
 */

int return_token_to_dll( HANDLE nph, char *token, unsigned int size)
{
  DWORD written;
  BOOL ret;
  ret = WriteFile( nph, token, size, &written, 0);

  if((ret == 0) || (written != size)) {
    log_message(LOG_ERR, "Failed to return token to password dll. Error was %s",
                str_oserr(GetLastError()));
    return -1;
  }
  return 0;
}
  
/*
 * Function to create a security descriptor containing all
 * access, inheritable to SYSTEM only.
 */

BOOL setup_security(SECURITY_DESCRIPTOR *psd)
{
  PSID systemSid = 0;
  SID_IDENTIFIER_AUTHORITY ntauth = SECURITY_NT_AUTHORITY;
  DWORD size;
  PACL pacl = 0;
  ACE_HEADER *ace_p;

  if(AllocateAndInitializeSid( &ntauth, 1,
 	SECURITY_LOCAL_SYSTEM_RID,
	0, 0, 0, 0, 0, 0, 0,
	&systemSid) == FALSE) {
    log_message(LOG_ERR, "AllocateAndInitializeSid failed. Error was %s",
	  str_oserr(GetLastError()));
    goto cleanup;
  }
  if(InitializeSecurityDescriptor( psd, SECURITY_DESCRIPTOR_REVISION) == FALSE) {
    log_message(LOG_ERR, "InitializeSecurityDescriptor failed. Error was %s",
		  str_oserr(GetLastError()));
    goto cleanup;
  }
  if(SetSecurityDescriptorOwner( psd, systemSid, 0) == FALSE) {
    log_message(LOG_ERR, "SetSecurityDescriptorOwner failed. Error was %s",
	  str_oserr(GetLastError()));
    goto cleanup;
  }

  /* Calculate the DACL size needed. */
  size = sizeof(ACL) +
	   sizeof(ACCESS_ALLOWED_ACE) +
	   sizeof(DWORD) +
	   GetLengthSid(systemSid);
  if((pacl = (PACL)LocalAlloc( LMEM_FIXED, size)) == 0) {
    log_message(LOG_ERR, "Failed to allocate memory for the ACL. Error was %s",
		  str_oserr(GetLastError()));
    goto cleanup;
  }

  if(InitializeAcl( pacl, size, ACL_REVISION) == FALSE) {
    log_message(LOG_ERR, "InitializeAce failed. Error was %s",
	  str_oserr(GetLastError()));
    goto cleanup;
  }
  if(AddAccessAllowedAce( pacl, ACL_REVISION, GENERIC_ALL, systemSid)	== FALSE) {
    log_message(LOG_ERR, "AddAccessAllowedAce failed. Error was %s",
	  str_oserr(GetLastError()));
    goto cleanup;
  }

  if(SetSecurityDescriptorDacl( psd, TRUE, pacl, FALSE) == FALSE) {
    log_message(LOG_ERR, "SetSecurityDescriptorDacl failed. Error was %s",
		  str_oserr(GetLastError()));
    goto cleanup;
  }

  return FALSE;

cleanup:

  if(pacl != 0)
    LocalFree((HLOCAL)pacl);
  if(systemSid != 0)
    FreeSid(systemSid);
  return TRUE;
}

/*
 * Function to split out the username, plaintext password and
 * uid from the buffer. Buffer holds the message in the form
 * [4 byte x86 int] uid
 * [ zero terminated bytes] name
 * [ zero terminated bytes ] plaintext password.
 * Returns 0 on success, -1 on failure.
 */

int split_message( wchar_t *buf, int buflen, wchar_t **username, wchar_t **password, int *uid)
{
  /* Convert buflen into wchar_t size. */
  buflen /= sizeof(wchar_t);
  
  if(buflen < sizeof(int)/sizeof(wchar_t)) {
    log_message(LOG_ERR, "buffer too small when getting user id.");
    return -1;
  }

  /* We can get away with the following as int's are guarenteed
     to be 4 byte x86 format on NT. Tacky but true.... */

  *uid = *(int *)buf;
  buf += (sizeof(int)/sizeof(wchar_t));
  buflen -= (sizeof(int)/sizeof(wchar_t));
  if(buflen <= 0) {
    log_message(LOG_ERR, "buffer too small when getting user name.");
    return -1;
  }

  *username = buf;
  for(; *buf && buflen > 0; buf++, buflen--)
    ;
  if(buflen <= 0) {
    log_message(LOG_ERR, "buffer too small when getting password.");
    return -1;
  }

  buf++;
  buflen--;

  *password = buf;
  for(; *buf && buflen > 0; buf++, buflen--)
    ;
  if(*buf != 0 || buflen < 0) {
    log_message(LOG_ERR, "error in buffer for password.");
    return -1;
  }

  return 0;
}

int original_main(int argc, char **argv)
{
  wchar_t buf[READ_BUF_SIZE];
  DWORD readlen;
  int number = 0;
  dll_instance *dll_list = 0;
  dll_instance *dllp;
  HANDLE nph;
  wchar_t *username;
  wchar_t *password;
  int uid;
  int i;
  int ret;
  char *token;
  unsigned int token_size;
  SECURITY_DESCRIPTOR sd;
  SECURITY_ATTRIBUTES sa;
  
  if(get_dll_token( &token, &token_size) != 0) {
    return -1;
  }

#ifdef KERBNET
  if(kerbnet_sync_init(argc, argv) != 0) {
    return -1;
  }
#endif /* KERBNET */
    
  /*
   * Get a list of the .DLL's we will be
   * using to sync with.
   */

#ifdef KERBNET
  get_dll_list( &dll_list, &number);
#else /* KERBNET */
  if(get_dll_list( &dll_list, &number) != 0) {
    return -1;
  }
#endif /* KERBNET */   

  /*
   * Create the named pipe server endpoint we 
   * will use to read passwords from the injected
   * .dll. Ensure that only the SYSTEM user
   * can connect to the named pipe.
   */

  if(setup_security( &sd ) != 0) {
    return -1;
  }

  sa.nLength = sizeof(SECURITY_ATTRIBUTES);
  sa.lpSecurityDescriptor = &sd;
  sa.bInheritHandle = FALSE;

  if((nph = CreateNamedPipe( PASSWD_SYNC_PIPE_NAME,
			     PIPE_ACCESS_DUPLEX,
			     PIPE_TYPE_MESSAGE |
			     PIPE_WAIT,
			     1, 0, 0, NMPWAIT_USE_DEFAULT_WAIT,
			     &sa)) == INVALID_HANDLE_VALUE ) {
    log_message( LOG_ERR, "Unable to create named pipe %s, Error was : %s",
		 PASSWD_SYNC_PIPE_NAME, str_oserr(GetLastError()) );
    return -1;
  }

  /*
   * Keep reading from the .dll.
   */

  while( global_do_exit == 0 ) {

    if((ConnectNamedPipe( nph, 0) == 0) && (GetLastError() != ERROR_PIPE_CONNECTED)) {
      log_message(LOG_ERR, "Error in ConnectNamedPipe. Error was : %s",
		  str_oserr(GetLastError()) );
      return -1;
    }

    /* Assure the connected user that we are running with Administrator
       privillage also. We do this by reading a token that the password syncing
       DLL has placed in the registry on startup, and return it back to
       the DLL. The security permissions on this entry are set such that
       only Administrator or SYSTEM can read it.
       */
    if(return_token_to_dll( nph, token, token_size) != 0)
      return -1;       

    /*
     * Now loop, reading password change notifications from the dll
     */
    while( global_do_exit == 0 ) {
      memset(buf, '\0', READ_BUF_SIZE);
      if(ReadFile( nph, buf, READ_BUF_SIZE, &readlen, 0) == FALSE) {
	if(GetLastError() == ERROR_MORE_DATA ) {
	  log_message(LOG_WARNING, "ReadFile: giant read.");
	  while((ReadFile(nph, buf, READ_BUF_SIZE, &readlen, 0) == FALSE) &&
		(GetLastError() == ERROR_MORE_DATA))
	    ;
	} else if( GetLastError() == ERROR_BROKEN_PIPE) {
	  break;
	} else {
	  log_message( LOG_ERR, "ReadFile error : %s", str_oserr(GetLastError()) );
	  return -1;
	}
      }

      if(readlen == 0 ) {
	log_message(LOG_WARNING, "ReadFile : zero bytes");
	break;
      }

      /* We should now have a password change message in the
	 buffer. Split it into the component parts. */
      if(split_message( buf, readlen, &username, &password, &uid)!= 0 )
	continue;
      
#ifdef KERBNET
      if(do_kerbnet_pw_sync( username, password, uid)!=0)
	log_message(LOG_ERR, "Password sync with kerbnet kerberos admin server failed.");
#endif /* ENDIF */

      /* Pass the change to each dll in turn. Maybe this should
	 be done 
	 a). In separate threads, 
	 b). With an exception handler per call. 
	 Future fix - for now call synchronously
	 and without exception handler.
	 */
      for(i = 0; i < number; i++ ) {
	dllp = &dll_list[i];
	if(dllp->password_changed) {
	  lock_exit(INFINITE);
	  if((*dllp->password_changed)(username, password, uid)!= 0) {
	    log_message(LOG_ERR, "Calling library %s to notify of password change for user %s failed.", dllp->lib_name, username);
	  }
	  unlock_exit();
	}
      }                   
    }
    /* Allow nph to be re-connected to. */
    DisconnectNamedPipe(nph);
    /* Tell password dll to re-connect. */
    update_watch_value();
  }
  return 0;
}
