/*
 * Copyright, OpenVision Technologies, Inc., 1996, All Rights Reserved
 *
 * WARNING:  Retrieving the OpenVision Kerberos Administration system
 * source code, as described below, indicates your acceptance of the
 * following terms.  If you do not agree to the following terms, do not
 * retrieve the OpenVision Kerberos administration system.
 *
 * You may freely use and distribute the Source Code and Object Code
 * compiled from it, but this Source Code is provided to you "AS IS"
 * EXCLUSIVE OF ANY WARRANTY, INCLUDING, WITHOUT LIMITATION, ANY
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, OR
 * ANY OTHER WARRANTY, WHETHER EXPRESS OR IMPLIED.  IN NO EVENT WILL
 * OPENVISION HAVE ANY LIABILITY FOR ANY LOST PROFITS, LOSS OF DATA OR
 * COSTS OF PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, OR FOR ANY
 * SPECIAL, INDIRECT, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THIS
 * AGREEMENT, INCLUDING, WITHOUT LIMITATION, THOSE RESULTING FROM THE
 * USE OF THE SOURCE CODE, OR THE FAILURE OF THE SOURCE CODE TO PERFORM,
 * OR FOR ANY OTHER REASON.
 *
 * OpenVision retains all rights, title, and interest in the donated
 * Source Code.  With respect to OpenVision's copyrights in the donated
 * Source Code, OpenVision also retains rights to derivative works of
 * the Source Code whether created by OpenVision or a third party.
 *
 * OpenVision Technologies, Inc. has donated this Kerberos
 * Administration system to MIT for inclusion in the standard Kerberos 5
 * distribution. This donation underscores our commitment to continuing
 * Kerberos technology development and our gratitude for the valuable
 * work which has been performed by MIT and the Kerberos community.
 */

#include "gss.h"
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

/*
 * Function: send_token
 *
 * Purpose: Writes a token to a file descriptor.
 *
 * Arguments:
 *
 *	s		(r) an open file descriptor
 *	tok		(r) the token to write
 *
 * Returns: 0 on success, -1 on failure
 *
 * Effects:
 *
 * send_token writes the token length (as a network long) and then the
 * token data to the file descriptor s.	 It returns 0 on success, and
 * -1 if an error occurs or if it could not write all the data.
 */
int send_token(SOCKET s, gss_buffer_t tok) {
    size_t ret;

    ret = send(s, (char *) &tok->length, 4, 0);

	if (ret < 0) {
		fprintf(stderr, "Error sending token length\r");
		return -1;
	}
	else if (ret != 4) {
		fprintf(stderr, "sending token length: %d of %d bytes written\r", ret, 4);
		return -1;
	}

    ret = send(s, tok->value, tok->length, 0);

	if (ret < 0) {
		fprintf(stderr, "Error sending data\r");
		return -1;
	}
	else if (ret != tok->length) {
		fprintf(stderr, "sending token data: %d of %d bytes written\r", ret, tok->length);
		return -1;
	}

    return 0;

} /* send_token */


/*
 * Function: recv_token
 *
 * Purpose: Reads a token from a file descriptor.
 *
 * Arguments:
 *
 *	s		(r) an open file descriptor
 *	tok		(w) the read token
 *
 * Returns: 0 on success, -1 on failure
 *
 * Effects:
 * 
 * recv_token reads the token length (as a network long), allocates
 * memory to hold the data, and then reads the token data from the
 * file descriptor s.  It blocks to read the length and data, if
 * necessary.  On a successful return, the token should be freed with
 * gss_release_buffer.	It returns 0 on success, and -1 if an error
 * occurs or if it could not read all the data.
 */
int
recv_token (SOCKET s, gss_buffer_t tok) {
    int ret;
    unsigned long len;

    ret = recv(s, (char *) &len, 4, 0);

    if (ret < 0) {
		fprintf(stderr, "Error reading token length\r");
	    return -1;
     } 
     else if (ret != 4) {
	     fprintf(stderr, "Error reading token length: %d of %d bytes read\r", ret, 4);
	     return -1;
     }

    tok->length = (size_t) len;

    tok->value = (char *) malloc(tok->length);

    if (tok->value == NULL) {
        fprintf(stderr, "Out of memory allocating token data\r");
        return -1;
     }

    ret = recv (s, (char *) tok->value, tok->length, 0);

    if (ret < 0) {
	     fprintf(stderr, "Error reading token data\r");
	     free(tok->value);
	     return -1;
    }

    return 0;
} /* recv_token */


/*
 * Function: display_status
 *
 * Purpose: displays GSS-API messages
 *
 * Arguments:
 *
 *	msg		a string to be displayed with the message
 *	maj_stat	the GSS-API major status code
 *	min_stat	the GSS-API minor status code
 *
 * Effects:
 *
 * The GSS-API messages associated with maj_stat and min_stat are
 * displayed on stderr, each preceeded by "GSS-API error <msg>: " and
 * followed by a newline.
 */
void
display_status (char *msg, OM_uint32 maj_stat, OM_uint32 min_stat) {
    display_status_1(msg, maj_stat, GSS_C_GSS_CODE);
    display_status_1(msg, min_stat, GSS_C_MECH_CODE);
}

static void
display_status_1(char *m, OM_uint32 code, int type) {
    OM_uint32 maj_stat, min_stat;
    gss_buffer_desc msg;
    OM_uint32 msg_ctx;
     
    msg_ctx = 0;
    while (1) {
        maj_stat = gss_display_status(
        	&min_stat, code, type, GSS_C_NULL_OID, &msg_ctx, &msg);

        fprintf (stderr, "GSS-API error %s: %s\r", m, (char *)msg.value);
        
        (void) gss_release_buffer(&min_stat, &msg);
	  
        if (!msg_ctx)
            break;
    }
} /* display_status */
