/*-
 * Copyright (c) 1991, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* based on @(#)enc_des.c	8.1 (Berkeley) 6/4/93 */

/* mostly based on appl/telnet/libtelnet/enc_des.c, but changed to not
   use global variables since we might have multiple connections with
   different states */

#ifndef ENCRYPTION
#define ENCRYPTION
#endif /* ENCRYPTION */

#ifndef AUTHENTICATION
#define AUTHENTICATION
#endif /* AUTHENTICATION */

#ifndef DES_ENCRYPTION
#define DES_ENCRYPTION
#endif /* DES_ENCRYPTION */

#ifdef	ENCRYPTION
# ifdef	AUTHENTICATION
#  ifdef DES_ENCRYPTION
#include "telnet.h"
#include "tnae.h"
#include <stdio.h>
#ifdef	__STDC__
#include <stdlib.h>
#endif

#include "encrypt.h"
#include "des.h"

#define	CFB	0
#define	OFB	1

#define	NO_SEND_IV	1
#define	NO_RECV_IV	2
#define	NO_KEYID	4
#define	IN_PROGRESS	(NO_SEND_IV|NO_RECV_IV|NO_KEYID)
#define	SUCCESS		0
#define	FAILED		-1


struct fb {
	Block krbdes_key;
	Schedule krbdes_sched;
	Block temp_feed;
	int type;
	int need_start;
	int state[2];
	int keyid[2];
	int once;
	struct stinfo {
		Block		str_output;
		Block		str_feed;
		Block		str_iv;
		Block		str_ikey;
		Schedule	str_sched;
		int		str_index;
		int		str_flagshift;
	} streams[2];
};

static struct fb fb[2];

struct keyidlist {
	char	*keyid;
	int	keyidlen;
	char	*key;
	int	keylen;
	int	flags;
} keyidlist [] = {
	{ "\0", 1, 0, 0, 0 },		/* default key of zero */
	{ 0, 0, 0, 0, 0 }
};

static krb5_pointer random_state;

#define	KEYFLAG_MASK	03

#define	KEYFLAG_NOINIT	00
#define	KEYFLAG_INIT	01
#define	KEYFLAG_OK	02
#define	KEYFLAG_BAD	03

#define	KEYFLAG_SHIFT	2

#define	SHIFT_VAL(a,b)	(KEYFLAG_SHIFT*((a)+((b)*2)))

#define	FB64_IV		1
#define	FB64_IV_OK	2
#define	FB64_IV_BAD	3


void fb64_stream_iv P((Block, struct stinfo *));
void fb64_init P((struct fb *));
void fb64_stream_key P((Block, struct stinfo *));

void cfb64_init(void **state)
{
	struct fb *fbp;
	fbp=(*state)=NewPtr(sizeof(struct fb));
	
	fb64_init(fbp);
	fbp->type = ENCTYPE_DES_CFB64;
	fbp->streams[0].str_flagshift = SHIFT_VAL(0, CFB);
	fbp->streams[1].str_flagshift = SHIFT_VAL(1, CFB);
}

void ofb64_init(void **state)
{
	struct fb *fbp;
	fbp=(*state)=NewPtr(sizeof(struct fb));
	
	fb64_init(fbp);
	fbp->type = ENCTYPE_DES_OFB64;
	fbp->streams[0].str_flagshift = SHIFT_VAL(0, OFB);
	fbp->streams[1].str_flagshift = SHIFT_VAL(1, OFB);
}

void fb64_init(struct fb *fbp)
{
	memset((void *)fbp, 0, sizeof(*fbp));
	fbp->state[0] = fbp->state[1] = FAILED;
}

/*
 * Returns:
 *	-1: some error.  Negotiation is done, encryption not ready.
 *	 0: Successful, initial negotiation all done.
 *	 1: successful, negotiation not done yet.
 *	 2: Not yet.  Other things (like getting the key from
 *	    Kerberos) have to happen before we can continue.
 */
int fb64_start(tnParams *tn, struct fb *fbp, int dir)
{
	Block b;
	int x;
	register int state;
	krb5_keyblock *keyblock;

	switch (dir) {
	case DIR_DECRYPT:
		/*
		 * This is simply a request to have the other side
		 * start output (our input).  He will negotiate an
		 * IV so we need not look for it.
		 */
		state = fbp->state[dir-1];
		if (state == FAILED)
			state = IN_PROGRESS;
		break;

	case DIR_ENCRYPT:
		state = fbp->state[dir-1];
		if (state == FAILED)
			state = IN_PROGRESS;
		else if ((state & NO_SEND_IV) == 0)
			break;

		if (!VALIDKEY(fbp->krbdes_key)) {
			fbp->need_start = 1;
			break;
		}
		state &= ~NO_SEND_IV;
		state |= NO_RECV_IV;
		/*
		 * Create a random feed and send it over.
		 */
		mit_des_random_key(NULL, random_state, &keyblock);
		memcpy(fbp->temp_feed, keyblock->contents, sizeof(Block));
		krb5_free_keyblock(NULL, keyblock);
		mit_des_ecb_encrypt(fbp->temp_feed, fbp->temp_feed,
				fbp->krbdes_sched, 1);

		tn_sendencsub(tn, TELOPT_ENCRYPT, ENCRYPT_IS, fbp->type, FB64_IV, fbp->temp_feed, sizeof(Block));
		break;
	default:
		return(FAILED);
	}
	return(fbp->state[dir-1] = state);
}

/*
 * Returns:
 *	-1: some error.  Negotiation is done, encryption not ready.
 *	 0: Successful, initial negotiation all done.
 *	 1: successful, negotiation not done yet.
 */
int fb64_is(tnParams *tn, struct fb *fbp, unsigned char *data, int cnt)
{
	int x;
	Block b;
	register int state = fbp->state[DIR_DECRYPT-1];

	if (cnt-- < 1)
		goto failure;

	switch (*data++) {
	case FB64_IV:
		if (cnt != sizeof(Block)) {
			state = FAILED;
			goto failure;
		}

		fb64_stream_iv((void *)data, &fbp->streams[DIR_DECRYPT-1]);

		tn_sendencsub(tn, TELOPT_ENCRYPT, ENCRYPT_REPLY, fbp->type, FB64_IV_OK, NULL, 0);

		state = fbp->state[DIR_DECRYPT-1] = IN_PROGRESS;
		break;

	default:
		/* FALL THROUGH */
	failure:
		/*
		 * We failed.  Send an FB64_IV_BAD option
		 * to the other side so it will know that
		 * things failed.
		 */
		tn_sendencsub(tn, TELOPT_ENCRYPT, ENCRYPT_REPLY, fbp->type, FB64_IV_BAD, NULL, 0);

		break;
	}
	return(fbp->state[DIR_DECRYPT-1] = state);
}

/*
 * Returns:
 *	-1: some error.  Negotiation is done, encryption not ready.
 *	 0: Successful, initial negotiation all done.
 *	 1: successful, negotiation not done yet.
 */
int fb64_reply(tnParams *tn, struct fb *fbp, unsigned char *data, int cnt)
{
	int x;
	Block b;
	register int state = fbp->state[DIR_ENCRYPT-1];

	if (cnt-- < 1)
		goto failure;

	switch (*data++) {
	case FB64_IV_OK:
		fb64_stream_iv(fbp->temp_feed, &fbp->streams[DIR_ENCRYPT-1]);
		if (state == FAILED)
			state = IN_PROGRESS;
		state &= ~NO_RECV_IV;
		tn_sendsub(tn, TELOPT_ENCRYPT, ENCRYPT_ENC_KEYID, "\0", 1);
		break;

	case FB64_IV_BAD:
		memset(fbp->temp_feed, 0, sizeof(Block));
		fb64_stream_iv(fbp->temp_feed, &fbp->streams[DIR_ENCRYPT-1]);
		state = FAILED;
		break;

	default:
		/* FALL THROUGH */
	failure:
		state = FAILED;
		break;
	}
	return(fbp->state[DIR_ENCRYPT-1] = state);
}

void fb64_session(tnParams *tn, struct fb *fbp, Session_Key *key)
{
	if (!key || key->type != SK_DES) {
		return;
	}
	memcpy((void *)fbp->krbdes_key, (void *)key->data, sizeof(Block));

	fb64_stream_key(fbp->krbdes_key, &fbp->streams[DIR_ENCRYPT-1]);
	fb64_stream_key(fbp->krbdes_key, &fbp->streams[DIR_DECRYPT-1]);

	if (fbp->once == 0) {
		krb5_encrypt_block eblock;
		krb5_keyblock keyblock;
		
		krb5_use_enctype(NULL, &eblock, ENCTYPE_DES_CBC_CRC);
		keyblock.enctype = ENCTYPE_DES_CBC_CRC;
		keyblock.length = sizeof(Block);
		keyblock.contents = (krb5_octet *)fbp->krbdes_key;
		mit_des_init_random_key(&eblock, &keyblock, &random_state);
		fbp->once = 1;
	}
	mit_des_key_sched(fbp->krbdes_key, fbp->krbdes_sched);
	/*
	 * Now look to see if krbdes_start() was was waiting for
	 * the key to show up.  If so, go ahead an call it now
	 * that we have the key.
	 */
	if (fbp->need_start) {
		fbp->need_start = 0;
		fb64_start(tn, fbp, DIR_ENCRYPT);
	}
}

/*
 * We only accept a keyid of 0.  If we get a keyid of
 * 0, then mark the state as SUCCESS.
 */
int fb64_keyid(tnParams *tn, struct fb *fbp, int dir, unsigned char *kp, int *lenp)
{
	register int state = fbp->state[dir-1];

	if (*lenp != 1 || (*kp != '\0')) {
		*lenp = 0;
		return(state);
	}

	if (state == FAILED)
		state = IN_PROGRESS;

	state &= ~NO_KEYID;

	return(fbp->state[dir-1] = state);
}

void fb64_stream_iv(Block seed, register struct stinfo *stp)
{

	memcpy((void *)stp->str_iv,     (void *)seed, sizeof(Block));
	memcpy((void *)stp->str_output, (void *)seed, sizeof(Block));

	mit_des_key_sched(stp->str_ikey, stp->str_sched);

	stp->str_index = sizeof(Block);
}

void fb64_stream_key(Block key, register struct stinfo *stp)
{
	memcpy((void *)stp->str_ikey, (void *)key, sizeof(Block));
	mit_des_key_sched(key, stp->str_sched);

	memcpy((void *)stp->str_output, (void *)stp->str_iv, sizeof(Block));

	stp->str_index = sizeof(Block);
}

/*
 * DES 64 bit Cipher Feedback
 *
 *     key --->+-----+
 *          +->| DES |--+
 *          |  +-----+  |
 *			|           v
 *  INPUT --(--------->(+)+---> DATA
 *          |             |
 *			+-------------+
 *         
 *
 * Given:
 *	iV: Initial vector, 64 bits (8 bytes) long.
 *	Dn: the nth chunk of 64 bits (8 bytes) of data to encrypt (decrypt).
 *	On: the nth chunk of 64 bits (8 bytes) of encrypted (decrypted) output.
 *
 *	V0 = DES(iV, key)
 *	On = Dn ^ Vn
 *	V(n+1) = DES(On, key)
 */

void cfb64_encrypt(struct fb *fbp, unsigned char *s, int c)
{
	register struct stinfo *stp = &fbp->streams[DIR_ENCRYPT-1];
	register int index;

	index = stp->str_index;
	while (c-- > 0) {
		if (index == sizeof(Block)) {
			Block b;
			mit_des_ecb_encrypt(stp->str_output, b, stp->str_sched, 1);
			memcpy((void *)stp->str_feed,(void *)b,sizeof(Block));
			index = 0;
		}

		/* On encryption, we store (feed ^ data) which is cypher */
		*s = stp->str_output[index] = (stp->str_feed[index] ^ *s);
		s++;
		index++;
	}
	stp->str_index = index;
}

int cfb64_decrypt(struct fb *fbp, int data)
{
	register struct stinfo *stp = &fbp->streams[DIR_DECRYPT-1];
	int index;

	if (data == -1) {
		/*
		 * Back up one byte.  It is assumed that we will
		 * never back up more than one byte.  If we do, this
		 * may or may not work.
		 */
		if (stp->str_index)
			--stp->str_index;
		return(0);
	}

	index = stp->str_index++;
	if (index == sizeof(Block)) {
		Block b;
		mit_des_ecb_encrypt(stp->str_output, b, stp->str_sched, 1);
		memcpy((void *)stp->str_feed, (void *)b, sizeof(Block));
		stp->str_index = 1;	/* Next time will be 1 */
		index = 0;		/* But now use 0 */ 
	}

	/* On decryption we store (data) which is cypher. */
	stp->str_output[index] = data;
	return(data ^ stp->str_feed[index]);
}

/*
 * DES 64 bit Output Feedback
 *
 * key --->+-----+
 *      +->| DES |--+
 *      |  +-----+  |
 *      +-----------+
 *                  v
 *  INPUT -------->(+) ----> DATA
 *
 * Given:
 *	iV: Initial vector, 64 bits (8 bytes) long.
 *	Dn: the nth chunk of 64 bits (8 bytes) of data to encrypt (decrypt).
 *	On: the nth chunk of 64 bits (8 bytes) of encrypted (decrypted) output.
 *
 *	V0 = DES(iV, key)
 *	V(n+1) = DES(Vn, key)
 *	On = Dn ^ Vn
 */
void ofb64_encrypt(struct fb *fbp, unsigned char *s, int c)
{
	register struct stinfo *stp = &fbp->streams[DIR_ENCRYPT-1];
	register int index;

	index = stp->str_index;
	while (c-- > 0) {
		if (index == sizeof(Block)) {
			Block b;
			mit_des_ecb_encrypt(stp->str_feed, b, stp->str_sched, 1);
			memcpy((void *)stp->str_feed,(void *)b,sizeof(Block));
			index = 0;
		}
		*s++ ^= stp->str_feed[index];
		index++;
	}
	stp->str_index = index;
}

int ofb64_decrypt(struct fb *fbp, int data)
{
	register struct stinfo *stp = &fbp->streams[DIR_DECRYPT-1];
	int index;

	if (data == -1) {
		/*
		 * Back up one byte.  It is assumed that we will
		 * never back up more than one byte.  If we do, this
		 * may or may not work.
		 */
		if (stp->str_index)
			--stp->str_index;
		return(0);
	}

	index = stp->str_index++;
	if (index == sizeof(Block)) {
		Block b;
		mit_des_ecb_encrypt(stp->str_feed, b, stp->str_sched, 1);
		memcpy((void *)stp->str_feed, (void *)b, sizeof(Block));
		stp->str_index = 1;	/* Next time will be 1 */
		index = 0;		/* But now use 0 */ 
	}

	return(data ^ stp->str_feed[index]);
}
#  endif /* DES_ENCRYPTION */
# endif	/* AUTHENTICATION */
#else	/* ENCRYPTION */
#include "misc-proto.h"
#endif	/* ENCRYPTION */
