/*
 * Copyright (c) 1991, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* based on @(#)encrypt.c	8.1 (Berkeley) 6/4/93 */

/*
 * Copyright (C) 1990 by the Massachusetts Institute of Technology
 *
 * Export of this software from the United States of America may
 * require a specific license from the United States Government.
 * It is the responsibility of any person or organization contemplating
 * export to obtain such a license before exporting.
 *
 * WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
 * distribute this software and its documentation for any purpose and
 * without fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that
 * the name of M.I.T. not be used in advertising or publicity pertaining
 * to distribution of the software without specific, written prior
 * permission.  M.I.T. makes no representations about the suitability of
 * this software for any purpose.  It is provided "as is" without express
 * or implied warranty.
 */

/* mostly based on appl/telnet/libtelnet/encrypt.c, but changed to not
   use global variables since we might have multiple connections with
   different states */

#ifndef ENCRYPTION
#define ENCRYPTION
#endif

#ifdef ENCRYPTION

#define	ENCRYPT_NAMES

#ifndef DES_ENCRYPTION
#define DES_ENCRYPTION
#endif /* DES_ENCRYPTION */

#include "telnet.h"
#include "tnae.h"
#include "encrypt.h"

#include <stdlib.h>
#include <strings.h>

#define	typemask(x)	((x) > 0 ? 1 << ((x)-1) : 0)

static long i_support_encrypt = typemask(ENCTYPE_DES_CFB64)
				| typemask(ENCTYPE_DES_OFB64);
static long i_support_decrypt = typemask(ENCTYPE_DES_CFB64)
				| typemask(ENCTYPE_DES_OFB64);

static Encryptions encryptions[] = {
#ifdef	DES_ENCRYPTION
    { "DES_CFB64",	ENCTYPE_DES_CFB64,
			cfb64_encrypt,	
			cfb64_decrypt,
			cfb64_init,
			fb64_start,
			fb64_is,
			fb64_reply,
			fb64_session,
			fb64_keyid},
    { "DES_OFB64",	ENCTYPE_DES_OFB64,
			ofb64_encrypt,	
			ofb64_decrypt,
			ofb64_init,
			fb64_start,
			fb64_is,
			fb64_reply,
			fb64_session,
			fb64_keyid},
#endif	/* DES_ENCRYPTION */
    { 0, },
};


Encryptions *findencryption(int type)
{
	Encryptions *ep = encryptions;

	if (!(i_support_encrypt & typemask(type)))
		return(0);
	while (ep->type && ep->type != type)
		++ep;
	return(ep->type ? ep : 0);
}

Encryptions *finddecryption(int type)
{
	Encryptions *ep = encryptions;

	if (!(i_support_decrypt & typemask(type)))
		return(0);
	while (ep->type && ep->type != type)
		++ep;
	return(ep->type ? ep : 0);
}


/*
 * Called when ENCRYPT SUPPORT is received.
 * If we can encrypt in a way the server supports, set up that method
 */
void encrypt_support(tnParams *tn, unsigned char *typelist, int cnt)
{
	k5_tn_enc_data *edata=tn->encryptdata;
	register int type;
	Encryptions *ep, *use=NULL;
	char support[32], numsupp=0;

	/* tell him what we support. (we should do this earlier, but there's no easy
	   way to in the ncsa interface) */
	for(ep=encryptions; ep->type; ep++)
		support[numsupp++] = ep->type;

	tn_sendsub(tn, TELOPT_ENCRYPT, ENCRYPT_REQSTART, NULL, 0);
	tn_sendsub(tn, TELOPT_ENCRYPT, ENCRYPT_SUPPORT, support, numsupp);
	
	/*
	 * Forget anything the other side has previously told us.
	 */
	edata->remote_supports_decrypt = 0;

	while (cnt-- > 0) {
		type = *typelist++;
		if ((type < ENCTYPE_CNT) &&
		    (i_support_encrypt & typemask(type))) {
			edata->remote_supports_decrypt |= typemask(type);
			if (use == NULL)
				use = findencryption(type);
		}
	}
	if (use) {
		edata->ep = use;
		if (edata->ep->init) (*edata->ep->init)(&(edata->estate));
		type = edata->ep->start ? (*edata->ep->start)(tn, edata->estate, DIR_ENCRYPT) : 0;
		if (type < 0)
			return;
		if (type == 0)
			encrypt_start_output(tn);
	}
}

/*
 * Called when ENCRYPT IS is received
 * Set up the decryption method for the indicated encryption type
 */
void encrypt_is(tnParams *tn, unsigned char *data, int cnt)
{
	k5_tn_enc_data *edata=tn->encryptdata;
	Encryptions *dp=NULL;
	register int type, ret;

	if(!cnt--) return;
	type = *data++;
	if (type < ENCTYPE_CNT)
		edata->remote_supports_encrypt |= typemask(type);
	if (!(dp = finddecryption(type)))
		return;

	if (dp == edata->ep) edata->dstate = edata->estate;
	else {
		if (dp->init) (*dp->init)(&(edata->dstate));
		if (dp->start) (*dp->start)(tn, edata->dstate, DIR_DECRYPT);
		if (dp->session && edata->key) (*dp->session)(tn, edata->dstate, edata->key);
	}

	if (!dp->is) ret = 0;
	else ret = (*dp->is)(tn, edata->dstate, data, cnt);

	if (ret < 0) edata->autodecrypt = 0;
	else {
		edata->dp = dp;
		if (ret == 0 && edata->autodecrypt)
			encrypt_send_request_start(tn);
	}
}

/*
 * Called when ENCRYPT REPLY is received
 * Pass on the reply to the chosen encryptor, and if we're done setup, start encrypting
 */
void encrypt_reply(tnParams *tn, unsigned char *data, int cnt)
{
	k5_tn_enc_data *edata = tn->encryptdata;
	Encryptions *ep;
	register int ret, type;

	if (!cnt--) return;
	type = *data++;
	if (!(ep = findencryption(type)))
		return;

	if (!ep->reply) ret = 0;
	else ret = (*ep->reply)(tn, edata->estate, data, cnt);

	if (ret < 0) edata->autoencrypt = 0;
	else {
		edata->ep = ep;
		if (ret == 0 && edata->autoencrypt)
			encrypt_start_output(tn);
	}
}

/*
 * Called when a ENCRYPT START command is received.
 * If we really are ready, then go, otherwise send an error
 */
void encrypt_start(tnParams *tn)
{
	k5_tn_enc_data *edata=tn->encryptdata;

	if (!edata->dp) {
		/*
		 * Something is wrong.  We should not get a START
		 * command without having already picked our
		 * decryption scheme.  Send a REQUEST-END to
		 * attempt to clear the channel...
		 */
		encrypt_send_request_end(tn);
		return;
	}

	edata->decrypting = tn->decrypting = 1;
	encrypt_start_output(tn);
//	if (encrypt_verbose)
//		printf("[ Input is now decrypted with type %s ]\r\n",
//			ENCTYPE_NAME(decrypt_mode));
}

/*
 * Called to set up the encryption session key
 */
void encrypt_session_key(tnParams *tn, Session_Key *key)
{
	k5_tn_enc_data *edata=tn->encryptdata;
	
	edata->key = key;

	if(edata->ep) edata->ep->session(tn, edata->estate, key);
	if(edata->dp && (edata->dp != edata->ep)) edata->dp->session(tn, edata->dstate, key);

	if (edata->autoencrypt) encrypt_start_output(tn);
	if (edata->autodecrypt) encrypt_send_request_start(tn);
}

/*
 * Called when ENCRYPT END is received.
 */
void encrypt_end(tnParams *tn)
{
	k5_tn_enc_data *edata=tn->encryptdata;

	edata->decrypting = tn->decrypting = 0;
//	if (encrypt_verbose)
//		printf("[ Input is now clear text ]\r\n");
}

/*
 * Called when ENCRYPT REQUEST-END is received.
 */
void encrypt_request_end(tnParams *tn)
{
	encrypt_send_end(tn);
}

/*
 * Called when ENCRYPT REQUEST-START is received.  If we receive
 * this before a type is picked, then that indicates that the
 * other side wants us to start encrypting data as soon as we
 * can. 
 */
void encrypt_request_start(tnParams *tn)
{
	k5_tn_enc_data *edata=tn->encryptdata;

	if (!edata->ep)  {
		edata->autoencrypt = 1;
		return;
	}
	encrypt_start_output(tn);
}

void encrypt_enc_keyid(tnParams *tn ,unsigned char *keyid, int len)
{
	k5_tn_enc_data *edata=tn->encryptdata;
	int ret=-1;
	
	if(!edata->ep) {
		if (len == 0) return;
		edata->enc_key_len = 0;
	} else if (len == 0) {
		if(edata->enc_key_len == 0) return;
		edata->enc_key_len = 0;
		if(edata->ep->keyid)
			(void)(*edata->ep->keyid)(tn, edata->estate, DIR_ENCRYPT, edata->enc_key, &(edata->enc_key_len));
	} else {
		if ((edata->enc_key_len != len) || (memcmp(keyid, edata->enc_key, len))) {
			edata->enc_key_len = len;
			memcpy(edata->enc_key, keyid, len);
			tn_sendsub(tn, TELOPT_ENCRYPT, ENCRYPT_ENC_KEYID, edata->enc_key, edata->enc_key_len);
		}
		if (edata->ep->keyid)
			ret = (*edata->ep->keyid)(tn, edata->estate, DIR_ENCRYPT, edata->enc_key, &(edata->enc_key_len));
		if ((ret == 0) && edata->autoencrypt)
			encrypt_start_output(tn);
		return;
	}

}

void encrypt_dec_keyid(tnParams *tn ,unsigned char *keyid, int len)
{
	k5_tn_enc_data *edata=tn->encryptdata;
	int ret;
	
	if(!edata->dp) {
		if (len == 0) return;
		edata->dec_key_len = 0;
	} else if (len == 0) {
		if(edata->dec_key_len == 0) return;
		edata->dec_key_len = 0;
		if(edata->dp->keyid)
			(void)(*edata->dp->keyid)(tn, edata->dstate, DIR_DECRYPT, edata->dec_key, &(edata->dec_key_len));
	} else {
		if ((edata->dec_key_len != len) || (memcmp(keyid, edata->dec_key, len))) {
			edata->dec_key_len = len;
			memcpy(edata->dec_key, keyid, len);
			tn_sendsub(tn, TELOPT_ENCRYPT, ENCRYPT_DEC_KEYID, edata->dec_key, edata->dec_key_len);
		}
		if (edata->dp->keyid)
			ret = (*edata->dp->keyid)(tn, edata->dstate, DIR_DECRYPT, edata->dec_key, &(edata->dec_key_len));
		return;
	}
}

void encrypt_start_output(tnParams *tn)
{
	k5_tn_enc_data *edata=tn->encryptdata;
	Encryptions *ep;
	register int i;

	if (edata->ep->start) {
		i = (*edata->ep->start)(tn, edata->estate, DIR_ENCRYPT);
		if (i)
			return;
	}
	tn_sendsub(tn, TELOPT_ENCRYPT, ENCRYPT_START, edata->enc_key, edata->enc_key_len);
	edata->encrypting = tn->startencrypting = 1;
//	if (encrypt_verbose)
//		printf("[ Output is now encrypted with type %s ]\r\n",
//			ENCTYPE_NAME(type));
}

void encrypt_send_end(tnParams *tn)
{
	k5_tn_enc_data *edata=tn->encryptdata;
	if (!edata->encrypting)
		return;

	tn_sendsub(tn, TELOPT_ENCRYPT, ENCRYPT_END, NULL, 0);
	edata->encrypting = tn->encrypting = 0;
//	if (encrypt_verbose)
//		printf("[ Output is now clear text ]\r\n");
}

void encrypt_send_request_start(tnParams *tn)
{
	k5_tn_enc_data *edata=tn->encryptdata;
	
	tn_sendsub(tn, TELOPT_ENCRYPT, ENCRYPT_REQSTART, edata->dec_key, edata->dec_key_len);
}

void encrypt_send_request_end(tnParams *tn)
{
	tn_sendsub(tn, TELOPT_ENCRYPT, ENCRYPT_REQEND, NULL, 0);
}

#endif /* ENCRYPTION */