%{
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include <tcl.h>

static char *errbuf;
static int lineno, level;
static Tcl_DString list;
typedef int YYSTYPE;

#define skip_whitespace_forward(v)	while(isspace(*v)) v++
#define skip_whitespace_reverse(v)	while(isspace(*v)) v--
#define seek_whitespace_forward(v)	while(!isspace(*v)) v++

static char *parse_tagname (char **ptr) {
  char *tag, *p = *ptr;
  skip_whitespace_forward (p);
  tag = p;
  while (*p && !isspace (*p) && *p != '=')
    p++;
  *ptr = p;
  return tag;
}

static void add_comment(char *s) {
  Tcl_DStringStartSublist (&list);
  Tcl_DStringAppendElement (&list, "comment");
  Tcl_DStringAppendElement (&list, s);
  Tcl_DStringEndSublist (&list);
}

#undef yywrap
%}
%s NORMAL
WS	[ \t]*
NL	(\r\n|\n)
TAGCHR	[A-Za-z0-9._:/$-]
STAGCHR	[A-Za-z0-9._:/$-]
COMMENT	[;#]
%%
%{
	lineno = 1;
	level = 0;
	Tcl_DStringInit (&list);
%}

{NL}					{
	/* Track line numbers, handle blank lines.  Note that since
	   we're trying to handle CR, CR-LF sequences as end-of-line,
	   but UNIX normally does not, we can't rely on using "$" in
	   the patterns below.  Thus they'll tend to use something
	   like "{WS}{NL}" and "lineno++" themselves.  */
	  lineno++;
	}
^{WS}{NL}				{ lineno++; }

\[{STAGCHR}+\]{WS}{NL}			{
	/* Stanza beginning is valid in initial (comment) and normal
	   (data-parsing) states.  */
	  char *cp = yytext, *cp2;
	  if (level > 1) {
	      sprintf (errbuf,
		       "%d: new top-level stanza started at level %d",
		       lineno, level);
	      return 1;
	  }
	  cp++;
	  cp2 = cp + strlen (cp) - 1;
	  skip_whitespace_reverse (cp2);
	  assert (*cp2 == ']');
	  *cp2 = 0;
	  if (level == 1) {
	    /* end list of data items */
	    Tcl_DStringEndSublist (&list);
	    /* end "section foo ..." list */
	    Tcl_DStringEndSublist (&list);
	  }
	  level = 1;
	  Tcl_DStringStartSublist (&list);
	  Tcl_DStringAppendElement (&list, "section");
	  Tcl_DStringAppendElement (&list, cp);
	  Tcl_DStringStartSublist (&list);
	  BEGIN NORMAL;
	  lineno++;
	}

\[{STAGCHR}+\]				{
	  sprintf (errbuf,
		   "extra text after stanza label at line %d\n", lineno);
	  return 1;
	}
\[.*\]					{
	  sprintf (errbuf,
		   "invalid stanza name at line %d\n", lineno);
	  return 1;
	}
\[					{
	  sprintf (errbuf,
		   "unbalanced `[' at %d\n", lineno);
	  return 1;
	}

<NORMAL>{WS}{TAGCHR}+{WS}={WS}{NL}?{WS}\{{WS}{NL}	{
	/* Look for start of a new subdivision.  */
	  char *cp, *cp2, *tag;
	  level++;
	  cp = yytext;
	  tag = parse_tagname (&cp);
	  *cp = 0;
	  while (*++cp != '{')
	    if (*cp == '\n' || *cp == '\r') {
	      lineno++;
	      break;
	    }
	  Tcl_DStringStartSublist (&list);
	  Tcl_DStringAppendElement (&list, "section");
	  Tcl_DStringAppendElement (&list, tag);
	  Tcl_DStringStartSublist (&list);
	  lineno++;
	}
<NORMAL>{WS}{TAGCHR}+{WS}={WS}.+{NL}	{
	/* Normal data definition.  */
	  char *tag, *cp, *value;
	  cp = yytext;
	  tag = parse_tagname (&cp);
	  value = strchr (cp, '=') + 1;
	  *cp = 0;
	  skip_whitespace_forward (value);
	  cp = value + strlen (value) - 1;
	  skip_whitespace_reverse (cp);
	  cp[1] = 0;
	  Tcl_DStringStartSublist (&list);
	  Tcl_DStringAppendElement (&list, "value");
	  Tcl_DStringAppendElement (&list, tag);
	  Tcl_DStringAppendElement (&list, value);
	  Tcl_DStringEndSublist (&list);
	  lineno++;
	}

<NORMAL>{WS}\}{WS}			{
	/* End of subdivision.  */
	  level--;
	  if (level < 1) {
	    sprintf (errbuf, "too many `}' at line %d\n", lineno);
	    return 1;
	  }
	  /* end list of subsection entities */
	  Tcl_DStringEndSublist (&list);
	  /* end "section foo ..." 3-element list */
	  Tcl_DStringEndSublist (&list);
	}

<INITIAL>^[^[].*$			{ add_comment (yytext); }
<NORMAL>^{COMMENT}.*$			{ add_comment (yytext + 1); }

<NORMAL>.				{
	/* This should catch syntax errors.  Basically, anything not
	   recognized above.  */
	  sprintf (errbuf, "syntax error at %d\n", lineno);
	  return 1;
	}

%%
#undef yywrap
int yywrap () { return 1; }

static int parse_profile (FILE *in, Tcl_DString *result, char *Perrbuf) {
  int x;
  yyin = in;
  errbuf = Perrbuf;
  errbuf[0] = 0;
  x = yylex ();
  if (x == 0) {
    if (level > 1) {
      sprintf (errbuf, "EOF at nesting level %d", level);
      return 1;
    }
    if (level == 1) {
      /* end list of data items for current stanza */
      Tcl_DStringEndSublist (&list);
      /* end current stanza 3-element list */
      Tcl_DStringEndSublist (&list);
    }
    *result = list;
    return 0;
  }
  return 1;
}

/* These have nothing at all to do with lex, but since the rest of the
   I/O and parsing are handled here, it seems logical.  */

int krb5tcl_read_profile (ClientData clientData, Tcl_Interp *interp,
			  int argc, char *argv[])
{
    FILE *f;
    Tcl_DString r;
    int x;
    char *s;
    char errbuf[100];

    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"",
			  argv[0], " filename\"", NULL);
	return TCL_ERROR;
    }

    if (argv[1][0] == 0) {
	Tcl_AppendResult (interp, "bad filename \"\"", NULL);
	return TCL_ERROR;
    }

    f = fopen (argv[1], "r");
    if (f == 0) {
	Tcl_AppendResult (interp, "can't read file \"", argv[1], "\"", NULL);
	return TCL_ERROR;
    }

    x = parse_profile (f, &r, errbuf);
    if (x) {
	Tcl_AppendResult (interp, "error parsing file: ",
			  errbuf[0] ? errbuf : "[unknown internal error]",
			  NULL);
	return TCL_ERROR;
    }

    Tcl_DStringResult (interp, &r);
    return TCL_OK;
}

static int wrong_num_parms (Tcl_Interp *interp, int got, int wanted,
			    char *clause)
{
    char tmpbuf[40];
    sprintf (tmpbuf, "%d, wanted %d", got, wanted);
    Tcl_AppendResult (interp,
		      "wrong number of parameters (", tmpbuf, ") in \"",
		      clause, "\" clause", NULL);
    return TCL_ERROR;
}

static int is_okay_tagname (char *name) {
    int x;

    if (*name == 0)
	return 0;
#if 0
    do {
	if (isalnum (*name))
	    continue;
	switch (*name) {
	default:
	    return 0;
	case '.':
	case '_':
	case '-':
	    continue;
	}
    } while (*++name);
#endif
    /* Would be nice to include local alphabetic characters, but
       we need to reject everything that the above lexer will.  */
    x = strspn (name, "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789._:/$-");
    return name[x] == 0;
}

int krb5tcl_is_okay_tagname (ClientData clientData, Tcl_Interp *interp,
			     int argc, char *argv[])
{
    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " tagname\"", NULL);
	return TCL_ERROR;
    }
    Tcl_SetResult (interp, is_okay_tagname (argv[1]) ? "1" : "0", TCL_STATIC);
    return TCL_OK;
}

static int is_okay_value (char *value) {
    for (; *value; value++) {
	if (iscntrl (*value) && *value != '\t')
	    return 0;
	switch (*value) {
	case '\r':
	case '\n':
	case '{':
	case '}':
	    return 0;
	}
    }
    return 1;
}

int krb5tcl_is_okay_value (ClientData clientData, Tcl_Interp *interp,
			   int argc, char *argv[])
{
    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " value\"", NULL);
	return TCL_ERROR;
    }
    Tcl_SetResult (interp, is_okay_value (argv[1]) ? "1" : "0", TCL_STATIC);
    return TCL_OK;
}

static int write_spaces (Tcl_Channel out, int n_spaces) {
    static const char spaces[] = "                                     ";
    int ret;
    while (n_spaces) {
	int n_to_write = n_spaces;
	if (n_to_write > sizeof (spaces))
	    n_to_write = sizeof (spaces);
	ret = Tcl_Write (out, spaces, n_to_write);
	if (ret < 0)
	    return TCL_ERROR;
	n_spaces -= ret;
    }
    return TCL_OK;
}
static int write_string (Tcl_Channel out, const char *str, int len) {
    return (Tcl_Write (out, str, len) == len) ? TCL_OK : TCL_ERROR;
}
static int write_list ();
static int write_data (Tcl_Interp *interp, Tcl_Channel out, char *string, int lvl) {
    char **argv;
    int argc, code, n_spaces, ret;

    code = Tcl_SplitList (interp, string, &argc, &argv);
    if (code != TCL_OK)
	return code;

#define A(n)	if (argc != (n)) { return wrong_num_parms (interp, argc, n, argv[0]); }
#define BAD(ARGS) do{ Tcl_AppendResult ARGS; return TCL_ERROR; }while(0)

    if (argc == 0)
	BAD ((interp, "empty list found in profile data; not valid"));

    n_spaces = 3 * lvl;

#define put(x) if (write_string(out, (x), strlen (x)) == TCL_ERROR) return TCL_ERROR
    if (!strcmp ("value", argv[0])) {
	A(3);
	if (!is_okay_tagname (argv[1]))
	    BAD ((interp, "bad tag name \"", argv[1], "\""));
	if (!is_okay_value (argv[2]))
	    BAD ((interp, "bad value \"", argv[2], "\" for tag \"",
		  argv[1], "\""));
	write_spaces (out, n_spaces);
	put (argv[1]);
	put (" = ");
	put (argv[2]);
	put ("\n");
    } else if (!strcmp ("section", argv[0])) {
	A(3);
	if (!is_okay_tagname (argv[1]))
	    BAD ((interp, "bad tag name \"", argv[1], "\""));
	write_spaces (out, n_spaces);
	if (lvl > 0) {
	    put (argv[1]);
	    put (" = {\n");
	    ret = write_list (interp, out, argv[2], lvl+1);
	    if (ret)
		return ret;
	    write_spaces (out, n_spaces);
	    put ("}\n");
	} else {
	    put ("[");
	    put (argv[1]);
	    put ("]\n");
	    ret = write_list (interp, out, argv[2], lvl+1);
	    if (ret)
		return ret;
	    put ("\n");
	}
    } else if (!strcmp ("comment", argv[0])) {
	/* no leading whitespace! */
	A(2);
	if (strchr (argv[1], '\n') || strchr (argv[1], '\r'))
	    BAD ((interp,
		  "comments may not contain newline or carriage-return characters",
		  NULL));
	if (lvl == 0) {
	    put (";");
	}
	put (argv[1]);
	put ("\n");
    } else {
	BAD ((interp, "unknown object type \"", argv[0], "\"", NULL));
    }
#undef A
    return TCL_OK;
}

static int write_list (Tcl_Interp *interp, Tcl_Channel out, char *list, int lvl) {
    char **argv;
    int argc, code, i, ret;

    code = Tcl_SplitList (interp, list, &argc, &argv);
    if (code != TCL_OK)
	return code;
    for (i = 0; i < argc; i++) {
	ret = write_data (interp, out, argv[i], lvl);
	if (ret)
	    return ret;
    }
    return TCL_OK;
}

int krb5tcl_put_profile (ClientData clientData, Tcl_Interp *interp,
			 int argc, char *argv[])
{
    Tcl_DString r;
    int ret;
    Tcl_Channel channel;
    int mode;
    char *fileh, *data;

    if (argc != 2 && argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"",
			  argv[0], " ?filehandle? profile-data\"", NULL);
	return TCL_ERROR;
    }
    if (argc == 2)
	fileh = "stdout", data = argv[1];
    else
	fileh = argv[1], data = argv[2];

    if (fileh[0] == 0) {
	Tcl_AppendResult (interp, "bad filehandle \"\"", NULL);
	return TCL_ERROR;
    }

    channel = Tcl_GetChannel (interp, fileh, &mode);
    if (channel == 0) {
	Tcl_AppendResult (interp, "invalid filehandle \"", fileh, "\"", NULL);
	return TCL_ERROR;
    }
    if ((mode & TCL_WRITABLE) == 0) {
	Tcl_AppendResult (interp, "can't write to filehandle \"", fileh, "\"", NULL);
	return TCL_ERROR;
    }

    ret = write_list (interp, channel, data, 0);

    if (ret)
	return ret;

    return Tcl_Flush (channel);
}

int krb5tcl_write_profile (ClientData clientData, Tcl_Interp *interp,
			   int argc, char *argv[])
{
    Tcl_Channel channel;
    int ret;
    char *s;

    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"",
			  argv[0], " filename profile-data\"", NULL);
	return TCL_ERROR;
    }

    if (argv[1][0] == 0) {
	Tcl_AppendResult (interp, "bad filename \"\"", NULL);
	return TCL_ERROR;
    }

    channel = Tcl_OpenFileChannel (interp, argv[1], "w", 0644);
    if (channel == 0) {
	Tcl_AppendResult (interp, "can't write file \"", argv[1], "\"", NULL);
	return TCL_ERROR;
    }

    ret = write_list (interp, channel, argv[2], 0);
    if (ret == 0)
	ret = Tcl_Flush (channel);

    if (ret) {
	Tcl_AppendResult (interp, "error writing to file \"", argv[1], "\"",
			  NULL);
	Tcl_Close (interp, channel);
	return ret;
    }
    ret = Tcl_Close (interp, channel);
    if (ret)
	Tcl_AppendResult (interp, "error writing to file \"", argv[1], "\"",
			  NULL);
    return ret;
}

#include <krb5.h>
#include "context.h"

void krb5tcl_init_profile (krb5tcl_context *ctx) {
#define X(NAME) krb5tcl_add_proc (#NAME, NAME, ctx)
    X (krb5tcl_read_profile);
    X (krb5tcl_write_profile);
    X (krb5tcl_put_profile);
    X (krb5tcl_is_okay_tagname);
    X (krb5tcl_is_okay_value);
}

#ifdef TEST
/* test program */

int Tcl_AppInit (Tcl_Interp *interp) {
  Tcl_DString d;
  if (parse_profile (stdin, &d) == 0) {
    Tcl_SetVar (interp, "profile", d.string, 0);
    write_list (interp, stdout, d.string, 0);
    fflush (stdout);
    return 0;
  } else {
    fprintf (stderr, "error!\n");
    return 1;
  }
}
int main (int argc, char *argv[]) {
  Tcl_Main (argc, argv, Tcl_AppInit);
}
#endif
