#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#define KERBNET_REG_NAME "SOFTWARE\\Cygnus Solutions\\Kerbnet\\1"
#define CEDENTIALS_CACHE_REG_NAME "CredentialsCacheDir"
#define KERBNET_HOME_REG_NAME "KERBNET_HOME"  
#define CCACHE_DIR "CCACHE"

static HWND global_parent_wind;
static char global_msgbuf[512];

/*
 * Convert system error to char. Returns 
 * memory allocated with LocalAlloc.
 */

static char *error_to_string(DWORD error)
{
  char *msgbuf;
  
  if(FormatMessage(
       FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
       NULL,
       error,
       MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), /* Default language */
       (char *)&msgbuf,
       0,
       NULL
       ) == 0)
    return 0;
  return msgbuf;
}

/*
 * Return a pointer to a string describing an os error.
 * error_to_string returns a pointer to LocalAlloc'ed
 * memory. Cache it and release when the next one is
 * requested.
 */

static char *str_oserr(DWORD err)
{
  static char *lastmsg = 0;

  if(lastmsg)
    LocalFree((HLOCAL)lastmsg);

  lastmsg = error_to_string(err);
  return lastmsg;
}

/*
 * Utility function to get allocate a SID from a name.
 * Looks on local machine. SID is allocated with LocalAlloc
 * and must be freed by the caller.
 * Returns TRUE on success, FALSE on fail.
 */

static BOOL get_sid(const char *name, SID **ppsid)
{
  SID_NAME_USE sid_use;
  DWORD sid_size = 0;
  DWORD dom_size = 0;
  char *domain;

  *ppsid = 0;
  if(LookupAccountName(0, name, 0, &sid_size, 0, &dom_size, &sid_use) == 0) {
    if(GetLastError() != ERROR_INSUFFICIENT_BUFFER) {
      _snprintf( global_msgbuf, sizeof(global_msgbuf),
	  "get_sid: LookupAccountName for size on name %s failed. \
Error was %s", name, str_oserr(GetLastError()));
	  MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
      return FALSE;
    }
  }

  *ppsid = (SID *)LocalAlloc( LMEM_FIXED, sid_size);
  domain = (char *)LocalAlloc( LMEM_FIXED, dom_size);
  if( *ppsid == 0 || domain == 0) {
    _snprintf( global_msgbuf, sizeof(global_msgbuf), 
		"get_sid: LocalAlloc failed. Error was %s",
                 str_oserr(GetLastError()));
    MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
    if(*ppsid)
      LocalFree((HLOCAL)*ppsid);
    if(domain)
      LocalFree((HLOCAL)domain);
    *ppsid = 0;
    return FALSE;
  }

  if(LookupAccountName(0, name, *ppsid, &sid_size, domain, &dom_size, &sid_use) == 0) {
    _snprintf( global_msgbuf, sizeof(global_msgbuf), 
         "get_sid: LookupAccountName failed for name %s. Error was %s",
         name, str_oserr(GetLastError()));
	MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
    LocalFree((HLOCAL)*ppsid);
    LocalFree((HLOCAL)domain);
    *ppsid = 0;
    return FALSE;
  }

  LocalFree((HLOCAL)domain);
  return TRUE;
}

/*
 * Utility function to setup a security descriptor
 * from a varargs list of char *name followed by a DWORD access
 * mask, followed by a DWORD inherit mask. The access control list is allocated with 
 * LocalAlloc and must be freed by the caller.
 * returns TRUE on success, FALSE on fail.
 */

static BOOL create_sd_from_list( SECURITY_DESCRIPTOR *sdout, int num, ...)
{
  va_list ap;
  SID **sids = 0;
  char *name;
  DWORD amask;
  DWORD imask;
  DWORD acl_size;
  PACL pacl = 0;
  int i;

  if((sids = (SID **)calloc(1,sizeof(SID *)*num)) == 0) {
        _snprintf( global_msgbuf, sizeof(global_msgbuf), 
			"create_sd_from_list: calloc fail.");
	MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
    return FALSE;
  }

  acl_size = num * (sizeof(ACL) +
             sizeof(ACCESS_ALLOWED_ACE) +
             sizeof(DWORD));

  /* Collect all the SID's */
  va_start( ap, num);
  for( i = 0; i < num; i++) {
    name = va_arg( ap, char *);
    amask = va_arg(ap, DWORD);
    imask = va_arg(ap, DWORD);
    if(get_sid( name, &sids[i]) == FALSE)
      goto cleanup;
    acl_size += GetLengthSid(sids[i]);
  }
  va_end(ap);
  if((pacl = (PACL)LocalAlloc( LMEM_FIXED, acl_size)) == 0) {
        _snprintf( global_msgbuf, sizeof(global_msgbuf), 
			"create_sd_from_list: LocalAlloc fail. Error was %s",
            str_oserr(GetLastError()));
    MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
	goto cleanup;
  }

  if(InitializeSecurityDescriptor( sdout, SECURITY_DESCRIPTOR_REVISION) == FALSE) {
        _snprintf( global_msgbuf, sizeof(global_msgbuf), 
			"create_sd_from_list: InitializeSecurityDescriptor fail. Error was %s",
                 str_oserr(GetLastError()));
    MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
	goto cleanup;
  }
  if(InitializeAcl( pacl, acl_size, ACL_REVISION) == FALSE) {
        _snprintf( global_msgbuf, sizeof(global_msgbuf), 
			"create_sd_from_list: InitializeAcl fail. Error was %s",
                 str_oserr(GetLastError()));
    MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
	goto cleanup;
  }
  va_start(ap, num);
  for( i = 0; i < num; i++) {
    ACE_HEADER *ace_p;
    name = va_arg( ap, char *);
    amask = va_arg( ap, DWORD);
    imask = va_arg( ap, DWORD);

    if(AddAccessAllowedAce( pacl, ACL_REVISION, amask, sids[i]) == FALSE) {
          _snprintf( global_msgbuf, sizeof(global_msgbuf), 
				"create_sd_from_list: AddAccessAllowedAce fail. Error was %s",
                 str_oserr(GetLastError()));
	  MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
      goto cleanup;
    }
    /* Set the ACE inherit mask */
    if(GetAce( pacl, i, (LPVOID *)&ace_p) == FALSE) {
          _snprintf( global_msgbuf, sizeof(global_msgbuf), 
				"create_sd_from_list: GetAce fail. Error was %s",
                 str_oserr(GetLastError()));
	  MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
      goto cleanup;
    }
    ace_p->AceFlags |= imask;
  }

  /* Add the ACL into the sd. */
  if(SetSecurityDescriptorDacl( sdout, TRUE, pacl, FALSE) == FALSE) {
        _snprintf( global_msgbuf, sizeof(global_msgbuf), 
				"create_sd_from_list: SetSecurityDescriptorDacl fail. Error was %s",
               str_oserr(GetLastError()));
    MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
    goto cleanup;
  }
  for( i = 0; i < num; i++)
    if(sids[i] != 0)
      LocalFree((HLOCAL)sids[i]);
  free(sids);

  return TRUE;

cleanup:

  if(sids != 0) {
    for( i = 0; i < num; i++)
      if(sids[i] != 0)
        LocalFree((HLOCAL)sids[i]);
    free(sids);
  }
  if(pacl != 0)
    LocalFree((HLOCAL)pacl);
  return FALSE;
}

/*
 * Utility function to cause a registry tree to be created.
 * If the supplied tree doesn't exist it will walk down the
 * tree creating the branches.
 */

static BOOL create_registry_tree( HKEY key_start, const char *reg_path, SECURITY_DESCRIPTOR *sd,
								const char *base_path)
{
  /* First see if the key already exists */
  HKEY key;
  DWORD err;
  DWORD disp;
  char *cur_path = 0;
  char *p;
  char *kerbnet_base;
  char *ccachedir;
  int len;
  SECURITY_ATTRIBUTES sa;

  sa.nLength = sizeof(SECURITY_ATTRIBUTES);
  sa.lpSecurityDescriptor = sd; 
  sa.bInheritHandle = FALSE;

  if((err = RegOpenKeyEx( key_start, reg_path, 0, KEY_ALL_ACCESS, &key)) == ERROR_SUCCESS) {
    /* Key exists */
	goto create_vals;
  }

  if((cur_path = malloc(strlen(reg_path) + 1)) == 0) {
        _snprintf( global_msgbuf, sizeof(global_msgbuf), 
			"create_registry_tree: malloc failed.");
	MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
    return FALSE;
  }

  p = strchr(reg_path, '\\');
  while(1) {
    if( p == 0 ) {
      strcpy(cur_path, reg_path);
    } else {
      strncpy( cur_path, reg_path, p - reg_path);
      cur_path[p-reg_path] = '\0';
    }

    if((err = RegCreateKeyEx( key_start, cur_path, 0, 0, REG_OPTION_NON_VOLATILE,
                               KEY_ALL_ACCESS, &sa, &key, &disp)) != ERROR_SUCCESS) {
      free(cur_path);
	  _snprintf( global_msgbuf, sizeof(global_msgbuf), 
            "create_registry_tree: RegCreateKeyEx failed for path %s. Error was %s.", 
            cur_path, str_oserr(err));
	  MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
	  return FALSE;

    }
    RegCloseKey(key);
    if(p == 0)
      break;
    p++;
    p = strchr(p, '\\');
  }

  /* Now add the two named values onto the last component. */
  if((err = RegOpenKeyEx( key_start, reg_path, 0, KEY_ALL_ACCESS, &key)) != ERROR_SUCCESS) {
      _snprintf( global_msgbuf, sizeof(global_msgbuf), 
            "create_registry_tree: RegOpenKeyEx failed for path %s. Error was %s.", 
            cur_path, str_oserr(err));
	  MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
	  return FALSE;
  }

create_vals:

  len = strlen(base_path);
  if((kerbnet_base = (char *)malloc(len + 2)) == 0) {
      _snprintf( global_msgbuf, sizeof(global_msgbuf), 
			"create_registry_tree: malloc failed."); 
	  MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
	  return FALSE;
  }
  if((ccachedir = (char *)malloc(len + 2 + strlen(CCACHE_DIR))) == 0 ) {
      _snprintf( global_msgbuf, sizeof(global_msgbuf), 
			"create_registry_tree: malloc failed.");
	  MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
	  free(kerbnet_base);
	  return FALSE;
  }
  strcpy( kerbnet_base, base_path);
  if(kerbnet_base[len-1] != '\\') {
	kerbnet_base[len] = '\\';
	kerbnet_base[len+1] = '\0';
  }
  strcpy( ccachedir, kerbnet_base);
  strcat( ccachedir, CCACHE_DIR);
  if((err = RegSetValueEx( key, KERBNET_HOME_REG_NAME, 0, REG_SZ, kerbnet_base,
					strlen(kerbnet_base)+1)) != ERROR_SUCCESS) {
      _snprintf( global_msgbuf, sizeof(global_msgbuf), 
			"create_registry_tree: RegSetValueEx for value %s failed. \
Error was %s.", KERBNET_HOME_REG_NAME, str_oserr(err));
	  MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
	  free(kerbnet_base);
	  free(ccachedir);
	  return FALSE;
  }
  if((err = RegSetValueEx( key, CEDENTIALS_CACHE_REG_NAME, 0, REG_SZ, ccachedir,
					strlen(ccachedir)+1)) != ERROR_SUCCESS) {
      _snprintf( global_msgbuf, sizeof(global_msgbuf), 
			"create_registry_tree: RegSetValueEx for value %s failed. \
Error was %s.", CEDENTIALS_CACHE_REG_NAME, str_oserr(err));
	  MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
	  free(kerbnet_base);
	  free(ccachedir);
	  return FALSE;
  }


  free(kerbnet_base);
  free(ccachedir);
  RegCloseKey( key );
  return TRUE;
}

/*
 * Function to recurse over all files/subdirectories in a tree and
 * set the given security on them.
 * Returns TRUE on success, FALSE on fail.
 */
BOOL set_directory_security( const char *start_dir, SECURITY_DESCRIPTOR *psd_dir,
							SECURITY_DESCRIPTOR *psd_file)
{
	HANDLE dh = INVALID_HANDLE_VALUE;
	WIN32_FIND_DATA fd;
	DWORD err = 0;
	char cur_dir[_MAX_PATH];
	char base_dir[_MAX_PATH];
	char new_name[_MAX_PATH];
	char *p;

	p = strrchr( start_dir, '\\');
	if( p ) {
		if( (p - start_dir) + 1 >= _MAX_PATH ) {
			_snprintf( global_msgbuf, sizeof(global_msgbuf),
						"set_directory_security: path name %s too long.", start_dir);
			MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
			return FALSE;
		}
		strncpy( base_dir, start_dir, (p - start_dir) + 1);
		base_dir[p - start_dir + 1] = '\0';
	} else 
		base_dir[0] = '\0';

	while( 1 ) {
		if(dh == INVALID_HANDLE_VALUE ) {
			if((dh = FindFirstFile( start_dir, &fd )) == INVALID_HANDLE_VALUE) {
				_snprintf( global_msgbuf, sizeof(global_msgbuf),
						"set_directory_security: FindFirstFile for directory %s failed. \
Error was %s.", 
					start_dir, str_oserr(GetLastError()));
				MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
				return FALSE;
			}
		} else {
			/* Get next file */
			if(FindNextFile( dh, &fd) == FALSE) {
				if((err = GetLastError()) == ERROR_NO_MORE_FILES)
					break;
				else {
					_snprintf( global_msgbuf, sizeof(global_msgbuf),
							"set_directory_security: FindNextFile for directory %s failed. \
Error was %s.", 
						start_dir, str_oserr(err));
					MessageBox( global_parent_wind, global_msgbuf, 0,
					  MB_OK| MB_ICONSTOP); 
					return FALSE;
				}
			}

		} 
		if( strcmp( fd.cFileName, "..") == 0)
			continue;

		if( strlen(	base_dir ) + strlen( fd.cFileName ) >= _MAX_PATH ) {
			_snprintf( global_msgbuf, sizeof(global_msgbuf),
						"set_directory_security: path %s%s too long.", 
					base_dir, fd.cFileName);
			MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
			return FALSE;
		}	
		strcpy( new_name, base_dir);
		strcat( new_name, fd.cFileName);

		if( (strcmp( fd.cFileName, ".") != 0) && 
			(fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )) {
			if(GetCurrentDirectory( sizeof(cur_dir), cur_dir ) == FALSE) {
				_snprintf( global_msgbuf, sizeof(global_msgbuf),
							"set_directory_security: GetCurrentDirectory failed. Error was %s.", 
						str_oserr(GetLastError()));
				MessageBox( global_parent_wind, global_msgbuf, 0,
					  MB_OK| MB_ICONSTOP); 
				return FALSE;
			}
			if(SetCurrentDirectory( new_name ) == FALSE) {
				_snprintf( global_msgbuf, sizeof(global_msgbuf),
							"set_directory_security: SetCurrentDirectory to %s failed. Error was %s.", 
						fd.cFileName, str_oserr(GetLastError()));
				MessageBox( global_parent_wind, global_msgbuf, 0,
					  MB_OK| MB_ICONSTOP); 
				return FALSE;
			}	
			if(set_directory_security( "*", psd_dir, psd_file) == FALSE)	{
				SetCurrentDirectory( cur_dir );
				return FALSE;
			}
			if(SetCurrentDirectory( cur_dir ) == FALSE) {
				_snprintf( global_msgbuf, sizeof(global_msgbuf),
							"set_directory_security: SetCurrentDirectory to %s failed. Error was %s.", 
						cur_dir, str_oserr(GetLastError()));
				MessageBox( global_parent_wind, global_msgbuf, 0,
					  MB_OK| MB_ICONSTOP); 
				return FALSE;
			}	
		}

		/* Set security */
		if(SetFileSecurity(new_name, DACL_SECURITY_INFORMATION, 
			(fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) ? psd_dir : psd_file) == FALSE) {
			_snprintf( global_msgbuf, sizeof(global_msgbuf),
				"set_directory_security: SetFileSecurity on filename %s \
failed. Error was %s.", fd.cFileName, str_oserr(GetLastError()));
			MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
			return FALSE;
		}
	}
	return TRUE;
}

/*
 * Function called from installshield.
 * base_path is where the kerberos directory is - by
 * default C:\CYGNUS\KERBNET
 * Returns TRUE on success, FALSE to fail.
 */

BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD fwdReason, LPVOID lpvReserved)
{
  return 1;
}


CHAR WINAPI do_install( HWND wnd, LPSTR dum1, LPSTR dum2, LPSTR base_path, LPSTR dum3)
{
  OSVERSIONINFO vers;
  char username[512];
  DWORD usize = sizeof(username);
  SECURITY_DESCRIPTOR sd_file;
  SECURITY_DESCRIPTOR sd_dir;
  SECURITY_DESCRIPTOR sd_reg;
  SECURITY_DESCRIPTOR sd_secure_dir;
  SECURITY_DESCRIPTOR sd_secure_file;
  SECURITY_DESCRIPTOR sd_ccache;
  DWORD dummy;
  DWORD fs_flags = 0;
  BOOL is_winnt = FALSE;

  global_parent_wind = wnd;
  /* Check if we are running on Windows NT */
  vers.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
  GetVersionEx( &vers );

  if(SetCurrentDirectory( base_path ) == FALSE) {
	_snprintf( global_msgbuf, sizeof(global_msgbuf), 
			  "Failed to change directory to %s. Error was %s", 
			  base_path, str_oserr(GetLastError()));
	MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
	return FALSE;
  }

  /* Create the ccache directory */
  if((CreateDirectory( CCACHE_DIR, 0) == FALSE) && (GetLastError() != ERROR_ALREADY_EXISTS)) {
	_snprintf( global_msgbuf, sizeof(global_msgbuf), 
			  "Failed to create directory  %s. Error was %s", 
			  CCACHE_DIR, str_oserr(GetLastError()));
	MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
	return FALSE;
  }

  if(vers.dwPlatformId == VER_PLATFORM_WIN32_NT) {
	  /*
	   * Check we are running as Administrator - exit otherwise.
	   */

	is_winnt = TRUE;

	if(GetUserName(username, &usize) == FALSE) {
		_snprintf( global_msgbuf, sizeof(global_msgbuf),"GetUserName failed. Error was %s.",
					str_oserr(GetLastError()));
		MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
		return FALSE;
	  }

	  if(stricmp(username, "Administrator") != 0) {
		  _snprintf( global_msgbuf, sizeof(global_msgbuf), 
			  "You must be running as user Administrator to install this program. You are currently user %s.", 
			username);
		MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
		return FALSE;
	  }

  /* Get the security descriptor for the files and registry keys. */
	if(create_sd_from_list( &sd_reg, 3, 
		"Administrators", GENERIC_ALL, CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE,
		"SYSTEM", GENERIC_ALL, CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE,
		"Users", GENERIC_READ, CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE) == FALSE)
		return FALSE;
	if(create_sd_from_list( &sd_dir, 6, 
		"Administrators", GENERIC_ALL, 0,
		"Administrators", GENERIC_ALL, CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE | INHERIT_ONLY_ACE,
		"SYSTEM", GENERIC_ALL, 0,
		"SYSTEM", GENERIC_ALL, CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE | INHERIT_ONLY_ACE,
		"Users", GENERIC_READ | GENERIC_EXECUTE, 0,
		"Users", GENERIC_READ | GENERIC_EXECUTE, 
			   CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE | INHERIT_ONLY_ACE) == FALSE)
		return FALSE;
	if(create_sd_from_list( &sd_file, 3, 
		"Administrators", GENERIC_ALL, 0,
		"SYSTEM", GENERIC_ALL, 0,
		"Users", GENERIC_READ | GENERIC_EXECUTE, 0) == FALSE)
		return FALSE;
	if(create_sd_from_list( &sd_secure_dir, 4, 
		"Administrators", GENERIC_ALL, 0,
		"Administrators", GENERIC_ALL, CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE | INHERIT_ONLY_ACE,
		"SYSTEM", GENERIC_ALL, 0,
		"SYSTEM", GENERIC_ALL, CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE | INHERIT_ONLY_ACE) == FALSE)
		return FALSE;

	if(create_sd_from_list( &sd_secure_file, 2, 
		"Administrators", GENERIC_ALL, 0,
		"SYSTEM", GENERIC_ALL, 0) == FALSE)
		return FALSE;

	if(create_sd_from_list( &sd_ccache, 8, 
			"Administrators", GENERIC_ALL, 0,
			"Administrators", GENERIC_ALL, CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE | INHERIT_ONLY_ACE,
			"SYSTEM", GENERIC_ALL, 0, 
			"SYSTEM", GENERIC_ALL, CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE | INHERIT_ONLY_ACE, 
			"Creator Owner", GENERIC_ALL, 0,
			"Creator Owner", GENERIC_ALL,CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE | INHERIT_ONLY_ACE,
			"Users", READ_CONTROL|FILE_GENERIC_WRITE|FILE_EXECUTE, 0,
			"Users", READ_CONTROL|FILE_GENERIC_WRITE|FILE_EXECUTE,
			CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE | INHERIT_ONLY_ACE) == FALSE)
		return FALSE;

	if(GetVolumeInformation( 0, 0, 0, 0, &dummy, &fs_flags, 0, 0) == FALSE) {
		_snprintf( global_msgbuf, sizeof(global_msgbuf), 
				  "GetVolumeInformation failed. Error was %s", 
				  str_oserr(GetLastError()));
		MessageBox( global_parent_wind, global_msgbuf, 0,
				  MB_OK| MB_ICONSTOP); 
		return FALSE;
	}
	
	if(fs_flags & FS_PERSISTENT_ACLS) {
		if(set_directory_security( "*", &sd_dir, &sd_file) == FALSE)
			return FALSE;

		if(GetFileAttributes( "LIB\\KRB5KDC") != 0xFFFFFFFF) {
			if(set_directory_security( "LIB\\KRB5KDC\\*", &sd_secure_dir, &sd_secure_file) == FALSE)
				return FALSE;
		}

		if(set_directory_security( CCACHE_DIR, &sd_ccache, 0) == FALSE)
			return FALSE;

	}

  }

  /* Create the registry tree. */
  if(create_registry_tree( HKEY_LOCAL_MACHINE, KERBNET_REG_NAME, 
	                       (is_winnt ? &sd_reg : 0), base_path) == FALSE)
    return FALSE;

  return TRUE;
}


