/*
 *
 * Phantom Cipher is Copyright (C) 1996 Kaz Kylheku
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * The author may be contacted at:
 *
 * Kaz Kylheku
 * 2869 East 14th Avenue
 * Vancouver, B.C.
 * CANADA
 * V5M 2H8
 * email: kaz@cafe.net
 *
 */

#include <stdlib.h>
#include <string.h>
#include "phantom.h"

static void make_internal_cblock(pha_cblock *in, pha_cblock_internal *out);
static void make_external_cblock(pha_cblock *in, pha_cblock_internal *out);
static unsigned long apply_sbox(int box, unsigned long wi);
static void rotate_left(unsigned long *rot);
static void xor_block(pha_cblock *dest, pha_cblock *src);
static void invert_block(pha_cblock *dest, pha_cblock *src);
static void concat_blocks(pha_key *key, pha_cblock *l, pha_cblock *r);

#define ROUNDS	16	/* must be even, between 2 and 32	*/
#define ROT	19	/* should be relatively prime to 128	*/

/*
 * This array determines which pieces of the key are used in which round.
 * In each of the 32 rounds, two pieces, chosen from a set of eight,
 * are used. Each piece is 32 bits (32 x 8 = 256).
 * In the first round, we use piece 0 and 4, in the second round piece
 * 1 and 5 and so forth. This feature of the algorithm is fixed.
 */

int key_order[32][2] =
{
    { 0, 4 },
    { 1, 5 },
    { 2, 6 },
    { 3, 7 },

    { 1, 0 },
    { 3, 2 },
    { 5, 4 },
    { 7, 6 },

    { 0, 4 },
    { 5, 1 },
    { 2, 6 },
    { 7, 3 },

    { 7, 6 },
    { 5, 4 },
    { 3, 2 },
    { 1, 0 },

    { 0, 4 },
    { 3, 7 },
    { 1, 5 },
    { 2, 6 },

    { 1, 3 },
    { 5, 7 },
    { 4, 6 },
    { 0, 2 },

    { 1, 0 },
    { 2, 3 },
    { 5, 4 },
    { 6, 7 },

    { 7, 6 },
    { 5, 4 },
    { 3, 2 },
    { 1, 0 }
};

/*
 * Load the whole set of 16 S-Boxes with custom data.
 * It's up to the user to ensure that this is valid S-Box data.
 * A valid S-Box is a permutation of the integers 0 through 255.
 * There are precisely 256 integers, and no integer appears twice.
 */

void pha_load_sboxes(pha_sbox_group *sbgroup)
{
    memcpy(&sbox, sbgroup, sizeof *sbgroup);
}

/*
 * Replace a single S-Box with a custom version.
 * It's up to the user to provide a valid 256 element S-Box.
 */

void pha_load_sbox(int s, pha_sbox *sbox)
{
    memcpy(&sbox[s], sbox, sizeof *sbox);
}

/*
 * Create a key schedule. This is done by first converting the byte-wise
 * key into a sequence of eight unsigned long integers which hold
 * 32 bit quantities. Then, pieces of the key are copied into a schedule
 * according to the key_order array. The schedule is just a table
 * which gives provides appropriate pair of key pieces for any given round.
 */

void pha_schedule_key(pha_key *key, pha_schedule *sched)
{
    int i, j;

    pha_key_internal kin;
    unsigned long (*piece)[2] = sched->piece;

    for (i = 0, j = 0; i < 8; i++, j += 4) {
	kin[i] = (unsigned long) (*key)[j] << 24
	    | (unsigned long) (*key)[j+1] << 16
	    | (unsigned) (*key)[j+2] << 8
	    | (*key)[j+3];
    }

    for (i = 0; i < ROUNDS; i++) {
	piece[i][0] = kin[key_order[i][0]];
	piece[i][1] = kin[key_order[i][1]];
    }
}

/*
 * Perform an S-Box substitution on each of the four octets stored in
 * the given unsigned long integer. Concatenate the resulting substituted
 * octets into a 32 bit quantity that's returned as an unsigned long
 * The box argument ranges from 0 to 3. It selects a group of 4 S-Boxes
 * out of the total 16 in a straight-forward way. 
 */

static unsigned long apply_sbox(int box, unsigned long wi)
{
    unsigned long wo;

    box *= 4;
    wo  = (unsigned long) sbox[box++][wi & 255];	wi >>= 8;
    wo |= (unsigned long) sbox[box++][wi & 255] << 8;	wi >>= 8;
    wo |= (unsigned long) sbox[box++][wi & 255] << 16;	wi >>= 8;
    wo |= (unsigned long) sbox[box  ][wi & 255] << 24;	wi >>= 8;
    return wo;
}

/*
 * Combine 32 bits of key material with 32 bits of block. This is 
 * done by splitting both into 16 bits, adding 1 to each 16 bit number,
 * and multiplying corresponding pieces modulo 65537 and then
 * subtracting 1 to get back a result 0-65535.
 */

static unsigned long mult_combine(unsigned long block, unsigned long key)
{
    unsigned bl = (block >> 16);
    unsigned br = (block & 65535);
    unsigned kl = (key >> 16);
    unsigned kr = (key & 65535);
    unsigned long rl, rr;

    rl = kl * (unsigned long) bl;
    if (rl) {
	kl = rl & 65535;
	bl = rl >> 16;
	rl = ((kl - bl) + (unsigned) (kl < bl)) & 0xffff;
    } else if (kl) {
	rl = (1 - kl) & 0xffff;
    } else {
	rl = (1 - bl) & 0xffff;
    }
    rl <<= 16;

    rr = kr * (unsigned long) br;
    if (rr) {
	kr = rr & 65535;
	br = rr >> 16;
	rl |= ((kr - br) + (unsigned) (kr < br)) & 0xffff;
    } else if (kr) {
	rl |= (1 - kr) & 0xffff;
    } else {
	rl |= (1 - br) & 0xffff;
    }
    return rl;			
}

/*
 * Treating a two-element array of unsigned long integers as one big
 * 64 bit quantity, perform a left rotate by a prime number of bits.
 * This is used in each encryption round to provide diffusion.
 */

static void rotate_left(unsigned long *rot)
{
    unsigned long carry_in, carry_out;


    carry_out = rot[0] >> (32 - ROT);
    rot[0] = ((rot[0] << ROT) & 0xffffffffUL);
    carry_in = carry_out;

    carry_out = rot[1] >> (32 - ROT);
    rot[1] = ((rot[1] << ROT) & 0xffffffffUL) | carry_in;

    rot[0] |= carry_out;
}

/*
 * Convert an externally represented cipher block (array of 16 unsigned
 * characters storing octets) into an internal representation (array of 
 * 4 unsigned long integers holding 32 bit quantities).
 */

static void make_internal_cblock(pha_cblock *in, pha_cblock_internal *out)
{
    int w, b;

    for (w = 0, b = 0; w < 4; w++, b += 4) {
	(*out)[w] = (unsigned long) (*in)[b] << 24
		| (unsigned long) (*in)[b+1] << 16
		| (unsigned) (*in)[b+2] << 8
		| (unsigned) (*in)[b+3];
    }
}

/*
 * Convert an internal cipherblock representation (4 unsigned long integers
 * holding 32 bit quantities) into the external representation (16 octets
 * stored in an array of unsigned characters).
 */

static void make_external_cblock(pha_cblock *out, pha_cblock_internal *in)
{
    int w, b;
    unsigned long wd;

    for (w = 0, b = 0; w < 4; w++) {
	wd = (*in)[w];
	(*out)[b++] = wd >> 24;
	(*out)[b++] = (wd >> 16) & 0xff;
	(*out)[b++] = (wd >> 8) & 0xff;
	(*out)[b++] = wd & 0xff;
    }
}

/*
 * Perform the Phantom encryption. This is a 32 round Feistel cipher, which
 * works as follows:
 * 
 * Divide the 128 bit cipherblock into two 64 bit halves, L and R
 * (left and right). Form a new cipher block composed of two halves L_new
 * and R_new. Compute the new cipher block:
 *
 *     L_new = R
 *     R_new = L xor leftrotate(sbox(combine(R,K)))
 *
 * This is repeated for 32 rounds, except that in the last round, no swapping
 * is performed:
 *
 *     R_new = R
 *     L_new = L xor leftrotate(sbox(combine(R,K)))
 *
 * The function sbox() takes the 64 bit quantity and applies a S-Box
 * substitution to each of its eight octets. In the even rounds (counting
 * from zero), S-Boxes 0 through 7 are used. In the odd rounds, S-Boxes 8
 * through 15 are used. The lowest numbered S-Box is applied to the least
 * significant octet.
 *
 * The function combine breaks the 64 bit quantity into four 16 bit words,
 * and then multiplies them with corresponding 16 bit words of the key
 * material, modulo 65537. (Zeros are mapped to 65536 prior to the 
 * multiplication, and 65536 is mapped back to zero afterward).
 *
 * The leftrotate() function rotates the 64 bit block left by four bits.
 * 
 * The quantity K is a 64 bit concatenation of two 32 bit pieces of key
 * material extracted according to the key schedule. If the key schedule
 * was produced by the pha_sched_key() function, the choice of key material
 * is consistent with the key_order[] array scheme.
 *
 * The objects referenced by pointers plain and cipher should either
 * not overlap, or overlap exactly.
 */

void pha_encipher(pha_cblock *plain, pha_cblock *cipher, pha_schedule *sched)
{
    pha_cblock_internal a, b;
    unsigned long (*piece)[2] = sched->piece;
    int r;

    make_internal_cblock(plain, &a);

    b[2] = a[0];
    b[3] = a[1];
    for (r = 0; r < ROUNDS; ) {
	a[0] = apply_sbox(0, (mult_combine(b[2], piece[r][0])));
	a[1] = apply_sbox(1, (mult_combine(b[3], piece[r][1])));
	rotate_left(a);
	a[2] = a[2] ^ a[0];
	a[3] = a[3] ^ a[1];
	r++;

	b[0] = apply_sbox(2, (mult_combine(a[2], piece[r][0])));
	b[1] = apply_sbox(3, (mult_combine(a[3], piece[r][1])));
	rotate_left(b);
	b[2] = b[2] ^ b[0];
	b[3] = b[3] ^ b[1];
	r++;
    }
    b[0] = a[2];
    b[1] = a[3];

    make_external_cblock(cipher, &b);
}

/*
 * The Phantom decryption is nearly the same as encryption. The difference
 * is that the order of the schedule is reversed, as is the order (odd versus
 * even) of the S-Boxes. This means that the key pieces that are used in round
 * 31 of encryption are used in round 0 of decryption, those that are used in
 * round 30 of encryption are used in round 1 of decryption and so on. This
 * simple reversal is made possible by the nature of the Feistel network.
 */

void pha_decipher(pha_cblock *cipher, pha_cblock *plain, pha_schedule *sched)
{
    pha_cblock_internal a, b;
    unsigned long (*piece)[2] = sched->piece;
    int r;

    make_internal_cblock(cipher, &a);

    b[2] = a[0];
    b[3] = a[1];
    for (r = ROUNDS-1; r >= 0; ) {
	a[0] = apply_sbox(2, (mult_combine(b[2], piece[r][0])));
	a[1] = apply_sbox(3, (mult_combine(b[3], piece[r][1])));
	rotate_left(a);
	a[2] = a[2] ^ a[0];
	a[3] = a[3] ^ a[1];
	r--;

	b[0] = apply_sbox(0, (mult_combine(a[2], piece[r][0])));
	b[1] = apply_sbox(1, (mult_combine(a[3], piece[r][1])));
	rotate_left(b);
	b[2] = b[2] ^ b[0];
	b[3] = b[3] ^ b[1];
	r--;
    }
    b[0] = a[2];
    b[1] = a[3];

    make_external_cblock(plain, &b);
}

/*
 * Exclusive-or one cipher block into another
 */

static void xor_block(pha_cblock *dest, pha_cblock *src)
{
    int i;
    for (i = 0; i < sizeof *src; i++)
	(*dest)[i] ^= (*src)[i];
}

/*
 * Invert one cipher block into another
 */

static void invert_block(pha_cblock *dest, pha_cblock *src)
{
    int i;
    for (i = 0; i < sizeof *src; i++)
	(*dest)[i] = ~(*src)[i];
}

/*
 * concatenate two cipherblocks to form a key
 */

static void concat_blocks(pha_key *key, pha_cblock *l, pha_cblock *r)
{
    memcpy(*key, *r, sizeof *r);
    memcpy(*key + 16, *l, sizeof *l);
}

/*
 * shift a block one byte to the left, discarding the most significant
 * byte.
 */

static void shift_left_block(pha_cblock *block)
{
    memmove(*block, *block + 1, sizeof *block - 1);
}

/*
 * Perform cipherblock chaining encryption. This is a standard ``mode'' which
 * is intended to mask regularities in the plaintext. Instead of encrypting
 * cipher blocks independently of one another, the cbc mode uses the 
 * ciphertext of the one block to XOR the plaintext of the subsequent block
 * before encrypting it. The first block is XORed with an initial vector.
 * This initial vector can (should) be randomly chosen, and can be prepended
 * to a message so that it can be extracted by the recipient. Alternately,
 * two parties can agree on a secret constant initial vector, or even use
 * a vector of all zeros. The latter methods are less secure because they do
 * not mask message header information, which often contains regularities
 * which make it easy to guess.
 *
 * This function will update the ivec block so that it can serve as an
 * ivec to continue the chaining. In other words, the ivec is set identical
 * to the last cipherblock.
 *
 * It's permissible for plain and cipher to overlap exactly, or to not
 * overlap at all.
 */

void pha_cbc_encipher(pha_cblock *plain, pha_cblock *cipher, pha_cblock *ivec,
	pha_schedule *sched, size_t nbytes)
{
    int nblocks = nbytes / sizeof (pha_cblock);
    int i;
    pha_cblock *iv = ivec;

    if (plain != cipher)
	memcpy(cipher, plain, nbytes);

    for (i = 0; i < nblocks; i++) {
	xor_block(cipher + i, iv);
	pha_encipher(cipher + i, cipher + i, sched);
	iv = cipher + i;
    }

    memcpy(ivec, iv, sizeof *iv);
}

/*
 * Decipher a message that was encrypted using the cipherblock chaining
 * mode. The initial vector must match the one that was used in the
 * encryption. Random initial vectors can be communicated
 * through a plaintext channel.
 * This function modifies the ivec so that it can be used to continue
 * a longer decryption.
 *
 * It's permissible for plain and cipher to overlap exactly, or to not
 * overlap at all.
 */

void pha_cbc_decipher(pha_cblock *cipher, pha_cblock *plain, pha_cblock *ivec,
	pha_schedule *sched, size_t nbytes)
{
    int nblocks = nbytes / sizeof (pha_cblock);
    pha_cblock cipher_save;
    int i;

    if (plain != cipher)
	memcpy(plain, cipher, nbytes);

    for (i = 0; i < nblocks; i++) {
	memcpy(&cipher_save, plain + i, sizeof cipher_save);
	pha_decipher(plain + i, plain + i, sched);
	xor_block(plain + i, ivec);
	memcpy(ivec, &cipher_save, sizeof cipher_save);
    }
}

/*
 * Encrypt a block of bytes using Phantom in cipher feedback mode.
 * The ivec argument is essentially the shift register. It is modified
 * by the function, so that it can be used as an argument to successive
 * calls to the function to continue the feedback stream.
 * The plain and cipher pointers should be identical, or else should
 * refer to non-overlapping objects.
 */

void pha_cfb_encipher(unsigned char *plain, unsigned char *cipher,
	pha_cblock *ivec, pha_schedule *sched, size_t nbytes)
{
    pha_cblock enciph;

    while (nbytes--) {
	pha_encipher(ivec, &enciph, sched);
	shift_left_block(ivec);
	*cipher++ = (*ivec)[0] = enciph[0] ^ *plain++;
    }
}

/*
 * Decrypt a block of bytes using Phantom in cipher feedback mode.
 * The ivec argument is the initial shift register. It is modified
 * by the function, so that it can chain together multiple calls
 * of the function to continue the feedback stream.
 * The plain and cipher pointers should be identical, or else should
 * refer to non-overlapping objects.
 */

void pha_cfb_decipher(unsigned char *cipher, unsigned char *plain,
	pha_cblock *ivec, pha_schedule *sched, size_t nbytes)
{
    pha_cblock enciph;

    while (nbytes--) {
	pha_encipher(ivec, &enciph, sched);
	shift_left_block(ivec);
	*plain++ = enciph[0] ^ ((*ivec)[0] = *cipher++);
    }
}

/*
 * Generate a key from a pass phase. This can also be used as a general-purpose
 * one-way hashing function.
 * The algorithm used is a pair of Davis-Meyer functions in conjunction
 * with the Phantom block cipher. The result is a hash that is twice
 * the block size.
 * Returns 1 on success, 0 if the given text is too long.
 */

int pha_phrase_to_key(char *text, size_t len, pha_key *key)
{
    pha_cblock g = {
    	0xa9, 0x4d, 0x5f, 0x03, 0x29, 0x7e, 0x3d, 0x76,
	0x54, 0xc0, 0x92, 0x1d, 0xbd, 0x38, 0xf2, 0xa0
    };
    pha_cblock h = {
	0x8d, 0x26, 0xd8, 0xc7, 0x13, 0x7d, 0x04, 0x51,
	0x93, 0x3b, 0xc9, 0x31, 0x30, 0x59, 0xbd, 0x35
    };
    pha_cblock temp, msg[32] = { { 0 } };
    pha_key kg, kh;
    pha_schedule sg, sh;

    int i, nblocks = (len + sizeof(pha_cblock) - 1) / sizeof(pha_cblock);

    if (nblocks > 32)
	return 0;

    memcpy(msg, text, len);

    for (i = 0; i < nblocks; i++) {
	concat_blocks(&kg, &g, msg + i);
	concat_blocks(&kh, msg + i, &h);
	pha_schedule_key(&kg, &sg);
	pha_schedule_key(&kh, &sh);
	invert_block(&temp, &g);
	pha_encipher(&temp, &temp, &sh);
	xor_block(&g, &temp);
	pha_encipher(&h, &temp, &sg);
	xor_block(&h, &temp);
    }

    concat_blocks(key, &g, &h);

    return 1;
}

/*
 * Generate a single cipher block from a pass phrase. This can serve
 * as an initial vector for instance.
 * The algorithm uses a modified Davis-Meyer algorithm in conjunction
 * with the Phantom cipher.
 * Returns 1 on success, 0 if the given text is too long.
 */

int pha_phrase_to_cblock(char *text, size_t len, pha_cblock *vec)
{
    pha_cblock h = {
    	0xa9, 0x4d, 0x5f, 0x03, 0x29, 0x7e, 0x3d, 0x76,
	0x54, 0xc0, 0x92, 0x1d, 0xbd, 0x38, 0xf2, 0xa0
    };
    pha_cblock msg[32] = { { 0 } };
    pha_key key;
    pha_schedule sched;

    int i, nblocks = (len + sizeof(pha_cblock) - 1) / sizeof(pha_cblock);

    if (nblocks > 32)
	return 0;

    memcpy(msg, text, len);

    for (i = 0; i < nblocks; i++) {
	concat_blocks(&key, &h, msg + i);
	pha_schedule_key(&key, &sched);
	pha_encipher(&h, &h, &sched);
    }

    memcpy(vec, &h, sizeof h);

    return 1;
}

/*
 * These are the default S-Boxes. These were chosen using a
 * linear-congruential random generator that was seeded with a large
 * vector derived from random keystrokes.  The boxes are not contrived
 * in any way to make linear or differential cryptanalysis difficult.
 * They are probably weak. On the other hand they are not part of the
 * algorithm. Phantom S-Boxes are left to the implementation, and
 * can even be tailored fo individual communication channels, yielding
 * a total of 33024 bits of secret information! (256 + 16 x 256 x 8).
 */

pha_sbox_group sbox = {
    {	/* sbox 0 */
	0xd6,0xca,0x12,0xe7,0xcd,0x89,0x7a,0xbc,
	0xad,0x79,0xa4,0x2c,0xc3,0x50,0x5e,0xb3,
	0x4b,0x11,0xe0,0x3d,0x28,0xae,0x01,0x55,
	0x59,0x45,0x35,0xb9,0x03,0x26,0x46,0x74,
	0xbd,0xf6,0x73,0xb0,0x5a,0x66,0xbe,0x18,
	0x78,0xaa,0xf9,0x76,0xc8,0x7f,0x70,0xa9,
	0x9d,0xab,0x75,0x02,0x71,0xa5,0x20,0xf3,
	0x0a,0x97,0x8c,0x3a,0x16,0x0f,0xcf,0x9b,
	0x83,0x5d,0x7c,0x86,0xda,0xb1,0xc6,0x92,
	0xd8,0x99,0xce,0x1c,0x32,0xfb,0x69,0x72,
	0x23,0x4a,0xbb,0xee,0xcb,0x93,0xfd,0xac,
	0x44,0xa3,0x8a,0x0d,0x53,0x2b,0x54,0x3e,
	0x09,0x77,0x64,0xd5,0x30,0x51,0x14,0x61,
	0xa0,0xe6,0x8b,0x5f,0x7d,0x34,0x1b,0x57,
	0xe3,0x5b,0xec,0xfe,0xa2,0xd9,0xd1,0x25,
	0x39,0x36,0x0b,0xef,0xa7,0x6f,0x9a,0xa8,
	0xde,0x00,0xb8,0xdf,0x4f,0xed,0x10,0x8e,
	0x21,0x2f,0xd4,0x84,0x04,0x6c,0xdc,0x94,
	0xf7,0xea,0x40,0x65,0xb5,0x07,0xa6,0xc7,
	0xf2,0x38,0x9f,0x6b,0x87,0x3c,0xc2,0xf5,
	0x60,0xd2,0x43,0xfa,0x4e,0x8f,0x2a,0xfc,
	0xdb,0xb4,0xc1,0x48,0x47,0xf8,0x1e,0x63,
	0xdd,0xb7,0x80,0xf1,0x6e,0xbf,0xe8,0xb6,
	0x91,0x1a,0x27,0x85,0x19,0x1f,0x15,0xe1,
	0x3b,0x56,0x41,0xff,0x31,0xcc,0x7b,0x3f,
	0x82,0x2e,0x1d,0x52,0x0c,0x7e,0x9e,0x5c,
	0xd3,0x6d,0x96,0x17,0x22,0x13,0xe9,0x4c,
	0x58,0x05,0x08,0x8d,0x0e,0x49,0xe4,0x24,
	0x9c,0x68,0x81,0xf4,0x6a,0xb2,0x98,0xc4,
	0x95,0x88,0xe2,0x42,0x4d,0x29,0xe5,0xeb,
	0x67,0x62,0x37,0xaf,0xc9,0xd7,0xba,0x90,
	0x2d,0x06,0xc5,0xd0,0xa1,0x33,0xf0,0xc0
    },
    {	/* sbox 1 */
	0x77,0xc5,0x2d,0x2c,0x69,0x19,0xfd,0x5d,
	0x83,0x81,0x94,0x5c,0x92,0x5b,0x03,0x41,
	0xc4,0x34,0x66,0x0e,0x89,0x32,0x4a,0x9a,
	0xe3,0xdc,0xc3,0xef,0x8f,0xb3,0xa5,0xc0,
	0x49,0x62,0xf6,0xfc,0x4b,0x85,0x71,0xa8,
	0x2f,0x67,0x9b,0x60,0xd0,0x42,0xe2,0x54,
	0x44,0x3f,0x98,0xf7,0x3c,0x07,0xc2,0xc8,
	0x23,0x2b,0xeb,0xc9,0x7e,0x35,0x65,0xb1,
	0x0b,0x21,0x68,0xdd,0xe0,0x51,0x20,0xf2,
	0xf3,0xd6,0x0a,0x10,0x9f,0x1f,0xcb,0x9c,
	0x64,0x52,0xec,0x4e,0x43,0x04,0x22,0x6c,
	0x7b,0xa0,0xd5,0x38,0xe4,0x8d,0x59,0xae,
	0x0d,0xbf,0xf9,0x48,0x7a,0xa2,0x57,0x87,
	0x1a,0x7d,0x06,0x5a,0xb4,0x97,0xba,0x4f,
	0x95,0x7c,0x09,0xce,0xa1,0xe7,0xe9,0xf5,
	0x53,0x8b,0x8e,0x02,0xcc,0x5e,0xee,0x36,
	0xd1,0x80,0xca,0xdb,0x3b,0x29,0x99,0x93,
	0xa3,0x8c,0x40,0xdf,0x13,0x91,0x90,0x1b,
	0x7f,0xd8,0xff,0x76,0x27,0x16,0x4c,0xe1,
	0x61,0xaf,0x33,0xf0,0xc6,0x55,0xb5,0xe8,
	0x75,0xa7,0x17,0x3a,0x96,0xab,0xed,0x6e,
	0x74,0x58,0xa4,0x78,0x31,0x82,0x00,0x45,
	0x01,0x25,0x86,0x08,0x84,0x37,0x30,0xd2,
	0x15,0x1c,0xaa,0xf8,0x28,0x2a,0x63,0x47,
	0xe5,0xd4,0x9e,0xad,0xb0,0xde,0xd7,0xcd,
	0x72,0x6b,0xb9,0xbc,0x05,0x6f,0xfe,0xb7,
	0xfa,0x70,0xa6,0x26,0x4d,0x12,0x14,0x11,
	0x3d,0x9d,0xd9,0xf4,0x1e,0x24,0xac,0xbe,
	0xf1,0xbb,0x5f,0xb8,0xa9,0xd3,0x88,0xc7,
	0xcf,0x50,0x56,0x0f,0x8a,0xda,0x3e,0x18,
	0x6d,0xb6,0x39,0x46,0x6a,0x2e,0xbd,0xe6,
	0x73,0xfb,0x79,0x0c,0xb2,0xc1,0x1d,0xea,
    },
    {	/* sbox 2 */
	0xda,0x55,0x62,0x1e,0x91,0x80,0xab,0x01,
	0x95,0x12,0xb6,0x35,0x07,0x5a,0xc1,0xaf,
	0x19,0xcb,0x8f,0x8a,0xfd,0xcd,0xdb,0x36,
	0xd7,0xb5,0x47,0x87,0x8d,0xaa,0xe3,0x85,
	0x89,0xc2,0x57,0xff,0x52,0xf1,0xf3,0x2c,
	0xc9,0x31,0x86,0x6d,0x4d,0x97,0x1b,0xe2,
	0x8b,0xdc,0xc8,0x16,0x53,0x7c,0x0b,0xd0,
	0x4b,0xcc,0x7f,0x61,0x11,0x20,0xf0,0x45,
	0xb8,0x17,0x66,0x0e,0xd9,0xed,0x2a,0x08,
	0x4c,0x02,0x73,0x7b,0x3f,0x26,0xa7,0xf8,
	0xc5,0x7a,0xad,0xbc,0x67,0x18,0x51,0xea,
	0xa6,0xd3,0xa1,0xa2,0x9a,0x4f,0x59,0xa8,
	0xfb,0x6a,0xb2,0x42,0xd1,0x1f,0x88,0xd4,
	0x28,0x2d,0x76,0x9b,0x23,0x5e,0x82,0x34,
	0x79,0xca,0x64,0xf4,0x30,0xf7,0x13,0x81,
	0x9e,0x39,0x06,0xe1,0x84,0x29,0x1a,0x83,
	0xe9,0x46,0xa9,0xc4,0xd5,0x3a,0x24,0x68,
	0xd8,0x41,0xe8,0xb1,0xb4,0x2f,0x65,0x7d,
	0x63,0x94,0xb3,0xde,0xb7,0x96,0x49,0x5d,
	0x5f,0x32,0x60,0x25,0xef,0xcf,0xb9,0x74,
	0xe5,0x78,0xba,0x92,0xac,0xc0,0x8c,0xbb,
	0x5c,0x6c,0x6b,0x09,0x9f,0xe4,0xee,0xf6,
	0x69,0xdd,0x37,0xf9,0x72,0x90,0x1c,0xa0,
	0x6f,0x4e,0x50,0xce,0x99,0x04,0x56,0xfa,
	0xae,0x0f,0x9d,0xe0,0x44,0x4a,0xec,0x2b,
	0xfc,0xbd,0x48,0xeb,0x3c,0x10,0xa5,0x5b,
	0x58,0x75,0x3d,0x38,0x0a,0xfe,0x05,0xf2,
	0xbf,0x22,0x93,0x33,0x9c,0x3e,0xf5,0xd2,
	0x40,0x0c,0x00,0xe6,0x7e,0x14,0x15,0xdf,
	0xc7,0xbe,0x27,0xa4,0x03,0xe7,0x21,0xa3,
	0x1d,0x71,0x43,0x2e,0xc6,0x77,0xc3,0x6e,
	0x54,0x0d,0xd6,0x3b,0x98,0x8e,0xb0,0x70,
    },
    {	/* sbox 3 */
	0x61,0xf3,0xa0,0x25,0xf2,0xfb,0x85,0xbd,
	0xd0,0x41,0xe9,0xc4,0x4f,0xe1,0x94,0x44,
	0x67,0x6d,0xec,0xa8,0x28,0xd8,0xd1,0x1e,
	0x83,0x7a,0x66,0xe5,0xb4,0xa6,0x10,0x56,
	0x62,0x04,0x5a,0x5c,0x00,0xa1,0x98,0x71,
	0x0e,0x8c,0x97,0x95,0xc1,0xd4,0x6c,0x53,
	0x4c,0x48,0xf4,0x0a,0xed,0x3e,0x3a,0x1f,
	0x1b,0x29,0x49,0xf6,0x75,0x9a,0x6a,0x6e,
	0xc9,0x14,0x86,0x2d,0xfc,0xd6,0x78,0xe8,
	0x09,0x0c,0x26,0xef,0x92,0x8a,0xbe,0x21,
	0x33,0xd3,0x20,0xff,0x34,0x6b,0x76,0xeb,
	0x8b,0xf0,0xc6,0x08,0x2b,0xe2,0x3c,0x96,
	0xdb,0xbb,0xae,0xf5,0xd2,0x52,0x32,0xce,
	0x03,0x11,0xc7,0x58,0xee,0x4e,0x89,0x39,
	0x15,0xbf,0xe7,0x38,0xf7,0xe4,0x06,0xfd,
	0x68,0x63,0xc3,0x55,0x5e,0x50,0x23,0xcc,
	0xad,0x7b,0x74,0x70,0x8d,0x47,0x2f,0x59,
	0xe0,0xdf,0x6f,0xb1,0x8f,0x05,0x31,0x3f,
	0x4b,0xd5,0x5d,0x45,0xa3,0xb2,0x87,0x22,
	0x0b,0xc5,0x2c,0xe3,0x57,0x8e,0x07,0x80,
	0x3d,0xba,0x81,0x2a,0x35,0xaf,0x9b,0xdc,
	0x37,0xaa,0x69,0xca,0x9f,0x99,0x1d,0xa5,
	0x30,0x51,0xc0,0xac,0x82,0x60,0x0d,0x77,
	0x1c,0xf1,0xd7,0x18,0x3b,0x84,0x5b,0x65,
	0xb0,0xb7,0x0f,0xc2,0xb3,0x9d,0x91,0x5f,
	0x16,0xcb,0x7f,0xb6,0xd9,0x7c,0xea,0x02,
	0xbc,0x01,0xa9,0xf8,0x17,0x27,0x42,0xfa,
	0x2e,0x88,0x24,0x93,0xab,0xfe,0xc8,0xa4,
	0x43,0xf9,0xb5,0xcf,0x40,0xdd,0x54,0x12,
	0x4d,0x79,0x9e,0x72,0x4a,0xde,0x73,0xda,
	0x46,0x64,0x19,0x90,0xb8,0xa2,0x7e,0x9c,
	0x36,0x7d,0xcd,0xb9,0xe6,0x13,0x1a,0xa7,
    },
    {	/* sbox 4 */
	0xe7,0xd2,0x52,0xe8,0x09,0x27,0xe1,0x72,
	0xfc,0x4d,0x53,0xa7,0xa6,0xf8,0x74,0xac,
	0xb7,0x40,0x76,0x25,0x7e,0x34,0xb8,0xc7,
	0x05,0xb3,0xd4,0x3c,0x49,0x59,0x79,0x86,
	0x2b,0x54,0x6f,0x67,0x81,0xa8,0xc1,0x6b,
	0x14,0xfb,0x5c,0x77,0x20,0x0c,0x75,0xfd,
	0x1b,0x51,0xed,0x4f,0xdb,0x38,0x85,0x43,
	0xfe,0xe6,0x91,0x8d,0x0f,0xcb,0xd0,0x6e,
	0x12,0x1e,0x9c,0xe0,0xc3,0xd1,0xff,0xce,
	0x5f,0x46,0x89,0xf3,0xca,0x71,0x68,0xcd,
	0x92,0x2d,0xd3,0xc5,0xf6,0x42,0xee,0x19,
	0x29,0xef,0x2f,0xc9,0x4e,0x50,0xdc,0x04,
	0x8f,0xea,0x66,0x5b,0x7a,0xd8,0x3a,0x55,
	0xa9,0x3b,0x11,0x2c,0xdf,0x1f,0xeb,0x1a,
	0x01,0xba,0x61,0x8a,0x39,0x0d,0x62,0x94,
	0x90,0x9b,0x3e,0x82,0x47,0xb5,0x07,0x36,
	0x97,0x6c,0xf2,0x93,0xcf,0x44,0x8c,0x41,
	0xbc,0x3f,0xbf,0x5a,0xa5,0xbb,0x6d,0x10,
	0xf7,0xaf,0x0e,0xd7,0x4c,0xb0,0x9f,0x63,
	0xb1,0x5d,0x8b,0x30,0xe5,0x1c,0xc8,0x03,
	0xb4,0x00,0x08,0x0b,0xf5,0x28,0xe2,0x33,
	0xb6,0x3d,0x02,0x88,0x21,0xa3,0x83,0xdd,
	0x8e,0x0a,0x22,0xa2,0x58,0x6a,0x7c,0x48,
	0xae,0xbd,0x26,0x7f,0xc6,0xec,0xc4,0xbe,
	0x4a,0xc2,0x24,0x57,0xf1,0x17,0xc0,0x4b,
	0x35,0xe4,0x84,0x16,0x69,0xad,0x73,0x96,
	0x5e,0x98,0x78,0x70,0x9e,0x1d,0x65,0x15,
	0xe9,0xb2,0xe3,0xf4,0x99,0x7b,0x18,0xd5,
	0x37,0x7d,0x9a,0xd9,0x9d,0xab,0xcc,0xa4,
	0x23,0xf0,0xde,0xfa,0x2e,0x45,0xa0,0xd6,
	0x60,0x56,0x95,0xb9,0x06,0xa1,0xaa,0xda,
	0x87,0x31,0x80,0x64,0x32,0xf9,0x13,0x2a,
    },
    {	/* sbox 5 */
	0x87,0x56,0x5b,0xf8,0x71,0x1e,0xc2,0xd5,
	0xf5,0xe2,0x10,0xf2,0xd1,0xfa,0xec,0xc3,
	0xe8,0xab,0x74,0x7f,0x8e,0x06,0x92,0xd7,
	0xa5,0x0c,0xa4,0x2c,0xb7,0x8f,0xfc,0x34,
	0xac,0xfd,0xb2,0xf7,0x14,0x5f,0x80,0xad,
	0xd6,0x0a,0x93,0x08,0x82,0xed,0x21,0x55,
	0x94,0xdc,0xb9,0xf1,0x0d,0x86,0x7d,0x8b,
	0xe3,0xe7,0x97,0x5a,0x83,0x44,0xd9,0xdf,
	0x61,0x03,0x30,0x4b,0xcd,0x57,0x1c,0xa3,
	0x33,0x53,0x9c,0x11,0x09,0x3a,0x29,0xb3,
	0x1b,0x1f,0xcc,0xe1,0xa2,0xaf,0xb6,0x13,
	0x7a,0xc5,0x1a,0x0b,0x20,0x6f,0x43,0x01,
	0x02,0x58,0x59,0x52,0x8d,0xf3,0xd3,0x3e,
	0xdd,0x5e,0x17,0x4f,0xba,0xbb,0x2d,0x4d,
	0xf9,0x66,0xb4,0x6a,0xeb,0x28,0x47,0x76,
	0xea,0xbf,0x88,0xe5,0xc7,0xa7,0x9e,0xc4,
	0xb0,0x2f,0x39,0xef,0x7c,0x69,0xa6,0xa9,
	0xda,0xe9,0xbe,0xcb,0x05,0x12,0x0e,0x4c,
	0x75,0x3b,0x62,0x6d,0xde,0xb1,0x00,0x32,
	0x5d,0x40,0xd0,0x45,0xb8,0xf6,0xe0,0x4a,
	0x50,0xce,0x3d,0xc9,0xbc,0x70,0xe4,0x6e,
	0xb5,0x79,0x9f,0x16,0xee,0x3c,0xfe,0x68,
	0xaa,0x91,0xae,0xc0,0x3f,0x38,0x54,0x07,
	0x49,0x2e,0xe6,0x98,0x8c,0x25,0x36,0x89,
	0x95,0xd4,0x1d,0x2b,0x63,0x6c,0x19,0x78,
	0x42,0xff,0x8a,0x72,0x24,0xbd,0x5c,0x77,
	0x27,0xa1,0x22,0xfb,0xd2,0xa8,0x31,0x35,
	0x9b,0x41,0x60,0x2a,0x9a,0x15,0x4e,0x84,
	0x7b,0x23,0x46,0x48,0x67,0x04,0xd8,0x26,
	0xf0,0x65,0xdb,0x9d,0xa0,0x99,0x7e,0x90,
	0x96,0x0f,0xf4,0x37,0x81,0x64,0xc1,0xc6,
	0x6b,0xcf,0x73,0x51,0xc8,0x18,0xca,0x85,
    },
    {	/* sbox 6 */
	0x0b,0x71,0xfd,0xdf,0x9b,0x97,0xcb,0xc5,
	0x9e,0x15,0xfb,0x76,0x8c,0x05,0x2b,0x04,
	0x1a,0xe1,0xd9,0x69,0x70,0x34,0x91,0x92,
	0xf0,0xb1,0xee,0xb3,0xea,0x25,0xe4,0xe9,
	0xc1,0xfe,0xf8,0x01,0x68,0x20,0xa8,0xf7,
	0x8a,0x14,0x29,0x35,0x67,0xba,0x66,0x6b,
	0xc2,0x90,0x32,0x0a,0x57,0x95,0xc3,0x1c,
	0x64,0x61,0xe0,0xcf,0xe6,0xb6,0xca,0xad,
	0xf3,0x86,0x1b,0xc9,0x5e,0x43,0x24,0xac,
	0x77,0xa2,0xb4,0x1e,0x54,0xa5,0xe8,0x4d,
	0xce,0x0e,0x49,0x82,0xf9,0xe2,0x2a,0x19,
	0xbc,0x31,0x11,0xa3,0xab,0x58,0xd4,0x99,
	0x80,0x3d,0xb2,0x41,0x16,0xeb,0xbb,0x6d,
	0x98,0xef,0x4a,0x50,0xfa,0xf4,0x6e,0x73,
	0x17,0xfc,0x12,0xdc,0x87,0x7d,0x5f,0xdb,
	0x0f,0xa4,0xc0,0x52,0xbd,0x3f,0xd3,0x1f,
	0xbe,0x96,0xbf,0x46,0xda,0x88,0x37,0x5b,
	0xc8,0x60,0x13,0xf6,0x6a,0xcc,0xaa,0x22,
	0x23,0x47,0x79,0x27,0xe7,0xc6,0x72,0x2f,
	0x09,0x5a,0x9c,0x02,0xdd,0x7f,0x8f,0x08,
	0x59,0x40,0x0c,0x2d,0xed,0xe5,0xb5,0x30,
	0xd6,0x94,0x4f,0x7e,0xec,0x5d,0xa1,0x78,
	0xd2,0xd5,0x53,0x75,0x07,0x9a,0x62,0x10,
	0xf2,0x4e,0x6f,0xd0,0x8b,0x39,0x45,0x28,
	0xc4,0x74,0x3e,0x42,0x84,0x38,0x8e,0x4c,
	0x51,0xb7,0xa6,0x21,0x7b,0x2c,0xa0,0x63,
	0xff,0x3a,0xcd,0x83,0x89,0xb0,0x85,0xf5,
	0x7a,0x7c,0x48,0xa9,0x44,0xf1,0x3c,0x9d,
	0xd7,0x8d,0xd1,0xae,0xde,0x93,0xaf,0x9f,
	0x2e,0xb8,0x03,0x56,0x18,0x65,0x5c,0xc7,
	0x06,0x4b,0x0d,0xe3,0x1d,0x00,0x3b,0x36,
	0x55,0x26,0x33,0xa7,0x6c,0xd8,0x81,0xb9,
    },
    {	/* sbox 7 */
	0x9c,0x12,0xcc,0x01,0xb0,0x1f,0x1c,0xfa,
	0x8b,0xb8,0xff,0x50,0xdd,0xe3,0x8f,0x97,
	0x5a,0x56,0x27,0x77,0xc0,0x9d,0xf5,0x85,
	0x33,0x9e,0x71,0x4c,0x0a,0xf4,0x61,0x23,
	0x04,0xc9,0x1d,0x40,0xec,0xc6,0xb5,0xbb,
	0xcf,0x45,0x35,0x2c,0xf7,0x0e,0xaa,0x76,
	0xb7,0x67,0xee,0xfe,0x6c,0x48,0x3a,0xf2,
	0xa8,0xf8,0x4b,0xf1,0x0d,0x53,0x54,0x51,
	0x7e,0x03,0x10,0x9a,0x13,0x70,0xb1,0xd7,
	0xba,0x86,0x7a,0x08,0xac,0xaf,0x2b,0x57,
	0x78,0x55,0x9b,0xf3,0xdb,0xa0,0x93,0x94,
	0x83,0x88,0xb9,0xef,0x3e,0x99,0x2f,0xe8,
	0xa5,0xe2,0xe1,0x0f,0x8a,0x29,0x9f,0x59,
	0x5c,0x02,0x47,0xce,0x46,0x3d,0x24,0x0b,
	0x38,0xc4,0x39,0x91,0x8e,0x19,0x73,0x43,
	0x58,0x31,0x4f,0x5e,0x1a,0x66,0x6d,0x42,
	0xbd,0xe9,0x5b,0x2a,0xd5,0xea,0xa2,0x82,
	0x7c,0x49,0x96,0x22,0xc1,0xe7,0x69,0xc2,
	0x20,0xb3,0x11,0x7b,0x37,0xab,0xad,0x64,
	0x92,0x4a,0xd4,0x60,0xda,0x98,0x14,0xe5,
	0xc3,0x81,0xe4,0x18,0xc5,0xf9,0xb4,0x84,
	0x72,0xa7,0x3f,0xde,0x16,0xa6,0xbc,0x2d,
	0xb2,0xdf,0xfc,0xbf,0xbe,0x5d,0x34,0x8c,
	0xcb,0xd9,0xae,0xc8,0xed,0x75,0x89,0xd8,
	0xeb,0xc7,0x3c,0x09,0xd2,0x32,0x63,0x1e,
	0xa4,0xd6,0xe6,0x79,0x7d,0xd3,0xa9,0x68,
	0xd0,0x62,0x0c,0x44,0xfb,0x41,0xe0,0x25,
	0x21,0x6e,0x06,0x15,0xf6,0xca,0x90,0xfd,
	0x8d,0xdc,0x36,0x80,0x4e,0x1b,0x6b,0x5f,
	0x3b,0x28,0xb6,0xf0,0x17,0xa3,0xcd,0xd1,
	0x87,0xa1,0x6f,0x30,0x2e,0x26,0x07,0x4d,
	0x65,0x95,0x52,0x00,0x74,0x7f,0x6a,0x05,
    },
    {	/* sbox 8 */
	0x72,0x7c,0x4c,0x24,0xd2,0x6f,0x1c,0x6b,
	0x28,0xd3,0xe9,0x34,0x19,0xbe,0x9f,0xf2,
	0xfe,0x31,0x8f,0xc5,0x51,0x97,0x12,0xa2,
	0x29,0xed,0xce,0x76,0x65,0xef,0x67,0xff,
	0x03,0x46,0x35,0x17,0x2e,0xa4,0x3b,0x4f,
	0x5d,0x0d,0xee,0x57,0xe1,0x4d,0x39,0x8d,
	0x77,0x00,0x6a,0x69,0xf7,0x70,0x50,0x2c,
	0xb3,0xbd,0xa0,0x45,0xf1,0x78,0xf0,0x40,
	0xab,0x92,0x14,0xc0,0xe6,0x53,0x52,0xf5,
	0x85,0x23,0x68,0xc9,0x60,0xfc,0xb2,0x48,
	0x42,0xa9,0x71,0x2a,0x90,0x22,0x07,0x0e,
	0x99,0x16,0xcd,0x01,0xb6,0x9d,0x83,0xec,
	0x87,0x59,0xc4,0xf9,0xde,0xd5,0x56,0xc3,
	0x06,0x8e,0xda,0xad,0x33,0xea,0xb7,0xd4,
	0x02,0x41,0x8b,0xae,0xb8,0x94,0xd9,0x81,
	0x74,0x91,0xe3,0x11,0x1a,0x26,0xd1,0xe7,
	0x9b,0x82,0x1d,0xa5,0x2b,0x4b,0x3f,0x49,
	0x1e,0x30,0x27,0x80,0x63,0x05,0x9c,0x55,
	0xc6,0xb1,0x7b,0xe4,0x96,0x32,0xeb,0xdf,
	0x8c,0x73,0x7d,0x0a,0x79,0x10,0x20,0x15,
	0x44,0xdb,0xcc,0x1f,0xe0,0xba,0xb0,0xa6,
	0x1b,0x93,0x88,0xd6,0xe5,0x62,0xe2,0x3d,
	0x04,0x08,0x7f,0xcf,0x18,0x86,0xb5,0x5f,
	0x0b,0x6e,0x95,0xbc,0xb9,0x75,0x4e,0x8a,
	0xc7,0x5a,0x9a,0x98,0xa7,0x25,0x2f,0x09,
	0xd8,0xdd,0x43,0x5b,0xaa,0xb4,0xf4,0x0c,
	0x6d,0xdc,0x64,0x2d,0xc2,0xc8,0x5c,0xe8,
	0xfb,0x7a,0x0f,0x36,0xcb,0xf8,0xac,0xaf,
	0x58,0x38,0x37,0xf3,0x6c,0xd0,0xa3,0x4a,
	0x3c,0xfa,0xf6,0xa1,0x84,0xbb,0x47,0x5e,
	0x61,0x66,0x21,0x3e,0x7e,0xc1,0x9e,0x54,
	0xd7,0x13,0xa8,0xca,0xfd,0xbf,0x3a,0x89,
    },
    {	/* sbox 9 */
	0xb7,0x7f,0xd0,0x1b,0x05,0x42,0xb0,0x33,
	0xff,0x32,0x6d,0x18,0x19,0xfd,0x0a,0xb5,
	0x51,0xc8,0x16,0x35,0x34,0x57,0xea,0x3f,
	0x0f,0xa9,0x27,0x38,0x0b,0xc7,0x94,0xe1,
	0x21,0xcc,0x00,0xe0,0x4e,0x90,0x77,0x80,
	0xed,0x93,0x3e,0x8e,0x3d,0x1a,0x99,0x43,
	0xd5,0xfe,0xef,0x83,0x0e,0x9b,0xb6,0x98,
	0xf9,0x72,0x65,0xa8,0x2b,0x59,0x28,0xb9,
	0x56,0x14,0x3c,0x17,0x88,0xe8,0x25,0x7b,
	0x53,0x23,0x5a,0x50,0x9c,0x09,0x1f,0xd9,
	0x95,0x4d,0x6c,0x10,0xd2,0x45,0x7c,0x44,
	0x2d,0xf8,0xb3,0xf6,0xf2,0x02,0xfa,0x97,
	0x8a,0x61,0xe5,0x9f,0x55,0x48,0x37,0xa5,
	0xbe,0x24,0x22,0x85,0x60,0x8b,0x63,0xad,
	0xca,0x4b,0x7e,0x03,0xbf,0xc9,0x5d,0xaf,
	0x74,0x78,0x84,0xbd,0x69,0x91,0x96,0xdf,
	0xdb,0xa2,0x11,0x89,0x7a,0xb1,0x87,0x76,
	0x9a,0x04,0xf0,0xf5,0x0c,0xe9,0xce,0xc6,
	0xcf,0xab,0xac,0x49,0x15,0x6a,0x8c,0xb2,
	0x4a,0xa0,0xe3,0xe6,0x70,0x36,0x39,0x66,
	0xd8,0x58,0xae,0xbc,0x4c,0x6f,0x31,0xc4,
	0xee,0xc0,0xd4,0xdd,0x54,0x47,0xc5,0xf3,
	0x71,0xaa,0x3b,0xd6,0x81,0xf1,0xd3,0xfb,
	0x5b,0x12,0x29,0xda,0x2a,0xd1,0x3a,0x1c,
	0x1e,0xdc,0x26,0xe2,0xc3,0xec,0x6b,0x08,
	0xa4,0x2e,0xa3,0xe7,0x5e,0x68,0x75,0xcd,
	0xa1,0x62,0x0d,0x8d,0xf7,0xcb,0x07,0xf4,
	0x4f,0xfc,0x9e,0xbb,0x6e,0xa6,0x1d,0x67,
	0x8f,0x86,0xe4,0x46,0x82,0xeb,0xc1,0x01,
	0x64,0x20,0x40,0xb4,0x7d,0x79,0xde,0x5f,
	0x41,0x9d,0x06,0x2f,0xd7,0xc2,0xa7,0x73,
	0xb8,0x5c,0x30,0x52,0x92,0x13,0x2c,0xba,
    },
    {	/* sbox 10 */
	0xaa,0xa5,0x81,0x88,0x61,0x7e,0xd7,0x47,
	0x23,0xbc,0x77,0xab,0x39,0x31,0x0a,0x3e,
	0x54,0x29,0x82,0xe2,0xfb,0x5c,0xc2,0xb9,
	0x03,0xe5,0x2b,0x95,0x20,0x22,0xad,0x70,
	0x9c,0x53,0x09,0x64,0x5e,0x43,0x3f,0xdc,
	0xbf,0x9d,0x9b,0x8b,0xec,0x4f,0x56,0xa7,
	0x32,0x69,0x67,0x87,0xbd,0x28,0x8e,0x58,
	0x91,0x21,0x6b,0x1c,0x2c,0xc1,0xf3,0x75,
	0x8a,0x83,0x78,0xbe,0xb3,0x5d,0x2e,0x96,
	0xeb,0x7c,0x71,0x93,0x85,0xae,0x33,0x3d,
	0xe9,0x79,0xd3,0xfd,0xea,0xcc,0xb7,0xd9,
	0x68,0x12,0x45,0x41,0x6e,0x44,0xf0,0xe1,
	0xf6,0xba,0x1f,0xf8,0xe6,0xa0,0xe0,0xfe,
	0xcb,0xc9,0xd8,0xc0,0x2f,0xa9,0x01,0x49,
	0xcf,0x11,0xd4,0xb6,0xee,0x52,0xde,0x6c,
	0x9f,0xca,0x1d,0x1b,0x90,0x1a,0x76,0x6f,
	0xcd,0x97,0xb4,0x3b,0x17,0x72,0xe8,0x9a,
	0x15,0xb1,0xb5,0x1e,0x51,0x98,0x7b,0x0e,
	0x14,0x94,0xb2,0x06,0x18,0xd6,0x2a,0xd1,
	0xff,0x92,0x8c,0x73,0x3c,0x36,0x0b,0x2d,
	0x37,0x57,0xa2,0x04,0x80,0xa6,0x26,0xc7,
	0xdd,0x99,0xbb,0xc6,0xf1,0xfa,0xa3,0x35,
	0x55,0x6a,0xd0,0x7f,0x7a,0x13,0x4d,0x0d,
	0x4c,0x38,0x84,0xe3,0x3a,0x30,0x07,0x0f,
	0x50,0xaf,0xf7,0x19,0x00,0x8f,0xc4,0xfc,
	0x4b,0xb0,0xa8,0xd2,0xa4,0x62,0xed,0x08,
	0xdf,0xce,0x7d,0x02,0xc5,0x42,0x5b,0xdb,
	0x60,0x40,0xf2,0x05,0x0c,0xda,0xb8,0x59,
	0x4a,0x5f,0xe7,0x66,0xf4,0x46,0x74,0x6d,
	0x48,0x4e,0xef,0xd5,0xf9,0xac,0x24,0xa1,
	0xf5,0xc8,0xc3,0x16,0x5a,0x89,0x27,0x10,
	0x86,0x34,0x63,0x25,0x8d,0xe4,0x9e,0x65,
    },
    {	/* sbox 11 */
	0xee,0xf0,0x5a,0xbc,0x6b,0xb8,0xc7,0x2a,
	0x7d,0x39,0xd9,0xf4,0x83,0x89,0xca,0x0d,
	0xb2,0xb9,0x3c,0x94,0xc1,0x5d,0x01,0x1d,
	0xaa,0xcb,0xc0,0xac,0x27,0x8d,0x73,0x9b,
	0x52,0x85,0x7f,0xc9,0x4c,0xd7,0x8c,0xd5,
	0x72,0xe3,0xcc,0x17,0x6f,0xc3,0xe5,0x25,
	0x96,0xa6,0x76,0x90,0x1a,0x97,0x09,0xf8,
	0xe1,0x5c,0xeb,0x7e,0x42,0x0a,0x71,0xc8,
	0x11,0x9e,0x28,0xb6,0x3d,0x65,0xd2,0x64,
	0xae,0xba,0xa4,0x9f,0xad,0x2e,0x9d,0xa1,
	0x78,0x15,0x7a,0x16,0x93,0x31,0x26,0x40,
	0xb3,0x02,0xd1,0x53,0x2f,0x5e,0x6d,0xd8,
	0x03,0x3e,0xce,0xcd,0xb4,0x21,0xf2,0x36,
	0xe0,0x2b,0xbe,0x4f,0xbd,0x66,0x0b,0x33,
	0xa0,0xe4,0x68,0x60,0x20,0x13,0x1b,0x5f,
	0xc4,0x4d,0x00,0x7c,0xff,0xec,0xaf,0x74,
	0x3a,0x51,0x32,0xe8,0x8f,0xfb,0x8b,0xc5,
	0x34,0xd3,0x3b,0x91,0x63,0xa3,0xdd,0x87,
	0x95,0x30,0xde,0xf7,0x84,0xcf,0xfe,0x98,
	0x4b,0x50,0x29,0x57,0xc6,0x48,0x59,0x2d,
	0xf1,0xa7,0xb0,0xea,0x81,0x88,0x12,0x3f,
	0x79,0x9a,0x6c,0xbf,0x7b,0x46,0x41,0x38,
	0x5b,0xd0,0xe6,0x82,0x49,0x43,0x61,0x77,
	0xa8,0xd4,0x86,0x6a,0x1c,0x04,0xdb,0x06,
	0x19,0x18,0xf6,0xe7,0xab,0x07,0xbb,0x23,
	0x56,0x99,0x4e,0x58,0x8e,0xf9,0xa2,0x0c,
	0xb7,0x69,0x0f,0x4a,0xc2,0xf5,0x05,0xfc,
	0x45,0x47,0x22,0x14,0xe2,0xf3,0xa9,0x80,
	0x67,0x75,0x9c,0x6e,0x62,0x54,0xa5,0x35,
	0xdc,0x10,0x1e,0x24,0x44,0x08,0x0e,0xd6,
	0xb1,0xed,0x55,0xe9,0x2c,0xdf,0x92,0x37,
	0x70,0xfd,0xfa,0xda,0x8a,0xef,0x1f,0xb5,
    },
    {	/* sbox 12 */
	0xb2,0x36,0x98,0xf5,0xe6,0x15,0x67,0x30,
	0xe7,0xe3,0xa1,0x17,0xaa,0x90,0x1b,0x66,
	0xb5,0x79,0x83,0x40,0xcc,0x6d,0x8b,0x3f,
	0x68,0x6a,0x84,0x9f,0x2f,0x72,0xda,0x59,
	0x65,0xdb,0x80,0xcf,0x39,0xc2,0xa7,0x76,
	0xf8,0x4b,0x1a,0xd6,0xf7,0xb1,0xd7,0x3b,
	0x99,0xc3,0xe2,0x38,0x70,0x1d,0x71,0x3c,
	0xd3,0x62,0x0d,0x87,0xcd,0x48,0xe5,0x12,
	0x78,0x5b,0xa3,0x06,0xbc,0xeb,0xfd,0x4f,
	0x85,0xd9,0x64,0xef,0x5d,0x58,0xc8,0xa5,
	0xa2,0x4a,0x34,0xd8,0x1f,0x03,0xa4,0x10,
	0xa9,0x75,0x52,0x8d,0x28,0x37,0x6f,0x46,
	0xe8,0x50,0xd5,0xf4,0xf1,0x20,0x7c,0xf6,
	0xc4,0xac,0xee,0x51,0x19,0xca,0x82,0x18,
	0xe4,0x43,0x41,0x7b,0x21,0xcb,0x4e,0x3a,
	0xf3,0x24,0x25,0xb0,0xfe,0xc1,0x89,0x86,
	0x32,0x9d,0x3d,0x0c,0x9e,0xfc,0x04,0x8f,
	0x0e,0x93,0x01,0x35,0x22,0x94,0xc0,0x60,
	0x7e,0x63,0xae,0x08,0xdc,0x57,0x4c,0xc7,
	0x53,0x0b,0xad,0xaf,0xde,0x6c,0xff,0x42,
	0x97,0x9b,0x13,0x09,0x2e,0xfa,0xce,0xc9,
	0x16,0x1c,0x23,0x5e,0x7d,0x05,0x7a,0x5f,
	0xb7,0x2b,0x7f,0xd4,0x3e,0x11,0x6b,0x0f,
	0xb8,0x0a,0x47,0x9a,0x2a,0x8e,0x8a,0x56,
	0xe0,0x49,0x45,0x55,0xd0,0x4d,0x69,0x61,
	0xa6,0x02,0x81,0x74,0xdd,0x92,0xba,0x07,
	0xf0,0x26,0x31,0x1e,0xf2,0x91,0xc6,0xa0,
	0x29,0x77,0xbf,0x33,0xb6,0x73,0x96,0x88,
	0x2c,0x9c,0xf9,0x54,0x44,0xfb,0x95,0xdf,
	0xbd,0x27,0xd2,0x5c,0xe9,0x6e,0xb9,0xec,
	0x14,0xea,0xa8,0xd1,0xb4,0xb3,0x5a,0xbe,
	0x8c,0xab,0x00,0xbb,0xe1,0x2d,0xc5,0xed,
    },
    {	/* sbox 13 */
	0xda,0x4b,0xc3,0x17,0xb0,0x67,0x6e,0x20,
	0x47,0x72,0x8e,0x7d,0x51,0x39,0xfa,0x93,
	0x8c,0xd9,0xe7,0xfc,0x5e,0xb4,0x57,0x3d,
	0xb3,0x1b,0xa9,0x1a,0xdf,0xd3,0xbb,0xdd,
	0xa8,0xc0,0x7b,0xb7,0x11,0x9b,0x5f,0x8f,
	0x21,0x74,0x98,0x61,0x53,0x12,0xc9,0x41,
	0x96,0xc1,0xb1,0xa0,0x78,0x31,0x5b,0xbf,
	0x5c,0x89,0xfd,0xd6,0x26,0x0e,0x80,0x43,
	0xef,0x1c,0x97,0x66,0x5a,0x65,0xf9,0x0f,
	0x1f,0xf0,0x92,0xe2,0xbe,0x64,0x22,0x23,
	0x09,0x95,0x38,0xde,0x60,0xdb,0x0d,0xd5,
	0x29,0xcc,0x56,0x49,0x06,0x46,0x10,0x86,
	0x79,0xd1,0xff,0xbc,0xc5,0x18,0x90,0xcd,
	0x4c,0x4e,0x13,0x62,0x71,0x24,0xb5,0x0a,
	0xa4,0xce,0x40,0x7e,0xf6,0x3e,0xae,0xaa,
	0x25,0x27,0xf2,0x94,0xd7,0x63,0x4f,0xfb,
	0xa7,0x2a,0x33,0x9c,0x45,0x05,0xd0,0x0c,
	0x3b,0x91,0x76,0x9f,0x99,0x2e,0xcf,0x7a,
	0xdc,0x44,0xa5,0x5d,0xe8,0xeb,0xad,0x83,
	0xe9,0x52,0xf8,0x70,0x34,0x19,0xf5,0x4a,
	0xa1,0x3a,0x03,0xa6,0x68,0x85,0x04,0x2c,
	0x48,0x8b,0x6d,0x07,0x9d,0x6c,0xba,0xc2,
	0x28,0x50,0x59,0xb8,0x37,0xe0,0x1d,0xe4,
	0x14,0x6b,0x58,0xd8,0xb6,0x9a,0xc6,0x08,
	0xc4,0x02,0x55,0xfe,0x84,0x7c,0xea,0xe6,
	0xb2,0xd2,0xee,0xa3,0x4d,0x73,0xf7,0xed,
	0x9e,0x16,0xab,0xf1,0xf4,0xb9,0xec,0xcb,
	0xe1,0x15,0x81,0xca,0x82,0xc8,0x2d,0x6f,
	0x77,0x3c,0x88,0x69,0xd4,0xbd,0x8d,0x3f,
	0xc7,0x35,0xf3,0xe5,0x75,0x6a,0x2b,0x30,
	0x32,0x1e,0xa2,0x00,0xaf,0x0b,0x8a,0x36,
	0x01,0x54,0x42,0xac,0xe3,0x7f,0x87,0x2f,
    },
    {	/* sbox 14 */
	0x52,0x34,0xc4,0x12,0xa8,0x53,0x03,0xbc,
	0x9c,0x96,0x0f,0xc3,0x29,0x2f,0x68,0xef,
	0x1c,0x6b,0x7e,0xa5,0xbe,0x5f,0x99,0xf4,
	0x95,0xb9,0x87,0xad,0xf1,0x90,0xe2,0xa6,
	0xba,0x22,0x98,0x00,0xb2,0x2d,0x30,0x25,
	0xbf,0x91,0x75,0xa7,0x83,0x08,0x04,0xc0,
	0x3d,0x70,0x1f,0xd7,0x26,0x48,0x7a,0x72,
	0x5e,0xac,0x65,0xb8,0xe9,0x74,0xf9,0x7b,
	0x88,0x05,0xd4,0x2c,0x63,0xbd,0xdc,0xe7,
	0x6d,0xd8,0xc1,0x6c,0x14,0x10,0x66,0x0e,
	0x4b,0x41,0x36,0x37,0x9b,0x31,0x8e,0xfc,
	0x57,0x62,0xc5,0x46,0xa4,0x56,0x3a,0x43,
	0x55,0x5c,0x3b,0x51,0x4d,0x3c,0x71,0xc9,
	0x78,0x16,0x61,0x15,0x73,0x2b,0xb5,0x1a,
	0x8c,0x4a,0xcc,0x7c,0x35,0xda,0xfa,0x80,
	0xd3,0xf3,0x9f,0xa0,0x60,0x3f,0xe5,0xe8,
	0x5a,0xc6,0xd0,0xce,0x4f,0x4e,0xf2,0x0c,
	0x23,0xf5,0xdb,0xe1,0x45,0x07,0xc2,0x02,
	0x0b,0x69,0x17,0x50,0xf8,0x5d,0xb4,0xf0,
	0xe6,0x92,0xb6,0x21,0xab,0xfd,0x67,0x9d,
	0x93,0x47,0x2e,0xdd,0xc7,0x9e,0xff,0xeb,
	0x79,0x1d,0xf6,0x27,0x40,0x6f,0x7d,0xd2,
	0x58,0x76,0x82,0x86,0x0a,0xb7,0xd5,0xb0,
	0x97,0xae,0xde,0xea,0x44,0x7f,0x84,0x8a,
	0xfb,0x28,0xaf,0x77,0xd9,0x1e,0x9a,0x59,
	0xcf,0x49,0xd6,0xee,0xe4,0x32,0xa1,0xb1,
	0x4c,0xec,0x64,0x18,0x06,0xa3,0x20,0xe0,
	0xed,0x01,0xd1,0xaa,0x19,0xbb,0x85,0x1b,
	0x39,0xc8,0xe3,0xca,0x54,0x38,0x6e,0x11,
	0x24,0x42,0xfe,0xdf,0x8f,0x6a,0xcb,0xa9,
	0x5b,0x89,0x94,0x0d,0x3e,0x2a,0x8b,0xcd,
	0xf7,0xb3,0x33,0x09,0x81,0xa2,0x13,0x8d,
    },
    {	/* sbox 15 */
	0xac,0xf0,0x73,0x4d,0x9e,0x41,0x3f,0x0c,
	0x2a,0x90,0x10,0xf8,0x4e,0x24,0x1c,0xa4,
	0xf5,0x16,0xf3,0x71,0xf7,0xba,0xd0,0x2d,
	0xa6,0x6c,0x68,0xd5,0x87,0x69,0x8d,0xc6,
	0xe1,0x7b,0xb8,0x21,0x81,0xed,0xb9,0x6f,
	0x5e,0x2b,0xa7,0x78,0x3c,0x5c,0x61,0xef,
	0xa5,0x58,0x79,0xff,0x20,0xda,0x4a,0x64,
	0x39,0xc2,0xf2,0x22,0x6a,0xdd,0xaa,0xfe,
	0x6b,0x75,0x28,0x63,0x57,0x85,0x26,0x38,
	0x9f,0xa0,0x3a,0xc4,0x3d,0x0f,0x43,0xcb,
	0xdc,0xd7,0xc0,0x72,0x1d,0xad,0x04,0x7a,
	0x98,0x8a,0x14,0xe3,0xb5,0x84,0x0d,0x1b,
	0x12,0xee,0xb7,0xf1,0x9d,0x09,0x66,0xd8,
	0x86,0x94,0xc9,0x96,0x80,0x8f,0x07,0x02,
	0xe0,0xfa,0x77,0x15,0x05,0x8c,0xb6,0x13,
	0x97,0x93,0xe4,0x65,0xf9,0x6e,0xc7,0x82,
	0x9c,0x17,0xcd,0xa9,0xd9,0x2e,0xea,0x74,
	0x7e,0xec,0xa8,0xbd,0x88,0xce,0x56,0xeb,
	0x59,0xbb,0xaf,0x4c,0xcf,0xcc,0x5f,0x3e,
	0xc1,0x40,0x4b,0x7d,0x08,0xf4,0xc5,0x36,
	0xfc,0xb3,0xca,0x52,0x8e,0x49,0x06,0xd3,
	0xbe,0x67,0xfb,0x03,0x9a,0xe8,0xd6,0x0b,
	0x7c,0x0a,0x27,0x33,0x2c,0x30,0x25,0x91,
	0xd4,0x37,0x9b,0x3b,0xc3,0x2f,0x29,0x62,
	0x60,0xa1,0x00,0x92,0x1e,0x31,0x50,0xbc,
	0xe6,0x23,0x70,0xf6,0x34,0x35,0xae,0x89,
	0xdb,0xde,0xa3,0xb4,0x44,0xab,0x18,0xd2,
	0x4f,0x53,0x32,0x54,0x76,0x5d,0x95,0xb2,
	0xbf,0xd1,0xe5,0x5b,0x1a,0x48,0xfd,0xe9,
	0x1f,0xa2,0x99,0x42,0xe7,0xb0,0x55,0x46,
	0xb1,0x19,0x47,0x6d,0x83,0x45,0x0e,0x5a,
	0x8b,0x51,0x7f,0xe2,0xdf,0x11,0x01,0xc8,
    }
};

#ifdef TEST_MAIN

#include <stdio.h>
#include <ctype.h>
#include <time.h>

int main(int argc, char **argv)
{
    pha_cblock plain[2] = { { 0 } }, cryp[2] = { { 0 } };
    pha_cblock decryp[2] = { { 0 } }, ivec = { 0 };
    pha_key key = { 0 };
    pha_schedule sched;
    const long iterations = 50000;
    long j;
    int i;
    clock_t start, stop;
    double bps;

    if (argc != 3) {
	fputs("arguments: <plaintext> <key>\n", stderr);
	exit(EXIT_FAILURE);
    }
    strncpy((char *) *plain, argv[1], 32);
    if (!pha_phrase_to_key(argv[2], strlen(argv[2]), &key)) {
	fprintf(stderr, "key too long (%lu chars)\n",
		(unsigned long) strlen(argv[2]));
	exit(EXIT_FAILURE);
    }
    printf("key: ");
    for (i = 0; i < 32; i++) {
	printf("%02x", (unsigned) key[i] & 0xff);
	if ((i & 15) == 15)
	    putchar(' ');
    }
    putchar('\n');
    pha_schedule_key(&key, &sched);

    printf("Running %ld iterations...\r", iterations);
    fflush(stdout);

    start = clock();
    for (j = 0; j < iterations; j++) {
	memset(&ivec, 0, sizeof ivec);
	pha_cbc_encipher(plain, cryp, &ivec, &sched, 32);
	memset(&ivec, 0, sizeof ivec);
	pha_cbc_decipher(cryp, decryp, &ivec, &sched, 32);
    }
    stop = clock();
    printf("cip: ");
    for (i = 0; i < 32; i++) {
	printf("%02x", (unsigned) (*cryp)[i] & 0xff);
	if ((i & 15) == 15)
	    putchar(' ');
    }
    putchar('\n');
    printf("pla: ");
    for (i = 0; i < 32; i++) {
	printf("%02x", (*decryp)[i] & 0xff);
	if ((i & 15) == 15)
	    putchar(' ');
    }
    putchar('\n');
    printf("txt: ");
    for (i = 0; i < 32; i++) {
	char ch = (*decryp)[i] & 0xff;
	printf(" %c", isprint(ch) ? ch : '.');
	if ((i & 15) == 15)
	    putchar(' ');
    }
    putchar('\n');
    if (stop < start) {
	printf("clock_t overflowed");
	exit(EXIT_FAILURE);
    }
    bps = 4.0 * iterations / (stop - start) * (double) CLOCKS_PER_SEC;
    printf("Blocks per second: %10.1f\n", bps);
    printf("Octets per second: %10.1f\n", bps * 16.0);
    return 0;
}

#endif

