/*
Copyright (C) 1992,1993,1994 Trusted Information Systems, Inc.

Export of this software from the United States of America or
Canada requires a specific license from the United States
Government.  This version of this software is not suitable for
export.

WITHIN THAT CONSTRAINT, the full text of the license agreement
that specifies the conditions under which this software may be
used is published in the file license.txt in the same directory
as that containing the TIS/PEM source.

Trusted Information Systems makes no representation about the
suitability of this software for any purpose.  It is provided
"as is" without express or implied warranty.
*/

/*****************************************************************************
 * encode_rsakey.c
 *---------------------------------------------------------------------------
 * ASN.1 encode RSA key per algorithm-specific syntax
 *****************************************************************************/

#include "config.h"
#include "general.h"

#include "asn1.h"
#include "crypto.h"
#include "bbuf.h"
#include "util.h"

#include "./rsa.h"

/*****************************************************************************
 * encode_rsakeyparms()
 *---------------------------------------------------------------------------
 * ASN.1 encode RSA key length (as an INTEGER) into bbuf
 ****************************************************************************/

int encode_rsakeyparms(len, parms)
int len;
struct bbuf **parms;
{
    int r;

    /* DOUBLE-CHECK FUNCTION PARAMETERS */
    if (parms == (struct bbuf **)0) 
	return(NOTOK);

    FREE_BBUF(*parms);

    r = encode_int(len, parms);

    return(r);
}


/*****************************************************************************
 * encode_rsapubkeydata()
 *---------------------------------------------------------------------------
 * ASN.1 encode key from an RSAREF public key structure into bbuf,
 * as following SEQUENCE, taken from X.509:
 *
 *     SEQUENCE
 *         modulus	    INTEGER,	-- n
 *         exponent	    INTEGER	-- e
 *     }
 * 
 ****************************************************************************/

int encode_rsapubkeydata(Pu, data)
R_RSA_PUBLIC_KEY *Pu;
struct bbuf **data;
{
    int r;
    struct bbuf *n = NULLBB, *e = NULLBB;
    struct bbuf *bb = NULLBB, *bb2 = NULLBB, *bb3 = NULLBB;

    /* DOUBLE-CHECK FUNCTION PARAMETERS */
    if (Pu == (R_RSA_PUBLIC_KEY *)0) 
	return(NOTOK);

    if (data == (struct bbuf **)0) 
	return(NOTOK);

    FREE_BBUF(*data);

    /* COPY MODULUS N INTO BBUF */
    if ((r = buf2bbuf(Pu->modulus, MAX_RSA_MODULUS_LEN, &n)) != OK)
	goto cleanup;

    /* COPY EXPONENT E INTO BBUF */
    if ((r = buf2bbuf(Pu->exponent, MAX_RSA_MODULUS_LEN, &e)) != OK)
	goto cleanup;

    /* ENCODE N AS INTEGER (STRING) */
    if ((r = encode_ints(n, &bb2)) != OK)
	goto cleanup;

    /* ENCODED E AS INTEGER (STRING) & APPEND */
    if ((r = encodeintsandappend(e, &bb, &bb2, &bb3)) != OK)
	goto cleanup;

    /* PREPEND ENCODED LENGTH */
    if ((r = encode_len(bb2->length, &bb)) != OK)
        goto cleanup;
    bb3 = bbufcat(bb, bb2);
    FREE_BBUF(bb);
    FREE_BBUF(bb2);
    bb2 = bb3;

    /* PREPEND UNIVERSAL CONSTRUCTED SEQUENCE TAG */
    if ((r = encode_tag(CLASS_UNIV|FORM_CONS|CONS_SEQ, &bb)) != OK)
	goto cleanup;
    *data = bbufcat(bb, bb2);
    FREE_BBUF(bb);
    FREE_BBUF(bb2);

    r = OK;

 cleanup:

    if (r != OK)
	FREE_BBUF(*data);

    FREE_BBUF(n);
    FREE_BBUF(e);
    FREE_BBUF(bb2);

    return(r);

}

/*****************************************************************************
 * encode_rsaprikeydata()
 *---------------------------------------------------------------------------
 * ASN.1 encode key from an RSAREF private key structure into bbuf,
 * as following SEQUENCE, taken from PKCS #1:
 *
 *     SEQUENCE
 *         version	    INTEGER,	-- 0 for PKCS #1 Version 1.4
 *         modulus	    INTEGER,	-- n
 *         publicExponent   INTEGER,	-- e
 *         privateExponent  INTEGER,	-- d
 *         prime1	    INTEGER,	-- p
 *         prime2	    INTEGER,	-- q
 *         exponent1	    INTEGER,	-- d mod (p-1)
 *         exponent2	    INTEGER,	-- d mod (q-1)
 *         coefficient	    INTEGER	-- q^(-1) mod p
 *     }
 *
 *****************************************************************************/

int encode_rsaprikeydata(Pr, data)
R_RSA_PRIVATE_KEY *Pr;
struct bbuf **data;
{
    int r;
    struct bbuf *n = NULLBB, *e = NULLBB, *d = NULLBB, *p = NULLBB, 
		*q = NULLBB, *dp = NULLBB, *dq = NULLBB, *cr = NULLBB;
    struct bbuf *bb = NULLBB, *bb2 = NULLBB, *bb3 = NULLBB;

    /* COPY MODULUS N INTO BBUF */
    if ((r = buf2bbuf(Pr->modulus, MAX_RSA_MODULUS_LEN, &n)) != OK)
	goto cleanup;
    
    /* COPY EXPONENT E INTO BBUF */
    if ((r = buf2bbuf(Pr->publicExponent, MAX_RSA_MODULUS_LEN, &e)) != OK)
	goto cleanup;
    
    /* COPY EXPONENT D INTO BBUF */
    if ((r = buf2bbuf(Pr->exponent, MAX_RSA_MODULUS_LEN, &d)) != OK)
	goto cleanup;
    
    /* COPY PRIME P INTO BBUF */
    if ((r = buf2bbuf(Pr->prime[0], MAX_RSA_PRIME_LEN, &p)) != OK)
	goto cleanup;
    
    /* COPY PRIME Q INTO BBUF */
    if ((r = buf2bbuf(Pr->prime[1], MAX_RSA_PRIME_LEN, &q)) != OK)
	goto cleanup;
    
    /* COPY CRT EXPONENT DP INTO BBUF */
    if ((r = buf2bbuf(Pr->primeExponent[0], MAX_RSA_PRIME_LEN, &dp)) != OK)
	goto cleanup;
    
    /* COPY CRT EXPONENT DQ INTO BBUF */
    if ((r = buf2bbuf(Pr->primeExponent[1], MAX_RSA_PRIME_LEN, &dq)) != OK)
	goto cleanup;
    
    /* COPY CRT COEFFICIENT INTO BBUF */
    if ((r = buf2bbuf(Pr->coefficient, MAX_RSA_PRIME_LEN, &cr)) != OK)
	goto cleanup;
    
    /* ENCODE VERSION AS INTEGER */
    if ((r = encode_int(PRIVATE_KEY_VERSION, &bb2)) != OK)
	goto cleanup;

    /* ENCODE MODULUS N AS INTEGER (STRING) & APPEND */
    if ((r = encodeintsandappend(n, &bb, &bb2, &bb3)) != OK)
	goto cleanup;

    /* ENCODE ENCRYPTION EXPONENT E AS INTEGER (STRING) & APPEND */
    if ((r = encodeintsandappend(e, &bb, &bb2, &bb3)) != OK)
	goto cleanup;

    /* ENCODE DECRYPTION EXPONENT D AS INTEGER (STRING) & APPEND */
    if ((r = encodeintsandappend(d, &bb, &bb2, &bb3)) != OK)
	goto cleanup;

    /* ENCODE PRIME P AS INTEGER (STRING) & APPEND */
    if ((r = encodeintsandappend(p, &bb, &bb2, &bb3)) != OK)
	goto cleanup;

    /* ENCODE PRIME Q AS INTEGER (STRING) & APPEND */
    if ((r = encodeintsandappend(q, &bb, &bb2, &bb3)) != OK)
	goto cleanup;

    /* ENCODE EXPONENT DP AS INTEGER (STRING) & APPEND */
    if ((r = encodeintsandappend(dp, &bb, &bb2, &bb3)) != OK)
	goto cleanup;

    /* ENCODE EXPONENT DQ AS INTEGER (STRING) & APPEND */
    if ((r = encodeintsandappend(dq, &bb, &bb2, &bb3)) != OK)
	goto cleanup;

    /* ENCODE COEFFICIENT CR AS INTEGER (STRING) & APPEND */
    if ((r = encodeintsandappend(cr, &bb, &bb2, &bb3)) != OK)
	goto cleanup;

    /* PREPEND ENCODED LENGTH */
    if ((r = encode_len(bb2->length, &bb)) != OK)
        goto cleanup;
    bb3 = bbufcat(bb, bb2);
    FREE_BBUF(bb);
    FREE_BBUF(bb2);
    bb2 = bb3;

    /* PREPEND UNIVERSAL CONSTRUCTED SEQUENCE TAG */
    if ((r = encode_tag(CLASS_UNIV|FORM_CONS|CONS_SEQ, &bb)) != OK)
	goto cleanup;
    *data = bbufcat(bb, bb2);
    FREE_BBUF(bb);
    FREE_BBUF(bb2);

    r = OK;

 cleanup:

    if (r != OK)
	FREE_BBUF(*data);

    FREE_BBUF(n);
    FREE_BBUF(e);
    FREE_BBUF(d);
    FREE_BBUF(p);
    FREE_BBUF(q);
    FREE_BBUF(dp);
    FREE_BBUF(dq);
    FREE_BBUF(cr);
    FREE_BBUF(bb2);

    return(r);
}

/****************************************************************************/

static int encodeintsandappend(x, bb, bb2, bb3)
struct bbuf *x;
struct bbuf **bb, **bb2, **bb3;
{
    int r;

    /* ENCODE AS INTEGER (STRING) & APPEND */
    if ((r = encode_ints(x, bb)) != OK)
	return(r);
    *bb3 = bbufcat(*bb2, *bb);
    FREE_BBUF(*bb);
    FREE_BBUF(*bb2);
    *bb2 = *bb3;
    return(OK);
}

/****************************************************************************/

static int buf2bbuf(b, l, bb)
unsigned char *b;
int l;
struct bbuf **bb;
{
    int i, j;

    /* SKIP OVER LEADING ZEROES (UNLESS 1ST BIT OF NEXT BYTE IS SET) */
    for (i=0; i<l-1 && b[i]==NULLUC && !(b[i+1] & 0x80); i++) ;

    /* PREPARE FOR EXTRA LEADING NULL BYTE, IF NECESSARY */
    j = (i==0 && (b[0] & 0x80)) ? 1 : 0;

    /* ALLOCATE BBUF */
    if ((*bb = alloc_bbuf()) == NULLBB ||
        ((*bb)->data = alloc_uchar(l-i+j)) == NULLUCP)
	return(NOTOK);

    /* COPY DATA */
    if (j==1) (*bb)->data[0] = NULLUC;
    BCOPY(b+i, (*bb)->data+j, l-i);
    (*bb)->length = l-i+j;

    return(OK);
}

/****************************************************************************/
