/*
Copyright (C) 1992,1993,1994 Trusted Information Systems, Inc.

Export of this software from the United States of America or
Canada requires a specific license from the United States
Government.  This version of this software is not suitable for
export.

WITHIN THAT CONSTRAINT, the full text of the license agreement
that specifies the conditions under which this software may be
used is published in the file license.txt in the same directory
as that containing the TIS/PEM source.

Trusted Information Systems makes no representation about the
suitability of this software for any purpose.  It is provided
"as is" without express or implied warranty.
*/

/*
 * sav_key - save the private key in "key" according to the user record
 * "user" with the password/authentication "pw"
 *
 * RETURN VALUES
 * 	OK	- the private key was saved
 * 	NOTOK	- the private key could not be saved
 */

#include "config.h"
#include "general.h"

#include "bbuf.h"
#include "crypto.h"
#include "key.h"
#include "new.h"
#include "util.h"

static int do_file();


sav_key(user, pw, key)
char        **user;		/* user record */
struct bbuf  *pw;		/* password data according to access */
struct bbuf  *key;		/* ASN.1 encoded private key */
{
    char **av_key, **av_access, **av_pkey;

    if (user == NULLVP)
	return NOTOK;

    if ((av_access = tag_user(user, "private-key-access")) == NULLVP)
	return NOTOK;

    if ((av_pkey = tag_user(user, "public-key")) == NULLVP)
	return NOTOK;

    av_key = tag_user(user, "private-key");

    if (!strcasecmp(*(av_access+1), "file")) {
	if (av_key == NULLVP || *(av_key+1) == NULLCP || **(av_key+1) == NULLC)
	    return NOTOK;
	return do_file(*(av_pkey+1), *(av_key+1), pw, key);
    }
    else
	return NOTOK;
}


/* combine two bbufs into one formatted as follows:
 * 
 * bb1 length, bb1 data, bb2 length, and bb2 data
 *
 * the lengths are byte swapped
 */

static struct bbuf *bbufjoin(bb1, bb2)
struct bbuf *bb1, *bb2;
{
    struct bbuf *ret = NULLBB;
    short length;

    if (bb1 == NULLBB || bb2 == NULLBB)
	return NULLBB;

    ret = alloc_bbuf();

    length = 2 * sizeof(bb1->length) + bb1->length + bb2->length;

    ret->data = alloc_uchar(length);
    ret->length = length;

    length = bb1->length;
    BSWAP(length);
    BCOPY((char *)&length, ret->data, sizeof(length));
    BCOPY(bb1->data, ret->data + sizeof(length), bb1->length);

    length = bb2->length;
    BSWAP(length);
    BCOPY((char *)&length, ret->data + sizeof(length) + bb1->length,
	  sizeof(length));
    BCOPY(bb2->data, ret->data + 2 * sizeof(length) + bb1->length,
	  bb2->length);

    return ret;
}


/* private key file is formatted as follows
 *
 * CLEARTEXT
 *
 * 	1 byte status - always 0x00
 * 	bbuf (len, data) which is md5 hash of key
 * 	bbuf (len, data) which is ASN.1 encoded key
 *
 * ENCRYPTED
 *
 * 	1 bytes status - always 0x01
 * 	bbuf (len, data) which is encrypted join of hash bbuf and key bbuf
 */

static do_file(pkey, file, pw, key)
char         *pkey;
char         *file;
struct bbuf  *pw;
struct bbuf  *key;
{
    int             fd = -1;
    unsigned char   status = 0;
    struct bbuf    *hash = NULLBB;
    int             ret = NOTOK;

    if (key == NULLBB || key->length <= 0)
	return NOTOK;

    /* if the key file exists, we use it; otherwise create a protected file */

    if (filchk(file, -1, 0) != OK)
	if (filchk(file, 0600, 1) != OK)
	    return NOTOK;

    if ((fd = lk_open(file, O_WRONLY|O_TRUNC)) < 0)
	return NOTOK;

    /* create hash of key */

    if (gen_md5(NULL_KEY, key, &hash) != OK)
	goto cleanup;

    /* set encrypted status and save it */

    if (pw != NULLBB && pw->length > 0)
	status = 0x01;

    if (write(fd, (char *)&status, 1) != 1)
	goto cleanup;

    /* save key according to encrypted status */

    if (status) {
	struct bbuf  *ekey = NULLBB,
                     *join = NULLBB,
                     *ejoin = NULLBB;
	struct key   *dkey = NULL_KEY;

	if (str2deskey((char *)pw->data, &dkey) != OK)
	    goto cleanup;
	if (encode_key(dkey, &ekey) != OK)
	    goto cleanup;
	if (set_key(ekey) != OK)
	    goto cleanup;
	if ((join = bbufjoin(hash, key)) == NULLBB)
	    goto cleanup;
	if (encipher(join, &ejoin) != OK)
	    goto cleanup;
	if (write_bbuf(fd, ejoin) != OK)
	    goto cleanup;

	FREE_BBUF(ejoin);
	FREE_BBUF(ekey);
	FREE_BBUF(join);

	FREE_KEY(dkey);
    }
    else {
	if (write_bbuf(fd, hash) != OK)
	    goto cleanup;

	if (write_bbuf(fd, key) != OK)
	    goto cleanup;
    }

    ret = OK;

 cleanup:

    if (fd != -1)
	(void) lk_close(fd);

    FREE_BBUF(hash);

    return ret;
}
