/*
Copyright (C) 1992,1993,1994 Trusted Information Systems, Inc.

Export of this software from the United States of America or
Canada requires a specific license from the United States
Government.  This version of this software is not suitable for
export.

WITHIN THAT CONSTRAINT, the full text of the license agreement
that specifies the conditions under which this software may be
used is published in the file license.txt in the same directory
as that containing the TIS/PEM source.

Trusted Information Systems makes no representation about the
suitability of this software for any purpose.  It is provided
"as is" without express or implied warranty.
*/

#include "config.h"
#include <stdio.h>
#include "general.h"

#include "bbuf.h"
#include "cbio.h"
#include "cert.h"
#include "crl.h"
#include "dname.h"
#include "new.h"
#include "util.h"


static int getheader(inputcb, fieldname, fieldvalue)
struct bbuf *(*inputcb)();
char **fieldname;
char **fieldvalue;
{
    int r = NOTOK;
    int ch;
    int pos;
    char *name = NULLCP;
    char *val = NULLCP;

    if (fieldname == NULLVP || fieldvalue == NULLVP) 
	goto cleanup;
  
    FREE(*fieldname); 
    FREE(*fieldvalue);

    name = alloc_char(BUFSIZ);
    val = alloc_char(BUFSIZ);

    /* get field name */

    pos = 0;
    while ((ch = getcharcb(inputcb)) != ':' && pos < BUFSIZ - 1) {
	if (ch == EOF)
	    goto cleanup;
	name[pos++] = ch;
    }
    name[pos] = '\0';

    /* skip whitespcae */

    while ((ch = getcharcb(inputcb)) == ' ' || ch == '\t')
	;

    /* get field value */

    pos = 0;
    while (ch != '\n' && ch != EOF && pos < BUFSIZ - 1) {
	val[pos++] = ch;
	ch = getcharcb(inputcb);
    }
    val[pos] = '\0';

    *fieldname = name;
    name = NULLCP;
    *fieldvalue = val;
    val = NULLCP;

    r = OK;

 cleanup:
  
    FREE(name);
    FREE(val);
    return r;
}



static int save_crl(crl, iocbs) 
char *crl;
struct cbstruct *iocbs;
{
    int r = NOTOK;
    struct bbuf *crlbb = NULLBB;
    struct bbuf *oldcrlbb = NULLBB;
    struct bbuf *dnbb = NULLBB;
    struct bbuf *certbb = NULLBB;
    struct bbuf bb;
    char *dnstr = NULLCP;
    char **user = NULLVP;
    char **usercrl = NULLVP;
    char **usercert = NULLVP;
    char *cp = NULLCP;

    if (crl == NULLCP || iocbs == NULL_IOCBS)
	goto cleanup;

    /* decode crl & extract issuer name */
    bb.data = (unsigned char *) crl;
    bb.length = strlen (bb.data);
    if (bdecode (&bb, &crlbb) != OK) {
	cp2cb(iocbs->out_errs, "Unable to decode CRL.\n");
	goto cleanup;
    }
    if (issuer_crl(&crlbb,&dnbb,READ)!=OK || (dnstr=dn2str(dnbb))==NULLCP) {
	cp2cb(iocbs->out_errs, "Unable to extract issuer name from CRL.\n");
	goto cleanup;
    }

    /* pull up user record with subject name the same as crl issuer's */
    rewind_indexfile();
    if ((user = get_tv_user("subject-name", dnstr)) == NULLVP) {
	cp = add2cp(NULLCP, "Unable to find certificate for CRL issuer ");
	cp = add2cp(cp, dnstr);
	cp = add2cp(cp, ".\n");
	cp2cb(iocbs->out_errs, cp);
	FREE(cp);
	goto cleanup;
    }

    /* extract & decode the user's certificate for CRL validation */
    if ((usercert = tag_user(user, "certificate")) == NULLVP) {
	cp = add2cp(NULLCP, "User record with CRL issuer name ");
	cp = add2cp(cp, dnstr);
	cp = add2cp(cp, " has no certificate.\n");
	cp2cb(iocbs->out_errs, cp);
	FREE(cp);
	goto cleanup;
    }

    bb.data = (unsigned char *) *(usercert+1);
    bb.length = strlen (bb.data);
    if (bdecode(&bb, &certbb) != OK) {
	cp = add2cp(NULLCP, "Unable to decode certificate for CRL issuer ");
        cp = add2cp(cp, dnstr);
        cp = add2cp(cp, ".\n");
        cp2cb(iocbs->out_errs, cp);
        FREE(cp);
        goto cleanup;
    }

    /* validate the crl */
    if (validate_crl(crlbb, certbb) != OK || 
	crlbb->status & (BNOVAL|BBADVAL|BNOSIG|BNOISS|BBADSIG|BNOIDN)) {
	cp = add2cp(NULLCP, "New CRL is invalid for issuer ");
	cp = add2cp(cp, dnstr);
	cp = add2cp(cp, ".\n");
        cp2cb(iocbs->out_errs, cp);
        FREE(cp);
	goto cleanup;
    }
	
    if ((usercrl = tag_user(user, "crl")) == NULLVP) {
	user = add2av(user, add2cp(NULLCP, "crl"));
	user = add2av(user, add2cp(NULLCP, crl));
	if (update_indexfile(NULLVP, user) != OK) {
	    cp = add2cp(NULLCP, "Unable to update CRL for issuer ");
	    cp = add2cp(cp, dnstr);
	    cp = add2cp(cp, ".\n");
	    cp2cb(iocbs->out_errs, cp);
	    FREE(cp);
	    goto cleanup;
	}
    }
    else if (!strcmp(crl, *(usercrl+1)))
	;
    else {
	bb.data = (unsigned char *) *(usercrl+1);
	bb.length = strlen (bb.data);
	if (bdecode(&bb, &oldcrlbb) != OK) {
	    FREE(*(usercrl+1));
	    *(usercrl+1) = add2cp(NULLCP, crl);
	    if (update_indexfile(NULLVP, user) != OK) {
		cp = add2cp(NULLCP, "Unable to update CRL for issuer ");
		cp = add2cp(cp, dnstr);
		cp = add2cp(cp, ".\n");
		cp2cb(iocbs->out_errs, cp);
		FREE(cp);
		goto cleanup;
	    }
	}
	else if (validate_crl(oldcrlbb, certbb) != OK || 
		 oldcrlbb->status &	
		 (BNOVAL|BBADVAL|BNOSIG|BNOISS|BBADSIG|BNOIDN))
	{
	    FREE(*(usercrl+1));
	    *(usercrl+1) = add2cp(NULLCP, crl);
	    if (update_indexfile(NULLVP, user) != OK) {
		cp = add2cp(NULLCP, "Unable to update CRL for issuer ");
                cp = add2cp(cp, dnstr);
                cp = add2cp(cp, ".\n");
                cp2cb(iocbs->out_errs, cp);
                FREE(cp);
                goto cleanup;
            }
	}
    }
    
    r = OK;

 cleanup:

    FREE_BBUF(crlbb);
    FREE_BBUF(oldcrlbb);
    FREE_BBUF(dnbb);
    FREE_BBUF(certbb);
    FREE(dnstr);
    FREE_AV(user);
    return r;
}



static int save_cert(cert, iocbs)
char *cert;
struct cbstruct *iocbs;
{
    int r = NOTOK;
    char **certuser = NULLVP;
    char **certkey = NULLVP;
    char **certdn = NULLVP;
    char **usercert = NULLVP;
    char **user = NULLVP;
    struct bbuf *dnbb = NULLBB;
    struct bbuf bb;
    char *dnstr = NULLCP;
    char *cp = NULLCP;

    if (cert == NULLCP || iocbs == NULL_IOCBS)
	goto cleanup;

    if ((certuser = explode_cert(cert)) == NULLVP ||
	(certkey = tag_user(certuser, "public-key")) == NULLVP ||
	(certdn = tag_user(certuser, "subject-name")) == NULLVP) {
	cp2cb(iocbs->out_errs, "Certificate is incomplete or corrupt.\n");
	goto cleanup;
    }

    bb.data = (unsigned char *) *(certdn+1);
    bb.length = strlen(bb.data);
    if (bdecode(&bb, &dnbb) != OK || (dnstr = dn2str(dnbb)) == NULLCP) {
	cp2cb(iocbs->out_errs,"Unable to extract certificate subject name.\n");
	goto cleanup;
    }

    rewind_indexfile();
    
    if ((user = get_tv_user("public-key", *(certkey+1))) == NULLVP) {
	if (update_indexfile(NULLVP, certuser) != OK) {
	    cp = add2cp(NULLCP, 
			"Unable to save certificate with subject name ");
	    cp = add2cp(cp, dnstr);
	    cp = add2cp(cp, ".\n");
	    cp2cb(iocbs->out_errs, cp);
	    FREE(cp);
	    goto cleanup;
	}
    }
    else if (tag_user(user, "trusted") == NULLVP) {
	if ((usercert = tagval_user(user, "certificate", cert)) == NULLVP) {
	    user = merge_user(user, certuser);
	    if (update_indexfile(NULLVP, user) != OK) {
            cp = add2cp(NULLCP,
         "Unable to update user record with certificate with subject name ");
            cp = add2cp(cp, dnstr);
            cp = add2cp(cp, ".\n");
            cp2cb(iocbs->out_errs, cp);
            FREE(cp);
            goto cleanup;
	    }
	}
	else if (strcmp(*(usercert+1), cert) != 0) {
            cp = add2cp(NULLCP,
                 "A different certificate already exists with subject name ");
            cp = add2cp(cp, dnstr);
            cp = add2cp(cp, ".\n");
            cp2cb(iocbs->out_errs, cp);
            FREE(cp);
	    goto cleanup;
	}
    }
	   
    r = OK;
	
 cleanup:
    
    FREE_AV(certuser);
    FREE_AV(user);
    FREE_BBUF(dnbb);
    FREE(dnstr);
    return r;
}



static int save_keyform(keyform, iocbs) 
char *keyform;
struct cbstruct *iocbs;
{
    int r = NOTOK;
    char *pkid = NULLCP;
    char *pk = NULLCP;
    char *id = NULLCP;
    char *keyid = NULLCP;
    char *idval = NULLCP;
    char *cp = NULLCP;
    char **user = NULLVP;
    char **newuser = NULLVP;

    if (iocbs == NULL_IOCBS)
	goto cleanup;

    /* pick out various fields by looking for commas */

    if ((pkid = keyform) == NULLCP || 
	(pk = INDEX(pkid, ',')) == NULLCP ||
	(id = INDEX(pk+1, ',')) == NULLCP ||
	(keyid = INDEX(id+1, ',')) == NULLCP || 
	(idval = INDEX(keyid+1, ',')) == NULLCP) {
	cp2cb(iocbs->out_errs, "Incomplete Key: header.\n");
	goto cleanup;
    }

    /* convert commas to NULLs and move pointers forward */

    *(pk++) = NULLC;
    *(id++) = NULLC;
    *(keyid++) = NULLC;
    *(idval++) = NULLC;

    /* remove both leading and trailing spaces around fields*/

    for (cp = pkid; isprint(*cp) && !isspace(*cp); cp++)
	;
    *cp = NULLC;
    while (isspace(*pk)) pk++;
    for (cp = pk; isprint(*cp) && !isspace(*cp); cp++)
	;
    *cp = NULLC;
    while (isspace(*id)) id++;
    for (cp = id; isprint(*cp) && !isspace(*cp); cp++)
	;
    *cp = NULLC;
    while (isspace(*keyid)) keyid++;
    for (cp = keyid; isprint(*cp) && !isspace(*cp); cp++)
	;
    *cp = NULLC;
    while (isspace(*idval)) idval++;
    for (cp = idval; isprint(*cp) && !isspace(*cp); cp++)
	;
    *cp = NULLC;

    /* Make sure we [still] have what we need */

    if (*pkid == NULLC || *pk == NULLC || *id == NULLC || 
	*keyid == NULLC || *idval == NULLC) {
	cp2cb(iocbs->out_errs, "Missing required fields in Key header.\n");
	goto cleanup;
    }

    if (strcasecmp(pkid, "pk") != 0) {
	cp = add2cp(NULLCP, "Incorrect identifier \"");
	cp = add2cp(cp, pkid);
	cp = add2cp(cp, "\" (should be PK).\n");
	cp2cb(iocbs->out_errs, cp);
	FREE(cp);
	goto cleanup;
    }
    
    /* Create user record out of supplied information */

    newuser = add2av(NULLVP, add2cp(NULLCP, "public-key"));
    newuser = add2av(newuser, add2cp(NULLCP, pk));

    if (!strcasecmp(id, "dn")) {
	newuser = add2av(newuser, add2cp(NULLCP, "id-dn"));
	newuser = add2av(newuser, add2cp(NULLCP, idval));
	newuser = add2av(newuser, add2cp(NULLCP, "id-dn-keyid"));
	newuser = add2av(newuser, add2cp(NULLCP, keyid));
    }
    else if (!strcasecmp(id, "is")) {
	newuser = add2av(newuser, add2cp(NULLCP, "issuer-name"));
	newuser = add2av(newuser, add2cp(NULLCP, idval));
	newuser = add2av(newuser, add2cp(NULLCP, "serial-number"));
	newuser = add2av(newuser, add2cp(NULLCP, keyid));
    }
    else if (!strcasecmp(id, "en")) {
	newuser = add2av(newuser, add2cp(NULLCP, "id-en"));
	newuser = add2av(newuser, add2cp(NULLCP, idval));
	newuser = add2av(newuser, add2cp(NULLCP, "id-en-keyid"));
	newuser = add2av(newuser, add2cp(NULLCP, keyid));
    }
    else if (!strcasecmp(id, "str")) {
	newuser = add2av(newuser, add2cp(NULLCP, "id-str"));
	newuser = add2av(newuser, add2cp(NULLCP, idval));
	newuser = add2av(newuser, add2cp(NULLCP, "id-str-keyid"));
	newuser = add2av(newuser, add2cp(NULLCP, keyid));
    }
    else if (!strcasecmp(id, "pgp2")) {	
	newuser = add2av(newuser, add2cp(NULLCP, "id-pgp"));
	newuser = add2av(newuser, add2cp(NULLCP, idval));
	newuser = add2av(newuser, add2cp(NULLCP, "id-pgp-keyid"));
	newuser = add2av(newuser, add2cp(NULLCP, keyid));
    }
    else {
	cp = add2cp(NULLCP, "Unknown name identifier \"");
	cp = add2cp(cp, id);
	cp = add2cp(cp, "\".\n");
	cp2cb(iocbs->out_errs, cp);
	FREE(cp);
	goto cleanup;
    }

    /* Look for existing user record with same public key */
        
    if ((user = get_tv_user("public-key", pk)) != NULLVP) 
	newuser = merge_user(newuser, user);

    if (tag_user(newuser, "trusted") != NULLVP) {
	cp2cb(iocbs->out_errs, "Will not update a trusted user.\n");
	goto cleanup;
    }
    else {
	if (update_indexfile(NULLVP, newuser) != OK) {
	    cp2cb(iocbs->out_errs, "Unable to update user record.\n");
	    goto cleanup;
	}
    }


    r = OK;

 cleanup:
    
    FREE_AV(newuser);
    FREE_AV(user);
    return r;
}



int pem_savkeydata(iocbs)
struct cbstruct *iocbs;
{
    int r = NOTOK;
    int iscert, iscrl, iskey;
    char *fieldname = NULLCP;
    char *fieldvalue = NULLCP;
    char *prevcrl = NULLCP;
    char *cp = NULLCP;

    /* Check args */

    if (iocbs == NULL_IOCBS || iocbs->in_hdrs == NULL_INCB) 
	goto cleanup;

    while (getheader(iocbs->in_hdrs, &fieldname, &fieldvalue) == OK) {

	iscert = !strcasecmp(fieldname, "certificate");
	iscrl  = !strcasecmp(fieldname, "crl");
	iskey  = !strcasecmp(fieldname, "key");

	if (!iscert && !iscrl && !iskey) {
	    cp = add2cp(NULLCP, "Unknown field name \"");
	    cp = add2cp(cp, fieldname);
	    cp = add2cp(cp, "\"\n");
	    cp2cb (iocbs->out_errs, cp);
	    FREE(cp);
	    goto cleanup;
	}

	if(iskey && save_keyform(fieldvalue, iocbs) != OK) {
	    goto cleanup;
	}

	if (iscert && save_cert(fieldvalue, iocbs) != OK) {
	    goto cleanup;
	}

	if (prevcrl != NULLCP && save_crl(prevcrl, iocbs) != OK) {
	    goto cleanup;
	}

	FREE(prevcrl);

	if (iscrl) {
	    prevcrl = fieldvalue;
	    fieldvalue = NULLCP;
	}
    }

    if (prevcrl != NULLCP && save_crl(prevcrl, iocbs) != OK) 
	goto cleanup;

    r = OK;
    
 cleanup:

    FREE(fieldname);
    FREE(fieldvalue);
    FREE(prevcrl);
	
    return (r);
}
