/*********************************************************************
**
**     File name:               ssh_ident.c
**                                  
**                              Copyright 1997, 1998 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 saves/loads ssh identities
**
**     Author/Date:             Tadayoshi Kohno, 14 December 1997
**
**     Notes:
**	The functions in this file can be used to load and generate
**	RSA identities.
**
**	I plan to provide a conversion program from Tatu Ylonen
**	and F-Secure's identity format.
**
**	I currently encrypted file format version with the RSA
**	key information.  This is probably not a good idea because
**	the format version is a rather "lengthy" string, so I should
**	probably pull it out and have some other method to make
**	sure I decrypt correctly.  Ylonen's file format sets
**	4 bytes and makes sure those are decrypted OK.  I think I will
**	do the same.
**
**	Re-modularize -- there is some duplicate effort with ssh_crypt.c
**
**     Functions:
**	ssh_identity_gen		generates an RSA identity
**	ssh_identity_load		loads an RSA identity
**
**	static ssh_identity_save	saves an RSA identity
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/cia/kohno/libssh/libssh/RCS/ssh_ident.c,v 3.2 1998/03/07 16:32:59 kohno Exp $";
#endif

#include <string.h>
#include <assert.h>

#include "ssh.h"
#include "ssh_debug.h"
#include "ssh_crypt.h"
#include "ssh_ident.h"
#include "ssh_mp_int.h"
#include "ssh_mp_int_dec.h"
#include "ssh_random.h"

#include "ssh_util.h"

#include <rsa.h>

/*
**	des3 encryption/decryption values
*/
static struct des3_values_identity
{
	des_cblock iv_encrypt_1;	/* initialization vertor (enc) */
	des_cblock iv_encrypt_2;	/* 2nd enc initialization vector */
	des_cblock iv_encrypt_3;	/* 3d enc initialization vector */
	des_cblock iv_decrypt_1;	/* init vertor (decrypt) */
	des_cblock iv_decrypt_2;	/* 2nd decrypt init vector */
	des_cblock iv_decrypt_3;	/* 3d decrypt init vector */
	des_key_schedule key_1;		/* First DES key */
	des_key_schedule key_2;		/* Second DES key */
	des_key_schedule key_3;		/* Third DES key */
} DES3ValuesIdentity;

/*
**	save our ssh identity
*/
static int ssh_identity_save(struct ssh_struct * ssh_info, char * priv_file,
	char * pub_file, char * msg, char * passphrase, MP_Int modulus,
	MP_Int private_exp, MP_Int public_exp);

/*********************************************************************
**
**     Function:                ssh_identity_gen
**
**     Purpose:                 generate and save an RSA identity
**
**     Entry (pre) conditions:  priv_file, pub_file are valid files
**
**     Parameters:              ssh_info	information about ssh
**				
**				priv_file	private  filename
**				pub_file	public filename
**				msg		public trailing message
**				passphrase	passphrase to use
**
**				bits		bits for modulus
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error on generation
**						error on saving
**				ssh_errno set
**
**     Side effects:            RSA identity generated and saved in
**				specified file
**
**     Author/Date:             Tadayoshi Kohno, 14 December 1997
**
**     References:                  
**	This routine also makes use of Eric Young's libeay library.
**
**     Notes:
**	I want to double check that my conversion from Eric Young's BN
**	to my MP_Ints are correct.  I'm pretty sure it is. xxx
**
**	In the ssh_identity case, the ssh_info variable is used
**	to determine if and how to display debug information
**
*********************************************************************/

int ssh_identity_gen
(
	struct ssh_struct * ssh_info, 	/* information about ssh */
	char * priv_file,	/* location to store RSA key (private) */
	char * pub_file,	/* location to store RSA key (public) */
	char * msg,		/* message to tail public file */
	char * passphrase,	/* passphrase to decrypt RSA key */
	int bits		/* number of bits to use */
)
{	
	RSA * private_key;	/* our generated private key */

	MP_Int modulus;		/* our modulus */
	MP_Int public_exp;	/* our public exponent */
	MP_Int private_exp;	/* our private exponent */

	/*
	**	Initialize our MP_Ints
	*/
	mp_int_new(&modulus);
	mp_int_new(&public_exp);
	mp_int_new(&private_exp);

	/*
	**	Generate our private key
	*/
	ssh_debugger_new(&(ssh_info->debug_info),
		"calling RSA_generate_key", "ssh_identity_gen");
	private_key = RSA_generate_key(bits, SSH_IDENT_EXP, NULL);

	/*
	**	And convert those private keys to MP_Ints
	*/
	modulus.data_bytes = BN_num_bytes(private_key->n);
	if ((modulus.num
		= (uint8_t *) malloc(sizeof(uint8_t)*(modulus.data_bytes + 1)))
		== NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"Error mallocing memory for modulus",
			"ssh_identity_gen");
		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}

	public_exp.data_bytes = BN_num_bytes(private_key->e);
	if ((public_exp.num
		= (uint8_t *) malloc(sizeof(uint8_t)
			* (public_exp.data_bytes + 1)))
		== NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"Error mallocing memory for public exp",
			"ssh_identity_gen");

		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}

	private_exp.data_bytes = BN_num_bytes(private_key->d);
	if ((private_exp.num
		= (uint8_t *) malloc(sizeof(uint8_t)
			* (private_exp.data_bytes + 1)))
		== NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"Error mallocing memory for private exp",
			"ssh_identity_gen");

		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}

	assert(modulus.data_bytes == BN_bn2bin(private_key->n, modulus.num));
	modulus.nbits = modulus.data_bytes * BITS_PER_BYTE;

	assert(public_exp.data_bytes
		== BN_bn2bin(private_key->e, public_exp.num));
	public_exp.nbits = public_exp.data_bytes * BITS_PER_BYTE;

	assert(private_exp.data_bytes 
		== BN_bn2bin(private_key->d, private_exp.num));
	private_exp.nbits = private_exp.data_bytes * BITS_PER_BYTE;

	/*
	**	Now save our identity
	*/
	if (ssh_identity_save(ssh_info, priv_file, pub_file, msg,
		passphrase, modulus, private_exp, public_exp) == S_BAD)
	{

		free(private_exp.num);
		free(public_exp.num);
		free(modulus.num);

		my_bzero(passphrase, strlen(passphrase));

		return(S_BAD);
	}

	/*
	**	Cleanup
	*/
	my_bzero(passphrase, strlen(passphrase));
	my_bzero(private_exp.num, private_exp.data_bytes);
	my_bzero(public_exp.num, public_exp.data_bytes);
	my_bzero(modulus.num, modulus.data_bytes);

	free(private_exp.num);
	free(public_exp.num);
	free(modulus.num);

	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_identity_load
**
**     Purpose:                 read private/public keys
**
**     Entry (pre) conditions:  identity previously stored
**				MP_Ints initialized
**
**				file at filename is a valid private
**				identity
**
**     Parameters:              ssh_info		info about ssh
**
**				filename		name of file to load
**				passphrase		decryption passphrase
**
**				*public_key_modulus	public modulus
**				*public_key_exponent	public exponent
**				*private_key_modulus	private modulus
**				*private_key_exponent	private exponent
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD			(ssh_errno set)
**
**     Side effects:            users' RSA key returned through paramaters
**
**     Author/Date:             Tadayoshi Kohno, 14 December 1997
**				Tadayoshi Kohno, 23 February 1998
**
**     Notes:
**	In the ssh_identity case, ssh_info is used to determine how
**	to display debug information
**
**	February, 1998.  Modified to support only new-format
**	version (agreed with the University of Colorado's JavaSSH
**	version)
**
*********************************************************************/

int ssh_identity_load
(
	struct ssh_struct * ssh_info,	/* information about ssh config */
	char * filename,		/* file to read from */
	char * passphrase,		/* user's passphrase */
	MP_Int * public_key_modulus,	/* public modulus */
	MP_Int * public_key_exponent,	/* public exponent */
	MP_Int * private_key_modulus,	/* private modulus */
	MP_Int * private_key_exponent	/* private exponent */
)
{
	FILE * private_file;			/* our private identity */

	uint8_t tmp_buf[SSH_MAX_PACKET];	/* temporary buffer */

	uint8_t tmp_buf1[SSH_MAX_PACKET];	/* more temporary buffers */
	uint8_t tmp_buf2[SSH_MAX_PACKET];	/* more temporary buffers */

	char head[SSH_IDENTITY_HEAD_LEN];	/* length of identity head */

	int bytes_read;				/* bytes read from file */
	
	MD5_CTX md5_context;			/* MD5 context for decipher */
	uint8_t md5_digest[MD5_DIGEST_LEN];	/* MD5 digest for decryption */
	int ret_val;				/* decryption return value */

	int offset;				/* offset into tmp_buf */
	int count;				/* number of bytes written */

	if ((private_file = fopen(filename, "rb")) == (FILE *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	
	bytes_read = fread((void *) tmp_buf, sizeof(uint8_t), SSH_MAX_PACKET,
		private_file);
	if (ferror(private_file))
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	
	if (feof(private_file))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"have EOF on identity file", "ssh_identity_load");
	}

	if (bytes_read <= 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}

	/*
	**	notice if we really read a multiple of SSH_UNITS (8)
	**	bytes
	*/
	if (bytes_read % SSH_UNITS)
	{
		ssh_debug_int_new(&(ssh_info->debug_info),
			"didn\'t read multiple of %d bytes",
			SSH_UNITS, "ssh_identity_load");
	} else
	{
		ssh_debug_int_new(&(ssh_info->debug_info),
			"good, read a multiple of %d bytes",
			SSH_UNITS, "ssh_identity_load");
	}

	/* BEGIN */

	/*
	**	Let's create the key for our deciphering
	*/
	MD5_Init(&md5_context);
	MD5_Update(&md5_context, (uint8_t *) passphrase, strlen(passphrase));
	MD5_Final(md5_digest, &md5_context);
	
	/*
	**	Now let's seed our 3des cipher
	*/
	if ((ret_val = des_set_key((C_Block *) md5_digest,
		DES3ValuesIdentity.key_1)) != 0)
	{
		ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
		return(S_BAD);
	}
	if ((ret_val = des_set_key((C_Block *) (md5_digest + DES_KEY_LEN),
		DES3ValuesIdentity.key_2)) != 0)
	{
		ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
		return(S_BAD);
	}
	if ((ret_val = des_set_key((C_Block *) md5_digest,
		DES3ValuesIdentity.key_3)) != 0)
	{
		ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
		return(S_BAD);
	}

	my_bzero(DES3ValuesIdentity.iv_encrypt_1, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_encrypt_2, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_encrypt_3, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_decrypt_1, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_decrypt_2, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_decrypt_3, DES_IV_LEN);

	/*
	**	And decrypt the identity
	*/
	des_ncbc_encrypt((C_Block *)(tmp_buf),
		(C_Block *)(tmp_buf1), (long) bytes_read,
		DES3ValuesIdentity.key_1,
		(C_Block *)DES3ValuesIdentity.iv_decrypt_1, DES_DECRYPT);		
	des_ncbc_encrypt((C_Block *)(tmp_buf1),
		(C_Block *)(tmp_buf2), (long) bytes_read,
		DES3ValuesIdentity.key_2,
		(C_Block *)DES3ValuesIdentity.iv_decrypt_2, DES_ENCRYPT);		
	des_ncbc_encrypt((C_Block *)(tmp_buf2),
		(C_Block *)(tmp_buf), (long) bytes_read,
		DES3ValuesIdentity.key_3,
		(C_Block *)DES3ValuesIdentity.iv_decrypt_3, DES_DECRYPT);

	/*
	**	Now lets read our data from the stream
	*/
	offset = 0;

#ifdef OLD_IDENTITY_FORMAT
	my_bcopy(tmp_buf + offset, head, strlen(SSH_IDENTITY_HEAD));
	offset += strlen(SSH_IDENTITY_HEAD);

	head[strlen(SSH_IDENTITY_HEAD)] = 0;

	/*
	**	First make sure we have a valid identity file (should *really*
	**	check for buffer stuff [am I reading too much!] xxx)
	*/
	if (strcmp(head, SSH_IDENTITY_HEAD))
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_DECRYPT);
		return(S_BAD);
	}
#else	/* OLD_IDENTITY_FORMAT */

	my_bcopy(tmp_buf + offset, head, SSH_IDENTITY_CHECKHEAD_LEN);
	offset += SSH_IDENTITY_CHECKHEAD_LEN;

	/*
	**	check our bytes to make sure they really work out
	*/
	if (head[0] != head[2] || head[1] != head[3])
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_DECRYPT);
		return(S_BAD);
	}

#endif	/* OLD_IDENTITY_FORMAT */

	ssh_debugger_new(&(ssh_info->debug_info),
		"correctly deciphered identity file",
		"ssh_identity_load");

	/*
	**	Now let's read the data
	*/
	count = mp_int_from_stream(tmp_buf + offset, public_key_modulus);
	count = mp_int_from_stream(tmp_buf + offset, private_key_modulus);
	if (count <= 0)
	{
		/* should also check not overflowing auth_offset */
		ssh_debugger_new(&(ssh_info->debug_info),
			"error reading modulus", "ssh_identity_load");

		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	offset = offset + count;

	count = mp_int_from_stream(tmp_buf + offset, private_key_exponent);
	if (count <= 0)
	{
		/* should also check not overflowing auth_offset */
		ssh_debugger_new(&(ssh_info->debug_info),
			"error reading private exp", "ssh_identity_load");

		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	offset = offset + count;
	
	count = mp_int_from_stream(tmp_buf + offset, public_key_exponent);
	if (count <= 0)
	{
		/* should also check not overflowing auth_offset */
		ssh_debugger_new(&(ssh_info->debug_info),
			"error reading public exp", "ssh_identity_load");

		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	offset = offset + count;

	ssh_debugger_new(&(ssh_info->debug_info),
		"identity read", "ssh_identity_load");

	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_identity_save
**
**     Purpose:                 save an RSA identity
**
**     Entry (pre) conditions:  n, d, e valid, filenames valid
**
**     Parameters:              ssh_info	info about ssh config
**
**				priv_file	private file
**				pub_file	public file
**				msg		message to end pub_file
**				passphrase	passphrase to encrypt by
**
**				modulus		RSA modulus		(n)
**				private_exp	private exponent	(d)
**				public_exp	public exponent		(e)
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		(ssh_errno set)
**
**     Side effects:            RSA identity encrypted and stored in
**				the specified file.
**
**     Author/Date:             Tadayoshi Kohno, 14 December 1997
**				Tadayoshi Kohno, 23 February 1998
**					(see "Notes")
**                              Tadayoshi Kohno, 14 March 1998
**					(now really uses random bytes)
**
**     Notes:
**
**	February, 1998.  Modified to support only the new-format
**	version (agreed with the University of Colorado's JavaSSH
**	version)
**
*********************************************************************/

static int ssh_identity_save
(
	struct ssh_struct * ssh_info,	/* information about ssh config */
	char * priv_file,	/* private filename to save to */
	char * pub_file,	/* public filename to save to */
	char * msg,		/* message to end public file */
	char * passphrase,	/* passphrase for the keys */
	MP_Int modulus,		/* modulus */
	MP_Int private_exp,	/* d, private exp */
	MP_Int public_exp	/* e, public exp */
)
{
	FILE * private_file;	/* private identity file */
	FILE * public_file;	/* public identity file */

	uint8_t tmp_buf[SSH_MAX_PACKET];	/* temporary buffer */

	uint8_t tmp_buf1[SSH_MAX_PACKET];	/* more temporary buffers */
	uint8_t tmp_buf2[SSH_MAX_PACKET];	/* more temporary buffers */
	
	MD5_CTX md5_context;			/* MD5 context for decipher */
	uint8_t md5_digest[MD5_DIGEST_LEN];	/* MD5 digest for decryption */
	int ret_val;				/* decryption return value */

	int offset;				/* offset into tmp_buf */
	int count;				/* number of bytes written */

	uint8_t rand_bytes[2];			/* two random bytes */

	/*
	**	Make sure our passphrase is long enough (some companies may
	**	want to make this rather large)
	*/
	if (strlen(passphrase) < SSH_PASSPHRASE_MIN_LEN)
	{
		ssh_errno_set(SSH_ERRNO_PHRASE_SHORT);
		return(S_BAD);
	}

	/*
	**	Open our identity file
	*/
	if ((private_file = fopen(priv_file, "wb")) == (FILE *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}

	ssh_debugger_new(&(ssh_info->debug_info),
		"private file open", "ssh_identity_save");

	offset = 0;

#ifdef OLD_IDENTITY_FORMAT

	/*
	**	write an identity header to the tmp_buf to be enciphered
	*/
	my_bcopy(SSH_IDENTITY_HEAD, tmp_buf + offset,
		strlen(SSH_IDENTITY_HEAD));
	offset += strlen(SSH_IDENTITY_HEAD);

#else	/* OLD_IDENTITY_FORMAT */
	/*
	**	use four checkbytes to ensure correct decipherment
	*/
	RAND_bytes(rand_bytes, 2);
	tmp_buf[offset] = tmp_buf[offset + 2] = rand_bytes[0];
	tmp_buf[offset + 1] = tmp_buf[offset + 3] = rand_bytes[1];

	offset += SSH_IDENTITY_CHECKHEAD_LEN;

#endif	/* OLD_IDENTITY_FORMAT */

	/*
	**	Write modules and exponents in MP_Int stream notation.
	*/
	if ((count = mp_int_to_stream((uint8_t *) (tmp_buf + offset),
		modulus, SSH_MAX_PACKET - offset)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	offset += count;
	if ((count = mp_int_to_stream((uint8_t *) (tmp_buf + offset),
		private_exp, SSH_MAX_PACKET - offset)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	offset += count;
	if ((count = mp_int_to_stream((uint8_t *) (tmp_buf + offset),
		public_exp, SSH_MAX_PACKET - offset)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	offset += count;

#ifndef OLD_IDENTITY_FORMAT
	/* these bytes should really be random.  xxx -- call a function */
	/* like we would for a real packet xxx */

	/*
	**	add padding to ensure tmp_buf is a multiple of
	**	SSH_UNITS (8) bytes
	*/
	offset += SSH_UNITS - (offset % SSH_UNITS);

#endif	/* OLD_IDENTITY_FORMAT */

	ssh_debug_int_new(&(ssh_info->debug_info),
		"length of offset to tmp_buf: %d", offset,
		"ssh_identity_save");

	assert(offset < SSH_MAX_PACKET);

	ssh_debugger_new(&(ssh_info->debug_info),
		"tmp_buf created", "ssh_identity_save");

	/*
	**	Let's create the key for our decryption
	*/
	MD5_Init(&md5_context);
	MD5_Update(&md5_context, (uint8_t *) passphrase, strlen(passphrase));
	MD5_Final(md5_digest, &md5_context);
	
	/*
	**	Now let's seed our 3des cipher
	*/
	if ((ret_val = des_set_key((C_Block *) md5_digest,
		DES3ValuesIdentity.key_1)) != 0)
	{
		ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
		return(S_BAD);
	}
	if ((ret_val = des_set_key((C_Block *) (md5_digest + DES_KEY_LEN),
		DES3ValuesIdentity.key_2)) != 0)
	{
		ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
		return(S_BAD);
	}
	if ((ret_val = des_set_key((C_Block *) md5_digest,
		DES3ValuesIdentity.key_3)) != 0)
	{
		ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
		return(S_BAD);
	}

	my_bzero(DES3ValuesIdentity.iv_encrypt_1, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_encrypt_2, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_encrypt_3, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_decrypt_1, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_decrypt_2, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_decrypt_3, DES_IV_LEN);

	/*
	**	And encrypt the identity
	*/
	des_ncbc_encrypt((C_Block *)(tmp_buf),
		(C_Block *)(tmp_buf1), (long) offset,
		DES3ValuesIdentity.key_3,
		(C_Block *)DES3ValuesIdentity.iv_encrypt_1, DES_ENCRYPT);		
	des_ncbc_encrypt((C_Block *)(tmp_buf1),
		(C_Block *)(tmp_buf2), (long) offset,
		DES3ValuesIdentity.key_2,
		(C_Block *)DES3ValuesIdentity.iv_encrypt_2, DES_DECRYPT);		
	des_ncbc_encrypt((C_Block *)(tmp_buf2),
		(C_Block *)(tmp_buf), (long) offset,
		DES3ValuesIdentity.key_1,
		(C_Block *)DES3ValuesIdentity.iv_encrypt_3, DES_ENCRYPT);

	/*
	**	and now write the encrypted file
	*/
	if (fwrite(tmp_buf, sizeof(uint8_t), offset, private_file)
		< (size_t)offset)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}

	my_bzero(tmp_buf, SSH_MAX_PACKET);
	my_bzero(tmp_buf1, SSH_MAX_PACKET);
	my_bzero(tmp_buf2, SSH_MAX_PACKET);
	fclose(private_file);

	ssh_debugger_new(&(ssh_info->debug_info),
		"private file written", "ssh_identity_save");

	/*
	**	now create public file
	*/

	if ((public_file = fopen((char *) pub_file, "w")) == (FILE *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}

	fprintf(public_file, "%d ", modulus.nbits);

	/* xxx note here that tmp_buf is an SSH_MAX_PACKET, but mp_int_to_dec
	   xxx takes string of size MP_INT_DEC_LEN -- here we're bigger
	   xxx but nice to make changes and make sure
	*/
	(void) mp_int_to_dec((char *) tmp_buf, public_exp);
	fprintf(public_file, "%s ", tmp_buf);

	(void) mp_int_to_dec((char *) tmp_buf, modulus);
	fprintf(public_file, "%s ", tmp_buf);

	fprintf(public_file, "%s\n", msg);

	fclose(public_file);

	ssh_debugger_new(&(ssh_info->debug_info),
		"done writing identity file", "ssh_identity_save");

	return(S_GOOD);
}

