/*********************************************************************
**
**     File name:               ssh_mp_int_dec.c
**                                  
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 handles printing an MP_Int in
**				decimal format
**
**     Author/Date:             Tadayoshi Kohno, 14 December 1997
**
**     Notes:
**	The functions in this file are of relatively limited use.  They're
**	primarily designed to save "uniform" (across different versions
**	of ssh, of which I only really know about Tatu Ylonen's)
**	RSA public keys (identity.pub).
**
**	Our interal data type is the MP_Dec, which is a reverse-order
**	decimal number of (arbitrary [up to MP_INT_DEC_LEN] length... I
**	would like to make this dynamic but haven't had a chance to yet).
**
**     Functions:
**	mp_int_to_dec		converts an MP_Int to decimal string
**
**	static mp_dec_add	adds two MP_Dec numbers in decimal
**	static mp_dec_set_1	initializes an MP_Dec to 1
**	static mp_dec_set_0	initializes an MP_Dec to 0
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/cia/kohno/libssh/libssh/RCS/ssh_mp_int_dec.c,v 1.1 1998/03/07 16:32:59 kohno Exp $";
#endif

#include <string.h>
#include <stdio.h>

#include <assert.h>

#include "ssh_mp_int_dec.h"
#include "ssh_debug.h"

#include "ssh_util.h"

/*
**	Define our local decimal datatype as discussed in the "Notes" section.
*/
typedef struct mp_int_dec
{
	char num_str[MP_INT_DEC_LEN];	/* our str of (rev) decimal digits */
	int data_bytes;			/* length of num_str (todo: dynamic) */
} MP_Dec;

/*
**	Define a mask for each bit in a byte (index 1..8)
*/
static int BitMask[BITS_PER_BYTE + 1] = {0, 1, 2, 4, 8, 16, 32, 64, 128};

/*
**	And some local functions
*/
static int mp_dec_add(MP_Dec num1, MP_Dec num2, MP_Dec * sum);
static int mp_dec_set_1(MP_Dec * num);
static int mp_dec_set_0(MP_Dec * num);

/*********************************************************************
**
**     Function:                mp_int_to_dec
**
**     Purpose:                 converts an MP_Int to a printable
**				decimal string
**
**     Entry (pre) conditions:  num a valid MP_Int
**				*str valid ptr of length MP_INT_DEC_LEN
**
**     Parameters:              *str	output string
**
**				num	input MP_Int
**
**     Return value:            0
**
**     Error codes:             -1	output string length exceeds max
**
**     Side effects:            *str is a printable string (int decimal)
**				for num
**				*str is NULL terminated
**
**     Author/Date:             Tadayoshi Kohno, 14 December 1997
**
**     Notes:
**
*********************************************************************/

int mp_int_to_dec
(
	char * str,			/* output string */
	MP_Int num			/* MP_Int to print to string */
)
{
	int byte_index;			/* index into the bytes of num */
	int bit_index;			/* index into the bits of num */
	int internal_bit_index;		/* byte-internal bit index */

	MP_Dec dec_num;			/* decimal version of num */
	char tmp_str[MP_INT_DEC_LEN];	/* temporary string to store output */
	int dec_index;			/* index into the decimal integer */

	MP_Dec running_power;	/* running power of two for bits add */

	/*
	**	Initialize our MP_Decs
	*/
	mp_dec_set_0(&dec_num);
	mp_dec_set_1(&running_power);

	/*
	**	Start looking from the least significant digit first
	*/
	bit_index = 1;
	assert(num.data_bytes == (num.nbits + 7) / 8);

	for (byte_index = num.data_bytes - 1; byte_index >=0; byte_index--)
	{
		/*
		**	look at each bit of the byte
		*/
		for (internal_bit_index = 1;
			internal_bit_index <= BITS_PER_BYTE;
			internal_bit_index++)
		{
			if (num.num[byte_index] & BitMask[internal_bit_index])
			{
				(void) mp_dec_add(dec_num, running_power,
					&dec_num);
			}
			bit_index++;
			if (mp_dec_add(running_power, running_power,
				&running_power))
			{
					return(-1);
			}
		}
	}

	/*
	**	if final string is to big, return an error
	*/
	if (dec_num.data_bytes >= MP_INT_DEC_LEN - 1)
	{
		return(-1);
	}

	/*
	**	And now lets format the string in a printable manner
	*/
	strcpy(str, "");

	for (dec_index = dec_num.data_bytes - 1; dec_index >= 0; dec_index--)
	{
		sprintf(tmp_str, "%d", dec_num.num_str[dec_index]);
		strcat(str, tmp_str);
	}

	return(0);
}

/*********************************************************************
**
**     Function:                mp_dec_add
**
**     Purpose:                 adds to MP_Dec numbers
**
**     Entry (pre) conditions:  num1, num2 valid MP_Decs,
**				*sum ptr valid
**
**     Parameters:              num1		first number to add
**				num2		second number to add
**
**				*sum		result
**
**     Return value:            0
**
**     Error codes:             1		sum too large to hold
**
**     Side effects:            *sum = num1 + num2
**
**     Author/Date:             Tadayoshi Kohno, 14 December 1997
**
*********************************************************************/

static int mp_dec_add
(
	MP_Dec num1,		/* first number to add */
	MP_Dec num2,		/* second number to add */
	MP_Dec * sum		/* sum of num1 and num2 */
)
{
	int sum_index;		/* index to our sum */

	int first_stop;		/* min number of digits across nums */

	/*
	**	First lets zero out anything in sum that might be remaining
	*/
	my_bzero(sum->num_str, MP_INT_DEC_LEN);

	first_stop = (num1.data_bytes < num2.data_bytes)
		? num1.data_bytes
		: num2.data_bytes;

	/*
	**	Now add num1 and num2 on common ground
	*/
	for(sum_index = 0; sum_index < first_stop; sum_index++)
	{
		sum->num_str[sum_index] += num1.num_str[sum_index]
			+ num2.num_str[sum_index];
		sum->num_str[sum_index + 1] += sum->num_str[sum_index] / 10;
		sum->num_str[sum_index] = sum->num_str[sum_index] % 10;
	}

	/*
	**	Now either num1 or num2 has more bytes of data
	**	(or they were the same)
	*/
	if (num2.data_bytes > first_stop)
	{
		for ( ; sum_index < num2.data_bytes; sum_index++)
		{
			sum->num_str[sum_index] += num2.num_str[sum_index];

			sum->num_str[sum_index + 1] +=
				sum->num_str[sum_index] / 10;
			sum->num_str[sum_index] = sum->num_str[sum_index] % 10;
		}
	} else if (num1.data_bytes > first_stop)
	{
		for ( ; sum_index < num1.data_bytes; sum_index++)
		{
			sum->num_str[sum_index] += num1.num_str[sum_index];

			sum->num_str[sum_index + 1] +=
				sum->num_str[sum_index] / 10;
			sum->num_str[sum_index] = sum->num_str[sum_index] % 10;
		}
	}

	if (sum->num_str[sum_index] == 0)
	{
		/* no carry on last add */
		sum->data_bytes = sum_index;
	} else
	{
		/* had a carry on the last add */
		sum->data_bytes = sum_index + 1;
	}

	/*
	**	make sure sum hasn't gotten too big
	*/
	if (sum->data_bytes >= MP_INT_DEC_LEN - 1)
	{
		return(-1);
	}

	return(0);
}

/*********************************************************************
**
**     Function:                mp_dec_set_{1,0}
**
**     Purpose:                 initializes an MP_Dec to either 0 or 1
**
**     Entry (pre) conditions:  *num valid pointer
**
**     Parameters:              *num	number to initialize
**
**     Return value:            0
**
**     Error codes:             1	not used
**
**     Side effects:            *num is initialized and can now be used
**
**     Author/Date:             Tadayoshi Kohno, 14 December 1997
**
*********************************************************************/

static int mp_dec_set_1
(
	MP_Dec * num		/* number to set to 1 */
)
{
	my_bzero(num->num_str, MP_INT_DEC_LEN);

	num->num_str[0] = 1;
	num->data_bytes = 1;
	return(0);
}

static int mp_dec_set_0
(
	MP_Dec * num		/* number to set to 0 */
)
{
	my_bzero(num->num_str, MP_INT_DEC_LEN);

	num->data_bytes = 1;
	return(0);
}

