/*********************************************************************
**
**     File name:               ssh_mp_stream.c
**                                  
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 used when dealing with strings
**				in SSHv1.5's data type.
**
**     Author/Date:             Tadayoshi Kohno, 23 November 1997
**
**     References:              draft-ylonen-ssh-protocol-00.txt
**
**     Notes:
**	The functions in this file are to be used when dealing with
**	the SSH Protocol's version of a string (v1.5).  Please see
**	T. Ylonen's Internet Draft draft-ylonen-ssh-protocol-00.txt
**	for a description of v1.5 of the protocol.
**
**     Functions:
**	string_to_stream	puts the string onto a stream in v1.5 form
**	string_to_stream_n	puts "n" bytes of string into stream (v1.5)
**	string_from_stream	takes a string from a stream in v1.5 form
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/cia/kohno/libssh/libssh/RCS/ssh_mp_stream.c,v 1.1 1998/03/07 16:32:59 kohno Exp $";
#endif

#include <string.h>

#include "ssh_mp_stream.h"

/*********************************************************************
**
**     Function:                string_to_stream
**
**     Purpose:                 convert a string into a stream
**
**     Entry (pre) conditions:  str and stream valid (have enough
**				memory).  str NULL terminated.
**
**     Parameters:              stream		stream to write to
**				str		string to read from
**
**     Return value:            total number of bytes written to stream
**
**     Error codes:             none, but we need at least 4 bytes for
**				the draft standard (so return value should
**				be at least 4)
**
**     Side effects:            str written to stream as specified by
**				SSHv1.5 protocol.
**
**     Author/Date:             Tadayoshi Kohno, 23 November 1997
**
**     Notes:
**	What if str has a '\0' in it.  maybe make a string_to_stream_n().
**
*********************************************************************/

int string_to_stream
(
	uint8_t * stream,	/* stream to write to */
	uint8_t * str		/* string to convert */
)
{
	uint32_t length;	/* length of the string */
	uint32_t nbo_len;	/* length in network byte order */

	length = strlen((char *)str);
	nbo_len = htonl(length);

	my_bcopy((void *) &nbo_len, (void *) stream, sizeof(nbo_len));

	my_bcopy((void *) str, (void *) (stream + sizeof(length)),
		length);

	return(length + sizeof(length));
}

/*********************************************************************
**
**     Function:                string_to_stream_n
**
**     Purpose:                 take n bytes of string and put it
**				into the stream
**
**     Entry (pre) conditions:  str, stream valid for length bytes
**
**     Parameters:              stream		output stream
**				str		input string
**				length		bytes of str to output
**
**				stream_size	size of the stream
**
**     Return value:            length of new data in stream
**
**     Error codes:             -1		stream_size too small
**
**     Side effects:            stream now points to <return value>
**				bytes that represent str in mp_stream
**				notation
**
**     Author/Date:             Tadayoshi Kohno, 28 November 1997
**
**     Notes:
**	This is useful if we're not certain a string will be NULL terminated
**	or if the string will have NULLs in it (string_to_stream uses
**	strlen() ).
**
*********************************************************************/

int string_to_stream_n
(
	uint8_t * stream,	/* stream to write to */
	uint8_t * str,		/* string to convert */
	uint32_t length,	/* length of string to convert */

	int stream_size		/* length of *stream (so don't overflow) */
)
{
	uint32_t nbo_len;	/* length in network byte order */

	/*
	**	if we would overflow the stream, return an error
	**	(is this too slow xxx)
	*/
	if (length + sizeof(length) >= (uint32_t) stream_size)
	{
		return(-1);
	}

	nbo_len = htonl(length);
	my_bcopy((void *) &nbo_len, (void *) stream, sizeof(nbo_len));

	my_bcopy((void *) str, (void *) (stream + sizeof(length)),
		length);

	return(length + sizeof(length));
}

/*********************************************************************
**
**     Function:                string_from_stream
**
**     Purpose:                 create a string from a stream
**
**     Entry (pre) conditions:  memory for string exists.
**
**     Parameters:              stream		stream to read from
**				str		output string
**
**				str_size	size of stream
**
**     Return value:            length of str
**
**     Error codes:             -1		str_size too small
**
**     Side effects:            str contains string represented by
**				stream
**
**				str not automatically NULL terminated.
**
**     Author/Date:             Tadayoshi Kohno, 23 November 1997
**
**     Notes:
**
*********************************************************************/

int string_from_stream
(
	uint8_t * stream,	/* stream to read from */
	uint8_t * str,		/* resulting string */

	int str_size		/* length of *str */
)
{
	uint32_t length;	/* length of string */

	/*
	**	Now lets computer then number of bits
	*/
	my_bcopy((void *) stream, (void *) &length, sizeof(length));
	length = ntohl(length);

	/*
	**	make sure there's enough space to copy this string
	*/
	if (length >= (uint32_t) str_size)
	{
		return(-1);
	}

	my_bcopy((void *) (stream + sizeof(length)), (void *) str,
		length);

	return(length);
}

