/*  pgpsendmail.c

    Source file for  PGPsendmail  (wrapper to sendmail).

    Copyright (C) 1994-1997  Richard Gooch

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Richard Gooch may be reached by email at  rgooch@atnf.csiro.au
    The postal address is:
      Richard Gooch, c/o ATNF, P. O. Box 76, Epping, N.S.W., 2121, Australia.
*/

/*  This programme intercepts messages sent by user mail agents to the
    sendmail daemon and checks to see if messages can be encrypted using the
    recipient's PGP public keys.


    Written by      Richard Gooch   3-APR-1994

    Updated by      Richard Gooch   9-APR-1994

    Updated by      Richard Gooch   27-MAY-1994: Created  freshen_keylist  .

    Updated by      Richard Gooch   30-MAY-1994: Created  check_recipients  and
  check_for_header_flags  .

    Updated by      Richard Gooch   2-JUN-1994: Included full message body on
  bounce.

    Updated by      Richard Gooch   5-JUN-1994: Added signing facility.

    Updated by      Richard Gooch   10-JUN-1994: Changed from using  stricmp
  to  st_icmp  ,since some platforms don't have  stricmp  .

    Updated by      Richard Gooch   12-JUN-1994: Fixed bug in  main  where a
  constant string was passed to  process_addresses  .

    Updated by      Richard Gooch   14-JUN-1994: Passed PID of real  sendmail
  to  check_for_header_flags  so that it can abort the sending process.

    Updated by      Richard Gooch   17-JUN-1994: Added warning if root runs
  with PGPPATH set and added debugging facility.

    Updated by      Richard Gooch   18-JUN-1994: Added startup test for
  $PGPPATH/ranseed.bin  .

    Updated by      Richard Gooch   26-JUN-1994: Added more debug logging
  output.

    Updated by      Richard Gooch   28-JUN-1994: Added more debug logging
  output in  read_line  .

    Updated by      Richard Gooch   30-JUN-1994: Made "GLOBAL" option the first
  set of flags to be processed.

    Updated by      Richard Gooch   5-JUL-1994: Changed to use of  m_copy  .

    Updated by      Richard Gooch   14-JUL-1994: Checked response flag that
  PGPdaemon now returns.

    Updated by      Richard Gooch   14-JUL-1994: Added more debug messages at
  the end and discarded  /tmp/PGPsendmail.err.<PID>  files even if receipt is
  discarded.

    Updated by      Richard Gooch   20-JUL-1994: Added ability to sign all
  outgoing messages (even those going out in plaintext).

    Updated by      Richard Gooch   21-JUL-1994: Added declarations in
  transmit_plaintext  for  errno  and  sys_errlist  .

    Updated by      Richard Gooch   15-AUG-1994: Added  add_key  parameter to
  transmit_plaintext  .

    Updated by      Richard Gooch   22-AUG-1994: Automatically execute real
  sendmail  if running as root (realUID = 0), and trap for invocation as
  mailq  and  newaliases  .

    Updated by      Richard Gooch  23-AUG-1994: Allow continuation of address
  lines with trailing ','.

    Updated by      Richard Gooch   21-NOV-1994: Supported  LOGNAME
  environment variable.

    Updated by      Richard Gooch   3-DEC-1994: Took account of changes to
  connect_to_pgpdaemon  .

    Updated by      Richard Gooch   6-DEC-1994: Wrote message in debug file
  if no $PGPPATH/randseed.bin file exists.

    Updated by      Richard Gooch   13-DEC-1994: Added support for address
  format:  "Real Name"@email.address  in  extract_address  .

    Updated by      Richard Gooch   17-DEC-1994: Fixed silly bug where username
  was not set if no debug file existed.

    Updated by      Richard Gooch   25-SEP-1997: Used new ERRSTRING macro.

    Updated by      Richard Gooch   27-SEP-1997: Used new general-purpose
  address extraction routines.

    Updated by      Richard Gooch   2-OCT-1997: Use paths.h instead of my own
  hardcoded values.

    Updated by      Richard Gooch   3-OCT-1997: Open config file only once.
  Add +language=en option to PGP.

    Last updated by Richard Gooch   4-OCT-1997: Added "HEADER_TAIL_FILE" and
  "BODY_HEAD_FILE" options. Attempt to wake PGPdaemon when message sent.


*/
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <paths.h>
#include <pgpsendmail.h>
#include <version.h>

#ifdef TEST1
#  undef VERSION
#  define VERSION "private-test1"
#endif

#define SEPARATOR_LENGTH 70


/*  Private data  */
static stringlist_type pgp_args = {NULL, 0, 0};
static char *pgppath = NULL;
static FILE *debug_fp = NULL;
static char *username = NULL;
static char real_sendmail[STRING_LENGTH];


STATIC_FUNCTION (flag process_addresses,
		 (decode_type *info, FILE *config_fp,
		  option_type *options, FILE *debug_fp) );
STATIC_FUNCTION (void return_badnews, (char *message) );
STATIC_FUNCTION (char *extract_address, (char *str, char **rest) );
STATIC_FUNCTION (flag check_for_header_flags,
		 (char *line, option_type *options, int sendmail_pid,
		  FILE *debug_fp) );
STATIC_FUNCTION (void sig_handler, () );
STATIC_FUNCTION (void transmit_plaintext, 
		 (flag sign, FILE *debug_fp, int sendmail_pid,
		  flag no_advertising, flag add_key, CONST char *pgppath) );
STATIC_FUNCTION (void include_file, (FILE *config_fp, FILE *debug_fp,
				     CONST char *keyword) );
STATIC_FUNCTION (void wake_pgpdaemon, (CONST char *pgppath, FILE *debug_fp) );


void main (argc, argv, envp)
int argc;
char *argv[];
char *envp[];
{
    int tmp;

    flag bool_val;
    flag eof;
    option_type options;
    flag header = TRUE;
    flag quoted_printables = FALSE;
    flag continuation_line = FALSE;
    flag dummy;
    int count;
    int ch;
    int tmpfd;
    int sendmail_in_fd, sendmail_pid;
    int out_fd, err_fd;
    int pgp_in_fd, pgp_pid;
    int status;
    int to_fd, from_fd;
    struct stat statbuf;
    decode_type info;
    char *bad_recipient;
    char *ptr1, *ptr2;
    FILE *config_fp;
    char txt[LINE_LENGTH];
    extern FILE *debug_fp;
    extern char *pgppath;
    extern char *username;
    extern char *sys_errlist[];

    if (argv[0] == NULL)
    {
	fprintf (stderr, "argv[0] is NULL!\n");
	exit (RV_UNDEF_ERROR);
    }
    m_clear ( (char *) &info, sizeof info );
    info.scan_to = TRUE;
    info.scan_cc = TRUE;
    info.scan_bcc = TRUE;
    sprintf (real_sendmail, "%s.real", _PATH_SENDMAIL);
    if (access (real_sendmail, X_OK) != 0)
    {
	fprintf (stderr, "File: \"%s\" is not executable\t%s\n",
		 real_sendmail, ERRSTRING);
	exit (RV_SYS_ERROR);
    }
    /*  Try to trap if real_sendmail references PGPsendmail. I hope this works.
	I only do this to protect people from themselves: if they *read* the
	documentation, this wouldn't be necessary. Gosh, I'm such a nice
	person ;-)  */
    ptr1 = strrchr (real_sendmail, '/');
    ptr1 = (ptr1 == NULL) ? real_sendmail : ptr1 + 1;
    ptr2 = strrchr (argv[0], '/');
    ptr2 = (ptr2 == NULL) ? argv[0] : ptr2 + 1;
    if (strcmp (ptr1, ptr2) == 0)
    {
	fprintf (stderr,
		 "PGPsendmail appears to have been installed as: \"%s\"\n",
		 real_sendmail);
	fprintf (stderr, "Someone didn't read the installation notes...\n");
	exit (RV_UNDEF_ERROR);
    }
    if (strcmp (ptr2, "sendmail") != 0)
    {
	sleep (1);
	execve (real_sendmail, argv, envp);
	exit (1);
    }
    if ( ( pgppath = getenv ("PGPPATH") ) == NULL )
    {
	/*  Delay 1 second in case the installer didn't read the instructions
	    and real_sendmail actually refers back to me. Without this delay,
	    the machine will thrash, continuously exec'ing real_sendmail.  */
	sleep (1);
	execve (real_sendmail, argv, envp);
	exit (1);
    }
    if (getuid () == 0)
    {
	/*  Hm. root shouldn't be using this.  */
#ifdef DISALLOW_ROOT
	sleep (1);
	execve (real_sendmail, argv, envp);
	exit (1);
#else
	fprintf (stderr,"It is not recommended for root to run PGPsendmail\n");
	fprintf (stderr, "with the environment variable PGPPATH set\n");
	fprintf (stderr,
		 "Sleeping 15 seconds to give you a chance to kill me\n");
	sleep (15);
#endif
    }
    sprintf (txt, "%s/PGPsendmail.debug", pgppath);
    if (access (txt, F_OK) == 0)
    {
	/*  Turn on verbose debugging  */
	sprintf ( txt, "/tmp/PGPsendmail.debug.%d", getuid () );
	if ( ( debug_fp = fopen (txt, "w") ) == NULL )
	{
	    fprintf (stderr, "Error opening debugging file: \"%s\"\t%s\n",
		     txt, ERRSTRING);
	    exit (RV_CANNOT_OPEN);
	}
	fprintf (debug_fp, "Debug file for PGPsendmail %s opened\n", VERSION);
	fflush (debug_fp);
    }
    else
    {
	debug_fp = NULL;
    }
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Arguments to PGPsendmail:  ");
	for (count = 0; count < argc; ++count)
	{
	    fprintf (debug_fp, "\"%s\"  ", argv[count]);
	}
	fprintf (debug_fp, "\n");
#ifdef NeXT
	fprintf (debug_fp, "Environment:\n");
	for (count = 0; envp[count] != NULL; ++count)
	{
	    fprintf (debug_fp, "  \"%s\"\n", envp[count]);
	}
#endif
	fflush (debug_fp);
    }
    if ( ( username = getenv ("USER") ) == NULL )
    {
	if ( ( username = getenv ("LOGNAME") ) == NULL )
	{
	    if (debug_fp != NULL)
	    {
		fprintf (debug_fp,
			 "No USER or LOGNAME environment variable!\n");
		fflush (debug_fp);
	    }
	    exit (RV_UNDEF_ERROR);
	}
    }
    signal (SIGPIPE, sig_handler);
    options.secure = FALSE;
    options.insecure = FALSE;
    options.receipt = TRUE;
    options.add_key = FALSE;
    options.no_advertising = FALSE;
    options.sign = FALSE;
    options.global_done = FALSE;
    sprintf ( txt, "/tmp/PGPsendmail.err.%d", getpid () );
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Opening file: \"%s\"\n", txt);
	fflush (debug_fp);
    }
    if ( ( tmpfd = open (txt, O_CREAT | O_TRUNC | O_WRONLY, 0644) )
	< 0 ) exit (1);
    if (dup2 (tmpfd, 2) == -1) exit (1);
    fprintf (stderr, "To: %s\nSubject: PGPsendmail response\n\n", username);
    fprintf (stderr, "PGPsendmail response message:\n\n");
    /*  Sanity checks for files the user should have  */
    sprintf (txt, "%s/randseed.bin", pgppath);
    if (access (txt, R_OK) != 0)
    {
	fprintf (stderr, "Could not find file: \"%s\"\t%s\n", txt, ERRSTRING);
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp,
		     "Could not find file: \"%s\"\t%s\nPGP would hang\n",
		     txt, ERRSTRING);
	    fflush (debug_fp);
	}
	return_badnews ("PGP would hang");
	exit (RV_SYS_ERROR);
    }
    /*  Open the configuration file only once (said with a French accent)  */
    sprintf (txt, "%s/PGPsendmail.config", pgppath);
    config_fp = fopen (txt, "r");
    /*  Process "GLOBAL" option in config file  */
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Processing GLOBAL option\n");
	fflush (debug_fp);
    }
    translate_recipient (NULL, config_fp, &options, debug_fp, NULL);
    /*  Check input descriptor  */
    if (fstat (INPUT_FD, &statbuf) != 0)
    {
	fprintf (stderr, "Error getting stats on  stdin\t%s\n", ERRSTRING);
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "Error getting stats on  stdin\t%s\n",
		     ERRSTRING);
	    fflush (debug_fp);
	}
	exit (RV_SYS_ERROR);
    }
    if (debug_fp != NULL)
    {
	fprintf (debug_fp,
		 "stdin  has: %d bytes with: %d hard links  mode: %o  owner: %d\n",
		 statbuf.st_size, statbuf.st_nlink, statbuf.st_mode,
		 statbuf.st_uid);
	fflush (debug_fp);
    }
    if ( !S_ISFIFO (statbuf.st_mode) && !S_ISREG (statbuf.st_mode) )
    {
	fprintf (stderr, "stdin  is not a pipe/FIFO, type is: %o\n",
		 statbuf.st_mode & S_IFMT);
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp,
		     "stdin  is not a pipe/FIFO, type is: %o\n",
		     statbuf.st_mode & S_IFMT);
	    fflush (debug_fp);
	}
	/*exit (RV_UNDEF_ERROR);*/
    }
    else
    {
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "stdin  is a %s: good\n",
		     S_ISFIFO (statbuf.st_mode) ? "pipe/FIFO" :
		     "regular file");
	    fflush (debug_fp);
	}
    }
#ifdef TEST1
    if (debug_fp != NULL)
    {
	fprintf ( debug_fp, "old_off: %ld\n",
		  lseek (INPUT_FD, 0, SEEK_SET) );
	fprintf (debug_fp, "Reading one line from header...\n");
	fflush (debug_fp);
    }
    if ( ( tmp = read (INPUT_FD, txt, statbuf.st_size) ) < statbuf.st_size )
    {
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "Error reading: got: %d bytes\t%s\n",
		     tmp, ERRSTRING);
	    fflush (debug_fp);
	}
	exit (RV_BAD_DATA);
    }
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Header: \"%s\"\n", txt);
	fflush (debug_fp);
    }
    exit (RV_OK);
#endif  /*  TEST1  */
    /*  Start real  sendmail  */
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Starting real sendmail: \"%s\"\n", real_sendmail);
	fflush (debug_fp);
    }
    sendmail_in_fd = -1;
    out_fd = OUTPUT_FD;
    err_fd = ERROR_FD;
    /*  Prevent fork bombs bringing the system to it's knees by waiting a bit.
	Fork bombs occur when real_sendmail references PGPsendmail
	Hopefully this is actually trapped earlier: but I like suspenders *and*
	belt.  */
    sleep (1);
    if ( ( sendmail_pid = spawn_job (real_sendmail, argv, &sendmail_in_fd,
				     &out_fd, &err_fd) ) < 0 )
    {
	return_badnews ("could not spawn real sendmail");
	exit (RV_UNDEF_ERROR);
    }
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Freshening keylist if not already fresh\n");
	fflush (debug_fp);
    }
    if ( !freshen_keylist (pgppath) )
    {
	return_badnews ("could not generate keylist");
	exit (RV_SYS_ERROR);
    }
    /*  Make stdout go to  sendmail  */
    dup2 (sendmail_in_fd, OUTPUT_FD);
    close (sendmail_in_fd);
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Adding special PGP arguments\n");
	fflush (debug_fp);
    }
    add_string (&pgp_args, "pgp", FALSE);
    add_string (&pgp_args, "-feat", FALSE);
    add_string (&pgp_args, "+batchmode", FALSE);
    add_string (&pgp_args, "+language=en", FALSE);
    /*  Process header  */
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Starting to process header...\n");
	fflush (debug_fp);
    }
    while (header)
    {
	if ( !read_line (INPUT_FD, &info, debug_fp, &eof) )
	{
	    if (debug_fp != NULL)
	    {
		fprintf (debug_fp, "Could not read header line\n");
		fflush (debug_fp);
	    }
	    return_badnews ("could not read header");
	    exit (1);
	}
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "  Processing header line: \"%s\"\n",
		     info.linebuf);
	    fflush (debug_fp);
	}
	if ( check_for_header_flags (info.linebuf, &options, sendmail_pid,
				     debug_fp) ) continue;
	if (info.line_length < 1)
	{
	    header = FALSE;
	    include_file (config_fp, debug_fp, "HEADER_TAIL_FILE");
	    fputc ('\n', stdout);
	    continue;
	}
	fputs (info.linebuf, stdout);
	fputc ('\n', stdout);
	if (st_icmp (info.linebuf,
		     "Content-Transfer-Encoding: quoted-printable") == 0)
	{
	    quoted_printables = TRUE;
	    continue;
	}
	if (st_nicmp (info.linebuf, "subject:", 8) == 0)
	{
	    fprintf (stderr, "> %s\n\n", info.linebuf);
	    continue;
	}
	if ( !process_addresses (&info, config_fp, &options, debug_fp) )
	{
	    return_badnews ("cannot process continuation header line");
	    kill (sendmail_pid, SIGKILL);
	    exit (1);
	}
    }
    fflush (stdout);
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Header processed\n");
	fflush (debug_fp);
    }
    if (options.insecure)
    {
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp,
		     "Insecure flag set: message will be sent as plaintext\n");
	    fflush (debug_fp);
	}
	transmit_plaintext (options.sign, debug_fp, sendmail_pid,
			    options.no_advertising, options.add_key, pgppath);
    }
    if (quoted_printables)
    {
	/*  Cannot encrypt this message  */
	if (options.secure)
	{
	    return_badnews ("cannot encrypt messages with quoted printables");
	    kill (sendmail_pid, SIGKILL);
	    exit (1);
	}
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp,
		     "Quoted printables flag set: message will be sent as plaintext\n");
	    fflush (debug_fp);
	}
	transmit_plaintext (options.sign, debug_fp, sendmail_pid,
			    options.no_advertising, options.add_key, pgppath);
    }
    /*  Determine if all recipients have PGP public keys  */
    switch ( check_recipients (pgp_args.strings + 4, pgppath, &bad_recipient) )
    {
      case CHECK_CAN_ENCRYPT:
	break;
      case CHECK_NO_KEY:
	if (options.secure)
	{
	    sprintf (txt, "recipient: \"%s\" does not have a PGP public key",
		     bad_recipient);
	    return_badnews (txt);
	    kill (sendmail_pid, SIGKILL);
	    exit (1);
	}
	/*  One of the recipients does not have a PGP key  */
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp,
		     "Not all recipients have a public key: message will be sent as plaintext\n");
	    fflush (debug_fp);
	}
	transmit_plaintext (options.sign, debug_fp, sendmail_pid,
			    options.no_advertising, options.add_key, pgppath);
	break;
      case CHECK_AMBIGUOUS_KEY:
	if (options.secure)
	{
	    sprintf (txt,
		     "ambiguous recipient: \"%s\" matches multiple PGP public keys",
		     bad_recipient);
	    return_badnews (txt);
	    kill (sendmail_pid, SIGKILL);
	    exit (1);
	}
	/*  One of the recipients does not have a PGP key  */
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp,
		     "A recipient has an ambiguous public key: message will be sent as plaintext\n");
	    fflush (debug_fp);
	}
	transmit_plaintext (options.sign, debug_fp, sendmail_pid,
			    options.no_advertising, options.add_key, pgppath);
	break;
      case CHECK_NO_KEYLIST:
	return_badnews ("could not read keylist");
	exit (RV_CANNOT_OPEN);
	/*break;*/
      default:
	break;
    }
    /*  Every recipient has a PGP public key: encrypt!  */
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Message will be encrypted\n");
	fflush (debug_fp);
    }
    if (options.no_advertising)
    {
	include_file (config_fp, debug_fp, "BODY_HEAD_FILE");
    }
    else
    {
	printf ("\nMessage has been automatically %sencrypted by PGPsendmail %s,\n",
		options.sign ? "signed and " : "", VERSION);
	printf ("available from ftp://ftp.atnf.csiro.au/pub/people/rgooch/\n");
	include_file (config_fp, debug_fp, "BODY_HEAD_FILE");
	printf ("\n\n");
	fflush (stdout);
    }
    if (options.sign)
    {
	/*  Sign message  */
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "Will attempt to sign message\n");
	    fflush (debug_fp);
	}
	if ( !connect_to_pgpdaemon (pgppath, &to_fd, &from_fd) )
	{
	    return_badnews ("could not connect to PGPdaemon");
	    kill (sendmail_pid, SIGKILL);
	    exit (RV_UNDEF_ERROR);
	}
	/*  Set close-on-exec flag for pipe to PGPdaemon  */
	if (fcntl (to_fd, F_SETFD, 1) == -1)
	{
	    return_badnews ("could not set close-on-exec flag");
	    kill (sendmail_pid, SIGKILL);
	    exit (RV_UNDEF_ERROR);
	}
	if (write (to_fd, "SIGN\n", 5) < 5)
	{
	    return_badnews ("could not write to PGPdaemon");
	    kill (sendmail_pid, SIGKILL);
	    exit (RV_UNDEF_ERROR);
	}
	/*  Check if OK so far  */
	if (read (from_fd, (char *) &bool_val, sizeof bool_val) <
	    sizeof bool_val)
	{
	    fprintf (stderr,
		     "Error reading response flag from PGPdaemon\t%s\n",
		     ERRSTRING);
	    return_badnews ("problem communicating with PGPdaemon");
	    kill (sendmail_pid, SIGKILL);
	    exit (RV_UNDEF_ERROR);
	}
	if (!bool_val)
	{
	    return_badnews ("PGPdaemon refused to sign: probably has no passphrase");
	    if (debug_fp != NULL)
	    {
		fprintf (debug_fp,
			 "PGPdaemon refused to sign: probably has no passphrase\n");
		fflush (debug_fp);
	    }
	    kill (sendmail_pid, SIGKILL);
	    exit (RV_UNDEF_ERROR);
	}
	pgp_in_fd = from_fd;
	out_fd = OUTPUT_FD;
	err_fd = ERROR_FD;
	if ( ( pgp_pid = spawn_job ("pgp", pgp_args.strings,
				    &pgp_in_fd, &out_fd, &err_fd) ) < 0 )
	{
	    return_badnews ("could not spawn PGP");
	    kill (sendmail_pid, SIGKILL);
	    exit (RV_UNDEF_ERROR);
	}
	close (from_fd);
	/*  Copy data  */
	if ( !copy_data (to_fd, INPUT_FD, FALSE) )
	{
	    return_badnews ("could not copy data to be signed");
	    kill (sendmail_pid, SIGKILL);
	    exit (RV_UNDEF_ERROR);
	}
	/*  Tell PGPdaemon the end of the message has arrived  */
	write (to_fd, "\0", 1);
	close (to_fd);
	fprintf (stderr, "Signed message\n");
    }
    else
    {
	pgp_in_fd = INPUT_FD;
	out_fd = OUTPUT_FD;
	err_fd = ERROR_FD;
	if ( ( pgp_pid = spawn_job ("pgp", pgp_args.strings,
				    &pgp_in_fd, &out_fd, &err_fd) ) < 0 )
	{
	    return_badnews ("could not spawn PGP");
	    kill (sendmail_pid, SIGKILL);
	    exit (RV_UNDEF_ERROR);
	}
    }
    if (waitpid (pgp_pid, &status, 0) < 0)
    {
	fprintf (stderr, "Error reaping child\t%s\n", ERRSTRING);
	return_badnews ("could not reap PGP process");
	kill (sendmail_pid, SIGKILL);
	exit (RV_UNDEF_ERROR);
    }
    if (status != 0)
    {
	fprintf (stderr, "Bad child status: %d\n", status);
	return_badnews ("PGP failure");
	kill (sendmail_pid, SIGKILL);
	exit (1);
    }
    if (options.add_key)
    {
	if ( !include_mykey (pgppath, stdout) )
	{
	    return_badnews ("could not include public key");
	    kill (sendmail_pid, SIGKILL);
	    exit (1);
	}
    }
    fprintf (stderr, "\n\n\n");
    if (debug_fp != NULL)
    {
	fprintf (debug_fp,
		 "Message should have been sent to real  sendmail\n");
	fflush (debug_fp);
    }
    if (options.receipt)
    {
	return_badnews (NULL);
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "Receipt returned\n");
	    fflush (debug_fp);
	}
    }
    else
    {
	sprintf ( txt, "/tmp/PGPsendmail.err.%d", getpid () );
	unlink (txt);
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "Receipt discarded\n");
	    fflush (debug_fp);
	}
    }
    fclose (config_fp);
    wake_pgpdaemon (pgppath, debug_fp);
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Exiting with status 0\n");
	fflush (debug_fp);
    }
    exit (RV_OK);
}   /*  End Function main  */

static flag process_addresses (decode_type *info, FILE *config_fp,
			       option_type *options, FILE *debug_fp)
/*  [SUMMARY] Process a line of addresses, adding to the list of recipients.
    <info> The line decoding information.
    [NOTE]: The line buffer is modified!
    <config_fp> The configuration file. If this is NULL, the recipient name
    will not be translated.
    <options> The contents may be modified.
    <debug_fp> Debugging information will be written here. If this is NULL, no
    debugging information is written.
    [RETURNS] TRUE on succes, else FALSE.
*/
{
    char *ptr;
    char error[STRING_LENGTH];

    while ( ( ptr = get_next_address (info, error) ) != NULL )
    {
	if (error[0] != '\0')
	{
	    fprintf (stderr, "Error decoding: %s\n", error);
	    return (FALSE);
	}
	if ( !translate_recipient (ptr, config_fp, options, debug_fp,
				   &pgp_args) )
	{
	    fprintf (stderr, "Error translating recipient: \"%s\"\n", ptr);
	    return (FALSE);
	}
    }
    return (TRUE);
}   /*  End Function process_addresses  */

static void return_badnews (message)
/*  This routine will mail off a message to the user, including the original
    input message.
    The message must be pointed to by  message  .If this is NULL, a "PGP OK"
    message is mailed and the original message is not included.
    The routine returns nothing.
*/
char *message;
{
    int count;
    int in_fd, out_fd, err_fd;
    int child;
    char txt[LINE_LENGTH];
    char separator[SEPARATOR_LENGTH + 1];
    FILE *err_fp;
    char *argv[6];
    extern char *username;
    extern char *sys_errlist[];

    /*  Make stdout go to  stderr  */
    fflush (stdout);
    dup2 (ERROR_FD, OUTPUT_FD);
    if (message == NULL)
    {
	fprintf (stderr, "PGPsendmail: PGP OK\n");
	sprintf ( txt, "%s -t -f %s -oi < /tmp/PGPsendmail.err.%d",
		  real_sendmail, username, getpid () );
	system (txt);
	sprintf ( txt, "/tmp/PGPsendmail.err.%d", getpid () );
	unlink (txt);
	return;
    }
    fprintf (stderr, "PGPsendmail: %s\n", message);
    sprintf ( txt, "/tmp/PGPsendmail.err.%d", getpid () );
    if ( ( err_fp = fopen (txt, "r") ) == NULL )
    {
	fprintf (stderr, "Could not open error file: \"%s\"\t%s\n",
		 txt, ERRSTRING);
	exit (RV_CANNOT_OPEN);
    }
    /*  Bounce message  */
    argv[0] = real_sendmail;
    argv[1] = "-t";
    argv[2] = "-f";
    argv[3] = username;
    argv[4] = "-oi";
    argv[5] = NULL;
    in_fd = -1;
    out_fd = OUTPUT_FD;
    err_fd = ERROR_FD;
    if ( ( child = spawn_job (real_sendmail, argv, &in_fd, &out_fd, &err_fd) )
	 < 0 )
    {
	fprintf (stderr, "Could not spawn  sendmail  to bounce message\n");
	exit (RV_SYS_ERROR);
    }
    /*  Read error messages and pass to  sendmail  */
    while (fgets (txt, STRING_LENGTH - 1, err_fp) != NULL)
    {
	write ( in_fd, txt, strlen (txt) );
    }
    fclose (err_fp);
    strcpy (txt, "\n\nBounced message follows:\n\n");
    write ( in_fd, txt, strlen (txt) );
    for (count = 0; count < SEPARATOR_LENGTH; ++count) separator[count] = '-';
    separator[SEPARATOR_LENGTH] = '\n';
    write (in_fd, separator, SEPARATOR_LENGTH + 1);
    /*  Read message and pass to  sendmail  */
    while (fgets (txt, STRING_LENGTH - 1, stdin) != NULL)
    {
	write ( in_fd, txt, strlen (txt) );
    }
    write (in_fd, separator, SEPARATOR_LENGTH + 1);
    close (in_fd);
    sprintf ( txt, "/tmp/PGPsendmail.err.%d", getpid () );
    unlink (txt);
}   /*  End Function return_badnews  */

static flag check_for_header_flags (line, options, sendmail_pid, debug_fp)
/*  This routine will check if the header line contains the "X-Secure:"
    keyword and will process it.
    The header line must be pointed to by  line  .
    The flags will modify the storage pointed to by  options  .See the
    documentation on the  process_flags  routine for details.
    The process ID of the real sendmail must be given by  sendmail_pid  .
    Debugging information will be written to the file pointed to by  debug_fp
    .If this is NULL, no debugging information is written.
    The routine returns TRUE if the header line contains the "X-Secure:"
    keyword, else it returns FALSE.
*/
char *line;
option_type *options;
int sendmail_pid;
FILE *debug_fp;
{
    char *ptr, *rest;

    if (st_nicmp (line, "X-Secure:", 9) != 0) return (FALSE);
    line += 9;
    if ( !process_flags (line, options) )
    {
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "bad X-Secure: line: \"%s\"\n", line);
	    fflush (debug_fp);
	}
	return_badnews ("bad X-Secure: line");
	kill (sendmail_pid, SIGKILL);
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "Real  sendmail  killed: exiting\n");
	    fflush (debug_fp);
	}
	exit (1);
    }
    return (TRUE);
}   /*  End Function check_for_header_flags  */

static void sig_handler (sig, code, scp, addr)
/*  This routine will handle signals.
    The signal number will be given by  sig  .
    The routine returns nothing.
*/
int sig;
int code;
struct sigcontext *scp;
char *addr;
{
    extern FILE *debug_fp;

    if (sig == SIGPIPE)
    {
	fprintf (stderr, "Caught SIGPIPE\n");
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "Caught SIGPIPE\n");
	    fflush (debug_fp);
	}
    }
}   /*  End Function sig_handler  */

static void transmit_plaintext (flag sign, FILE *debug_fp, int sendmail_pid,
				flag no_advertising, flag add_key,
				CONST char *pgppath)
/*  This routine will transmit the Email message the message in plaintext form.
    If the value of  sign  is TRUE, the message will be.
    Debugging information will be written to the file pointed to by  debug_fp
    .If this is NULL, no debugging information is written.
    The process ID of the real sendmail must be given by  sendmail_pid  .
    If advertising is not desired the value of  no_advertising  must be TRUE.
    The routine does not return: upon completion the process exits.
*/
{
    flag bool_val;
    int ch;
    int child_pid, status;
    int to_fd, from_fd;
    char buffer[STRING_LENGTH];
    extern char *sys_errlist[];

    if (!sign)
    {
	if (debug_fp != NULL)
	{
	    fputs ("Copying unsigned, plaintext message to  sendmail\n",
		   debug_fp);
	    fflush (debug_fp);
	}
	/*  Copy message to  sendmail  */
	while ( ( ch = fgetc (stdin) ) != EOF ) fputc (ch, stdout);
	fflush (stdout);
	if (add_key)
	{
	    if ( !include_mykey (pgppath, stdout) )
	    {
		return_badnews ("could not include public key");
		kill (sendmail_pid, SIGKILL);
		exit (1);
	    }
	}
	/*  Clean up and exit  */
	sprintf ( buffer, "/tmp/PGPsendmail.err.%d", getpid () );
	unlink (buffer);
	wake_pgpdaemon (pgppath, debug_fp);
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "Exiting with status 0\n");
	    fflush (debug_fp);
	}
	exit (RV_OK);
    }
    /*  Message should be signed  */
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Will attempt to sign message\n");
	fflush (debug_fp);
    }
    if ( !connect_to_pgpdaemon (pgppath, &to_fd, &from_fd) )
    {
	return_badnews ("could not connect to PGPdaemon");
	kill (sendmail_pid, SIGKILL);
	exit (RV_UNDEF_ERROR);
    }
    if (write (to_fd, "SIGN\n", 5) < 5)
    {
	return_badnews ("could not write to PGPdaemon");
	kill (sendmail_pid, SIGKILL);
	exit (RV_UNDEF_ERROR);
    }
    /*  Check if OK so far  */
    if (read (from_fd, (char *) &bool_val, sizeof bool_val) < sizeof bool_val)
    {
	return_badnews ("problem communicating with PGPdaemon");
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp,
		     "Error reading response flag from PGPdaemon\t%s\n",
		     ERRSTRING);
	    fflush (debug_fp);
	}
	kill (sendmail_pid, SIGKILL);
	exit (RV_UNDEF_ERROR);
    }
    if (!bool_val)
    {
	return_badnews ("PGPdaemon refused to sign: probably has no passphrase");
	if (debug_fp != NULL)
	{
	    fprintf(debug_fp,
		    "PGPdaemon refused to sign: probably has no passphrase\n");
	    fflush (debug_fp);
	}
	kill (sendmail_pid, SIGKILL);
	exit (RV_UNDEF_ERROR);
    }
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Copying signed, plaintext message to  sendmail\n");
	fflush (debug_fp);
    }
    if (!no_advertising)
    {
	printf ("\nMessage has been automatically signed by PGPsendmail %s,\n",
		VERSION);
	printf ("available from ftp://ftp.atnf.csiro.au/pub/people/rgooch/");
	printf ("\n\n");
	fflush (stdout);
    }
    /*  Now have to copy stdin to PGPdaemon input and PGPdaemon output to
	stdout  */
    /*  Setup a child process. Parent copies stdin to PGPdaemon.
	Child copies PGPdaemon output to stdout  */
    switch ( child_pid = fork () )
    {
      case 0:
	/*  Child  */
	close (to_fd);
	if ( !copy_data (OUTPUT_FD, from_fd, FALSE) )
	{
	    fprintf (stderr, "Error copying data from PGPdaemon\n");
	    if (debug_fp != NULL)
	    {
		fprintf (debug_fp, "Error copying data from PGPdaemon\n");
		fflush (debug_fp);
	    }
	    exit (RV_UNDEF_ERROR);
	}
	/*  Done  */
	close (from_fd);
	exit (RV_OK);
	/*break;*/
      case -1:
	/*  Error  */
	fprintf (stderr, "Could not fork\t%s\n", ERRSTRING);
	copy_data (OUTPUT_FD, INPUT_FD, FALSE);
	exit (RV_OK);
	/*break;*/
      default:
	/*  Parent  */
	close (from_fd);
	break;
    }
    /*  Copy  stdin  to PGPdaemon  */
    if ( !copy_data (to_fd, INPUT_FD, FALSE) )
    {
	return_badnews ("Error copying data to PGPdaemon");
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "Error copying data to PGPdaemon\n");
	    fflush (debug_fp);
	}
	kill (sendmail_pid, SIGKILL);
	exit (RV_UNDEF_ERROR);
    }
    /*  Tell PGPdaemon the end of message has arrived  */
    write (to_fd, "\0", 1);
    close (to_fd);
    /*  Reap child  */
    waitpid (child_pid, &status, 0);
    if (status != 0)
    {
	return_badnews ("Child returned bad status");
	if (debug_fp != NULL)
	{
	    fprintf (debug_fp, "Bad child status: %d\n", status);
	    fflush (debug_fp);
	}
	kill (sendmail_pid, SIGKILL);
	exit (RV_OK);
    }
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Signed message\n");
	fflush (debug_fp);
    }
    if (add_key)
    {
	if ( !include_mykey (pgppath, stdout) )
	{
	    return_badnews ("could not include public key");
	    kill (sendmail_pid, SIGKILL);
	    exit (1);
	}
    }
    /*  Clean up and exit  */
    sprintf ( buffer, "/tmp/PGPsendmail.err.%d", getpid () );
    unlink (buffer);
    wake_pgpdaemon (pgppath, debug_fp);
    if (debug_fp != NULL)
    {
	fprintf (debug_fp, "Exiting with status 0\n");
	fflush (debug_fp);
    }
    exit (RV_OK);
}   /*  End Function transmit_plaintext  */

static void include_file (FILE *config_fp, FILE *debug_fp, CONST char *keyword)
/*  [SUMMARY] Include file into message if keyword provided.
    <config_fp> The configuration file. This may be NULL.
    <debug_fp> Debugging information will be written here. If this is NULL no
    debugging information is written.
    <keyword> The keyword that must be specified in the configuration file.
    [RETURNS] Nothing.
*/
{
    flag eof;
    int fd, len;
    char *ptr;
    char buffer[STRING_LENGTH];
    static char function_name[] = "include_file";

    if (config_fp == NULL) return;
    if (fseek (config_fp, 0, SEEK_SET) == -1)
    {
	if (debug_fp == NULL) return;
	fprintf (debug_fp, "%s: error rewinding config file\t%s\n",
		 function_name, ERRSTRING);
	return;
    }
    len = strlen (keyword);
    while (TRUE)
    {
	if ( !read_fp_line (config_fp, buffer, STRING_LENGTH, debug_fp, &eof) )
	{
	    if (debug_fp != NULL)
		fprintf (debug_fp,
			 "%s: error reading configuration file\t%s\n",
			 function_name, ERRSTRING);
	    return;
	}
	if (eof) return;
	if (strncmp (buffer, keyword, len) != 0) continue;
	for (ptr = buffer + len; isspace (*ptr) && (*ptr != '\0'); ++ptr);
	if ( ( fd = open (ptr, O_RDONLY, 0) ) == -1 )
	{
	    if (debug_fp != NULL)
		fprintf (debug_fp, "Error opening file: \"%s\"\t%s\n",
			 ptr, ERRSTRING);
	    return;
	}
	fflush (stdout);
	copy_data (OUTPUT_FD, fd, FALSE);
	close (fd);
    }
}   /*  End Function include_file  */

static void wake_pgpdaemon (CONST char *pgppath, FILE *debug_fp)
/*  [SUMMARY] Send SIGUSR2 to PGPdaemon if it exists to wake it.
    <pgppath> The contents of the PGPPATH environment variable.
    <debug_fp> Debugging information will be written here. If this is NULL no
    debugging information is written.
    [RETURNS] Nothing.
*/
{
    int daemon_pid;
    FILE *fp;
    char hostname[STRING_LENGTH], pid_filename[STRING_LENGTH];

    if (gethostname (hostname, STRING_LENGTH - 1) != 0)
    {
	if (debug_fp != NULL)
	    fprintf (debug_fp, "Error getting hostname\t%s\n", ERRSTRING);
	return;
    }
    hostname[STRING_LENGTH - 1] = '\0';
    /*  Worry about communication files  */
    sprintf (pid_filename, "%s/.pgpd.PID.%s", pgppath, hostname);
    if ( ( fp = fopen (pid_filename, "r") ) == NULL ) return;
    if (fscanf (fp, "%d", &daemon_pid) != 1)
    {
	if (debug_fp != NULL) fprintf (debug_fp, "Error reading: \"%s\"\t%s\n",
				       pid_filename, ERRSTRING);
	return;
    }
    fclose (fp);
    if (kill (daemon_pid, SIGUSR2) == -1)
    {
	if (debug_fp != NULL)
	    fprintf (debug_fp, "Error signalling PGPdaemon\t%s\n", ERRSTRING);
    }
    else
    {
	if (debug_fp != NULL) fprintf (debug_fp, "Woke PGPdaemon\n");
    }
}   /*  End Function wake_pgpdaemon  */
