/*
 * Copyright (C) 1995, 1996 Systemics Ltd (http://www.systemics.com/)
 * All rights reserved.
 *
 */

package cryptix.pgp;

/**
 * <p>Copyright (C) 1995, 1996 Systemics Ltd (http://www.systemics.com/)
 * All rights reserved.
 */
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.io.EOFException;
import cryptix.math.BigInteger;
import cryptix.math.MPI;
import cryptix.math.RandomStream;
import cryptix.crypt.rsa.PublicKey;
import cryptix.crypt.rsa.SecretKey;

public final class PublicKeyEncrypted extends Packet
{
	private BigInteger enckey;
	private byte keyId[];
	
	public PublicKeyEncrypted( PublicKey publickey, byte Ideakey[], RandomStream rand )
	{
		keyId = publickey.id();
		enckey = BigIntFromHash( publickey, Ideakey, rand );
	}
	
	public PublicKeyEncrypted( DataInput in, int length )
		throws IOException
	{
		super( in, length );
	}

	public void 
	read( DataInput in, int length )
		throws IOException
	{
		in.readByte(); // version.
		keyId = new byte[8];
		in.readFully( keyId ); // key id;
		if ( in.readByte() != 0x01 ) // RSA algorythm identifier.
			throw new FormatException( "Incorrect algorithm identifier." );
		try
		{
			enckey = MPI.read( in );
		}
		catch ( EOFException e )
		{
			throw new IOException("MPI too long.");
		}
	}

	public int
	write( DataOutput out )
		throws IOException
	{
		out.write( 0x02 );
		out.write( keyId );		
		out.write( 0x01 );
		byte tmp[] = MPI.save( enckey );
		out.write( tmp );
		return tmp.length + 10;
	}
	
	private static final BigInteger
	BigIntFromHash( PublicKey key, byte hash[], RandomStream rand )
	{
		if ( hash.length != 16 )
			throw new Error( "The code for hashes of lengths other than 16 is not in place yet." );
		int byteLength = key.bitLength() / 8;
		int fillLength = byteLength - ( 6 + hash.length );
		byte buffer[] = new byte[byteLength];
		int i=0;
		buffer[i++] =0;
		buffer[i++] =2;
		for (int count = fillLength; count > 0 ; count--)
		{
			byte fill;
			do		
				fill =(byte)rand.read();
			while ( fill == 0 );
			buffer[i++] = fill;
		}
		buffer[i++] =0;
		buffer[i++] =1;
		System.arraycopy( hash, 0, buffer, i, hash.length );
		int sum = 0;
		for (int count = hash.length - 1; count >= 0; count--)
			sum += hash[count] & 0xFF;
		i += hash.length;
		buffer[i++] = (byte)( ( sum >>> 8 ) & 0xFF );
		buffer[i++] = (byte)( sum & 0xFF );

		return key.encrypt( new BigInteger( buffer ) );
	}

	public 	byte[]
	getKey( SecretKey key )
	{
		byte buf[] = key.decrypt( enckey ).toByteArray();

		int sum = 0;
		for ( int i = buf.length - 18; i < buf.length-2; i++)
			sum += buf[i] & 0xFF;
		int check = ( ( buf[buf.length-2] & 0xFF ) << 8 ) + ( buf[buf.length-1] & 0xFF );
		if (sum != check)
			throw new Error( "sum mismatch sum:" + sum + " check:" + check );
		byte ideaKey[] = new byte[16];
		System.arraycopy( buf, buf.length - 18, ideaKey, 0, 16 );
		return ideaKey;
	}
	
	public String
	toString()
	{
		return "Public key encrypted packet - number: " + enckey.toString();
	}
}
