/*
 * The author of this software is J. Andrew Fingerhut.
 * Copyright (c) 1995 by J. Andrew Fingerhut.  All rights reserved.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, THE AUTHOR DOES NOT MAKE ANY CLAIM OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

/* versions.c
 *
 * REVISION HISTORY
 *
 * DATE      RESPONSIBLE PARTY  DESCRIPTION
 * -------------------------------------------------------------------------
 * 95/08/24  J. A. Fingerhut    Initial version written.
 */


#include "stdio.h"
#include "nautilus.h"


/* external variables */
extern struct param_t params;   /* operating parameters */


/*
 * This is the maximum number of characters that a lower version of
 * Nautilus can receive in the upgrade information.  It should be
 * somewhat larger than we ever expect the string upgrade_info to
 * become.  An upper limit is made here only to prevent the receiver
 * from receiving data indefinitely.
 */
#define MAX_INFO_SIZE     16384


#define RECEIVE_TIMEOUT   3	/* seconds */


/*
 * The following string contains information on how to upgrade to the
 * most recently released version of Nautilus.  This information
 * should remain as stable as possible.
 */

static char *upgrade_info =
"Nautilus is available in three different formats:\n\
\n\
nautilus-XXX.tar.gz   - full source code\n\
nautXXX.zip           - MSDOS executable and associated documentation\n\
nautXXXs.zip          - full source code\n\
\n\
It is available at the following FTP sites:\n\
\n\
ftp://ftp.csn.org:/mpj/I_will_not_export/crypto_???????/voice/\n\
This is an export controlled ftp site: read /mpj/README for\n\
information on access.\n\
\n\
ftp://miyako.dorm.duke.edu/mpj/crypto/voice/\n\
This is an export controlled ftp site: read /mpj/GETTING_ACCESS for\n\
information on access.\n\
\n\
ftp://ripem.msu.edu/pub/crypt/other/nautilus-phone-XXX-source.tar.gz\n\
ftp://ripem.msu.edu/pub/crypt/msdos/nautilus-phone-XXX-source.zip\n\
ftp://ripem.msu.edu/pub/crypt/msdos/nautilus-phone-XXX-exe.zip\n\
This is an export controlled ftp site: read /pub/crypt/GETTING_ACCESS\n\
for information on access.\n\
\n\
It is also available at:\n\
\n\
Colorado Catacombs BBS - (303) 772-1062\n";



/*

   FUNCTION: OpenUpgradeFile

   PURPOSE:

   Open the file with name UPGRADE_FILE for writing, unless that file
   already exists, or it cannot be opened for writing.

   ARGUMENTS:

   char *file_name

   The file name to open.

   RETURN VALUE:

   A pointer to the FILE structure if successful, or NULL if the file
   already exists, or cannot be opened for writing.
   
   SIDE EFFECTS:

   If NULL is returned, an error message is printed to stderr
   explaining the reason.
*/

FILE *
OpenUpgradeFile()
{
    FILE        *f;

    f = fopen(params.upgrade_file, "r");
    if (f == NULL) {
	/*
	 * Then the file does not exist already.  Attempt to open for
	 * writing.
	 */
	f = fopen(UPGRADE_FILE, "w");
	if (f == NULL) {
	    fprintf(stderr, "Upgrade file %s cannot be written.\n",
		    UPGRADE_FILE);
	    fflush(stderr);
	}
    }
    else {
	fprintf(stderr,
		"Upgrade file %s already exists.  It will not be overwritten.\n",
		UPGRADE_FILE);
	fflush(stderr);
	fclose(f);
	f = NULL;
    }

    return f;
}



/*

   FUNCTION: VersionsCompatible

   PURPOSE:

   Determine whether the current version is compatible with the older
   version specified in the arguments.

   ARGUMENTS:

   UINT8 old_major
   UINT8 old_minor

   The major and minor version number of the older version of Nautilus
   that wants to communicate with this one.

   RETURN VALUE:
   
   TRUE if the two versions are compatible, otherwise FALSE.

   SIDE EFFECTS: None
*/

int
VersionsCompatible(UINT8 old_major, UINT8 old_minor)
{
    /*
     * It is guaranteed that old_major >= 1, because this check is
     * made before VersionsCompatible is ever called.
     * 
     * Since this is the code for version 1.0, this function always
     * returns true.  In the future, it will likely contain
     * comparisons between the arguments and the preprocessor symbols
     * VERSION_MAJOR and VERSION_MINOR.
     */

    return TRUE;
}



/*

   FUNCTION: SendCompatible

   PURPOSE:

   This function is called by the newer of two versions of Nautilus
   participating in a call.  It sends a packet to the other program
   merely indicating one bit of information, TRUE or FALSE.

   ARGUMENTS:

   int compatible

   compatible is TRUE if and only if the newer version is compatible
   with the older one.

   RETURN VALUE: None
   
   SIDE EFFECTS: None
*/

void
SendCompatible(int compatible)
{
    UINT8 data;

    if (compatible) {
	data = (UINT8) 1;
    }
    else {
	data = (UINT8) 0;
    }
    SendPkt(DATA, &data, 1);
}



/*

   FUNCTION: ReceiveCompatible

   PURPOSE:

   This function is called by the older of two versions of Nautilus
   participating in a call.  It receives the one packet sent by
   SendCompatible.

   ARGUMENTS:

   int *compatible

   If a data packet with 1 byte is successfully received, and the 1
   byte is either 1 or 0, then the integer pointed to by this argument
   is set to TRUE or FALSE, respectively.  In all other cases, this
   value is not written.

   RETURN VALUE:

   SUCCESS if the integer pointed to by the argument is written,
   otherwise FAIL.
   
   SIDE EFFECTS: None
*/

int
ReceiveCompatible(int *compatible)
{
    struct packet_t    packet;
    int                done;
    long               start;

    done = FALSE;
    start = Clock();
    do {
	if (IncomingPort()) {
	    if (ReadPkt(&packet) == SUCCESS) {
		done = TRUE;
		break;
	    }
	}
    } while (Clock() - start <= RECEIVE_TIMEOUT);

    if (!done || (packet.type != DATA) || (packet.length != 1)) {
	return FAIL;
    }
    
    if (packet.data[0] == (UINT8) 1) {
	*compatible = TRUE;
	return SUCCESS;
    }
    else if (packet.data[0] == (UINT8) 0) {
	*compatible = FALSE;
	return SUCCESS;
    }
    else {
	return FAIL;
    }
}



/*

   FUNCTION: SendUpgradeInfo

   PURPOSE:

   This function is called by the newer of two versions of Nautilus
   participating in a call.  It sends a short file of unencrypted text
   that describes where to get the most recent version of Nautilus.
   It also displays a message on the screen of the participant with
   the newer version, so they also know why the call was terminated
   prematurely.

   ARGUMENTS:

   UINT8 old_major
   UINT8 old_minor

   The major and minor version numbers of the old version that the
   newer version is failing to communicate with.  This is only used
   in the display to the user of the newer version.

   RETURN VALUE:
   
   Currently always SUCCESS, but this could be changed so that it only
   returns SUCCESS if the receiver acknowledges receiving the message.
   The current version does not do this.

   SIDE EFFECTS: None
*/

int
SendUpgradeInfo(UINT8 old_major, UINT8 old_minor)
{
    char *local_notification_fmt =
"The other Nautilus user has version %d.%d, which is not compatible with\n\
your version, %s.\n\
\n\
Transmitting instructions on how to obtain the newest version...\n";
    char local_notification[512];

    int len, total_sent, send_this_time;


    sprintf(local_notification, local_notification_fmt,
	    old_major, old_minor, VERSION_STRING);

    error(MSG_ERROR, local_notification);

    /*
     * Send the contents of the string 'upgrade_info' to the older
     * version, where all packets except possibly the last one are
     * the maximum size allowed.  End with an EOT packet.
     */

    len = strlen(upgrade_info);

    total_sent = 0;
    while (total_sent < len) {
	send_this_time = len - total_sent;
	if (send_this_time > MAX_PKT_DATA) {
	    send_this_time = MAX_PKT_DATA;
	}
	SendPkt(DATA, &upgrade_info[total_sent], send_this_time);
	total_sent += send_this_time;
    }
    SendPkt(EOT, NULL, 0);

    error(MSG_ERROR, "Transmission complete.");

    return SUCCESS;
}



/*

   FUNCTION: ReceiveUpgradeInfo

   PURPOSE:

   This function is called by the older of two versions of Nautilus
   participating in a call.  It receives a short file of unencrypted
   text that describes where to get the most recent version of
   Nautilus.  This message is displayed on the screen, and it is saved
   in a file with name UPGRADE_FILE, assuming that file does not exist
   already.  We don't want to clobber an existing file with that name,
   since it might have nothing to do with Nautilus.

   Of course it would be even nicer if the program asked the user for
   a file name in which to save the information, and asked for
   confirmation if the specified file already exists, but I just want
   to get something basic working first.

   ARGUMENTS:

   UINT8 new_major
   UINT8 new_minor

   The major and minor version numbers of the new version that the
   older version is failing to communicate with.  This is only used
   in the display to the user of the older version.

   RETURN VALUE:

   SUCCESS if the file was completely received from the newer version,
   and displayed on the screen, even if it wasn't saved in a file.
   FAIL otherwise.
   
   SIDE EFFECTS: None
*/

int
ReceiveUpgradeInfo(UINT8 new_major, UINT8 new_minor)
{
    char           *local_notification_fmt =
"The other Nautilus user has version %d.%d, which is not compatible with\n\
your version, %s.\n\
\n\
Receiving instructions on how to obtain the newest version...\n\
\n\
\n";
    char           local_notification[512];
    struct packet_t packet;
    long           last_receive_time;
    int            total_received, received_this_time;
    int            done, receive_overflow, received_bad_packet;
    FILE           *upgrade_f;
    int            i;


    sprintf(local_notification, local_notification_fmt,
	    new_major, new_minor, VERSION_STRING);

    error(MSG_ERROR, local_notification);

    /*
     * Check if the file UPGRADE_FILE already exists, and if not, if
     * it can be opened for writing.
     */
    upgrade_f = OpenUpgradeFile();

    /*
     * Continue reading DATA packets from the newer version,
     * displaying the received data on the screen, and saving the
     * received data into the file 'upgrade_f', if upgrade_f is not NULL.
     * Stop if any other type of packet is received, if more than
     * MAX_INFO_SIZE bytes are received, or if more than
     * RECEIVE_TIMEOUT seconds elapse between consecutive packet
     * arrivals.
     */
    done = FALSE;
    receive_overflow = FALSE;
    received_bad_packet = FALSE;
    total_received = 0;
    last_receive_time = Clock();
    do {
	if (IncomingPort()) {
	    if (ReadPkt(&packet) == SUCCESS) {
		last_receive_time = Clock();
		if (packet.type == EOT) {
		    done = TRUE;
		    break;
		}
		else if (packet.type == DATA) {
		    /* Avoid receiving more than MAX_INFO_SIZE bytes. */
		    received_this_time = packet.length;
		    if (total_received + received_this_time > MAX_INFO_SIZE) {
			received_this_time = MAX_INFO_SIZE - total_received;
			receive_overflow = TRUE;
		    }
		    for (i = 0; i < received_this_time; i++) {
			fputc(packet.data[i], stderr);
			if (upgrade_f != NULL) {
			    fputc(packet.data[i], upgrade_f);
			}
		    }
		    if (receive_overflow) {
			break;
		    }
		}
		else {
		    received_bad_packet = TRUE;
		    break;
		}
	    }
	}
    } while (Clock() - last_receive_time <= RECEIVE_TIMEOUT);

    if (!done) {
	fprintf(stderr,
"More than %d seconds elapsed between consecutive packet arrivals.\n\
Aborting.\n",
		RECEIVE_TIMEOUT);
	fflush(stderr);
	return FAIL;
    }
    
    if (receive_overflow) {
	fprintf(stderr,
"The newer version attempted to send more than %d bytes.  Truncating.\n",
		MAX_INFO_SIZE);
	fflush(stderr);
	return FAIL;
    }
    
    if (received_bad_packet) {
	fprintf(stderr,
"A packet type other than DATA or EOT was received.  Aborting.\n");
	fflush(stderr);
	return FAIL;
    }
    
    if (upgrade_f != NULL) {
	fclose(upgrade_f);
	fprintf(stderr,
"\n\nThe upgrade information was also saved in the file '%s'.\n",
		UPGRADE_FILE);
	fflush(stderr);
	return FAIL;
    }
    
    error(MSG_ERROR, "Reception complete.");
    return SUCCESS;
}



/*

   FUNCTION: CantNotifyBetaVersion

   PURPOSE:

   This function is called by the newer of two versions of Nautilus
   participating in a call, but only when the older version is a
   pre-1.0 version.  A message is displayed on the screen explaining
   this, and suggests contacting the pre-1.0 user by some other means
   to let them know how to upgrade.  This message is also saved in the
   file with name UPGRADE_FILE, if it does not already exist, so that
   it may be more easily sent to the pre-1.0 user.

   ARGUMENTS: None

   RETURN VALUE:

   SUCCESS if the message was saved in the file, FAIL otherwise.
   
   SIDE EFFECTS: None
*/

int
CantNotifyBetaVersion()
{
    FILE         *upgrade_f;

    error(MSG_ERROR, "The other Nautilus user has a version earlier than 1.0,\n\
and it is incompatible with your version.  You should ask them to upgrade\n\
their version.  Send them the following instructions...\n\
\n\
\n");

    error(MSG_ERROR, upgrade_info);

    upgrade_f = OpenUpgradeFile();
    if (upgrade_f != NULL) {
	fputs(upgrade_info, upgrade_f);
	fclose(upgrade_f);
	fprintf(stderr,
"\n\nThe upgrade information was also saved in the file '%s'.\n",
		UPGRADE_FILE);
	fflush(stderr);
	return SUCCESS;
    }
    
    return FAIL;
}



/*

   FUNCTION: WaitForAck

   PURPOSE:

   This function is called by the newer of two versions of Nautilus
   participating in a call.  It receives the one ACK packet sent by
   the older version in response to the upgrade directions.

   ARGUMENTS: None

   RETURN VALUE:

   SUCCESS if the first packet is received within RECEIVE_TIMEOUT
   seconds, and it is an ACK, otherwise FAIL.
   
   SIDE EFFECTS: None
*/

int
WaitForAck()
{
    struct packet_t    packet;
    int                done;
    long               start;

    done = FALSE;
    start = Clock();
    do {
	if (IncomingPort()) {
	    if (ReadPkt(&packet) == SUCCESS) {
		done = TRUE;
		break;
	    }
	}
    } while (Clock() - start <= RECEIVE_TIMEOUT);

    if (done && (packet.type == ACK)) {
	return SUCCESS;
    }

    return FAIL;
}
