/*____________________________________________________________________________
	Copyright (C) 1996-1998 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	$Id: CDriver.cp,v 1.14.8.1 1998/11/12 03:05:40 heller Exp $
____________________________________________________________________________*/

#include <string.h>
#include <sound.h>
#include <FSM.h>
#include "pgpMacMemory.h"
#include "MacTraps.h"
#include "MacStrings.h"
#include "MacEnvirons.h"
#include <DriverGestalt.h>

#include "PGPDisk.h"
#include "PGPDiskEncryptDecrypt.h"
#include "A4Stuff.h"

#include "CDriveList.h"
#include "CDrive.h"
#include "CipherProcGlue.h"

#include "PGPMountUtils.h"

#include "PGPDiskDRVRMain.h"
#include "AddDriveStruct.h"
#include "PGPDiskGlobals.h"
#include "SleepAlert.h"

#include "CDriver.h"



#if USE_LOG
CLog *	gLog	= nil;
#endif

enum	{ kNoDriverErr	= -1 };
	
#define MAX( a, b ) 	( (a) >= (b) ? (a) : (b) )
#define MIN( a, b ) 	( (a) <= (b) ? (a) : (b) )


/* Control codes */
#define		killIOCC		1			/* kill I/O */
#define		verifyDiskCC	5			/* verify disk */
#define		formatDiskCC	6			/* format disk */
#define		ejectDiskCC		ejectCode	/* eject disk (ejectable media only) */
#define		setTagBufferCC	8			/* set tag buffer (.SONY) */
#define		trackCacheCC	9,			/* control track cache (.SONY) */
#define		physicalIconCC	21			/* return physical location icon and
										   where string */
#define		mediaIconCC		22			/* return media icon */
#define		driveInfoCC		23			/* return drive info */
#define		getDriveSizeCC	24
#define		trackDumpCC		18244		/* diagnostic raw track dump (.SONY) */
	
/* Status codes */
#define		returnFormatListSC	6			/* return format list (.SONY) */
#define		driveStatusSC		drvStsCode		/* drive status */



const ulong	kDefaultScratchBufferSize	= 128UL * 1024UL;
const ulong	kMinScratchBufferSize		= 8UL * 1024UL;
const ulong	kMaxScratchBufferSize		= 768UL * 1024UL;


const ResID	kDriverStringsRID	= 1003;
enum
{
	kPleaseReinsertDiskStrIndex = 1
};


/*_________________________________________________________________________
	Utility class whose constructor sets the zone to the system zone and
	whose destructor restores the previous zone.
_________________________________________________________________________*/
class CSystemZoneSetter
	{
	private:
		THz		mSaveZone;
		
	public:
		CSystemZoneSetter( void )
			{ mSaveZone	= GetZone(); SetZone( SystemZone() ); }
		~CSystemZoneSetter( void)
			{ SetZone( mSaveZone ); }
	};
	
	
static Boolean		PowerMgrExists( void );

	
	

CDriver *	CDriver::sSelf	= nil;

	void
CDriver::DisposeScratchBuffer( )
	{
	if ( IsntNull( mScratchBuffer ) )
		{
		pgpFreeMac( (Ptr)mScratchBuffer );
		mScratchBuffer	= nil;
		CDrive::sSetScratchBuffer( nil, 0 );
		}
	}
	
	
	void
CDriver::DisposePBs( )
	{
	if ( IsntNull( mPBArray ) )
		{
		DisposePtr( (Ptr)mPBArray );
		mPBArray	= nil;
		}
	}
	
	




	static OSErr
UnmountAndEjectDrive( short driveNumber )
	{
	ParamBlockRec	pb;
	OSErr			err	= noErr;
	
	pb.volumeParam.ioVRefNum	= driveNumber;
	pb.volumeParam.ioNamePtr	= nil;
	err	= PBUnmountVol( & pb );
	if ( IsntErr( err ) )
		{
		pb.volumeParam.ioCompletion	= nil;
		pb.volumeParam.ioVRefNum	= driveNumber;
		pb.volumeParam.ioNamePtr	= nil;
		err	= PBEject( &pb );
		}
		
	return( err );
	}


#pragma mark -


CDriver::CDriver( void )
	{
	sSelf					= this;
	
	mDriverIsOpen			= false;
	mDCE					= nil;
	mSleep.installed		= false;
	mPGPDiskGlobals			= nil;
	mHaveShownSleepAlert	= false;
	
	pgpClearMemory( &mSleep, sizeof( mSleep ) );
	pgpClearMemory( &mPrefs, sizeof( mPrefs ) );
	
	mPBArray	= nil;
	mNumPBs		= 0;
	
	mScratchBuffer		= nil;
	mScratchBufferSize	= 0;
	mCipherProcDamaged	= false;
	
	GetIndString( mEjectStringTemplate, kDriverStringsRID,
			kPleaseReinsertDiskStrIndex );
	pgpAssert( StrLength( mEjectStringTemplate ) != 0 );
	}
	
	
	
CDriver::~CDriver( void )
	{
	pgpAssert( ! mDriverIsOpen );
	
	// most deletion is done in Close()
	pgpAssert( IsNull( mScratchBuffer ) );
	pgpAssert( IsNull( mPBArray ) );
	
	sSelf				= nil;
	}
	


	
/*_________________________________________________________________________
	No drives remain.  Get rid of any unnecessary stuff.
_________________________________________________________________________*/
	void
CDriver::NoDrivesLeft( void )
	{
	DisposeScratchBuffer();
	
	DisposePBs();
	
	mDCE->dCtlDelay		= 0;	// we don't want time anymore
	
	RemoveSleepProc();
	}


	void
CDriver::DriveHousekeeping( Boolean deleteUnmountedDrives )
	{
	CDriveList *	driveList	= CDriveList::sGetList();
	
	if ( IsntNull( driveList ) )
		{
		ushort 	driveIndex = 0;
		
		while ( driveIndex < driveList->GetNumDrives() )
			{
			CDrive *	drive;
			ushort		driveNumber;
			Boolean		isMounted;
			
			drive	= driveList->GetIndDrive( driveIndex );
	
			driveNumber	= drive->GetDriveNumber();
			isMounted	= IsntNull( GetVCBForDrive( driveNumber ) );
			if ( ! isMounted )
				{
				Boolean	removeDrive = deleteUnmountedDrives;
				
				if ( drive->HasBeenMounted() && ! removeDrive )
					{
					ProcessSerialNumber	unmountedAppPSN;
					
					unmountedAppPSN = drive->GetUnmountedAppPSN();
					if( unmountedAppPSN.highLongOfPSN == 0 &&
						unmountedAppPSN.lowLongOfPSN == kNoProcess )
						{
						// We have no unmounting process for this drive.
						// Assume immediate removal.
						removeDrive = TRUE;
						}
					else
						{
						ProcessInfoRec	processInfo;
						FSSpec			unusedFileSpec;
						Str255			unusedProcessName;
						
						pgpClearMemory( &processInfo, sizeof( processInfo ) );
						
						processInfo.processInfoLength 	= sizeof(processInfo);
						processInfo.processName			= unusedProcessName;
						processInfo.processAppSpec		= &unusedFileSpec;
						
						if( IsntErr( GetProcessInformation( &unmountedAppPSN,
								&processInfo ) ) )
							{
							if( ( processInfo.processMode &
										modeOnlyBackground ) == 0 )
								{
								// Is not a faceless background app
								
								if( processInfo.processSignature == 'MACS' ||
									processInfo.processSignature ==
											kPGPDiskFileCreator )
									{
									// Unmounting app is the Finder or PGPdisk.
									// Removed immedaitely
									removeDrive = TRUE;
									}
								}
							else
								{
								// Always remove the drive when unmounted by
								// a background app.
								removeDrive = TRUE;	
								}
							}
						else
							{
							// Unmounting process no longer exists. Remove
							// the drive
							removeDrive = TRUE;
							}
						}
					}

				if( removeDrive )
					{
					CDriveList::sGetList()->RemoveDrive( drive );
					
					delete drive;
					// play sound here?
					
					continue;	// don't increment counter
					}
				}
			else
				{
				drive->SetHasBeenMounted( );
				}
				
			++driveIndex;
			}
		
		if ( driveList->GetNumDrives() == 0 )
			{
			NoDrivesLeft();
			}
		}
	}


	void
CDriver::UnmountAnyIdleDrives(  )
	{
	static ulong	sLastTryTime		= 0;
	const ulong		kTryIntervalSeconds	= 60;
	ulong			now;
	
	GetDateTime( &now );
	
	// attempt to unmount no more often than 'kTryIntervalSeconds'
	if( mPrefs.unmountAfterIdleTime &&
		( now - sLastTryTime ) >= kTryIntervalSeconds )
		{
		CDriveList *	driveList	= CDriveList::sGetList();
		ushort			numDrives;
		
		numDrives	= driveList->GetNumDrives();
			
		for( ushort driveIndex = 0; driveIndex < numDrives; ++driveIndex )
			{
			CDrive *		drive;
			
			drive	= driveList->GetIndDrive( driveIndex );
			
			if( now > ( drive->GetLastAccessTime() +
					mPrefs.unmountIdleTimeSeconds ) )
				{
				OSErr				err	= noErr;
				ProcessSerialNumber	noProcessPSN;
				
				pgpDebugPStr(
					"\pDrivePeriodic: calling UnmountAndEjectDrive;g");
				(void)UnmountAndEjectDrive( drive->GetDriveNumber() );
				
				noProcessPSN.highLongOfPSN	= 0;
				noProcessPSN.lowLongOfPSN	= kNoProcess;
				
				drive->SetUnmountedAppPSN( noProcessPSN );
				}
			}
		
		sLastTryTime	= now;
		}
	}
	
	
	
	void
CDriver::DrivePeriodic( )
	{
	CDriveList *	driveList	= CDriveList::sGetList();
	
	if ( IsntNull( driveList ) )
		{
		ushort numDrives	= driveList->GetNumDrives();
		
		for( ushort driveIndex = 0; driveIndex < numDrives; ++driveIndex )
			{
			CDrive *	drive;
			
			drive	= driveList->GetIndDrive( driveIndex );
			
			drive->CheckBaseDriveEjected( mEjectStringTemplate );

			if ( IsNull( drive->GetVCB() ) && drive->HasBeenMounted() )
				{
				const VCB *	vcb;
				
				vcb	= GetVCBForDrive( drive->GetDriveNumber() );
				// could be nil if drive is being created, or was unmounted
				drive->SetVCB( vcb );
				}
			
			drive->UpdateFileModificationDate();
			}
		}
	}
	
	
	void
CDriver::HandleToggleKeyBitsCC( void )
	{
	CDriveList *	driveList	= CDriveList::sGetList();
	
	if ( IsntNull( driveList ) )
		{
		ushort numDrives	= driveList->GetNumDrives();
		
		for( ushort driveIndex = 0; driveIndex < numDrives; ++driveIndex )
			{
			CDrive *	drive;
			
			drive	= driveList->GetIndDrive( driveIndex );
			drive->ToggleKeyBits();
			}
		}
	}
	
	
/*_________________________________________________________________________
	Caution:
		AccRuns are made via immediate control calls. The driver could start
	doing I/O while AccRun() is executing or it could already be doing I/O.
_________________________________________________________________________*/
	void
CDriver::AccRun()
	{
	OSErr			err	= noErr;
	CDriveList *	driveList	= CDriveList::sGetList();
	ulong			now;
	
	// Call the gestalt idle proc to install patches in this context,
	// if necessary
	if( IsntNull( mPGPDiskGlobals->idleProc ) )
		{
		(*mPGPDiskGlobals->idleProc)();
		}
		
	UnmountAnyIdleDrives();
	
	// be sure to do this first, since drives may be deleted
	DriveHousekeeping();
	
	DrivePeriodic();
	
	
	GetDateTime( &now );
	
	// CAUTION: toggling key bits must be serialized with I/O; we can't
	// allow I/O to be processed while toggling a key. So we have to make
	// a synchronous control call to do it.
	static ulong	sLastToggleTime	= 0;
	const ulong		kToggleTimeSeconds	= 5 * 60;
	if ( now - sLastToggleTime >= kToggleTimeSeconds )
		{
		ParamBlockRec	pb;
		
		pb.cntrlParam.ioVRefNum	= 0;
		pb.cntrlParam.ioCRefNum	= mDCE->dCtlRefNum;
		pb.cntrlParam.csCode	= kToggleKeyBitsCC;
		(void)PBControlSync( &pb );
		
		GetDateTime( &sLastToggleTime );
		}
	
	
	// periodically verify that the cipher proc is not damaged.
	// This is to avoid data loss; a single bit flipped in the S-boxes
	// will damage any data subsequently ciphered.
	static ulong	sLastVerifyTime			= 0;
	const ulong		kVerifyIntervalSeconds	= 5 * 60;
	if ( now - sLastVerifyTime >= kVerifyIntervalSeconds )
		{
		if ( mCipherProcRef != kInvalidCipherProcRef )
			{
			if ( IsErr( CipherProc_CallVerify( mCipherProcRef ) ) )
				{
				mCipherProcDamaged	= TRUE;
				}
			}
			
		GetDateTime( &sLastVerifyTime );
		}
	
	
#if USE_LOG
	gLog->Flush();
#endif
	}



	void
CDriver::InstallSleepProc( void )
	{
	if( PowerMgrExists() && ! mSleep.installed)
		{
		mSleep.sleepQRec.sleepQLink		= nil;
		mSleep.sleepQRec.sleepQType		= slpQType;
		mSleep.sleepQRec.sleepQProc		= (SleepQUPP)sSleepDismount;
		mSleep.sleepQRec.sleepQFlags	= 0;
		SleepQInstall( & mSleep.sleepQRec );
		mSleep.installed		= TRUE;
		}
	}
	
	
	
	void
CDriver::RemoveSleepProc( void )
	{
	if ( mSleep.installed )
		{
		SleepQRemove( & mSleep.sleepQRec );
		mSleep.installed	= false;
		}
	}
	

	void
CDriver::InstallOrRemoveSleepProcAsAppropriate()
	{
	pgpAssert( IsntNull( CDriveList::sGetList() ) );
	
	ushort	numDrives	= CDriveList::sGetList()->GetNumDrives();
	
	if( mPrefs.unmountDrivesOnPBSleep &&
		PowerMgrExists() &&
		numDrives != 0)
		{
		InstallSleepProc();
		}
	else
		{
		RemoveSleepProc();
		}
	}
	
	
	void
CDriver::HandleSetPrefsCC( const PGPDiskDriverPrefs *	newPrefs)
	{
	pgpAssertAddrValid( newPrefs, PGPDiskDriverPrefs );
	
	mPrefs	= *newPrefs;
	
	InstallOrRemoveSleepProcAsAppropriate();
	}


	OSErr
CDriver::HandleDriveUnmountedCC(const PGPDriveUnmountedStruct *info)
	{
	CDriveList *	driveList	= CDriveList::sGetList();
	OSErr			err	= noErr;
	
	pgpAssertAddrValid( info, PGPDriveUnmountedStruct );

	if ( IsntNull( driveList ) )
		{
		CDrive *drive = driveList->GetDriveForDriveNumber(info->driveNumber);
		
		if ( IsntNull( drive ) )
			{
			drive->SetUnmountedAppPSN( info->unmountingProcess );
			}
		else
			{
			pgpDebugPStr( "\pHandleDriveUnmountedCC: no such drive" );
			err	= nsDrvErr;
			}
		}
		
	return( err );	
	}


	OSErr
CDriver::HandleChangeIOOptionsCC( const IOOptionsStruct	*options)
	{
	OSErr			err	= noErr;
	CDrive *		drive;

	drive = CDriveList::sGetList()->GetDriveForDriveNumber(
					options->driveNumber );
	
	if ( IsntNull( drive ) )
		{
		if ( options->tunerType == IOOptionsStruct::kReadTuner )
			{
			drive->SetReadTuner( &options->tuner );
			}
		else if ( options->tunerType == IOOptionsStruct::kWriteTuner )
			{
			drive->SetWriteTuner( &options->tuner );
			}
		else
			{
			err	= paramErr;
			}
		}
	else
		{
		err	= paramErr;
		}
	
	AssertNoErr( err, "CDriver::HandleChangeIOOptionsCC" );
	return( err );
	}
	
	


struct DriverGestaltVersResponse
{
	NumVersion driverVersion;
};
typedef struct DriverGestaltVersResponse DriverGestaltVersResponse;

struct DriverGestaltWideResponse
{
	Boolean	 supportsIOWPosOffset;
};
typedef struct DriverGestaltWideResponse DriverGestaltWideResponse;



/*
	The DriverGestaltVMOptionsResponse is returned by a disk driver in
	response to a kdgVMOptions Driver Gestalt request. This allows a disk
	driver to tell VM a few things about a disk drive. For example:

	* A drive that should never be in the page fault path should return
		kAllowVMNoneMask. Examples of this are drives that have manual
		eject buttons that are not disabled by software, drives with very slow
		throughput, or drives that depend on a network connection.
		
	* A drive that should never be written to but is safe for read-only file
		mapping should return kAllowVMReadOnlyMask. Examples of this are WORM
		drives where each write eats write-once space on the disk and CD-ROM
		drives which are read-only media.
		
	* A drive that should allow VM to create its main backing store
		file should return kAllowVMReadWriteMask. Examples of this are fast
		read/write drives that don't allow manual eject and don't use a network
		connection.

	A disk driver must look at the ioVRefNum field of the DriverGestaltParam
	to determine what disk drive this call is for. This is a per-drive call,
	not a per-driver call.

	The only three valid responses to kdgVMOptions at this time are
	kAllowVMNoneMask, kAllowVMReadOnlyMask, and kAllowVMReadWriteMask
	(i.e., setting only kAllowVMWriteBit is not valid).

	Important: All bits not defined here are reserved and should be set
	to zero until they are defined for a specific purpose.
*/


	OSErr
CDriver::HandleDriverGestalt( DriverGestaltParam *pb )
	{
	OSErr	err	= statusErr;
	void *	responseBytes	= &pb->driverGestaltResponse;
	
	switch( pb->driverGestaltSelector )
		{
		default:
			break;
			
		case kdgVersion:
			{
			DriverGestaltVersResponse *response;
			
			response = (DriverGestaltVersResponse *)responseBytes;

			response->driverVersion.majorRev =
					kPGPDiskDriverVersion >> 8;
			response->driverVersion.minorAndBugRev =
					kPGPDiskDriverVersion & 0xFF;
			response->driverVersion.stage = finalStage;
			response->driverVersion.nonRelRev = 0;
			err	= noErr;
			break;
			}
			
		case kdgDeviceType:	// this one is called frequently
			{
			DriverGestaltDevTResponse *response;
			
			response = (DriverGestaltDevTResponse *)responseBytes;

			response->deviceType	= kdgFileType;
			err	= noErr;
			break;
			}
			
		case kdgSync:
			{
			DriverGestaltSyncResponse *response;
			
			response = (DriverGestaltSyncResponse *)responseBytes;

			response->behavesSynchronously	= false;
			err	= noErr;
			break;
			}
			
		case kdgInterface:
			{
			DriverGestaltIntfResponse *response;
			
			response = (DriverGestaltIntfResponse *)responseBytes;

			response->interfaceType	= 'pgpD';
			err	= noErr;
			break;
			}
			
		case kdgWide:
			{
			DriverGestaltWideResponse *response;
			
			response = (DriverGestaltWideResponse *)responseBytes;
			
			response->supportsIOWPosOffset	= TRUE;
			err	= noErr;
			break;
			}
			
		case kdgVMOptions:
			{
			DriverGestaltVMOptionsResponse *response;

			response = (DriverGestaltVMOptionsResponse *)responseBytes;

			response->vmOptions	= kAllowVMNoneMask;
			err	= noErr;
			break;
			}
		}
	
	return( err );
	}
	
	
	OSErr
CDriver::HandleDriverConfigure( ParamBlockRec *pb )
	{
	OSErr	err	= controlErr;
	
	// there's nothing we support
	pb	= pb;
	
	return( err );
	}
	
	
/*_________________________________________________________________________
	Our drive has been ejected *or* unmounted.  We may get this call even for
	an unmount in the Finder because we set the 'kDriveInDriveMask' flag.
	
	With VM on, we have to mark our drives as ejectable so that they won't
	be asked to service a page fault.
_________________________________________________________________________*/
	OSErr
CDriver::HandleEjectDiskCC( CDrive *theDrive )
	{
	OSErr					err	= controlErr;
	
	CDrive::sAssertValid( theDrive );
			
	VCB *	vcb	= (VCB *)theDrive->GetVCB();
	
	if ( IsntNull( vcb ) )
		{
		// before accessing it, verify that it's still in the VCB list;
		// it may have been destroyed if the drive was unmounted versus
		// just being ejected.
		Boolean			stillInQueue	= false;
		const VCB *		curVCB			= (const VCB  *)(GetVCBQHdr()->qHead);
		
		while ( IsntNull( curVCB ) )
			{
			if ( curVCB == vcb )
				{
				stillInQueue	= TRUE;
				break;
				}
			curVCB	= (const VCB *)curVCB->qLink;
			}
		
		if ( stillInQueue )
			{
			// undo the effects of Eject
			vcb->vcbDrvNum	= theDrive->GetDriveNumber();
			vcb->vcbDRefNum	= mDCE->dCtlRefNum;
			
			pgpAssert(IsntNull(GetDriveQElem(theDrive->GetDriveNumber())));
			}
		}
	else
	{
		pgpDebugPStr( "\pCDriver::HandleEjectDiskCC: no vcb" );
	}
	
	return( noErr );
	}
	
	
	OSErr
CDriver::HandleRemoveDriveCC( short	driveNumber )
	{
	CDriveList *	driveList	= CDriveList::sGetList();
	OSErr			err	= noErr;
	
	if ( IsntNull( driveList ) )
		{
		CDrive *	drive	= driveList->GetDriveForDriveNumber( driveNumber );
		
		if ( IsntNull( drive ) )
			{
			pgpDebugPStr(
				"\pHandleRemoveDriveCC: removing drive from list and deleting");
			driveList->RemoveDrive( drive );
			delete drive;
			drive	= nil;
			}
		else
			{
			pgpDebugPStr( "\pHandleRemoveDriveCC: no such drive" );
			err	= nsDrvErr;
			}
		}
	return( err );
	}

	
	OSErr
CDriver::HandleOtherCC( ParamBlockRec *	pb)
	{
	OSErr			err	= controlErr;
	const short		csCode	= pb->cntrlParam.csCode;
	const short		driveNumber	= pb->cntrlParam.ioVRefNum;
	CDrive *		drive	= nil;
	
	// for those cases that require it, get the drive
	switch( csCode )
		{
		default:
			drive	= nil;
			break;
		
		case verifyDiskCC:
		case formatDiskCC:
		case physicalIconCC:
		case mediaIconCC:
		case driveInfoCC:
		case getDriveSizeCC:
			drive	= CDriveList::sGetList()->GetDriveForDriveNumber(
						driveNumber );
			CDrive::sAssertValid( drive );
			break;
		}
	
	
	switch( csCode )
		{
		default:
			pgpDebugFmtMsg( ( pgpaFmtPrefix,
					"HandleOtherCC: Unrecognized control call %ld",
					(long)csCode) );
			err	= controlErr;
			break;
		
		case ejectDiskCC:
			{
			short	 driveNumber	= pb->cntrlParam.ioVRefNum;
			
			// depending on the context, it could be the drive number or its
			// negative
			drive	= CDriveList::sGetList()->GetDriveForDriveNumber(
							driveNumber < 0 ? -driveNumber : driveNumber );
			CDrive::sAssertValid( drive );
			
			if ( IsntNull( drive ) )
				{
				err	= HandleEjectDiskCC( drive );
				}
			break;
			}
	
		case kDriverConfigureCode:
			err	= HandleDriverConfigure( pb );
			break;
			
		case kAddDriveCC:
			{
			CSystemZoneSetter	setter;
			AddDriveStruct *	addInfo;
			
			err	= ExtractFromCSParam( pb, &addInfo, sizeof( addInfo ) );
			if ( IsntErr( err ) )
				{
				err	= HandleAddDriveCC( addInfo );
				MarkCSParamUnderstood( pb );
				}
			break;
			}
			
		case kSetPrefsCC:
			{
			PGPDiskDriverPrefs *	prefs;
			
			err	= ExtractFromCSParam( pb, &prefs, sizeof( prefs ) );
			if ( IsntErr( err ) )
				{
				HandleSetPrefsCC( prefs );
				MarkCSParamUnderstood( pb );
				}
			break;
			}
		
		case kRemoveDriveCC:
			{
			long	driveNumber;	// yes, a long
			
			err	= ExtractFromCSParam( pb, &driveNumber,
						sizeof( driveNumber ) );
			if ( IsntErr( err ) )
				{
				err	= HandleRemoveDriveCC( (short)driveNumber );
				MarkCSParamUnderstood( pb );
				}
			break;
			}
			
		case kChangeWriteBufferSizeCC:
			{
			ulong	newBufferSize;
			
			pgpAssert( ! PBIsAsync( pb ) );
			
			err	= ExtractFromCSParam( pb, &newBufferSize,
						sizeof( newBufferSize ) );
			if ( IsntErr( err ) )
				{
				err	= ChangeScratchBufferSize( newBufferSize );
				MarkCSParamUnderstood( pb );
				}
			break;
			}
			
		
		case kSetIOOptionsCC:
			{
			IOOptionsStruct	*options;
			
			pgpAssert( ! PBIsAsync( pb ) );
			
			err	= ExtractFromCSParam( pb, &options, sizeof( options ) );
			if ( IsntErr( err ) )
				{
				err	= HandleChangeIOOptionsCC( options );
				MarkCSParamUnderstood( pb );
				}
			break;
			}
		
		case kToggleKeyBitsCC:
			HandleToggleKeyBitsCC();
			err	= noErr;
			break;
			
		case kDriveUnmountedCC:
			{
			PGPDriveUnmountedStruct	*info;
			
			pgpAssert( ! PBIsAsync( pb ) );
			
			err	= ExtractFromCSParam( pb, &info, sizeof( info ) );
			if ( IsntErr( err ) )
				{
				err = HandleDriveUnmountedCC( info );
				MarkCSParamUnderstood( pb );
				}
			break;
			}

		case killIOCC:
			// we don't support this...return an error
			err	= controlErr;
			break;
			
		case verifyDiskCC:
			err = noErr;
			break;
			
		case formatDiskCC:
			err = noErr;
			break;
			
		case physicalIconCC:
			/* return pointer to icon and where string */
			*(Ptr *) &pb->cntrlParam.csParam[ 0 ] =
					(Ptr)drive->GetPhysicalIcon( );
			err = noErr;
			break;
			
		case mediaIconCC:
			/* return pointer to icon and where string */
			*(Ptr *) &pb->cntrlParam.csParam[ 0 ] =
					(Ptr)drive->GetMediaIcon( );
			err = noErr;
			break;
			
		case driveInfoCC:
			// we do NOT want to respond to this
			err	= controlErr;
			break;
			
		case getDriveSizeCC:	/* Return SCSI csCode Partition Size */
			*(ulong *)&pb->cntrlParam.csParam[ 0 ] =
					drive->GetLogicalSizeInBlocks( );
			err = noErr;
			break;
			
		case accRun:
			{
			AccRun();
			err	= noErr;
			break;
			}
		}
	
	return( err );
	}


	
	
		


	
	
	
	OSErr
CDriver::HandleAddDriveCC( AddDriveStruct *	addInfo )
	{
	OSErr					err	= noErr;

	mPrefsSpec	= addInfo->prefsSpec;
	
	err	= SetNumPBs( addInfo->numExtents );
	if ( IsntErr( err ) )
		{
		// if this drive is writeable and we don't have a scratch buffer then
		// create one.
		if ( (! addInfo->mountReadOnly ) && IsNull( mScratchBuffer ) )
			{
			ulong	desiredSize;
			
			if ( addInfo->minimizeWriteBufferSize )
				{
				desiredSize	= kMinScratchBufferSize;
				}
			else
				{
				desiredSize	= CalcScratchBufferSize();
				}
			
			err	= ChangeScratchBufferSize( desiredSize );
			}
		}
		
		
	if ( IsntErr( err ) )
		{
		CDrive *				newDrive;
		CDrive::InitInfo		info;
			
		pgpClearMemory( &info, sizeof( info ) );
		info.info				= addInfo;
		info.numPBs				= mNumPBs;
		info.pbArray			= mPBArray;
		info.cipherProcRef		= mCipherProcRef;
		
		err	= CDrive::sCreate( &newDrive, &info );
		if ( IsntErr( err ) )
			{
			pgpAssert( IsNull( newDrive->GetVCB() ) );
			
			CDriveList::sGetList()->AddDrive( newDrive );
			
			FlagsDrvQEl	*driveQEl;
			
			err = AddDriveToQueue( addInfo->numBlocks, mDCE->dCtlRefNum,
							addInfo->mountReadOnly, &driveQEl);
			if( IsntErr( err ) )
				{
				newDrive->SetDriveQElement( driveQEl );
				addInfo->resultDriveNumber = driveQEl->el.dQDrive;
				
				mDCE->dCtlDelay		= 60;	// 60 ticks...call once per second
				}
			else
				{
				delete( newDrive );
				}
			}
		}
	else
		{
		pgpDebugPStr( "\pCDriver::HandleAddDriveCC: can't allocate enough pbs "
			"for drive" );
		}
	
	InstallOrRemoveSleepProcAsAppropriate();
		
	return( err );
	}




	
	ulong
CDriver::CalcScratchBufferSize( )
	{
	ulong	bufferSize	= kDefaultScratchBufferSize;
	long	gestaltResult;
	OSErr	err;
	
	err	= Gestalt( gestaltPhysicalRAMSize, &gestaltResult );
	if ( IsntErr( err ) )
		{
		ulong	ramSize	= gestaltResult;
		
		// use 128K for every 32MB of physical RAM, which is about 0.3%
		ulong	num32MBChunks	= ramSize / ( 32UL * 1024UL * 1024UL );
		if ( num32MBChunks != 0 )
			{
			bufferSize	*= num32MBChunks;
			}
		
		if ( bufferSize > kMaxScratchBufferSize )
			bufferSize	= kMaxScratchBufferSize;
		}

	return( bufferSize );
	}
	
/*_________________________________________________________________________
	Resize the scratch buffer by destroying the old one and creating a new
	one.
_________________________________________________________________________*/
	OSErr
CDriver::ChangeScratchBufferSize( ulong preferredSize)
	{
	OSErr	err	= noErr;
	void *	newBuffer;
	UInt32	newBufferSize;
		
	newBuffer	= pgpNewPtrMost( preferredSize, kMinScratchBufferSize,
						kMacMemory_UseSystemHeap, &newBufferSize );
	if ( IsntNull( newBuffer ) )
		{
		if ( preferredSize > mScratchBufferSize &&
				newBufferSize < mScratchBufferSize )
			{
			// we wanted a larger one than what we have, but we got less
			// than what we have now
			pgpDebugPStr( "\pChangeScratchBufferSize: unable to allocate a "
				"larger buffer" );
			pgpFreeMac( newBuffer );
			err	= memFullErr;
			}
		else
			{
			DisposeScratchBuffer( );
		
			mScratchBuffer		= newBuffer;
			mScratchBufferSize	= ( newBufferSize / kPGPDiskBlockSize ) *
					kPGPDiskBlockSize;
		
			CDrive::sSetScratchBuffer( mScratchBuffer, mScratchBufferSize );
			}
		}
	else
		{
		err	= memFullErr;
		}
	
	
	return( err );
	}

	
/*_________________________________________________________________________
	Allocate enough PBs to cover the maximum number of extents for the file
	representing this drive.  Always allocate some minimum number so that
	we can split read and write requests up into multiple requests in order
	to overlap I/O and ciphering.
_________________________________________________________________________*/
	void *
CDriver::AllocatePBs(
	const ushort	minRequired,
	ushort *		numAllocated)
	{
	const ulong kMinPBs	= 32;
	void *		pbArray	= nil;
	ushort		numPBs;
	
	numPBs			= MAX( minRequired, kMinPBs );
	*numAllocated	= numPBs;
	
	pbArray	= NewPtrSysClear( CDrive::kPBSize * numPBs  );
	pgpAssert( IsntNull( pbArray ) );
	
	return( pbArray );
	}


	
/*_________________________________________________________________________
	Allocate enough PBs to cover the maximum number of extents for the file
	representing this drive.  Always allocate some minimum number so that
	we can split read requests up into multiple requests in order to overlap
	decryption and reading
_________________________________________________________________________*/
	OSErr
CDriver::SetNumPBs( const ushort newNumber )
	{
	OSErr	err	= noErr;
	
	// we never reduce the number of pbs, only increase it,
	// since an existing drive may require the current number
	if ( newNumber > mNumPBs || IsNull( mPBArray ) )
		{
		void *	newArray;
		ushort	numNew;
		
		newArray	= AllocatePBs( newNumber, &numNew );
		if ( IsntNull( newArray ) )
			{
			DisposePBs();
			
			mPBArray	= newArray;
			mNumPBs		= numNew;
			
			CDriveList *	driveList	= CDriveList::sGetList();
			for( ushort driveIndex = 0;
					driveIndex < driveList->GetNumDrives(); ++driveIndex )
				{
				driveList->GetIndDrive( driveIndex )->SetPBs( newArray,
							numNew );
				}
			}
		else
			{
			err	= memFullErr;
			}
		}
	
	return( err );
	}

	
	
	
	OSErr
CDriver::InitCipherProc(  )
	{
	OSErr	err	= noErr;
	
	err	= CipherProc_Load( &mCipherProcRef );
	return( err );
	}
	
	
	

	void
CDriver::DisposeCipherProc( )
	{
	OSErr	err	= noErr;
	
	if ( mCipherProcRef != kInvalidCipherProcRef )
		{
		err	= CipherProc_Dispose( mCipherProcRef );
		mCipherProcRef	= kInvalidCipherProcRef;
		}
	AssertNoErr( err, "CDriver::DisposeCipherProc" );
	}
	





#pragma mark -

	

	static Boolean
PowerMgrExists()
	{
	Boolean	routinesExist = false;
	long		pmgrAttributes;
	
	if( Gestalt(gestaltPowerMgrAttr, &pmgrAttributes) == noErr )
		{
		if( ( pmgrAttributes & (1<<gestaltPMgrDispatchExists)) != 0 )
			{
			routinesExist	= (PMSelectorCount() >= 7);
			}
		}
	
	return( routinesExist );
	}




	pascal Boolean
CDriver::sAlertHook(
	DialogPtr		theDialog,
	EventRecord *	theEvent,
	short *			itemHit)
{
	Boolean		handledIt	= false;
	
	EnterCodeResource();
	
	(void) theDialog;
	(void) theEvent;
	
	CDriver *	theDriver	= CDriver::sSelf;
	if ( IsntNull( theDriver ) )
	{
		const ulong	kAlertTimeoutSeconds	= 15;
		ulong		now;
		
		GetDateTime( & now );
		
		if ( ( now - CDriver::sSelf->mAlertDisplayTime) >=
					kAlertTimeoutSeconds )
		{
			*itemHit	= 1;	// default item
			handledIt	= TRUE;
		}
	}
	
	ExitCodeResource();
	
	return( handledIt );
}

	Boolean
CDriver::TellUserSleepingWithMountedDrives(
	ConstStr255Param 	volumeName,
	Boolean 			canCancelSleep )
	{
	short	saveResFile	= CurResFile();
	THz		saveZone	= GetZone();
	Boolean	okToSleep	= TRUE;
	short	fileRef;
	
	SetZone( SystemZone() );
	
	SysBeep( 1 );
	
	fileRef	= FSpOpenResFile( &mPrefsSpec, fsRdPerm );
	if ( fileRef > 0 )
	{
		typedef struct MyQDGlobals
			{
			QDGlobals	qdGlobals;
			long		a5PointsHere;
			long		appleStorage[8];
			} MyQDGlobals;
			
		MyQDGlobals			myQDGlobals;
		void *				saveA5;
		void *				saveCurrentA5;
		short				result;
		UniversalProcPtr	saveDeskHook;
		DragGrayRgnUPP	saveDragHook;
		WindowRef			saveGhostWindow;
		CGrafPort			grafPort;
		
		// set up our own A5 world, in case its not valid

		saveDeskHook 	= LMGetDeskHook();
		saveDragHook 	= LMGetDragHook();
		saveGhostWindow = LMGetGhostWindow();
		saveCurrentA5	= LMGetCurrentA5();
		
		LMSetDeskHook( NULL );
		LMSetDragHook( NULL );
		LMSetGhostWindow( NULL );
		
		LMSetCurrentA5( (char *) &myQDGlobals.a5PointsHere );
		saveA5 = (void *) SetA5( (long) &myQDGlobals.a5PointsHere );
		
		InitGraf( &myQDGlobals.qdGlobals.thePort );
		OpenCPort( &grafPort );
		
		GetDateTime( &mAlertDisplayTime );
		
		ParamText( volumeName, nil, nil, nil );
		result = CautionAlert( canCancelSleep ? kSleepRequestAlertResID :
					kSleepDemandAlertResID, sAlertHook );
		okToSleep = ( result == 1 );
		
		CloseCPort( &grafPort );

		LMSetDeskHook( saveDeskHook );
		LMSetDragHook( saveDragHook );
		LMSetGhostWindow( saveGhostWindow );

		SetA5( (long) saveA5 );
		LMSetCurrentA5( (char *) saveCurrentA5 );
		
		CloseResFile( fileRef );
	}
	
	UseResFile( saveResFile );
	SetZone( saveZone );
	
	return( okToSleep );
	}
	


	Boolean
CDriver::SleepDismount(short selector)
{
	OSErr 	err	= noErr;
	short	result	= selector;
	Boolean	okToSleep = TRUE;
		
	if( (selector == sleepRequest) ||
		(selector == sleepDemand) || 
		(selector == dozeRequest ) ||
		(selector == dozeDemand ) )
	{
		CDriveList *	driveList;
		ushort			numDrives;
		ushort			numDrivesWeCouldntDo	= 0;
		Str255			problemDriveName;
		Boolean			canCancelSleep = FALSE;
		
		driveList	= CDriveList::sGetList();
		numDrives	= driveList->GetNumDrives();
		
		for( ushort driveIndex = 0; driveIndex < numDrives; ++driveIndex )
		{
			CDrive *		drive;
			
			drive	= driveList->GetIndDrive( driveIndex );
			
			err	= UnmountAndEjectDrive( drive->GetDriveNumber() );
			if ( IsErr( err ) )
			{
				const VCB *	vcb;
				
				++numDrivesWeCouldntDo;
				
				vcb	= GetVCBForDrive( drive->GetDriveNumber() );
				CopyPString( vcb->vcbVN, problemDriveName );
			}
		}
		
		// make sure any unmounted drives are actually deleted
		DriveHousekeeping( TRUE );
		
		if ( numDrivesWeCouldntDo != 0 )
		{
			if( ! mHaveShownSleepAlert )
			{
				Boolean	canCancelSleep;
				
				if( selector == sleepRequest ||
					selector == dozeRequest )
				{
					canCancelSleep = TRUE;
				}
				else
				{
					canCancelSleep = FALSE;
				}
				
				okToSleep = TellUserSleepingWithMountedDrives(
								problemDriveName, canCancelSleep );
								
				mHaveShownSleepAlert = TRUE;
			}
		}
	}
	else
	{
		mHaveShownSleepAlert = FALSE;
	}
	
	return( okToSleep );
}
	
	
	pascal void
CDriver::sSleepDismount()
	{
	long		selector;
	Boolean		okToSleep;
	long		saveD0;
	
#define GetSleepSelector()		GetD0();
#define SetSleepOK(void)		SetD0( 0 );

	saveD0		= GetD0();
	selector	= GetSleepSelector();
	okToSleep	= TRUE;
	
	EnterCodeResource();
	
	CDriver *	theDriver	= CDriver::sSelf;
	pgpAssert( IsntNull( theDriver ) );
	if ( IsntNull( theDriver ) )
		{
		okToSleep = theDriver->SleepDismount( selector );
		}
	
	ExitCodeResource();
	
	if( okToSleep )
		{
		SetSleepOK();	// indicate OK to go ahead
		}
	else
		{
		SetD0( saveD0 );
		}
	}




#pragma mark -

	OSErr
CDriver::Open( DCtlEntry *	dce)
	{
	OSErr				err = noErr;
	
	if ( ! DriverIsOpen() )
		{
		/* set dCtlFlags because Device Mgr copied header */
		dce->dCtlFlags |= dCtlEnableMask | dStatEnableMask |
								dWritEnableMask | dReadEnableMask |
								dNeedTimeMask | dNeedLockMask;
		dce->dCtlDelay		= 60;	// 60 ticks...call once per second
		
		dce->dCtlFlags	|= kmDriverGestaltEnableMask;
		
		mDriverIsOpen		= TRUE;
		mSleep.installed	= false;
		mDCE				= dce;
		
		err	= InitCipherProc();
		if ( IsntErr( err ) )
			{
			err	= CDriveList::sCreateList();
			if( IsntErr( err ) )
				{
				err = GetPGPDiskGloblalsPtr( &mPGPDiskGlobals );
				pgpAssertAddrValid( mPGPDiskGlobals, PGPDiskGlobals );
				}
			}
		}
		
	return( err );
	}



	OSErr
CDriver::Prime(
	ParamBlockRec *	pb,
	DCtlEntry *		dce)
	{
	OSErr				err	= noErr;
	
	if ( mCipherProcDamaged )
		{
		pgpDebugPStr( "\pCDriver::Prime: cipherProc is damaged" );
		err						= kCipherProcDamagedError;
		pb->ioParam.ioResult	= kCipherProcDamagedError;
		}
	else if ( DriverIsOpen() )
		{
		CDrive *			drive;
		CDriveList *		driveList;
		Boolean				isRead;
		const ushort		kIOTrapMask	= 0xA1FF;
		const ushort		trapWordMasked	= pb->ioParam.ioTrap &
									kIOTrapMask;
		
		pb->ioParam.ioResult	= kAsyncIOInProgress;
	
		pgpAssert( trapWordMasked == (ushort)_Read ||
						trapWordMasked == (ushort)_Write );
		isRead	= (trapWordMasked == (ushort)_Read);
		
		driveList	= CDriveList::sGetList();
		drive		= driveList->GetDriveForDriveNumber(
							pb->ioParam.ioVRefNum );
		CDrive::sAssertValid( drive );
		
		#if USE_LOG
		LogMsg( isRead ? "read" : "write");
		LogMsg( ( pb->ioParam.ioTrap & kAsyncMask ) != 0 ?
					", async: " : ", sync: ");
		Str255	str;
		pgpFormatPStr( str, sizeof( str ), TRUE,
				"startBlock = %lu, numBlocks = %lu [+" /*]*/,
			GetPBStartBlock( pb ), GetPBNumBlocks( pb ));
		LogMsg( str );
		#endif
		
		if ( isRead )
			{
			drive->Read( pb, dce );
			}
		else
			{
			drive->Write( pb, dce );
			}
		
		LogMsg( "+" );
		
		// we ALWAYS return this error in ALL cases
		err	= kAsyncIOInProgress;
		}
	else
		{
		LogMsg( "CDriver::Prime: not open" );
		err = notOpenErr;
		pb->ioParam.ioResult	= err;
		}
	
	// do *not* set pb->ioParam.ioResult here!
	
	return( err );
	}



	OSErr
CDriver::Close(
	ParamBlockRec *	pb,
	DCtlEntry *		dce)
{
	OSErr	err	= noErr;
	
	pb	= pb;
	
	if ( DriverIsOpen() )
		{
		DriveHousekeeping( TRUE );
		
		if ( CDriveList::sGetList()->GetNumDrives() == 0 )
			{
			dce->dCtlStorage = nil;	
			
			mDriverIsOpen		= false;
			
			CDriveList::sDisposeList();
			
			DisposeCipherProc();
			
			#if USE_LOG
			if ( IsntNull( gLog ) )
				{
				delete gLog;
				gLog	= nil;
				}
			#endif
			
			CDrive::sCleanup();
			}
		else
			{
			// drives still exist!
			err	= -1;
			pgpDebugPStr( "\pdrives still exist!");
			}
		}
	else
		{
		err	= notOpenErr;
		}
	
	
	pb->ioParam.ioResult	= err;
	
	return( err );
}



	OSErr
CDriver::ExtractFromCSParam(
	const ParamBlockRec *	pb,
	void *					bytes,
	ulong					numBytes)
	{
	OSErr	err	= paramErr;
	
	const DriverAPIStruct *	api	=
				(const DriverAPIStruct *)&pb->cntrlParam.csParam[ 0 ];
	
	if ( api->magic == DriverAPIStruct::kMagicIn )
		{
		err	= noErr;
		BlockMoveData( &api->data, bytes, numBytes );
		}
	
	AssertNoErr( err, "ExtractFromCSParam" );
	return( err );
	}


	void
CDriver::MarkCSParamUnderstood(
	ParamBlockRec *	pb)
	{
	DriverAPIStruct *	api	= (DriverAPIStruct *)&pb->cntrlParam.csParam[ 0 ];
	
	api->magic	= DriverAPIStruct::kMagicOut;
	}
	

	OSErr
CDriver::Control(
	ParamBlockRec *	pb,
	DCtlEntry *		/*dce*/)
	{
	OSErr 				err	= controlErr;
	const short			csCode	= pb->cntrlParam.csCode;
	
	if ( DriverIsOpen() )
		{
		err	= HandleOtherCC( pb );
		}
	else
		{
		err	= notOpenErr;
		}
	
	pb->cntrlParam.ioResult	= err;
	return( err );
}





	OSErr
CDriver::HandleGetDriverInfoSC( PGPDriverInfoStruct * info)
	{
	OSErr	err	= noErr;
	
	info->versionOut			= kPGPDiskDriverVersion;
	info->writeBufferSizeOut	= mScratchBufferSize;
	info->numDrivesOut			= 0;
	
	CDriveList *	driveList	= CDriveList::sGetList();
	if ( IsntNull( driveList ) )
		{
		info->numDrivesOut	= driveList->GetNumDrives();
		}
	
	pgpClearMemory( info->reserved, sizeof( info->reserved ) );
	
	return( err );
	}
	


	OSErr
CDriver::HandleGetDriveInfoSC( PGPDriveInfoStruct * info)
	{
	OSErr		err	= noErr;
	CDrive *	drive	= nil;
	
	pgpClearMemory( info->reserved, sizeof( info->reserved ) );
	
	drive	= CDriveList::sGetList()->GetDriveForDriveNumber(
						info->driveNumberIn );
	if ( IsntNull( drive ) )
		{
		info->baseDriveNumberOut	= drive->GetBaseDriveNumber();
		info->fileRefNum			= drive->GetFileRefNum();
		info->breaksUpRequests		= drive->GetBreaksUpRequests();
		drive->GetReadTuner( &info->readTuneOut );
		drive->GetWriteTuner( &info->writeTuneOut );
		}
	else
		{
		err	= nsDrvErr;
		}
	
	return( err );
	}
	
	
	
	OSErr
CDriver::HandleDriveStatusSC(
	short		driveNumber,
	DrvSts *	driveStatus )
	{
	OSErr			err	= paramErr;
	CDrive *		drive;
			
	drive	= CDriveList::sGetList()->GetDriveForDriveNumber( driveNumber );
	CDrive::sAssertValid( drive );
	if ( IsntNull( drive ) )
		{
		const DrvQEl *		driveQEl;
		
		err	= noErr;
		
		// Drive Stats...
		// See IM-II, p.215 for documentation
		
		driveStatus->track			= 0;				/* not applicable */
		driveStatus->writeProt		= drive->IsWriteProtected() ? 0x80 : 0;
		driveStatus->diskInPlace	= 1;				/* disk is in drive */
		driveStatus->installed		= 1;				/* drive installed */
		driveStatus->sides			= 0;				/* not applicable */
		driveStatus->twoSideFmt		= 0;				/* not applicable */
		driveStatus->needsFlush		= 0;				/* not applicable */
		driveStatus->diskErrs		= 0;				/* not applicable */
		
		/* Copy qLink through dQFSID from our DrvQEl */
		driveQEl = (DrvQElPtr)(GetDrvQHdr()->qHead);
		while( IsntNull( driveQEl ) )
			{
			if( driveQEl->dQDrive == driveNumber )
				{
				driveStatus->qLink		= driveQEl->qLink;
				driveStatus->qType		= driveQEl->qType;
				driveStatus->dQDrive	= driveQEl->dQDrive;
				driveStatus->dQRefNum	= driveQEl->dQRefNum;
				driveStatus->dQFSID		= driveQEl->dQFSID;
				break;
				}
				
			driveQEl = (const DrvQEl *)(driveQEl->qLink);
			}
		}
	
	return( err );
	}


	
	OSErr
CDriver::Status(
	ParamBlockRec *	pb,
	DCtlEntry *		/*dce*/)
	{
	OSErr				err	= statusErr;
	
	pgpAssert( DriverIsOpen() );
	
	if ( DriverIsOpen() )
		{
		const short			csCode	= pb->cntrlParam.csCode;
	
		if ( csCode == kGetDriverInfoSC ) /* "Driver" */
			{
			PGPDriverInfoStruct	*info;
			
			err	= ExtractFromCSParam( pb, &info, sizeof( info ) );
			if ( IsntErr( err ) )
				{
				err	= HandleGetDriverInfoSC( info );
				MarkCSParamUnderstood( pb );
				}
			}
		else if ( csCode == kGetDriveInfoSC ) /* "Drive" */
			{
			PGPDriveInfoStruct	*info;
			
			err	= ExtractFromCSParam( pb, &info, sizeof( info ) );
			if ( IsntErr( err ) )
				{
				err	= HandleGetDriveInfoSC( info );
				MarkCSParamUnderstood( pb );
				}
			}
		else
			{
			short	driveNumber;
			
			driveNumber	= pb->cntrlParam.ioVRefNum;
	
			switch( csCode )
				{
				default:
					pgpDebugFmtMsg( (pgpaFmtPrefix,
							"DRVRStatus: unknown csCode %ld", (long)csCode) );
					break;
					
				case returnFormatListSC:
					// fill in FormatListRec
					// we do not need to respond to this.
					// See FSM.h for comment.
					err	= controlErr;
					break;
					
				case driveStatusSC:
					{
					// Drive Stats...
					// See IM-II, p.215 for documentation
					err	= HandleDriveStatusSC( driveNumber,
								(DrvSts *)pb->cntrlParam.csParam );
					break;
					}
					
				case kDriverGestaltCode:	// see DriverGestalt.h
					{
					err	= HandleDriverGestalt( (DriverGestaltParam *)pb );
					break;
					}
			
				}
			}
		}
	else
		{
		err = notOpenErr;
		}

	pb->ioParam.ioResult	= err;
	
	return (err);
	}

#pragma mark -


	OSErr
CDriver::CreateDriver(
	CDriver **	driver)
	{
	OSErr	err	= noErr;
	
	*driver	= new CDriver;
	if ( IsNull( driver ) )
		{
		err		= memFullErr;
		*driver	= nil;
		}
	
	return( err );
	}


	inline CDriver *
GetDriverObject( const DCtlEntry * dce )
	{
	return( (CDriver *)dce->dCtlStorage );
	}


	pascal OSErr
DRVROpen(
	ParamBlockRec *	pb,
	DCtlEntry *		dce)
	{
	CDriver *	driver	= GetDriverObject( dce );
	OSErr		err	= noErr;
	
	if ( IsNull( driver ) )
		{
		CSystemZoneSetter	setter;
	
		pgpLeaksBeginSession( "DRVROpen" );
		
	#if USE_LOG
		gLog	= new CLogToFile( "\pPGPDisk DRVR Log" );
		pgpAssert( IsntNull( gLog ) );
		gLog->Append( "\r\r BEGIN LOG\r\r" );
	#endif
	
		err	= CDriver::CreateDriver( &driver );
		if ( IsntErr( err ) )
			{
			err	= driver->Open( dce );
			
			if ( IsErr( err ) )
				{
				delete driver;
				}
			}
		
		if ( IsntErr( err ) )
			{
			dce->dCtlStorage	= (Handle)driver;
		
			}
		}
	else
		{
		err	= noErr;
		}
	
	pb->ioParam.ioResult	= err;
	return( err );
	}



	pascal OSErr
DRVRPrime(
	ParamBlockRec *	pb,
	DCtlEntry *		dce)
	{
	CDriver *	driver	= GetDriverObject( dce );
	
	if ( IsntNull( driver ) )
		{
		return( driver->Prime( pb, dce ) );
		}
		
	pb->ioParam.ioResult	= kNoDriverErr;
	return( kNoDriverErr );
	}


	pascal OSErr
DRVRStatus(
	ParamBlockRec *	pb,
	DCtlEntry *		dce)
	{
	CDriver *	driver	= GetDriverObject( dce );
	OSErr		err	= -1;
	
	if ( IsntNull( driver ) )
		{
		err	= driver->Status( pb, dce );
		}
	else
		{
		err	= kNoDriverErr;
		}
		
	pgpAssert( err != kAsyncIOInProgress );
	pb->cntrlParam.ioResult	= err;
	return( err );
	}
	
	
	pascal OSErr
DRVRControl(
	ParamBlockRec *	pb,
	DCtlEntry *		dce)
	{
	CDriver *	driver	= GetDriverObject( dce );
	OSErr		err	= -1;
	
	if ( IsntNull( driver ) )
		{
		err	= driver->Control( pb, dce );
		}
	else
		{
		err	= kNoDriverErr;
		}
	
	pgpAssert( err != kAsyncIOInProgress );
	pb->cntrlParam.ioResult	= err;
	return( err );
	}
	
	
	pascal OSErr
DRVRClose(
	ParamBlockRec *	pb,
	DCtlEntry *		dce)
	{
	CDriver *	driver	= GetDriverObject( dce );
	OSErr		err	= kNoDriverErr;
	
	if ( IsntNull( driver ) )
		{
		err	= driver->Close( pb, dce );
		if ( IsntErr( err ) )
			{
			delete driver;
			pgpLeaksEndSession( );
			}
		}
		
	return( err );
	}
	
	











	
	