/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	

	$Id: CSearchWindow.cp,v 1.66.8.1 1998/11/12 03:08:14 heller Exp $
____________________________________________________________________________*/

#include <LowMem.h>

#include <UDesktop.h>
#include <UGAColorRamp.h>
#include <UReanimator.h>
#include <UKeyFilters.h>
#include <LPushButton.h>
#include <LCheckBox.h>
#include <LIconPane.h>
#include <LTextGroupBox.h>
#include <LPlacard.h>
#include <LPopupButton.h>
#include <LBevelButton.h>
#include <LChasingArrows.h>
#include <PP_Messages.h>
#include <PP_KeyCodes.h>
#include <string.h>

#include "CSearchWindow.h"

#include "CSearchPanel.h"
#include "CKeyView.h"
#include "CKeyTable.h"
#include "CPGPKeys.h"
#include "CWarningAlert.h"
#include "ResourceConstants.h"

#include "MacSecureMemory.h"
#include "MacStrings.h"
#include "pgpClientPrefs.h"
#include "pgpKeyServer.h"
#include "pgpClientLib.h"
#include "pgpEncode.h"
#include "pgpUserInterface.h"
#include "pgpErrors.h"

const Int16		kKeyViewTopSlop			= 7;
const Int16		kKeyViewSideSlop		= -1;
const Int16		kSearchBoxBorderSlop	= 4;
const ResIDT	kSearchPanelID			= 134;
const Int16		kMaxSearches			= 7;

enum	{
			kSWStringListID 	= 1013,
			kLocalKeyringStringID			= 1,
			kCurrentResultsStringID,
			kCancelSearchStringID,
			kSearchStringID,
			kKeysFoundStringID,
			kUnsupportedHKPFilterErrorStringID,
			kVerifyingSignaturesStringID,
			kAuthenticatePhraseStringID,
			kAuthRemoteServerStringID,
			kImportServerKeyStringID,
			kDeletedKeysStringID,
			kDisabledKeysStringID,
			kServerKeyChangedStringID,
			kServerLimitExceedStringID,
			kTLSAuthStringID
		};
enum	{
			kKSStringListID		= 1009
		};

static PGPError ServerCallback(	PGPContextRef	context,
									PGPEvent		*event,
									PGPUserValue	userValue);
static PGPError SigCheckCallback(	PGPContextRef		context,
									PGPEvent			*event,
									PGPUserValue		userValue);
static void ServerThreadProc(LThread& thread, void *arg);

CSearchWindow::CSearchWindow()
{
}

CSearchWindow::CSearchWindow(LStream *inStream)
	:	LWindow(inStream)
{
}

CSearchWindow::~CSearchWindow()
{
	SearchPanelList		*panelWalk,
						*panelNext;
	Int32				selection;
		
	if( IsntNull( mServers ) )
		PGPDisposePrefData( gPrefRef, mServers );

	panelWalk = mSearchPanelList;
	while(IsntNull(panelWalk))
	{
		panelNext = panelWalk->next;
		pgpFree(panelWalk);
		panelWalk = panelNext;
	}
	selection = mServerPopup->GetValue() - 1;
	PGPSetPrefNumber( gPrefRef, kPGPPrefLastSearchWindowSelection, selection );
	
	if( PGPKeySetRefIsValid( mActionInfo.authKeySet ) )
		PGPFreeKeySet( mActionInfo.authKeySet );
}

	void
CSearchWindow::FinishCreateSelf()
{
	LWindow::FinishCreateSelf();
	mServers				= NULL;
	mNumSearches			= 0;
	mActionInfo.done		= TRUE;
	mActionInfo.actionKeys	= kInvalidPGPKeySetRef;
	mActionInfo.authKeySet	= kInvalidPGPKeySetRef;
	mWorking				= FALSE;
	mLastServerValid		= FALSE;
	mSearchPanelList		= NULL;
	mPanelHeight			= 0;
	mSearchButton			= (LPushButton *)FindPaneByID(kSearchButton);
	mClearSearchButton		= (LPushButton *)FindPaneByID(kClearSearchButton);
	mMoreButton				= (LPushButton *)FindPaneByID(kMoreButton);
	mFewerButton			= (LPushButton *)FindPaneByID(kFewerButton);
	mServerPopup			= (LPopupButton *)FindPaneByID(kServerPopup);
	mSearchesBox			= (LTextGroupBox *)FindPaneByID(kSearchesBox);
	mPendingCheckBox		= (LCheckBox *)FindPaneByID( kPendingCheckbox );
	mProgressArrows			= (LChasingArrows *)FindPaneByID( kProgressArrows );
	mHeaderPlacard			= (LPlacard *)FindPaneByID( kHeaderPlaque );

	FillServerPopup();
	
	mSearchButton->SetDefaultButton(true);
	mSearchButton->AddListener(this);
	mClearSearchButton->AddListener(this);
	mMoreButton->AddListener(this);
	mFewerButton->AddListener(this);
	mFewerButton->Disable();

	if( !gIsAdminConfigured )
		mPendingCheckBox->Show();
	
	LCommander::SetDefaultCommander( this );
	LPane::SetDefaultView( this );

	mKeyView = (CKeyView *)UReanimator::ReadObjects( 'PPob', kKeyViewResID );
	pgpAssertAddrValid(mKeyView, CKeyView *);
	mKeyView->FinishCreate();
	mKeyView->ResizeFrameBy(1,0,false);

	mStatusCaption			= mKeyView->GetCaptionPane();
	mLockButton				= mKeyView->GetLockButton();
	mUnlockButton			= mKeyView->GetUnlockButton();
	mLockMessage			= mLockButton->GetValueMessage();
	mLockButton->AddListener(this);
	
	mUnlockButton->Show();

	AddEmptySearch(TRUE);
	
	SetThemeWindowBackground( GetMacPort(),
				kThemeActiveModelessDialogBackgroundBrush, FALSE );
	Show();
}

	void
CSearchWindow::ApplyForeAndBackColors(void) const
{
	LWindow::ApplyForeAndBackColors();
	if( IsActive() )
		SetThemeWindowBackground( GetMacPort(),
				kThemeActiveModelessDialogBackgroundBrush, FALSE );
	else
		SetThemeWindowBackground( GetMacPort(),
				kThemeInactiveModelessDialogBackgroundBrush, FALSE );
}

	void
CSearchWindow::AttemptClose()
{
	if(!mWorking)
	{
		BroadcastMessage(kSearchWindowClosed, NULL);
		LWindow::AttemptClose();
	}
	else
		::SysBeep(1);
}

	void
CSearchWindow::AddEmptySearch(Boolean	first)
{
	SearchPanelList *panelList,
					*panelWalk;
	Rect			frame;
	
	if(IsNull(mSearchPanelList))
		panelList = mSearchPanelList =
			(SearchPanelList *)pgpAlloc(sizeof(SearchPanelList));
	else
	{
		panelList = (SearchPanelList *)pgpAlloc(sizeof(SearchPanelList));
		for(panelWalk = mSearchPanelList; IsntNull(panelWalk->next) ;
			panelWalk = panelWalk->next)
			;
		panelWalk->next = panelList;
	}
	panelList->next = nil;
	mNumSearches++;
	LCommander::SetDefaultCommander( this );
	LPane::SetDefaultView( mSearchesBox );
	panelList->panel = (CSearchPanel *)
		UReanimator::ReadObjects( 'PPob', kSearchPanelID );
	pgpAssertAddrValid(panelList->panel, LView *);
	panelList->panel->FinishCreate();
	panelList->panel->AddListener(this);
	panelList->panel->AdjustCategory(first);
	panelList->panel->CalcLocalFrameRect(frame);
	mPanelHeight = frame.bottom - frame.top;
	mSearchesBox->ResizeFrameBy(0, mPanelHeight, true);
	panelList->panel->PlaceInSuperFrameAt(frame.left,
		mPanelHeight * (mNumSearches - 1), true);
	AdjustSizeToSearches();
	ResizeWindowBy(0, mPanelHeight);
	panelList->panel->Show();
	GetMinMaxSize(frame);
	frame.top += mPanelHeight;
	SetMinMaxSize(frame);
}

	void
CSearchWindow::RemoveLastSearch()
{
	SearchPanelList *panelLast,
					*panelWalk;
	Rect			frame;
					
	if(IsNull(mSearchPanelList))
		return;
	for(panelWalk = panelLast = mSearchPanelList;
		IsntNull(panelWalk->next) ;)
	{
		panelLast = panelWalk;
		panelWalk = panelWalk->next;
	}
	delete panelWalk->panel;
	if(panelWalk == mSearchPanelList)
		mSearchPanelList = NULL;
	panelLast->next = NULL;
	pgpFree(panelWalk);
	mSearchesBox->ResizeFrameBy(0, -mPanelHeight, true);
	AdjustSizeToSearches();
	mNumSearches--;
	ResizeWindowBy(0, -mPanelHeight);
	GetMinMaxSize(frame);
	frame.top -= mPanelHeight;
	SetMinMaxSize(frame);
}

	void
CSearchWindow::FillServerPopup()
{
	PGPSize		len;
	PGPError	err;
	Str255		itemString;
	PGPUInt32	selection = 0;
	Int32		numItems;
	
	numItems = mServerPopup->GetMaxValue();
	while( numItems > 3 )
		mServerPopup->DeleteMenuItem( numItems-- );
	
	mNumServers = 0;
	err = PGPGetPrefData(	gPrefRef, 
							kPGPPrefKeyServerList, 
							&len, 
							&mServers);
	pgpAssertNoErr(err);
	if(IsntPGPError(err))
	{
		char	url[kMaxServerNameLength + 1];
		
		mNumServers = len / sizeof(PGPKeyServerEntry);
		for(Int16 serverIndex = 0; serverIndex < mNumServers; serverIndex++)
		{
			if( IsKeyServerListed( mServers[serverIndex].flags ) )
			{
				PGPGetKeyServerURL( &mServers[serverIndex], url );
				CToPString( url, itemString );
				mServerPopup->AppendMenu( itemString );
			}
		}
	}
	PGPGetPrefNumber( gPrefRef, kPGPPrefLastSearchWindowSelection, &selection );
	mServerPopup->SetValue( selection + 1 );
}

	void
CSearchWindow::AdjustSizeToSearches()
{
	Rect	frame,
			sbFrame,
			mbFrame,
			fbFrame,
			pbFrame,
			kvFrame,
			plFrame;
	
	CalcPortFrameRect( frame );
	mSearchesBox->CalcPortFrameRect(sbFrame);
	mMoreButton->CalcPortFrameRect(mbFrame);
	mFewerButton->CalcPortFrameRect(fbFrame);
	mPendingCheckBox->CalcPortFrameRect(pbFrame);
	mHeaderPlacard->CalcPortFrameRect(plFrame);
	mMoreButton->PlaceInSuperFrameAt(mbFrame.left,
									sbFrame.bottom + kSearchBoxBorderSlop,
									true);
	mFewerButton->PlaceInSuperFrameAt(fbFrame.left,
									sbFrame.bottom + kSearchBoxBorderSlop,
									true);
	mPendingCheckBox->PlaceInSuperFrameAt(pbFrame.left,
									sbFrame.bottom + kSearchBoxBorderSlop,
									true);
	mMoreButton->CalcPortFrameRect(mbFrame);
	mHeaderPlacard->ResizeFrameTo(plFrame.right - plFrame.left,
							mbFrame.bottom + kKeyViewTopSlop,
							true);
	mHeaderPlacard->CalcPortFrameRect(plFrame);
	mKeyView->PlaceInSuperFrameAt(kKeyViewSideSlop,
									plFrame.bottom - 1,
									true);
	mKeyView->CalcPortFrameRect(kvFrame);
	mKeyView->ResizeFrameTo(kvFrame.right - kvFrame.left,
							frame.bottom - kvFrame.top + 1,
							true);
}

	PGPError
CSearchWindow::MakeFilterFromPanels(PGPFilterRef *outFilter)
{
	PGPError		err;
	PGPFilterRef	filter,
					filter2;
	SearchPanelList	*walk;
	
	*outFilter = filter = kInvalidPGPFilterRef;
	err = kPGPError_UnknownError;
	walk = mSearchPanelList;
	while(IsntNull(walk))
	{
		walk->panel->MakeFilter(&filter2);
		if(!PGPFilterRefIsValid(filter2))
		{
			if(PGPFilterRefIsValid(filter))
				PGPFreeFilter(filter);
			err = kPGPError_UnknownError;
			goto fatalErr;
		}
		if(PGPFilterRefIsValid(filter))
		{
			if(PGPFilterRefIsValid(filter2))
			{
				err = PGPIntersectFilters(filter, filter2, &filter);
				pgpAssertNoErr(err);
				if(IsPGPError(err))
					goto fatalErr;
			}
		}
		else
			filter = filter2;
		walk = walk->next;
	}
	*outFilter = filter;
	err = kPGPError_NoErr;
fatalErr:
	return err;
}
	
	void
CSearchWindow::FindCommandStatus(
	CommandT	inCommand,
	Boolean		&outEnabled,
	Boolean		&outUsesMark,
	Char16		&outMark,
	Str255		outName)
{
	switch(inCommand) 
	{
		case cmd_ServerDisable:
		case cmd_ServerDelete:
		{
			PGPUInt32	numKeys,
						numUserIDs,
						numSigs;
			
			mKeyView->GetKeyTable()->GetSelectionStats(
					&numKeys, &numUserIDs, &numSigs );
			if( !mWorking && mLastServerValid && numKeys > 0 )
				outEnabled = true;
			else
				outEnabled = false;
			break;
		}
		case cmd_Close:
			if(mWorking)
			{
				outEnabled = false;
				break;
			}
		default:
			 LWindow::FindCommandStatus(inCommand, outEnabled,
										outUsesMark, outMark, outName);
			break;
	}
}

	Boolean
CSearchWindow::ObeyCommand(
	CommandT	inCommand,
	void		*ioParam)
{
	Boolean		cmdHandled = true;
	
	switch( inCommand )
	{
		case cmd_ServerDelete:
		case cmd_ServerDisable:
		{
			PGPKeySetRef	selKeys;
			
			mKeyView->GetKeyTable()->GetSelectedKeySet( &selKeys );
			if( PGPKeySetRefIsValid( selKeys ) )
			{
				Str255	str;
				
				mStatusCaption->SetDescriptor("\p");
				::GetIndString(str, kSWStringListID, kCancelSearchStringID);
				mSearchButton->SetDescriptor(str);
				mSearchButton->Draw(NULL);
				mClearSearchButton->Disable();
				mProgressArrows->Show();
				mLockButton->Hide();
				mUnlockButton->Show();
				StartServerThread(	inCommand, kInvalidPGPFilterRef,
									selKeys,
									&mLastServer );
			}
			break;
		}
		default:
			cmdHandled = LWindow::ObeyCommand(inCommand, ioParam);
			break;
	}
	
	return cmdHandled;
}

	void
CSearchWindow::ListenToMessage(MessageT inMessage, void *ioParam)
{
	PGPError	err;
	
	if( inMessage == mLockMessage )
	{
		Str255			string;
		char			cstring[256],
						serverURL[256];
		PGPKeyRef		authKey;
		PGPKeyListRef	keyList;
		PGPKeyIterRef	keyIter;

		if( PGPKeySetRefIsValid( mActionInfo.authKeySet ) )
		{
			err = PGPOrderKeySet( mActionInfo.authKeySet, kPGPAnyOrdering,
							&keyList );	CKERR;
			err = PGPNewKeyIter( keyList, &keyIter );	CKERR;
			if( IsntPGPError( PGPKeyIterNext( keyIter, &authKey ) ) )
			{
				GetIndString( string, kSWStringListID, kTLSAuthStringID );
				PToCString( string, cstring );
				PGPGetKeyServerURL( &mLastServer, serverURL );
				PGPConfirmRemoteKeyDialog( cstring, serverURL, authKey,
										mActionInfo.tlsCipher, FALSE );
			}
			PGPFreeKeyIter( keyIter );
			PGPFreeKeyList( keyList );
		}
	}
	else switch(inMessage)
	{
		case kMoreButton:
			if(mNumSearches < kMaxSearches)
			{
				AddEmptySearch(false);
				mFewerButton->Enable();
				if(mNumSearches == kMaxSearches)
					mMoreButton->Disable();
			}
			break;
		case kFewerButton:
			if(mNumSearches > 1)
			{
				RemoveLastSearch();
				if(mNumSearches == kMaxSearches - 1)
					mMoreButton->Enable();
				if(mNumSearches == 1)
					mFewerButton->Disable();
			}
			break;
		case kSearchButton:
			if(mActionInfo.done)
				Search();
			else
				mActionInfo.abort = TRUE;
			break;
		case kClearSearchButton:
			while(mNumSearches > 0)
				RemoveLastSearch();
			AddEmptySearch(TRUE);
			mStatusCaption->SetDescriptor("\p");
			mFewerButton->Disable();
			mMoreButton->Enable();
			break;
		case kRebuildServersMessage:
			if( IsntNull( mServers ) )
				PGPDisposePrefData( gPrefRef, mServers );
			FillServerPopup();
			break;
	}
done:
	return;
}

	static PGPError
ImportNewServerKey(
	PGPKeyRef			key,
	PGPInt32			keyAlg,
	PGPKeyID			*keyID,
	PGPKeyServerEntry	*entry )
{
	PGPError		err = kPGPError_NoErr;
	PGPKeySetRef	keySet;
	
	err = PGPGetKeyIDString( keyID, kPGPKeyIDString_Full,
					entry->authKeyIDString );	CKERR;
	entry->authAlg = (PGPPublicKeyAlgorithm) keyAlg;

	err = PGPNewSingletonKeySet( key, &keySet );	CKERR;
	CPGPKeys::TheApp()->GetDefaultKeyView()->GetKeyTable()->
		ImportKeysFromKeySet( keySet );
	
done:
	return err;
}

	PGPError
ServerCallback(
	PGPContextRef	context,
	PGPEvent		*event,
	PGPUserValue	userValue)
{
	PGPError		err = kPGPError_NoErr;
	KSCallbackInfo	*callbackInfo = (KSCallbackInfo *)userValue;
	Str255			caption,
					moreCaption;
	char			cstr[256];
	
	switch( event->type )
	{
		case kPGPEvent_KeyServerEvent:
			if(LMGetTicks() - callbackInfo->lastIncrement > 5)
			{
				callbackInfo->lastIncrement = LMGetTicks();
				if(callbackInfo->lastState != event->data.keyServerData.state)
				{
					char		url[kMaxServerNameLength + 1];
					
					moreCaption[0] = 0;
					PGPGetKeyServerURL( &callbackInfo->serverInfo, url );
					CToPString( url, moreCaption );
					GetIndString(caption, kKSStringListID,
									event->data.keyServerData.state);
					AppendPString(moreCaption, caption);
					callbackInfo->statusCaption->SetDescriptor(caption);
					callbackInfo->lastState = event->data.keyServerData.state;
				}
			}
			if(callbackInfo->abort)
				err = kPGPError_UserAbort;
			break;
		case kPGPEvent_KeyServerSignEvent:
		{
			PGPKeyRef		authKey = kInvalidPGPKeyRef;
			char			*passphrase = NULL;
			
			GetIndString( caption, kSWStringListID,
							kAuthenticatePhraseStringID );
			PToCString( caption, cstr );
			err = PGPSigningPassphraseDialog( gPGPContext,
						CPGPKeys::TheApp()->GetKeySet(),
						&authKey,
						PGPOUIDialogPrompt( gPGPContext, cstr ),
						PGPOUIOutputPassphrase( gPGPContext, &passphrase ),
						PGPOLastOption( gPGPContext ) );
			if( IsntPGPError( err ) && IsntNull( passphrase ) )
			{
				err = PGPAddJobOptions( event->job,
						PGPOSignWithKey( gPGPContext, authKey,
							PGPOPassphraseBuffer( gPGPContext, passphrase,
												strlen( passphrase ) ),
							PGPOLastOption( gPGPContext ) ),
						PGPOClearSign( gPGPContext, TRUE ),
						PGPOLastOption( gPGPContext ) );
				PGPFreeData( passphrase );
			}
			break;
		}
		case kPGPEvent_KeyServerTLSEvent:
		{
			PGPtlsSessionRef	tls = event->data.keyServerTLSData.tlsSession;
			PGPKeyRef			remoteKey;
			PGPInt32			remoteKeyAlg;
			PGPKeyID			keyID,
								expectedID;
			char				authUserID[256];
			PGPSize				authUserIDLen;
			char				url[256];
			
			err = PGPtlsGetRemoteAuthenticatedKey( tls, &remoteKey );
			if( IsPGPError( err ) )
				break;
			err = PGPNewSingletonKeySet( remoteKey,
										&callbackInfo->authKeySet );
			if( IsPGPError( err ) )
				break;
			PGPtlsGetNegotiatedCipherSuite( tls, &callbackInfo->tlsCipher );
			PGPGetPrimaryUserIDNameBuffer( remoteKey, sizeof(authUserID),
											authUserID, &authUserIDLen );
			GetIndString( caption, kSWStringListID, kAuthRemoteServerStringID );
			CToPString( authUserID, moreCaption );
			AppendPString(moreCaption, caption);
			callbackInfo->statusCaption->SetDescriptor(caption);
			
			err = PGPGetKeyIDFromKey( remoteKey, &keyID );
			if( IsPGPError( err ) )
				break;
			err = PGPGetKeyNumber( remoteKey, kPGPKeyPropAlgID,
									&remoteKeyAlg );
			if( IsPGPError( err ) )
				break;
			if( callbackInfo->serverInfo.authAlg !=
				kPGPPublicKeyAlgorithm_Invalid )
				err = PGPGetKeyIDFromString(
						callbackInfo->serverInfo.authKeyIDString,
						&expectedID );
			if( IsPGPError( err ) )
				break;
			if( ( callbackInfo->serverInfo.authAlg ==
					kPGPPublicKeyAlgorithm_Invalid ) ||
					PGPCompareKeyIDs( &expectedID, &keyID ) )
			{
				// Put up Authentication Confirmation Dialog
				// with customized text stating that the
				// server key has changed.  If the user does
				// not approve the key, return an error and
				// do not set lock icon.  If the user does
				// approve the key, set the new key in the
				// KeyServerEntry and save it
				
				GetIndString( caption, kSWStringListID,
							kServerKeyChangedStringID );
				PToCString( caption, cstr );
				PGPGetKeyServerURL( &callbackInfo->serverInfo, url );
				UDesktop::Deactivate();
				err = PGPConfirmRemoteKeyDialog( cstr, url, remoteKey,
					 callbackInfo->tlsCipher, TRUE );
				UDesktop::Activate();
				if( IsntPGPError( err ) )
				{
					err = ImportNewServerKey( remoteKey, remoteKeyAlg,
								&keyID, &callbackInfo->serverInfo );
					callbackInfo->serverChanged = TRUE;
				}
			}
			if( IsntPGPError( err ) )
			{
				callbackInfo->unlockButton->Hide();
				callbackInfo->lockButton->Show();
			}
			break;
		}
		case kPGPEvent_KeyServerIdleEvent:
			LThread::Yield();
			if( callbackInfo->abort && IsntNull( callbackInfo->server ) )
				PGPCancelKeyServerCall( callbackInfo->server );
			break;
	}	
	return err;
}

	void
CSearchWindow::SpendTime(const EventRecord	&inMacEvent)
{
	if(mActionInfo.done)
	{
		PGPUInt32	numKeys = 0;
		Str255		numStr,
					str;
		PGPError	err;
		
		StopRepeating();
		mProgressArrows->Hide();
		
		str[0] = 0;
		if( PGPKeySetRefIsValid( mActionInfo.actionKeys ) )
		{
			err = PGPCountKeys(mActionInfo.actionKeys, &numKeys);
			pgpAssertNoErr(err);
			switch( mActionInfo.command )
			{
				case cmd_Search:
					if( IsntPGPError( mActionInfo.result ) ||
						( numKeys > 0 ) )
					{
						mKeyView->CloseKeys();
						mKeyView->SetKeyDBInfo(mActionInfo.actionKeys,
												false, false);
					}
					GetIndString( str, kSWStringListID, kKeysFoundStringID );
					break;
				default:
					PGPFreeKeySet( mActionInfo.actionKeys );
					mActionInfo.actionKeys = kInvalidPGPKeySetRef;
					GetIndString( str, kSWStringListID,
						mActionInfo.command == cmd_ServerDelete ?
						kDeletedKeysStringID :  kDisabledKeysStringID );
					break;
			}
		}
		if( mActionInfo.result != kPGPError_UserAbort )
		{
			if( mActionInfo.result == kPGPError_ServerPartialSearchResults )
			{
				GetIndString( str, kSWStringListID,
								kServerLimitExceedStringID );
				NumToString( numKeys, numStr );
				PrintPString( str, str, numStr );
			}
			else if( IsPGPError( mActionInfo.result ) )
			{
				if(mActionInfo.result == kPGPError_UnsupportedHKPFilter)
				{
					CWarningAlert::Display(kWACautionAlertType,
									kWAOKStyle,
									kSWStringListID,
									kUnsupportedHKPFilterErrorStringID);
				}
				else
					ReportPGPError(mActionInfo.result);
				str[0] = 0;
			}
			else
			{
				NumToString( numKeys, numStr );
				PrintPString( str, str, numStr );
			}
		}
		else
			str[0] = 0;
		mStatusCaption->SetDescriptor( str );
		if( PGPFilterRefIsValid( mActionInfo.filter ) )
		{
			err = PGPFreeFilter( mActionInfo.filter );
			pgpAssertNoErr(err);
		}
		GetIndString( str, kSWStringListID, kSearchStringID );
		mSearchButton->SetDescriptor(str);
		mClearSearchButton->Enable();
		BroadcastMessage(kSearchComplete, NULL);
		mWorking = FALSE;
		
		if( mActionInfo.serverChanged )
		{
			PGPUInt32	serverIndex;
			
			for( serverIndex = 0; serverIndex < mNumServers; serverIndex++ )
			{
				if( ( mServers[serverIndex].protocol ==
					mActionInfo.serverInfo.protocol ) &&
					( mServers[serverIndex].serverPort ==
					mActionInfo.serverInfo.serverPort ) &&
					!strcmp( mServers[serverIndex].domain,
					mActionInfo.serverInfo.domain ) )
				{
					pgpCopyMemory( &mActionInfo.serverInfo,
									&mServers[serverIndex],
									sizeof( PGPKeyServerEntry ) );
					PGPSetPrefData(	gPrefRef, kPGPPrefKeyServerList, 
									mNumServers * sizeof( PGPKeyServerEntry ),
									mServers);
					break;
				}
			}
		}
	}
	LThread::Yield();
}

	PGPError
SigCheckCallback(
	PGPContextRef		context,
	PGPEvent			*event,
	PGPUserValue		userValue)
{
	KSCallbackInfo *callbackInfo = (KSCallbackInfo *)userValue;
	
	if(LMGetTicks() - callbackInfo->lastIncrement > 5)
	{
		callbackInfo->lastIncrement = LMGetTicks();
		LThread::Yield();
	}
	return kPGPError_NoErr;
}

	void
CSearchWindow::SearchServer(void *arg)
{
	PGPKeyServerRef					server = kInvalidPGPKeyServerRef;
	PGPError						err;
	PGPKeyServerAccessType			accessType = kPGPKeyServerAccessType_Administrator;
	PGPKeyServerKeySpace			keySpace = kPGPKeyServerKeySpace_Normal;
	PGPKeySetRef					failedKeys = kInvalidPGPKeySetRef;
	StPGPPreserveKeyServerStorage	keyserverStorage;

	
	/*if( !gIsAdminConfigured )
		accessType = kPGPKeyServerAccessType_Administrator;*/

	if( mPendingCheckBox->GetValue() )
		keySpace = kPGPKeyServerKeySpace_Pending;

	err = PGPNewKeyServerFromHostName( gPGPContext,
				mActionInfo.serverInfo.serverDNS,
				mActionInfo.serverInfo.serverPort,
				mActionInfo.serverInfo.protocol,
				accessType, keySpace, &server );
	pgpAssertNoErr(err);
	if( IsntPGPError( err ) )
	{
		PGPtlsSessionRef	tls = kInvalidPGPtlsSessionRef;
		
		mActionInfo.server = server;
		err = PGPSetKeyServerEventHandler( server,
					ServerCallback, &mActionInfo );
		pgpAssertNoErr( err );
		err = PGPSetKeyServerIdleEventHandler( ServerCallback,
												&mActionInfo );
		pgpAssertNoErr( err );
		if( mActionInfo.serverInfo.protocol == kPGPKeyServerType_LDAPS )
		{
			err = PGPNewTLSSession( gTLSContext, &tls );
			pgpAssertNoErr( err );
		}
		if( IsntPGPError( err ) && ( mActionInfo.command != cmd_Search ) &&
			PGPtlsSessionRefIsValid( tls ) )
		{
			PGPKeyRef		authKey		= kInvalidPGPKeyRef;
			char			*passphrase	= NULL;
			Str255			pstr;
			char			cstr[256];
			
			::GetIndString( pstr, kSWStringListID,
							kAuthenticatePhraseStringID );
			PToCString( pstr, cstr );
			err = PGPSigningPassphraseDialog( gPGPContext,
						CPGPKeys::TheApp()->GetKeySet(),
						&authKey,
						PGPOUIDialogPrompt( gPGPContext, cstr ),
						PGPOUIOutputPassphrase( gPGPContext, &passphrase ),
						PGPOLastOption( gPGPContext ) );
			if( IsntPGPError( err ) )
				err = PGPtlsSetLocalPrivateKey( tls, authKey,
						PGPOPassphrase( gPGPContext, passphrase ),
						PGPOLastOption( gPGPContext ) );
			if( IsntNull( passphrase ) )
				PGPFreeData( passphrase );
		}
		err = PGPKeyServerOpen( server, tls );
		if( IsntPGPError( err ) )
		{
			
			mLastServerValid = TRUE;
			pgpCopyMemory( &mActionInfo.serverInfo, &mLastServer,
							sizeof(PGPKeyServerEntry ) );
			if( IsntPGPError( err ) )
				switch( mActionInfo.command )
				{
					case cmd_Search:
						err = PGPQueryKeyServer( server, mActionInfo.filter,
												&mActionInfo.actionKeys );
						break;
					case cmd_ServerDelete:
						err = PGPDeleteFromKeyServer( server,
									mActionInfo.actionKeys, &failedKeys );
						break;
					case cmd_ServerDisable:
						err = PGPDisableFromKeyServer( server,
									mActionInfo.actionKeys, &failedKeys );
						break;
				}
			mActionInfo.result = err;
			PGPKeyServerClose( server );
		}
		else
			mActionInfo.result = err;
		LThread::Yield();
		if( IsntPGPError( err ) && mActionInfo.command == cmd_Search
			&& PGPKeySetRefIsValid( mActionInfo.actionKeys ) )
		{
			PGPKeySetRef	combinedSet;
			Str255			caption;
			
			::GetIndString(caption, kSWStringListID,
							kVerifyingSignaturesStringID);
			mStatusCaption->SetDescriptor(caption);
			err = PGPNewKeySet(gPGPContext, &combinedSet);
			pgpAssertNoErr(err);
			if( IsntPGPError( err ) )
			{
				err = PGPAddKeys(mActionInfo.actionKeys, combinedSet);
				pgpAssertNoErr(err);
				err = PGPAddKeys(CPGPKeys::TheApp()->GetKeySet(), combinedSet);
				pgpAssertNoErr(err);
				err = PGPCheckKeyRingSigs(	mActionInfo.actionKeys,
											combinedSet,
											FALSE, SigCheckCallback,
											&mActionInfo);
				pgpAssertNoErr(err);
				LThread::Yield();
				err = PGPPropagateTrust(combinedSet);
				pgpAssertNoErr(err);
				LThread::Yield();
				PGPFreeKeySet(combinedSet);
			}
		}
		err = PGPFreeKeyServer(server);
		pgpAssertNoErr(err);
		if( PGPtlsSessionRefIsValid( tls ) )
			PGPFreeTLSSession( tls );
	}
	else
		mActionInfo.result = err;
	LThread::Yield();
	mActionInfo.done = TRUE;
	if( PGPKeySetRefIsValid( failedKeys ) )
		PGPFreeKeySet( failedKeys );
}

	void
ServerThreadProc(LThread& thread, void *arg)
{
	CSearchWindow *searchWindow = (CSearchWindow *)arg;
	
	searchWindow->SearchServer(NULL);
}

	void
CSearchWindow::StartServerThread(
	CommandT			inCommand,
	PGPFilterRef		inFilter,
	PGPKeySetRef		inKeys,
	PGPKeyServerEntry	*inServerInfo )
{
	mWorking = TRUE;
	BroadcastMessage(kSearchInProgress, NULL);
	mActionInfo.command			= inCommand;
	mActionInfo.progressBar 	= mProgressArrows;
	mActionInfo.lockButton		= mLockButton;
	mActionInfo.unlockButton	= mUnlockButton;
	mActionInfo.statusCaption	= mStatusCaption;
	mActionInfo.lastIncrement	= 0;
	mActionInfo.lastState		= 0;
	mActionInfo.filter			= inFilter;
	mActionInfo.actionKeys		= inKeys;
	mActionInfo.server			= kInvalidPGPKeyServerRef;
	mActionInfo.serverChanged	= FALSE;
	mActionInfo.result			= kPGPError_NoErr;
	mActionInfo.abort			= mActionInfo.done	= FALSE;
	pgpCopyMemory( inServerInfo, &mActionInfo.serverInfo,
					sizeof(PGPKeyServerEntry) );
	mServerThread = new LSimpleThread( ServerThreadProc, this );
	mServerThread->Resume();
	StartRepeating();
}

	void
CSearchWindow::Search()
{
	PGPFilterRef			filter = NULL;
	PGPKeySetRef			resultKeys = kInvalidPGPKeySetRef,
							indKeySet;
	Int16					targetValue;
	PGPKeyServerEntry		serverInfo;
	Str255					str;
	PGPError				err = kPGPError_NoErr;
	
	mLastServerValid = FALSE;
	mStatusCaption->SetDescriptor("\p");
	::GetIndString(str, kSWStringListID, kCancelSearchStringID);
	mSearchButton->SetDescriptor(str);
	mSearchButton->Draw(NULL);
	mClearSearchButton->Disable();
	mProgressArrows->Show();
	mLockButton->Hide();
	mUnlockButton->Show();
	
	if( PGPKeySetRefIsValid( mActionInfo.authKeySet ) )
	{
		PGPFreeKeySet( mActionInfo.authKeySet );
		mActionInfo.authKeySet = kInvalidPGPKeySetRef;
	}

	pgpClearMemory( &mActionInfo, sizeof(KSCallbackInfo) );
	err = MakeFilterFromPanels(&filter);
	if( IsntPGPError( err ) && PGPFilterRefIsValid( filter ) )
	{
		targetValue = mServerPopup->GetValue() - 1;
		switch(targetValue)
		{
			case 0:		// Local Keyring
				{
					err = PGPFilterKeySet( CPGPKeys::TheApp()->GetKeySet(),
											filter, &indKeySet );
					pgpAssertNoErr(err);
					if(IsntPGPError(err) && PGPKeySetRefIsValid(indKeySet))
					{
						err = PGPNewKeySet(gPGPContext, &resultKeys);
						pgpAssertNoErr(err);
						if(IsntPGPError(err) && PGPKeySetRefIsValid(resultKeys))
						{
							err = PGPAddKeys(indKeySet, resultKeys);
							pgpAssertNoErr(err);
							err = PGPCommitKeyRingChanges(resultKeys);
							pgpAssertNoErr(err);
						}
						err = PGPFreeKeySet(indKeySet);
						pgpAssertNoErr(err);
					}
					mActionInfo.command		= cmd_Search;
					mActionInfo.filter		= filter;
					mActionInfo.actionKeys	= resultKeys;
					mActionInfo.result		= err;
					mActionInfo.done		= TRUE;
					StartRepeating();
					mWorking				= TRUE;
					BroadcastMessage(kSearchInProgress, NULL);
				}
				break;
			case 1:		// Current Results
				{
					if(PGPKeySetRefIsValid(mKeyView->GetKeySet()))
					{
						err = PGPFilterKeySet(
								mKeyView->GetKeySet(),
								filter, &indKeySet);
						pgpAssertNoErr(err);
						if(IsntPGPError(err) && PGPKeySetRefIsValid(indKeySet))
						{
							err = PGPNewKeySet(gPGPContext, &resultKeys);
							pgpAssertNoErr(err);
							if(IsntPGPError(err) &&
								PGPKeySetRefIsValid(resultKeys))
							{
								err = PGPAddKeys(indKeySet, resultKeys);
								pgpAssertNoErr(err);
							}
							err = PGPFreeKeySet(indKeySet);
							pgpAssertNoErr(err);
						}
					}
					mActionInfo.command			= cmd_Search;
					mActionInfo.filter			= filter;
					mActionInfo.actionKeys		= resultKeys;
					mActionInfo.result			= err;
					mActionInfo.done			= TRUE;
					StartRepeating();
					mWorking					= TRUE;
					BroadcastMessage( kSearchInProgress, NULL );
				}
				break;
			default:
			{
				Int16	serverInx,
						listedServerInx;
				
				targetValue -= 2;
				for( listedServerInx = serverInx = 0;
						serverInx < mNumServers; serverInx++ )
				{
					if( IsKeyServerListed( mServers[serverInx].flags ) )
						listedServerInx++;
					if( listedServerInx == targetValue )
						break;
				}
				
				pgpCopyMemory( &mServers[serverInx],
								&serverInfo,
								sizeof(PGPKeyServerEntry) );
				if( IsntPGPError(err) )
					StartServerThread(	cmd_Search,
										filter,
										kInvalidPGPKeySetRef,
										&serverInfo );
				break;
			}
		}
	}
	if( !mWorking )
	{
		if( IsntNull( filter ) )
			PGPFreeFilter( filter );
		StartRepeating();
		mActionInfo.result = kPGPError_UserAbort;
		mActionInfo.done = TRUE;
		mActionInfo.filter = kInvalidPGPFilterRef;
	}
	mKeyView->Empty();
}

	Boolean
CSearchWindow::HandleKeyPress(const EventRecord& inKeyEvent)
{
	Boolean		keyHandled = false;

	if(inKeyEvent.modifiers & cmdKey)
		keyHandled = LCommander::HandleKeyPress(inKeyEvent);
	else if(UKeyFilters::IsActionKey(inKeyEvent.message))
	{
		switch (inKeyEvent.message & charCodeMask)
		{
			case char_Return:
			case char_Enter: 
				mSearchButton->SimulateHotSpotClick(kControlButtonPart);
				keyHandled = true;
				break;
		}
	}

	return keyHandled;
}

