/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	$Id: CPGPRecipientGrafPort.cp,v 1.22.10.1 1998/11/12 03:20:43 heller Exp $
____________________________________________________________________________*/

#include <LCheckBox.h>
#include <LGAFocusBorder.h>
#include <LPopupButton.h>
#include <LPushButton.h>
#include <LStaticText.h>
#include <PP_KeyCodes.h>
#include <PP_Messages.h>
#include <UDrawingUtils.h>
#include <UEnvironment.h>
#include <UGAColorRamp.h>

#include "pgpDebug.h"
#include "pgpErrors.h"
#include "pgpEncode.h"
#include "pgpGroups.h"
#include "pgpKeys.h"
#include "pgpKeyServer.h"
#include "pgpMem.h"

#include <string.h>
#include "MacStrings.h"

#include "CPassphraseEdit.h"
#include "CPGPRecipientGrafPort.h"
#include "CUserIDTable.h"
#include "PGPsdkUILibDialogs.h"
#include "PGPsdkUILibUtils.h"
#include "pgpDialogs.h"

enum
{
	kMacBinaryPopupNoneMenuItem	= 1,
	kMacBinaryPopupSmartMenuItem,
	kMacBinaryPopupYesMenuItem
};

CPGPRecipientGrafPortView::CPGPRecipientGrafPortView(LStream *inStream)
	: CPGPUIGrafPortView(inStream)
{
	mUpdateFromServerButton		= NULL;
	mWindowIsMoveable			= TRUE;
	mWindowIsResizeable			= TRUE;
	mMinMaxSize.top				= 300;	// Minimum resize height
	mMinMaxSize.bottom			= 32000;// Maximum resize height
	mMinMaxSize.left			= 440;	// Minimum resize width
	mMinMaxSize.right			= 640;	// Maximum resize width
	mOptionsButton				= NULL;
	mOptions					= NULL;
	
	pgpClearMemory( &mRecipients, sizeof( mRecipients ) );
}

CPGPRecipientGrafPortView::~CPGPRecipientGrafPortView()
{	
	PGPDisposeRecipientsList( &mRecipients );
}

	void
CPGPRecipientGrafPortView::FinishCreateSelf()
{
	LGAFocusBorder	*userIDBorder;

	CPGPUIGrafPortView::FinishCreateSelf();
	
	mRecipientTable =
		(CUserIDTable *) FindPaneByID( kRecipientTablePaneID );
	mUserIDTable =
		(CUserIDTable *) FindPaneByID( kUserIDTablePaneID );
	mOptionsButton = 
		(LPushButton *) FindPaneByID( kOptionsButtonPaneID );
	mUpdateFromServerButton =
		(LPushButton *) FindPaneByID( kUpdateFromServerButtonPaneID );
	
	mUserIDTable->InsertCols( 1, 1, NULL, 0, TRUE );
	mUserIDTable->AddListener( this );
	
	mRecipientTable->InsertCols( 1, 1, NULL, 0, TRUE );
	mRecipientTable->AddListener( this );
	mRecipientTable->SetAllowDeleteKeyToSend( TRUE );

	userIDBorder = (LGAFocusBorder *)
					FindPaneByID( kUserIDFocusBorderPaneID );

	SwitchTarget( mUserIDTable );
	AdjustButtons();
}

	PGPUInt32
CPGPRecipientGrafPortView::MoveSelectedUsers(
	PGPRecipientUserLocation destinationList)
{
	PGPError					err = kPGPError_NoErr;
	PGPRecipientUser			**userList;
	PGPUInt32					numMovedUsers;
	PGPUInt32					numExtraMovedUsers;
	TableIndexT					numRows;
	TableIndexT					numColumns;
	CUserIDTable				*sourceTable;
	CUserIDTable				*destTable;
	PGPRecipientUserLocation	sourceList;
	
	numMovedUsers 		= 0;
	numExtraMovedUsers	= 0;
	
	if( destinationList == kPGPRecipientUserLocation_UserList )
	{
		sourceTable = mRecipientTable;
		destTable	= mUserIDTable;
		sourceList	= kPGPRecipientUserLocation_RecipientList;
	}
	else
	{
		sourceTable = mUserIDTable;
		destTable	= mRecipientTable;
		sourceList	= kPGPRecipientUserLocation_UserList;
	}
	
	sourceTable->GetTableSize( numRows, numColumns );
	
	/* Fast and dirty: Allocate list assuming all cells selected */
	
	userList = (PGPRecipientUser **) PGPNewData(
						PGPGetContextMemoryMgr( mContext ),
						numRows * sizeof( PGPRecipientUser * ), 0 );
	if( userList != NULL )
	{
		PGPUInt32	numSelectedUsers;
		STableCell	cell(0, 0);
		
		numSelectedUsers = 0;
		
		while( sourceTable->GetNextSelectedCell( cell ) )
		{
			sourceTable->GetCellUser( cell, &userList[numSelectedUsers] );
			numSelectedUsers++;
		}
		
		if( numSelectedUsers != 0 )
		{
			PGPBoolean	movedARRs;
			
			err = PGPMoveRecipients( &mRecipients, destinationList,
						numSelectedUsers, userList, &numMovedUsers,
						&movedARRs );
			if( IsntPGPError( err ) && numMovedUsers != 0 )
			{
				sourceTable->GetStorageArray()->SetKeepSorted( FALSE );
				destTable->GetStorageArray()->SetKeepSorted( FALSE );

				/* Move items between tables */
				
				cell.col = 1;
				for( cell.row = numRows; cell.row >= 1; cell.row-- )
				{
					PGPRecipientUser	*curUser;
						
					sourceTable->GetCellUser( cell, &curUser );
					if( curUser->location != sourceList )
					{
						sourceTable->RemoveRows( 1, cell.row, FALSE );
					}
					
					if( curUser->location == destinationList )
					{
						destTable->InsertRows( 1, 30000, &curUser,
									sizeof( curUser ), FALSE );
					}
				}
				
				sourceTable->GetStorageArray()->SetKeepSorted( TRUE );
				destTable->GetStorageArray()->SetKeepSorted( TRUE );
				
				sourceTable->Refresh();
				destTable->Refresh();
				
				AdjustButtons();
			}
		}
		
		PGPFreeData( userList );
	}
	else
	{
		err = kPGPError_OutOfMemory;
	}
	
	if( IsPGPError( err ) )
	{
		SysBeep( 1 );
	}
	
	return( numMovedUsers + numExtraMovedUsers );
}

	PGPUInt32
CPGPRecipientGrafPortView::SendSelectedToRecipients(void)
{
	return( MoveSelectedUsers( kPGPRecipientUserLocation_RecipientList ) );
}


	PGPUInt32
CPGPRecipientGrafPortView::SendSelectedToUserIDs(void)
{
	return( MoveSelectedUsers( kPGPRecipientUserLocation_UserList ) );
}

	void
CPGPRecipientGrafPortView::ListenToMessage(MessageT inMessage, void *ioParam)
{
	switch( inMessage )
	{
		case 'Send':
		{
			TableIndexT	numRows;
			TableIndexT	numColumns;
			
			// Send the selected cells from the table object in ioParam to
			// the other table.
			
			if( (CUserIDTable *) ioParam == mUserIDTable )
			{
				SendSelectedToRecipients();
			}
			else if( (CUserIDTable *) ioParam == mRecipientTable ) 
			{
				SendSelectedToUserIDs();
			}
								
			mRecipientTable->GetTableSize( numRows, numColumns );
			
			break;
		}
		
		case 'Recv':
		{
			TableIndexT	numRows;
			TableIndexT	numColumns;
			
			// Send the selected cells to the table object in ioParam from
			// the other table.
			
			if( (CUserIDTable *)ioParam == mUserIDTable )
			{
				SendSelectedToUserIDs();
			}
			else if( (CUserIDTable *) ioParam == mRecipientTable )
			{
				SendSelectedToRecipients();
			}
								
			mRecipientTable->GetTableSize( numRows, numColumns );

			break;
		}
		
		case kUpdateFromServerButtonPaneID:
		{
			PGPError	err;
			PGPBoolean	haveNewKeys;
			
			err = PGPUpdateMissingRecipients( NULL, &mRecipients,
						&haveNewKeys );
			if( IsntPGPError( err ) && haveNewKeys )
			{
				err = BuildTables();
			}
			
			AdjustButtons();
			
			break;
		}

		case msg_Options:
			DoOptionsDialog();
			break;

		default:
			CPGPUIGrafPortView::ListenToMessage( inMessage, ioParam );
			break;
	}
}

	Boolean
CPGPRecipientGrafPortView::HandleKeyPress(const EventRecord& inKeyEvent)
{
	Boolean		keyHandled 	= TRUE;
	Int16		theKey 		= inKeyEvent.message & charCodeMask;
	
	if( ( inKeyEvent.modifiers & cmdKey ) != 0 )
	{
		switch( theKey )
		{
			case 'O':
			case 'o':
			{
				if( mOptionsButton->IsEnabled() )
				{
					mOptionsButton->SimulateHotSpotClick( kControlButtonPart );
				}
				
				break;
			}

			default:
				keyHandled = CPGPUIGrafPortView::HandleKeyPress(
															inKeyEvent );
				break;
		}
	}
	else
	{
		keyHandled = CPGPUIGrafPortView::HandleKeyPress( inKeyEvent );
	}
	
	return keyHandled;
}

	void
CPGPRecipientGrafPortView::AdjustButtons(void)
{
	PGPBoolean	haveMissingRecipients;
	STableCell	cell(0, 0);
	
	haveMissingRecipients = FALSE;
	
	while( mRecipientTable->GetNextCell( cell ) )
	{
		PGPRecipientUser	*curUser;
		
		mRecipientTable->GetCellUser( cell, &curUser );
		
		if( curUser->kind == kPGPRecipientUserKind_MissingRecipient )
		{
			haveMissingRecipients = TRUE;
			break;
		}
		else if( curUser->kind == kPGPRecipientUserKind_Group &&
				 curUser->groupInfo.numMissingKeys != 0 )
		{
			haveMissingRecipients = TRUE;
			break;
		}
	}

	if( haveMissingRecipients && mUpdateFromServerButton->IsVisible() )
	{
		mUpdateFromServerButton->Enable();
	}
	else
	{
		mUpdateFromServerButton->Disable();
	}
}

	
	PGPError
CPGPRecipientGrafPortView::BuildTable(
	CUserIDTable 				*whichTable,
	PGPRecipientUserLocation 	location)
{
	PGPError			err = kPGPError_NoErr;
	Boolean				keepListSorted;
	PGPRecipientKey		*curKey;
	PGPRecipientUser	*curGroup;
	
	keepListSorted = whichTable->GetStorageArray()->IsKeptSorted();
	whichTable->GetStorageArray()->SetKeepSorted( FALSE );
	
	curKey = mRecipients.keys;
	while( IsntNull( curKey ) )
	{
		PGPRecipientUser	*curUser;
		
		curUser = curKey->users;
		while( IsntNull( curUser ) )
		{
			if( curUser->location == location )
			{
				whichTable->InsertRows( 1, 10000, &curUser,
							sizeof( curUser ), FALSE );
			}
			
			curUser = curUser->nextUser;
		}
	
		curKey = curKey->nextKey;
	}
	
	curGroup = mRecipients.groups;
	while( IsntNull( curGroup ) )
	{
		pgpAssert( curGroup->kind == kPGPRecipientUserKind_Group );
		
		if( curGroup->location == location )
		{
			whichTable->InsertRows( 1, 10000, &curGroup, sizeof( curGroup ),
							FALSE );
		}

		curGroup = curGroup->nextUser;
	}
	
	whichTable->GetStorageArray()->SetKeepSorted( keepListSorted );
	
	return( err );
}

	PGPError
CPGPRecipientGrafPortView::BuildTables(void)
{
	PGPError	err;
	TableIndexT	numRows;
	TableIndexT	numColumns;
	
	mUserIDTable->GetTableSize( numRows, numColumns );
	if( numRows != 0 )
		mUserIDTable->RemoveRows( numRows, 1, FALSE );

	mRecipientTable->GetTableSize( numRows, numColumns );
	if( numRows != 0 )
		mRecipientTable->RemoveRows( numRows, 1, FALSE );
	
	err = BuildTable( mUserIDTable, kPGPRecipientUserLocation_UserList );
	if( IsntPGPError( err ) )
	{
		err = BuildTable( mRecipientTable,
					kPGPRecipientUserLocation_RecipientList );
		if( IsntPGPError( err ) )
		{
			PGPRecipientUser	*curUser;
			
			// Add missing users
			
			curUser = mRecipients.missingRecipients;
			while( IsntNull( curUser ) )
			{
				pgpAssert( curUser->kind ==
									kPGPRecipientUserKind_MissingRecipient );

				if( curUser->location ==
									kPGPRecipientUserLocation_RecipientList )
				{
					mRecipientTable->InsertRows( 1, 10000, &curUser,
								sizeof( curUser ), FALSE );
				}
				
				curUser = curUser->nextUser;
			}
		}
	}
	
	mUserIDTable->Refresh();
	mRecipientTable->Refresh();
	
	return( err );
}

	PGPError
CPGPRecipientGrafPortView::SetOptions(
	PGPContextRef 				context,
	CPGPRecipientDialogOptions 	*options,
	PGPBoolean					*showDialog)
{
	PGPError	err = kPGPError_NoErr;
	
	*showDialog = options->mAlwaysDisplay;
	
	err = CPGPUIGrafPortView::SetOptions( context, options );
	if( IsntPGPError( err ) )
	{
		mOptions = options;
		
		if( IsntNull( options->mPrompt ) )
		{
			Str255	pPrompt;
			
			CToPString( options->mPrompt, pPrompt );
			SetDescriptor( pPrompt );
		}

		// Hide the options button unless the client has asked for options
		
		if( IsntNull( options->mDialogOptions ) &&
			pgpGetOptionListCount( options->mDialogOptions ) > 0 )
		{
			mOptionsButton->AddListener( this );
		}
		else
		{
			mOptionsButton->Disable();
			mOptionsButton->Hide();
		}

		// Hide the "Update From Server" button if client requests it
		if( IsntNull( options->mServerList ) )
		{
			mUpdateFromServerButton->AddListener( this );
		}
		else
		{
			mUpdateFromServerButton->Hide();
		}
		
		// If the options button is hidden and the update button is not,
		// move the update button to the options button location.
		
		if( mUpdateFromServerButton->IsVisible() &&
			! mOptionsButton->IsVisible() )
		{
			SPoint32	optionsButtonLoc;
			SPoint32	updateButtonLoc;
			
			mOptionsButton->GetFrameLocation( optionsButtonLoc );
			mUpdateFromServerButton->GetFrameLocation( updateButtonLoc );
		
			mUpdateFromServerButton->MoveBy(
						optionsButtonLoc.h - updateButtonLoc.h,
						optionsButtonLoc.v - updateButtonLoc.v,
						TRUE );
		}
		
		mRecipientTable->SetPreferences( mOptions->mDisplayMarginalValidity,
					mOptions->mIgnoreMarginalValidity );
		mUserIDTable->SetPreferences( mOptions->mDisplayMarginalValidity,
					mOptions->mIgnoreMarginalValidity );

		if( IsntPGPError( err ) )
		{
			PGPBoolean	haveDefaultARRs;

			err = PGPBuildRecipientsList( NULL, context, options->mClientKeySet,
							options->mGroupSet, options->mNumDefaultRecipients,
							options->mDefaultRecipients, options->mServerCount,
							options->mServerList, options->mTLSContext,
							options->mSearchBeforeDisplay,
							options->mEnforcement, EventHandler, 0,
							&mRecipients, &haveDefaultARRs );
			if( IsntPGPError( err ) )
			{
				err = BuildTables();
			}
			
			if( haveDefaultARRs &&
				mOptions->mAlwaysDisplayWithARRs )
			{
				*showDialog = TRUE;
			}
			
			if( IsntPGPError( err ) && *showDialog == FALSE )
			{
				STableCell	cell(0,0);
				PGPBoolean	haveRecipients = FALSE;
				PGPInt32	minimumValidity;
				
				if( options->mIgnoreMarginalValidity )
				{
					minimumValidity = (PGPInt32) kPGPValidity_Complete;
				}
				else
				{
					minimumValidity = (PGPInt32) kPGPValidity_Marginal;
				}
				
				/*
				** Check to see if the dialog needs to be displayed.
				** Show the dialog there are missing, ambiguous, invalid,
				** or no default recipients
				*/

				while( mRecipientTable->GetNextCell( cell ) )
				{
					PGPRecipientUser	*user;
					
					haveRecipients = TRUE;
					
					mRecipientTable->GetCellUser( cell, &user );
					
					if( user->multipleMatch ||
						user->kind == kPGPRecipientUserKind_MissingRecipient ||
						user->validity < minimumValidity )
					{
						*showDialog = TRUE;
						break;
					}
				}
				
				if( ! haveRecipients )
					*showDialog = TRUE; 
			}
		}
		
		AdjustButtons();
	}
		
	return( err );
}

	void
CPGPRecipientGrafPortView::DoOptionsDialog(void)
{
	pgpAssertAddrValid( mOptions->mDialogOptions, VoidAlign );
	
	/*
	** mOptions->dialogOptions is probably a volatile PGPOptionList i.e.
	** passing it to PGPOptionsDialog() will dispose of it. Make a copy
	** here for our purposes.
	*/

	(void) PGPOptionsDialog( mContext,
					pgpCopyOptionList( mOptions->mDialogOptions ),
					PGPOLastOption( mContext ) );
}

	MessageT
CPGPRecipientGrafPortView::HandleMessage(MessageT theMessage)
{
	if( theMessage == msg_OK )
	{
		mDialogError = PGPGetRecipientKeys( &mRecipients,
					mOptions->mRecipientKeysPtr, mOptions->mNewKeys,
					mOptions->mRecipientCount, mOptions->mRecipientList );
	}
	
	return( theMessage );
}

	PGPError
CPGPRecipientGrafPortView::EventHandler(
	PGPRecipientsList 	*recipients,
	PGPRecipientEvent 	*event,
	PGPUserValue 		userValue)
{
	PGPError	err 		= kPGPError_NoErr;
	short		strIndex 	= 0;
	const char	*userName	= NULL;
	
	(void) recipients;
	(void) userValue;
	
	switch( event->type )
	{
		case kPGPRecipientEvent_MoveUserFailedEvent:
		{
			if( event->user->kind == kPGPRecipientUserKind_Key )
			{
				strIndex = kKeyADKNotFoundErrorStrIndex;
			}
			else
			{
				strIndex = kGroupADKNotFoundErrorStrIndex;
			}
		
			userName = PGPGetRecipientUserNamePtr( event->user );
			break;
		}
		
		case kPGPRecipientEvent_MoveUserWarningEvent:
		{
			if( event->user->kind == kPGPRecipientUserKind_Key )
			{
				strIndex = kKeyADKNotFoundWarningStrIndex;
			}
			else
			{
				strIndex = kGroupADKNotFoundWarningStrIndex;
			}
		
			userName = PGPGetRecipientUserNamePtr( event->user );
			break;
		}
		
		case kPGPRecipientEvent_MovedARRWarningEvent:
		{
			strIndex = kRemoveADKViolatesPolicyStrIndex;
			userName = PGPGetRecipientUserNamePtr( event->user );
			
			break;
		}
	}
	
	if( strIndex != 0 )
	{
		Str255	str;
		
		CToPString( userName, str );
		SysBeep( 1 );
		PGPUIWarningAlert( kPGPUIGenericOKAlert, kPGPLibDialogsStringListResID,
					strIndex, str );
	}
	
	return( err );
}

	void
CPGPRecipientGrafPortView::SetDescriptor(ConstStringPtr prompt)
{
	LStaticText	*captionObj;
	
	pgpAssertAddrValid( prompt, char );

	captionObj = (LStaticText *) FindPaneByID( kPromptTextPaneID );
	pgpAssertAddrValid( captionObj, VoidAlign );
	
	captionObj->SetDescriptor( prompt );
}
