 /*
  * tcpdumpx - annotate "tcpdump -x" output of IPV4 datagrams.
  * 
  * Usage: tcpdump -x ... | tcpdumpx
  * 
  * This program puts header field names under header bytes, and puts ASCII
  * values under data bytes, producing a kind-of "od -x" display of packets.
  * 
  * Author: Wietse Venema, IBM T.J. Watson Research, Hawthorne, NY. USA.
  */
#include <sys/types.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#ifdef linux
#define __FAVOR_BSD
#endif
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <stdio.h>

 /*
  * IP packet header annotation.
  */
static char *ip_annotation[] = {
    "vhl", "tos", "len", "len",
    "id", "id", "off", "off",
    "ttl", "pro", "sum", "sum",
    "src", "src", "src", "src",
    "dst", "dst", "dst", "dst",
};

 /*
  * TCP packet header annotation.
  */
static char *tcp_annotation[] = {
    "src", "src", "dst", "dst",
    "seq", "seq", "seq", "seq",
    "ack", "ack", "ack", "ack",
    "off", "flg", "win", "win",
    "sum", "sum", "urp", "urp",
};

 /*
  * UDP packet header annotation.
  */
static char *udp_annotation[] = {
    "src", "src", "dst", "dst",
    "len", "len", "sum", "sum",
};

 /*
  * Print packets in 10 columns so that the default 20-byte IP and TCP
  * headers come out nicely separated from any option bytes.
  */
#define COLUMNS	10

/* annotate_header - print annotated header */

static void annotate_header(char *tag, unsigned char *data, int len,
			            char **annotation)
{
    int     i;
    int     j;

    for (i = 0; i < len; i += COLUMNS) {
	printf("    %-8s", tag);
	for (j = 0; j < COLUMNS && i + j < len; j++)
	    printf(" %02x ", data[i + j]);
	printf("\n%-12s", "");
	for (j = 0; j < COLUMNS && i + j < len; j++)
	    printf("%-4s", annotation[i + j]);
	printf("\n");
    }
}

/* annotate_options - print annotated options */

static void annotate_options(char *tag, unsigned char *data, int len,
			             char *annotation)
{
    int     i;
    int     j;

    for (i = 0; i < len; i += COLUMNS) {
	printf("    %-8s", tag);
	for (j = 0; j < COLUMNS && i + j < len; j++)
	    printf(" %02x ", data[i + j]);
	printf("\n%-12s", "");
	for (j = 0; j < COLUMNS && i + j < len; j++)
	    printf("%-4s", annotation);
	printf("\n");
    }
}

/* annotate_data - print annotated data */

static void annotate_data(char *tag, unsigned char *data, int len)
{
    int     i;
    int     j;

    for (i = 0; i < len; i += COLUMNS) {
	printf("    %-8s", tag);
	for (j = 0; j < COLUMNS && i + j < len; j++)
	    printf(" %02x ", data[i + j]);
	printf("\n%-12s", "");
	for (j = 0; j < COLUMNS && i + j < len; j++) {
	    if (!isascii(data[i + j]))
		printf(" %02x ", data[i + j]);
	    else if (iscntrl(data[i + j]))
		printf(" ^%c ", data[i + j] ^ 0100);
	    else
		printf(" %c  ", data[i + j]);
	}
	printf("\n");
    }
}

/* show_packet - display one datagram, annotated */

static void show_packet(unsigned char *payload, int len)
{
    struct ip *ip = (struct ip *) payload;
    int     ip_pkt_len;
    int     ip_hdr_len;
    int     ip_opt_len;
    struct tcphdr *tcp;
    int     tcp_hdr_len;
    int     tcp_opt_len;
    unsigned char *tcp_data;
    int     tcp_data_len;
    struct udphdr *udp;
    int     udp_hdr_len;
    unsigned char *udp_data;
    int     udp_data_len;
    unsigned char *other;
    int     other_len;
    int     i;
    static int count = 0;

    /*
     * Print header/option/data sizes and flags first.
     */
    ip_pkt_len = ntohs(ip->ip_len);
    ip_hdr_len = ip->ip_hl * 4;
    ip_opt_len = ip_hdr_len - sizeof(*ip);
    printf("\n    IP_HDR=%d IP_OPT=%d", sizeof(*ip), ip_opt_len);

    switch (ip->ip_p) {
    case IPPROTO_TCP:
	tcp = (struct tcphdr *) (payload + ip_hdr_len);
	tcp_hdr_len = 4 * tcp->th_off;
	tcp_opt_len = tcp_hdr_len - sizeof(*tcp);
	tcp_data = ((unsigned char *) tcp) + tcp_hdr_len;
	tcp_data_len = ip_pkt_len - ip_hdr_len - tcp_hdr_len;
	printf(" TCP_HDR=%d TCP_OPT=%d DATA=%d FLAGS=",
	       sizeof(*tcp), tcp_opt_len, tcp_data_len);
	if (tcp->th_flags & TH_FIN)
	    printf("FIN ");
	if (tcp->th_flags & TH_SYN)
	    printf("SYN ");
	if (tcp->th_flags & TH_RST)
	    printf("RST ");
	if (tcp->th_flags & TH_PUSH)
	    printf("PUSH ");
	if (tcp->th_flags & TH_ACK)
	    printf("ACK ");
	if (tcp->th_flags & TH_URG)
	    printf("URG ");
	break;
    case IPPROTO_UDP:
	udp = (struct udphdr *) (payload + ip_hdr_len);
	udp_data = ((unsigned char *) udp) + sizeof(*udp);
	udp_data_len = ip_pkt_len - ip_hdr_len - sizeof(*udp);
	printf(" UDP_HDR=%d DATA=%d", sizeof(*udp), udp_data_len);
	break;
    default:
	other = payload + ip_hdr_len;
	other_len = ip_pkt_len - ip_hdr_len;
	printf(" PROTO=%d OTHER=%d", ip->ip_p, other_len);
    }
    printf("\n\n");

    /*
     * Packet content, annotated.
     */
    annotate_header("IP_HDR", (unsigned char *) ip, sizeof(*ip),
		    ip_annotation);
    annotate_options("IP_OPT", ((unsigned char *) ip) + sizeof(*ip),
		     ip_opt_len, "opt");

    switch (ip->ip_p) {
    case IPPROTO_TCP:
	annotate_header("TCP_HDR", (unsigned char *) tcp, sizeof(*tcp),
			tcp_annotation);
	annotate_options("TCP_OPT", ((unsigned char *) tcp) + sizeof(*tcp),
			 tcp_opt_len, "opt");
	annotate_data("DATA", tcp_data, tcp_data_len);
	break;
    case IPPROTO_UDP:
	annotate_header("UDP_HDR", (unsigned char *) udp, sizeof(*udp),
			udp_annotation);
	annotate_data("DATA", udp_data, udp_data_len);
	break;
    default:
	annotate_data("OTHER", other, other_len);
	break;
    }

    printf("\n");
}

main(int argc, char **argv)
{
    char    buf[BUFSIZ];
    char   *cp;
    unsigned char pkt[8192];		/* XXX whatever */
    unsigned char *ptr;
    unsigned value;
    int     count = 0;

    /*
     * Split tcpdump output into the "-x" packet content and the other stuff.
     * Replace the "-x" packet content by our annotated form. The other stuff
     * is copied unchanged.
     */
    while (fgets(buf, sizeof(buf), stdin) != 0) {
	if (!isascii(buf[0]) || !isspace(buf[0])) {	/* other */
	    if (count > 0) {
		show_packet(pkt, count);
		count = 0;
	    }
	    fputs(buf, stdout);
	} else {				/* -x content */
	    for (cp = buf, ptr = pkt + count; *cp; count++, cp += 2, ptr++) {
		if (ptr >= pkt + sizeof(pkt)) {
		    fprintf(stderr, "packet exceeds buffer\n");
		    exit(1);
		}
		while (*cp && isspace(*cp))
		    cp++;
		if (sscanf(cp, "%2x", &value) != 1)
		    break;
		*ptr = value;
	    }
	}
    }

    /*
     * Flush remainder of last packet.
     */
    if (count > 0) {
	show_packet(pkt, count);
	count = 0;
    }
    return (0);
}
