/* ELTS.C

   Module for creating and storing picture elements (boxes, arrows, etc.).

   $Header: elts.c,v 1.2 91/08/30 20:15:21 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "box-type.h"
#include "elts.h"
#include "rels.h"
#ifdef MEASURE
#include "rels.g"
#endif MEASURE

/* MACRO DEFINITIONS ======================================================= */

/* size of ArrowPairTable hash table */
#define INIT_ARROW_PAIR_TBL_SZ (17389L)
#define ARROW_PAIR_TBL_THRESHOLD (0.9)
#define ARROW_PAIR_TBL_GROWTH (2.1)

/* TYPE DEFINITIONS ======================================================== */

typedef struct arrow_pair_entry {
    struct arrow_pair_entry *next;
    ASysname a1_name,a2_name;	/* sysname's of the two arrows */
    Boolean overrides_result;	/* does 'a1_name' override 'a2_name'? */
} ArrowPairEntry;

typedef struct arrow_pair_tbl {
    ULint size;			/* number of buckets in table */
    ULint threshold;		/* max number of elements in table */
    ULint cnt;			/* number of elements in table */
    ArrowPairEntry **tbl;	/* hash table of ArrowPairEntry's */
} *ArrowPairTable,ArrowPairTableStruct;

/* PRIVATE GLOBALS ========================================================= */

/* hash table of arrow pairs */
ArrowPairTable ArrowPairTbl;

/* LOCAL FUNCTIONS ========================================================= */

static void AddToBoxList(bl_ptr,b)
  BoxList **bl_ptr;
  Box *b;
{
    BoxList *temp;

    temp = AllocOne(BoxList);
    temp->b = b;
    temp->next = *bl_ptr;
    *bl_ptr = temp;
}

Boolean AncestorBoxHelper(b1,b2)
  Box *b1,*b2;
/* Return True iff 'b1' is an ancestor (not nec. proper) of 'b2'.

   IMPLEMENTATION: This implementation does not use the idea suggested in
   Chapter 3 of the thesis for marking visited nodes with a search number to
   avoid searching parts of the containment DAG multiple times. Measurements
   should be taken to see if this technique would actually lead to speed
   increases.
*/
{
    BoxList *parent;

#ifdef MEASURE
    AncestorBoxCallCnt++;
    fprintf(stderr,"%d ",b2->sysname);
#endif MEASURE
    /* quick tests: equal boxes or b2 at a higher level than b1 */
    if (b1 == b2) {
#ifdef MEASURE
	EqlBoxesCnt++;
#endif MEASURE
	return(True);
    }
    if (b1->level <= b2->level) {
#ifdef MEASURE
	UneqlLevelsCnt++;
#endif MEASURE
	return(False);
    }

    StepLinkedList(parent,b2->parents) {
	if (AncestorBoxHelper(b1,parent->b)) return(True);
    }
    return(False);
}

#ifdef MEASURE
AncestorBox(b1,b2)
  Box *b1,*b2;
{
    fprintf(stderr,"\nAncestorBox(%d): ",b1->sysname);
    AncestorBoxHelper(b1,b2);
    fputc('\n',stderr);
}
#else
#define AncestorBox(_b1,_b2) AncestorBoxHelper(_b1,_b2)
#endif MEASURE

#define HashArrowPair(_a1,_a2,_size)\
    (((_a1)*(_a2)) % (_size))

static ArrowPairEntry **EmptyArrowPairTable(size)
  ULint size;
{
    register int i;
    ArrowPairEntry **result = AllocPtrArray(ArrowPairEntry,size);
    StepIndex(i,0,size) { result[i] = (ArrowPairEntry *)NULL; }
    return(result);
}

static void RehashArrowPairTable(tbl)
  INOUT ArrowPairTable tbl;
/* The hash table 'tbl' is full, so increase its size by a factor of the
   constant ARROW_PAIR_TBL_GROWTH, and rehash all of its current elements into
   the new table. This also has the side-effect of updating 'tbl->size' and
   'tbl->threshold'.
*/
{
    ULint new_size = (ULint)((float)tbl->size * ARROW_PAIR_TBL_GROWTH);
    ArrowPairEntry **new_tbl = EmptyArrowPairTable(new_size);
    ArrowPairEntry **old_tbl = tbl->tbl;
    ArrowPairEntry *entry,*temp;
    int hash_ix;
    register int i;

    StepIndex(i,0,tbl->size) {
	for (entry=old_tbl[i]; entry; /*empty*/) {
	    temp = entry->next;
	    hash_ix = HashArrowPair(entry->a1_name,entry->a2_name,new_size);
	    SpliceIntoList(new_tbl[hash_ix],entry);
	    entry = temp;
	}
    }
    tbl->size = new_size;
    tbl->threshold = (long)(ARROW_PAIR_TBL_THRESHOLD * (float)new_size);
    Dealloc(old_tbl);
    tbl->tbl = new_tbl;
}

/* GLOBAL FUNCTIONS ======================================================== */

Box *NewBox()
{
    Box *result;

    result = AllocOne(Box);
    result->parents = (BoxList *)NULL;
    result->level = NO_LEVEL;
    result->u.children = (BoxList *)NULL;
    result->a_incidence = 0;
    result->blk = UnvisitedBlk;
    return(result);
}

void FreeBox(b)
  Box *b;
{
    /* free other things here? */
    Dealloc(b);
}

Arrow *NewArrow()
{
    Arrow *result;

    result = AllocOne(Arrow);
    result->perms[(int)Pos] = result->perms[(int)Neg] = 0x0;
    return(result);
}

void FreeArrowList(al)
  ArrowList *al;
{
    register ArrowList *curr,*last;
    curr = al;
    while (curr != NULL) {
	last = curr;
	curr = curr->next;
	Dealloc(last);
    }
}

void AssertBoxContainment(parent,child)
  Box *parent,*child;
{
    /* add 'child' to the 'children' list of 'child' */
    AddToBoxList(&(parent->u.children),child);

    /* add 'parent' to the 'parents' list of 'child' */
    AddToBoxList(&(child->parents),parent);
}

void InitOverridesCache()
{
    register int i;

    ArrowPairTbl = (ArrowPairTable)AllocOne(ArrowPairTableStruct);
    ArrowPairTbl->size = INIT_ARROW_PAIR_TBL_SZ;
    ArrowPairTbl->cnt = 0L;
    ArrowPairTbl->threshold =
	(long)(ARROW_PAIR_TBL_THRESHOLD * (float)ArrowPairTbl->size);
    ArrowPairTbl->tbl = EmptyArrowPairTable(ArrowPairTbl->size);
}

Boolean ArrowOverrides(a1,a2)
  Arrow *a1,*a2;
{
    ASysname a1_name,a2_name;	/* sysname's of two arrows */
    int hash_ix;		/* hash table index */
    ArrowPairEntry *curr;

    /* look for this pair in the hash table */
    a1_name = a1->sysname;	/* temp vars for fast execution of... */
    a2_name = a2->sysname;	/* ...HashArrowPair() macro */
    hash_ix = HashArrowPair(a1_name,a2_name,ArrowPairTbl->size);
    StepLinkedList(curr,ArrowPairTbl->tbl[hash_ix]) {
	if (curr->a1_name == a1_name && curr->a2_name == a2_name) break;
    }
    if (curr == (ArrowPairEntry *)NULL) {
	/*
	 * grow table if necessary */
	if (ArrowPairTbl->cnt >= ArrowPairTbl->threshold) {
	    RehashArrowPairTable(ArrowPairTbl);
	    hash_ix = HashArrowPair(a1_name,a2_name,ArrowPairTbl->size);
	}
	/*
	 * create and install new entry */
	curr = AllocOne(ArrowPairEntry);
	curr->a1_name = a1_name;
	curr->a2_name = a2_name;
	SpliceIntoList(ArrowPairTbl->tbl[hash_ix],curr);
	ArrowPairTbl->cnt++;
	/*
	 * do ancestorship checks */
#ifdef MEASURE
	ArrowCompareCnt++;
#endif MEASURE
	curr->overrides_result =
	    (((a1->from->level == a2->from->level)
	      && (a1->to->level == a2->to->level)) ? False
	     : (Boolean)(AncestorBox(a2->from,a1->from)
			 && AncestorBox(a2->to,a1->to)));
    }
#ifdef MEASURE
    else { ArrowPairHitCnt++; }
#endif MEASURE
    return(curr->overrides_result);
}
