/* ELTS.H

   Type definitions and function declarations for the elts.c module, which
   provides facilities for creating and storing box and arrow elements.

   $Header: elts.h,v 1.1 91/08/01 20:05:46 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#ifndef ELTS_H
#define ELTS_H

#include <my-types.h>

#include "sysname-types.h"
#include "box-type.h"

/* MACRO DEFINITIONS ======================================================= */

/* Unknown entries in Box and Arrow elements */
#define NO_SYSNAME (-1)
#define NO_LEVEL (-1)
#define SCANNING_LEVEL (-2)	/* to detect cycles in containment graph */

/* sysname of "super" Subject and Object boxes */
#define SUPER_BOX_SYSNAME 0

/* all permissions turned on */
#define ALL_PERMS (~ 0)		/* all 1's */

/* TYPE DEFINITIONS ======================================================== */

/* Permission-Related Types ------------------------------------------------ */

typedef USint PermSet;		/* permissions are a bit-map */

typedef struct perm_val {
    BoxTypeSet domain;		/* bit-pattern for BoxType's in domain */
    BoxTypeSet range;		/* bit-pattern for BoxType's in range */
    PermSet perm_set;		/* bit-pattern for this permission */
    int index;			/* index for this permission */
} PermVal;

typedef struct perm_list {
    struct perm_list *next;
    String perm_name;		/* name of access permission */
} PermList;

/* Box-Related Types ------------------------------------------------------- */

typedef struct box_list {
    struct box_list *next;
    struct box *b;
} BoxList;

/* For each permission, the 'a_incidence' bit for that permission is OFF iff
   the box has exactly one parent and at least one arrow of the given perm-
   ission incident on it.
*/

typedef struct box {
    struct box *next;		/* pointer to next box in main list */
    BSysname sysname;		/* sysname of element */
    BoxList *parents;		/* parents of this box */
    int level;			/* level in containment DAG */
    union {
	BoxList *children;	/* children of this box for Objects */
	int index;		/* index into WBlock array for Subjects */
    } u;
    PermSet a_incidence;	/* arrow incidence by permission */
    BoxTypeSet box_type_val;	/* bit-mask for the type of this box */
    struct w_block *blk;	/* block of witness sets (Objects only) */
} Box;

/* Arrow-Related Types ----------------------------------------------------- */

typedef enum {
    Pos=0, Neg=1
} ArrowParity;

typedef struct arrow {
    struct arrow *next;
    ASysname sysname;		/* CantorPair() of from & to boxes */
    Box *from,*to;		/* tail and head boxes */
    PermSet perms[PARITY_CNT];	/* set of pos/neg permissions */
} Arrow;

/* The following structure is used only to pass a group of parameters between
   functions. Only temporary space is allocated for these structures.
*/
typedef struct full_arrow {
    Box *from,*to;
    ArrowParity parity;
    PermSet perms;
} FullArrow;

typedef struct arrow_list {
    struct arrow_list *next;
    Arrow *a;
} ArrowList;

/* FUNCTION DECLARATIONS =================================================== */

Box *NewBox( /* void */ );
/* Return a pointer to a new, initialized Box element.
*/

void FreeBox( /* Box *b */ );
/* Deallocate the memory associated with the Box 'b'.
*/

Arrow *NewArrow( /* void */ );
/* Return a pointer to a new, initialized Arrow element.
*/

void FreeArrowList( /* ArrowList *al */ );
/* Deallocate the memory associated with the ArrowList structures in the list
   'al'. This routine does *not* deallocate the actual Arrow structures
   pointed to by the ArrowList structures (as the Arrow's are assumed to live
   in the ASysnameTable for the life of the program).
*/

void AssertBoxContainment( /* Box *parent, Box *child */ );
/* Updates data structures so 'parent' contains 'child'.
*/

void InitOverridesCache( /* void */ );
/* Initialize the cache storing previously computed overrides results. This
   function *must* be called prior to any call to ArrowOverrides() below.
*/

Boolean ArrowOverrides( /* Arrow *a1, Arrow *a2 */ );
/* Returns True iff 'a1' \overrides 'a2'.

   Pre-condition: This assumes 'a1' != 'a2', so if the levels of both the
   "from" boxes are equal and those of the "to" boxes are also equal, then the
   routine can return False (since it knows the arrows themselves are not
   equal).
*/

#ifdef MACRO_FUNCTION

Boolean FastArrowOverrides( /* Arrow *a1, Arrow *a2 */ );
/* Returns True iff 'a1'=='a2' or 'a1' \overrides 'a2'. This calls
   ArrowOverrides() above when necessary, but first performs quick checks to
   determine answer quickly if possible.
*/
#endif MACRO FUNCTION

#ifdef MEASURE
#define FastArrowOverrides(_a1,_a2)					\
  ((_a1)==(_a2) ? (EqlArrowCnt++,True)					\
   : (((_a2)->from->level < (_a1)->from->level				\
       || (_a2)->to->level < (_a1)->to->level) ? (WrongLevelCnt++,False)\
      : ArrowOverrides(_a1,_a2)))
#else
#define FastArrowOverrides(_a1,_a2)					\
  ((_a1)==(_a2) ? True							\
   : (((_a2)->from->level < (_a1)->from->level				\
       || (_a2)->to->level < (_a1)->to->level) ? False			\
      : ArrowOverrides(_a1,_a2)))
#endif MEASURE

#endif ELTS_H
