/* RELS.C

   Module for storing and manipulating relations between subject and object
   boxes represented by witness sets and blocks of witness sets.

   $Header: rels.c,v 1.6 91/10/28 11:45:20 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "sysname-types.h"
#include "rels.h"
#include "elts.h"

/* GLOBAL VARIABLES ======================================================= */

FILE *OutFile;
Boolean PrintCompactMatrix;
Boolean PrintFullMatrix;
Boolean PrintLisp;
int AmbigEntryCnt;
WSetTable WSetTbl;
WSet *DefaultSet;
String PermName;

#ifdef MEASURE
int WBlkInheritedCnt = 0, WBlkComputedCnt = 0;
int WSetFromArrowCnt = 0, WSetInheritedCnt = 0, WSetComputedCnt = 0;
int EqSets = 0, FirstSetDef = 0, SecondSetDef = 0, MinWSetCnt = 0;
int WSetHitCnt = 0, WSetMissCnt = 0;
int EqlArrowCnt=0, WrongLevelCnt=0, ArrowPairHitCnt=0, ArrowCompareCnt=0;
int AncestorBoxCallCnt = 0, EqlBoxesCnt = 0, UneqlLevelsCnt = 0;
#endif MEASURE

/* MACRO DEFINITIONS ====================================================== */

/* WSetTable hash table size, threshold, rehash factor */
#define INIT_W_SET_TBL_SZ (17389L) /* Prime[2000] */
#define W_SET_TBL_THRESHOLD (0.9)
#define W_SET_TBL_GROWTH (2.1)

/* AccessBlockTable hash table size, threshold, rehash factor */
#define INIT_ACCESS_TBL_SZ (211L)
#define ACCESS_TBL_THRESHOLD (0.9)
#define ACCESS_TBL_GROWTH (2.1)

/* LOCAL TYPES ============================================================= */

typedef enum {
    PosFound=0, NegFound=1, AmbigFound=2
} RelFound;

/* LOCAL VARIABLES ========================================================= */

/* AccessBlock hash table */
static AccessTable AccessTbl;

/* # of 32-bit words in each BitVector stored in the table */
static int BitVectSize;

/* static bit vector filled by BooleanToBitVector() */
static BitVector BitVect;

/* LOCAL FUNCTIONS ========================================================= */

#ifdef MACRO_FUNCTION

static int HashFingerPrint(hash_val,size)
  FingerPrint hash_val;
  ULint size;
/* Returns an integer index into an array corresponding to 'hash_val'
*/
#endif MACRO_FUNCTION

#define HashFingerPrint(_hash_val,_size) \
    ((int)(((ULint)(_hash_val)) % (_size)))

static FingerPrint HashWSet(w_set)
  WSet *w_set;
/* Returns a fingerprint for 'w_set'. The idea behind the fingerprint is that
   HashWSet(s1) == HashWSet(s2) <==> (equal s1 s2). In practice, all we really
   have is the '<==' direction.

  IMPLEMENTATION: The FingerPrint is simply the bitwise XOR of the pointers to
  the arrows in the witness set. We may also want to incorporated the number
  of positive and negative arrows into this calculation.
*/
{
    ArrowList *curr;
    FingerPrint result = 0L;
    FingerPrint pos_bit,neg_bit;

    pos_bit = 0x00000004L; neg_bit = 0x00020000L;
    StepLinkedList(curr,w_set->al[(int)Pos]) {
	result ^= (FingerPrint)(curr->a);
	pos_bit <<= 1;
    }
    StepLinkedList(curr,w_set->al[(int)Neg]) {
	result ^= (FingerPrint)(curr->a);
	neg_bit <<= 1;
    }
    result ^= (pos_bit | neg_bit);
    Assert((result & 0x3) == 0L);
    result >>= 2;        /* divide by 4 since pointers are on word boundries */
    return(result);
}

#ifdef MACRO_FUNCTION

static void DeallocWSetSpace(w_set)
  WSet *w_set;
/* Deallocate the space used by the ArrowList structures in 'w_set' and by the
   WSet itself.
*/
#endif MACRO_FUNCTION

#define DeallocWSetSpace(_w_set)			\
    /* deallocate each of the ArrowList structures */	\
    FreeArrowList((_w_set)->al[(int)Pos]);		\
    FreeArrowList((_w_set)->al[(int)Neg]);		\
							\
    /* deallocate the WSet itself */			\
    Dealloc(_w_set)

static void ShowRelation(fp,rel,s_sysname,o_sysname)
  FILE *fp;
  RelFound rel;
  BSysname s_sysname,o_sysname;
{
    /* Arrays for converting from a value of type 'RelFound' to a string */
    static String LispRelStrings[] = {
	"pos", "neg", "ambig"
    };
    static String RelStrings[] = {
	"[+] Positive: ", "[-] Negative: ", "[*] Ambiguous:"
    };

    if (PrintLisp) {
	fprintf(fp,"(sem %u %u '%s '%s)\n",s_sysname,o_sysname,
		PermName,LispRelStrings[(int)rel]);
    } else {
	fprintf(fp,"%s perm = %-8s subj = %-6u obj = %-6u\n",
	       RelStrings[(int)rel],PermName,s_sysname,o_sysname);
    }
}

static WSet **EmptyWSetTable(size)
  ULint size;
{
    register int i;
    WSet **result = AllocPtrArray(WSet,size);
    StepIndex(i,0,size) { result[i] = (WSet *)NULL; }
    return(result);
}

static void RehashWSetTable(tbl)
  INOUT WSetTable tbl;
/* The hash table 'tbl' is full, so increase its size by a factor of the
   constant W_SET_TBL_GROWTH, and rehash all of its current elements into the
   new table. This also has the side-effect of updating 'tbl->size' and
   'tbl->threshold'.
*/
{
    ULint new_size = (ULint)((float)tbl->size * W_SET_TBL_GROWTH);
    WSet **new_tbl = EmptyWSetTable(new_size);
    WSet **old_tbl = tbl->tbl;
    WSet *entry,*temp;
    int hash_ix;
    register int i;

    StepIndex(i,0,tbl->size) {
	for (entry=old_tbl[i]; entry; /*empty*/) {
	    temp = entry->next;
	    hash_ix = HashFingerPrint(entry->hash_val,new_size);
	    SpliceIntoList(new_tbl[hash_ix],entry);
	    entry = temp;
	}
    }
    tbl->size = new_size;
    tbl->threshold = (ULint)(W_SET_TBL_THRESHOLD * (float)new_size);
    Dealloc(old_tbl);
    tbl->tbl = new_tbl;
}

static FingerPrint HashBitVector(vector)
  BitVector vector;
{
    register FingerPrint result = 0L;
    register int i;

    StepIndex(i,0,BitVectSize) { result ^= (FingerPrint)vector[i]; }
    return(result);
}

AccessBlock **EmptyAccessTable(size)
  ULint size;
{
    register int i;
    AccessBlock **result = AllocPtrArray(AccessBlock,size);
    StepIndex(i,0,size) { result[i] = (AccessBlock *)NULL; }
    return(result);
}

static void RehashAccessTable(tbl)
  AccessTable *tbl;
{
    ULint new_size = (ULint)((float)tbl->size * ACCESS_TBL_GROWTH);
    AccessBlock **new_tbl = EmptyAccessTable(new_size);
    AccessBlock **old_tbl = tbl->tbl;
    AccessBlock *entry,*temp;
    int hash_ix;
    register int i;

    StepIndex(i,0,tbl->size) {
	for (entry=old_tbl[i]; entry; /*empty*/) {
	    temp = entry->next;
	    hash_ix = HashFingerPrint(entry->hash_val,new_size);
	    SpliceIntoList(new_tbl[hash_ix],entry);
	    entry = temp;
	}
    }
    tbl->size = new_size;
    tbl->threshold = (ULint)(ACCESS_TBL_THRESHOLD * (float)new_size);
    Dealloc(old_tbl);
    tbl->tbl = new_tbl;
}

void BooleanToBitVector(vector)
  BoolVector vector;
/* Fills the local global BitVect to be the bit vector having bits
   corresponding to the values in 'vector'. The first BitVectSize *
   sizeof(ULint) values are read from 'vector', even if it is not that large.
   Therefore, we may read garbage, but the extra garbage bits won't ever be
   read during constraint processing.
*/
{
    register ULint val,mask;
    register int cnt = 0;
    int i;

    StepIndex(i,0,BitVectSize) {
	val = 0L;
	for (mask=1L; mask; mask <<= 1) {
	    if (vector[cnt++]) { val |= mask; }
	}
	BitVect[i] = val;
    }
}

BitVector CopyBitVector(vect)
  BitVector vect;
{
    register int i;
    BitVector result;

    result = AllocArray(ULint,BitVectSize);
    StepIndex(i,0,BitVectSize) {
	result[i] = vect[i];
    }
    return(result);
}

static Boolean SameBitVectors(v1,v2)
  BitVector v1,v2;
{
    register int i;
    StepIndex(i,0,BitVectSize) {
	if (v1[i] != v2[i]) { return(False); }
    }
    return(True);
}

void DisplayBitVector(vect,size)
  BitVector vect;
  int size;
/* Note: This will write an even number of words, even if there are not that
   many subjects. Those bits will never be read, so we only lose a small
   space-savings (if anything at all, since Allegro may represent bit vectors
   as true words of bits anyhow).
*/
{
    register ULint mask,val;
    register int i;

    StepIndex(i,0,BitVectSize) {
	val = vect[i];
	for (mask=1L; mask; mask <<= 1) {
	    fputc(((val & mask) ? '1' : '0'),OutFile);
	}
    }
}

/* DEBUGGING FUNCTIONS ===================================================== */

#ifdef DEBUG

static Boolean SameWSets(s1,s2)
  WSet *s1,*s2;
/* Returns True iff 's1' and 's2' contain the exact same sets of arrows.
*/
{
    ArrowList *curr1,*curr2;
    int pos_cnt=0,neg_cnt=0;

    /* verify that both contain same number of each kind of arrow */
    StepLinkedList(curr1,s1->al[(int)Pos]) pos_cnt++;
    StepLinkedList(curr2,s2->al[(int)Pos]) pos_cnt--;
    if (pos_cnt != 0) return(False);
    StepLinkedList(curr1,s1->al[(int)Neg]) neg_cnt++;
    StepLinkedList(curr2,s2->al[(int)Neg]) neg_cnt--;
    if (neg_cnt != 0) return(False);

    /* check that actual arrows also match */
    StepLinkedList(curr1,s1->al[(int)Pos]) {
	StepLinkedList(curr2,s2->al[(int)Pos]) {
	    if (curr1->a == curr2->a) break;
	}
	if (curr2 == NULL) return(False);
    }
    StepLinkedList(curr1,s1->al[(int)Neg]) {
	StepLinkedList(curr2,s2->al[(int)Neg]) {
	    if (curr1->a == curr2->a) break;
	}
	if (curr2 == NULL) return(False);
    }
    return(True);
}
#endif DEBUG
  
/* GLOBAL FUNCTIONS ======================================================= */

WSet *NewWSet()
{
    WSet *result = AllocOne(WSet);
    result->ref_cnt = 0;
#ifdef MEASURE
    result->tot_ref_cnt = 0;
#endif MEASURE
    result->al[(int)Pos] = result->al[(int)Neg] = (ArrowList *)NULL;
    return(result);
}

#ifdef MEASURE
int WSetSize(w_set)
  WSet *w_set;
{
    int result = 0;
    ArrowList *curr;
    StepLinkedList(curr,w_set->al[(int)Pos]) result++;
    StepLinkedList(curr,w_set->al[(int)Neg]) result++;
    return(result);
}
#endif MEASURE

void DeallocWSet(w_set)
  WSet *w_set;
{
    WSet **curr,*temp;

    curr = WSetTbl->tbl + HashFingerPrint(w_set->hash_val,WSetTbl->size);
    while (*curr != w_set) curr = &((*curr)->next);
    temp = *curr;
    *curr = temp->next;
#ifdef MEASURE
    fprintf(stderr,"\nDeallocWSet(): total= %u size= %d\n",
	    temp->tot_ref_cnt,WSetSize(temp));
#endif MEASURE
    DeallocWSetSpace(temp);
}

WSet *NewWSetFromArrow(a,perm)
  Arrow *a;
  PermSet perm;
{
    ArrowList *temp;
    WSet *result = NewWSet();

    if (a->perms[(int)Pos] & perm) {
	temp = AllocOne(ArrowList);
	temp->next = (ArrowList *)NULL;
	temp->a = a;
	result->al[(int)Pos] = temp;
    }
    if (a->perms[(int)Neg] & perm) {
	temp = AllocOne(ArrowList);
	temp->next = (ArrowList *)NULL;
	temp->a = a;
	result->al[(int)Neg] = temp;
    }
    return(result);
}

WBlock *NewWBlock(size,w_set)
  int size;
  WSet *w_set;
{
    register int i;
    WSetGrp w_set_grp;

    WBlock *result = AllocOne(WBlock);
    result->ref_cnt = 1;	/* make this the first reference */
#ifdef MEASURE
    result->tot_ref_cnt = 1;
#endif MEASURE
    result->w_set_grp = w_set_grp = AllocPtrArray(WSet,size);
    w_set_grp[0] = CopyWSet(DefaultSet);
    if (w_set != (WSet *)NULL) {
	StepIndex(i,1,size) { w_set_grp[i] = CopyWSet(w_set); }
    } else {
	StepIndex(i,1,size) { w_set_grp[i] = (WSet *)NULL; }
    }
    return(result);
}

void DeallocWBlock(blk,size)
  WBlock *blk;
  int size;
{
    register int i;
    WSetGrp w_set_grp = blk->w_set_grp;
    StepIndex(i,0,size) {
	FreeWSet(w_set_grp[i]);
    }
    Dealloc(w_set_grp);
#ifdef MEASURE
    fprintf(stderr,"\nDeallocWBlock(): total= %u\n",blk->tot_ref_cnt);
#endif MEASURE
    Dealloc(blk);
}

Boolean DisplayParity(fp,w_set,s_sysname,o_sysname)
  FILE *fp;
  WSet *w_set;
  BSysname s_sysname,o_sysname;
{
    if (PosWSet(w_set)) {
	ShowRelation(fp,PosFound,s_sysname,o_sysname);
	return(True);
    } else if (NegWSet(w_set)) {
	ShowRelation(fp,NegFound,s_sysname,o_sysname);
	return(False);
    } else {
	AmbigEntryCnt++;
	ShowRelation(fp,AmbigFound,s_sysname,o_sysname);
	return(False);
    }
}

WSetTable NewWSetTable()
{
    WSetTable result = (WSetTable)AllocOne(WSetTableStruct);
    result->size = INIT_W_SET_TBL_SZ;
    result->cnt = 0L;
    result->threshold = (long)(W_SET_TBL_THRESHOLD * (float)result->size);
    result->tbl = EmptyWSetTable(result->size);
    return(result);
}

WSet *FindOrAddWSet(w_set)
  INOUT WSet *w_set;
{
    WSet *curr;
    int hash_ix;
    FingerPrint hash_val = HashWSet(w_set);

    hash_ix = HashFingerPrint(hash_val,WSetTbl->size);
    StepLinkedList(curr,WSetTbl->tbl[hash_ix]) {
	if (curr->hash_val == hash_val) { break; }
    }
    if (curr != (WSet *)NULL) {
	/* 'w_set' already in table, so deallocate it */
#ifdef DEBUG
	Assert(SameWSets(curr,w_set));
#endif DEBUG
	DeallocWSetSpace(w_set);
#ifdef MEASURE
	WSetHitCnt++;
#endif MEASURE
	return(curr);
    } else {
	/*
	 * grow table if necessary */
	if (WSetTbl->cnt >= WSetTbl->threshold) {
	    RehashWSetTable(WSetTbl);
	    hash_ix = HashFingerPrint(hash_val,WSetTbl->size);
	}
	/*
	 * 'w_set' not in table, so add it */
	w_set->hash_val = hash_val;
	SpliceIntoList(WSetTbl->tbl[hash_ix],w_set);
	WSetTbl->cnt++;
#ifdef MEASURE
	WSetMissCnt++;
#endif MEASURE
	return(w_set);
    }
}

#define CopyArrowsOfParity(_from,_to,_parity)				\
    StepLinkedList(curr,(_from)->al[(int)(_parity)]) {			\
        new = AllocOne(ArrowList);					\
        new->a = curr->a;						\
        SpliceIntoList((_to),new);					\
    }

#define DelOverriddenArrows(_del_list,_comp_list)			\
    if ((_comp_list) != (ArrowList *)NULL) {				\
	curr_ptr = &(_del_list);					\
	while (*curr_ptr != NULL) {					\
	    StepLinkedList(curr,(_comp_list)) {				\
		a1 = curr->a; a2 = (*curr_ptr)->a;			\
		if (FastArrowOverrides(a1,a2)) { break; }		\
	    }								\
	    if (curr != NULL) {						\
		/* delete arrow from (_del_list) */			\
		temp = *curr_ptr;					\
		*curr_ptr = temp->next;					\
		Dealloc(temp);						\
	    } else {							\
		/* skip over this arrow (it stays in the list) */	\
		curr_ptr = &((*curr_ptr)->next);			\
	    }								\
	}								\
    }

#define AppendArrowList(_first,_second)					     \
    for (curr_ptr = &(_first); *curr_ptr; curr_ptr = &((*curr_ptr)->next)) ; \
    *curr_ptr = (_second)

WSet *MinWSet(s1,s2)
  WSet *s1,*s2;
{
    WSet *result = NewWSet();
    ArrowList *a1_pos,*a1_neg,*a2_pos,*a2_neg;
    ArrowList *curr,*new,*temp,**curr_ptr;
    Arrow *a1,*a2;

    /* form linked lists of s1 and s2 arrows */
    a1_pos = a1_neg = a2_pos = a2_neg = (ArrowList *)NULL;
    CopyArrowsOfParity(s1,a1_pos,Pos);
    CopyArrowsOfParity(s1,a1_neg,Neg);
    CopyArrowsOfParity(s2,a2_pos,Pos);
    CopyArrowsOfParity(s2,a2_neg,Neg);

    /* remove overridden arrows */
    DelOverriddenArrows(a1_pos,a2_pos); DelOverriddenArrows(a1_pos,a2_neg);
    DelOverriddenArrows(a1_neg,a2_pos); DelOverriddenArrows(a1_neg,a2_neg);
    DelOverriddenArrows(a2_pos,a1_pos); DelOverriddenArrows(a2_pos,a1_neg);
    DelOverriddenArrows(a2_neg,a1_pos); DelOverriddenArrows(a2_neg,a1_neg);

    /* merge pos/neg lists */
    result->al[(int)Pos] = a1_pos;
    AppendArrowList(result->al[(int)Pos],a2_pos);
    result->al[(int)Neg] = a1_neg;
    AppendArrowList(result->al[(int)Neg],a2_neg);

    /* see if this witness set is in the hash table, or register it */
    return(FindOrAddWSet(result));
}

int ceil(f)
  float f;
{
    int result = (int)f;
    if ((float)result < f) result++;
    return(result);
}

int InitAccessTable(subj_cnt)
  int subj_cnt;
{
    AccessTbl.size = INIT_ACCESS_TBL_SZ;
    AccessTbl.threshold = (long)(ACCESS_TBL_THRESHOLD * (float)AccessTbl.size);
    AccessTbl.cnt = 0L;
    AccessTbl.tbl = EmptyAccessTable(AccessTbl.size);
    BitVectSize = ceil((float)subj_cnt / (8.0 * (float)sizeof(ULint)));
    BitVect = AllocArray(ULint,BitVectSize);
    return(BitVectSize * 8 * sizeof(ULint));
}

ULint FindOrAddAccess(vector)
  BoolVector vector;
{
    AccessBlock *curr;
    int hash_ix;
    FingerPrint hash_val;

    BooleanToBitVector(vector);
    hash_val = HashBitVector(BitVect);
    hash_ix = HashFingerPrint(hash_val,AccessTbl.size);
    StepLinkedList(curr,AccessTbl.tbl[hash_ix]) {
	if (curr->hash_val==hash_val && SameBitVectors(BitVect,curr->bits)) {
	    break;
	}
    }
    if (curr == (AccessBlock *)NULL) {
	/*
	 * grow table if necessary */
	if (AccessTbl.cnt >= AccessTbl.threshold) {
	    RehashAccessTable(&AccessTbl);
	    hash_ix = HashFingerPrint(hash_val,AccessTbl.size);
	}
	/*
	 * 'BitVect' not in table, so add it */
	curr = AllocOne(AccessBlock);
	curr->index = (AccessTbl.cnt)++;
	curr->hash_val = hash_val;
	curr->bits = CopyBitVector(BitVect);
	SpliceIntoList(AccessTbl.tbl[hash_ix],curr);
    }
    return(curr->index);
}

void DisplayAccessBlocks(subj_cnt)
  int subj_cnt;
{
    register int i;
    register AccessBlock *curr;

    fprintf(OutFile,"(init-sem %d %d)\n",subj_cnt,AccessTbl.cnt);
    StepIndex(i,0,AccessTbl.size) {
	StepLinkedList(curr,AccessTbl.tbl[i]) {
	    fprintf(OutFile,"(blk %d #*",curr->index);
	    DisplayBitVector(curr->bits,BitVectSize);
	    fputs(")\n",OutFile);
	}
    }
}
