/* SYSNAME.C

   Functions implementing a hash table of sysname's.

   $Header: sysname.c,v 1.3 91/08/30 19:20:14 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   IMPLEMENTATION NOTES

   This module implements hash tables for Box and Arrow Sysnames. The
   implmentation uses bucket hashing. The hash tables grow dynamically when
   the hash table load factor exceeds a certain threshold. In this way, (so
   long as the hash function distributes items evenly), insert and lookup
   operations take O(1) amortized time.

   Currently, the buckets are represented as linked lists, but these should
   eventually be converted to a more efficient structure such as AVL trees.
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "sysname-types.h"
#include "elts.h"
#include "sysname.h"

/* DEFINITIONS ============================================================ */

/* initial sizes of sysname hash tables (both primes) */
#define INIT_B_SYSNAME_TABLE_SZ (6133L)
#define INIT_A_SYSNAME_TABLE_SZ (10007L)

/* threshold values */
#define B_TABLE_THRESHOLD (0.9)
#define A_TABLE_THRESHOLD (0.9)

/* growth factors */
#define B_TBL_GROWTH_FACTOR (2.1)
#define A_TBL_GROWTH_FACTOR (2.1)

/* LOCAL FUNCTIONS ======================================================== */

#define HashBSysname(_sysname,_size)\
    ((int)((_sysname) % (_size)))

#define HashASysname(_sysname,_size)\
    ((int)((_sysname) % (_size)))

static BSysnameEntry **EmptyBTable(size)
  long size;
{
    register int i;
    BSysnameEntry **tbl = AllocPtrArray(BSysnameEntry,size);
    StepIndex(i,0,size) { tbl[i] = (BSysnameEntry *)NULL; }
    return(tbl);
}

static Arrow **EmptyATable(size)
  long size;
{
    register int i;
    Arrow **tbl = AllocPtrArray(Arrow,size);
    StepIndex(i,0,size) { tbl[i] = (Arrow *)NULL; }
    return(tbl);
}

static BSysnameEntry *FindBSysname(tbl,sysname)
  BSysnameTable tbl;
  BSysname sysname;
/* RETURNS a pointer to the BTableEntry having Sysname 'sysname' in the table
   'tbl', or NULL if no such entry exists in 'tbl'.
*/
{
    register BSysnameEntry *entry;

    StepLinkedList(entry,tbl->u.b_tbl[HashBSysname(sysname,tbl->size)]) {
	if (entry->sysname == sysname) break;
    }
    return(entry);
}

static Arrow *FindASysname(tbl,sysname)
  ASysnameTable tbl;
  ASysname sysname;
/* RETURNS a pointer to the Arrow having Sysname 'sysname' in the table
   'tbl', or NULL if no such arrow exists in 'tbl'.
*/
{
    register Arrow *curr;

    StepLinkedList(curr,tbl->u.a_tbl[HashASysname(sysname,tbl->size)]) {
	if (curr->sysname == sysname) break;
    }
    return(curr);
}

static void RehashBSysnameTable(tbl)
  BSysnameTable tbl;
/* The hash table 'tbl->u.b_tbl' is full, so increase its size by a factor of
   the constant B_TBL_GROWTH_FACTOR, and rehash all of its current elements
   into the new table. This also has the side-effect of updating 'tbl->size'
   and 'tbl->threshold'.
*/
{
    register int i;
    int hash_ix;
    BSysnameEntry *entry,*temp;
    ULint new_size = (ULint)((float)tbl->size * B_TBL_GROWTH_FACTOR);
    BSysnameEntry **new_tbl = EmptyBTable(new_size);
    BSysnameEntry **old_tbl = tbl->u.b_tbl;

    StepIndex(i,0,tbl->size) {
	for (entry=old_tbl[i]; entry; /*empty*/) {
	    temp = entry->next;
	    hash_ix = HashBSysname(entry->sysname,new_size);
	    SpliceIntoList(new_tbl[hash_ix],entry);
	    entry = temp;
	}
    }
    tbl->size = new_size;
    tbl->threshold = (long)(B_TABLE_THRESHOLD * (float)new_size);
    Dealloc(old_tbl);
    tbl->u.b_tbl = new_tbl;
}

static void RehashASysnameTable(tbl)
  ASysnameTable tbl;
/* The hash table 'tbl->u.a_tbl' is full, so increase its size by a factor of
   the constant A_TBL_GROWTH_FACTOR, and rehash all of its current elements
   into the new table. This also has the side-effect of updating 'tbl->size'
   and 'tbl->threshold'.
*/
{
    register int i;
    int hash_ix;
    ULint new_size = (ULint)((float)tbl->size * A_TBL_GROWTH_FACTOR);
    Arrow **new_tbl = EmptyATable(new_size);
    Arrow **old_tbl = tbl->u.a_tbl;
    Arrow *entry,*temp;

    StepIndex(i,0,tbl->size) {
	for (entry=old_tbl[i]; entry; /*empty*/) {
	    temp = entry->next;
	    hash_ix = HashASysname(entry->sysname,new_size);
	    SpliceIntoList(new_tbl[hash_ix],entry);
	    entry = temp;
	}
    }
    tbl->size = new_size;
    tbl->threshold = (long)(A_TABLE_THRESHOLD * (float)new_size);
    Dealloc(old_tbl);
    tbl->u.a_tbl = new_tbl;
}

/* GLOBAL DEBUGGING FUNCTIONS ============================================== */

#ifdef DEBUG

void DisplayBSysnameTable(tbl)
  BSysnameTable tbl;
{
    int i,count;
    register BSysnameEntry *curr;

    fprintf(stderr,"\nBox Sysname Hash Table:\n");
    StepIndex(i,0,tbl->size) {
	count = 0;
	StepLinkedList(curr,tbl->u.b_tbl[i]) { count++; }
	if (count) {
	    fprintf(stderr,"  Bucket: %4d, Count: %d\n",i,count);
	    StepLinkedList(curr,tbl->u.b_tbl[i]) {
		fprintf(stderr,"    Sysname: %5d, ",curr->sysname);
		fprintf(stderr,"Value: 0x%x\n",(int)(curr->b));
	    }
	}
    }
    printf("Total number of entries: %d\n",tbl->cnt);
    printf("Total number of buckets: %d\n",tbl->size);
}

void DisplayASysnameTable(tbl)
  ASysnameTable tbl;
{
    int i,count;
    register Arrow *curr;

    fprintf(stderr,"\nArrow Sysname Hash Table:\n");
    StepIndex(i,0,tbl->size) {
	count = 0;
	StepLinkedList(curr,tbl->u.a_tbl[i]) { count++; }
	if (count) {
	    fprintf(stderr,"  Bucket: %4d, Count: %d\n",i,count);
	    StepLinkedList(curr,tbl->u.a_tbl[i]) {
		fprintf(stderr,"    Sysname: %5d, ",curr->sysname);
		fprintf(stderr,"Pos = 0x%02x, Neg = 0x%02x\n",
			(int)curr->perms[0],(int)curr->perms[1]);
	    }
	}
    }
    printf("Total number of entries: %d\n",tbl->cnt);
    printf("Total number of buckets: %d\n",tbl->size);
}

#endif DEBUG

/* GLOBAL FUNCTIONS ======================================================= */

BSysnameTable NewBSysnameTable()
{
    BSysnameTable result = (BSysnameTable)AllocOne(SysnameTable);
    result->size = INIT_B_SYSNAME_TABLE_SZ;
    result->cnt = 0L;
    result->threshold = (long)(B_TABLE_THRESHOLD * (float)result->size);
    result->u.b_tbl = EmptyBTable(result->size);
    return(result);
}

ASysnameTable NewASysnameTable()
{
    ASysnameTable result = (ASysnameTable)AllocOne(SysnameTable);
    result->size = INIT_A_SYSNAME_TABLE_SZ;
    result->cnt = 0L;
    result->threshold = (long)(A_TABLE_THRESHOLD * (float)result->size);
    result->u.a_tbl = EmptyATable(result->size);
    return(result);
}

ASysname CantorPair(x,y)
  BSysname x,y;
{
    ASysname temp = x+y;
    return(x + (temp * (temp-1))/2);
}

void AddBoxSysname(tbl,sysname,b)
  BSysnameTable tbl;
  BSysname sysname;
  Box *b;
{
    int hash_ix;
    BSysnameEntry *entry;

    /* grow table if necessary */
    if (tbl->cnt >= tbl->threshold) {
	RehashBSysnameTable(tbl);
    }

    /* create the BSysnameEntry */
    entry = AllocOne(BSysnameEntry);
    entry->sysname = sysname;
    entry->b = b;

    /* add it to the hash table */
    hash_ix = HashBSysname(sysname,tbl->size);
    SpliceIntoList(tbl->u.b_tbl[hash_ix],entry);
    tbl->cnt++;
}

Arrow *AddArrowSysname(tbl,sysname,arrow)
  ASysnameTable tbl;
  ASysname sysname;
  FullArrow *arrow;
{
    register Arrow *curr;
    int hash_ix = HashASysname(sysname,tbl->size);

    /* see if an arrow of this sysname already exists */
    StepLinkedList(curr,tbl->u.a_tbl[hash_ix]) {
	if (curr->sysname == sysname) { break; }
    }

    /* if not, create and add a new one */
    if (curr == (Arrow *)NULL) {
	/*
	 * grow the table if necessary */
	if (tbl->cnt >= tbl->threshold) {
	    RehashASysnameTable(tbl);
	    hash_ix = HashASysname(sysname,tbl->size);
	}
	/*
	 * create new ASysnameEntry for this arrow */
	curr = NewArrow();
	curr->sysname = sysname;
	curr->from = arrow->from;
	curr->to = arrow->to;
	SpliceIntoList(tbl->u.a_tbl[hash_ix],curr);
	tbl->cnt++;
    }

    /* add the permission information for the arrow */
    curr->perms[(int)(arrow->parity)] |= arrow->perms;
    return(curr);
}

Box *FindBoxSysname(tbl,sysname)
  BSysnameTable tbl;
  BSysname sysname;
{
    BSysnameEntry *entry;

    if ((entry=FindBSysname(tbl,sysname)) == NULL)
	return((Box *)NULL);
    else return(entry->b);
}

Arrow *FindArrowSysname(tbl,sysname,perm)
  ASysnameTable tbl;
  ASysname sysname;
  PermSet perm;
{
    Arrow *a;

    if ((a=FindASysname(tbl,sysname)) != NULL) {
	if (((a->perms[(int)Neg] | a->perms[(int)Pos]) & perm) != 0) {
	    return(a);
	}
    }
    return((Arrow *)NULL);
}
