/* TOP.C - top-level control file for Miro Ambiguity checker.

   $Header: top.c,v 1.4 91/10/29 14:46:11 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   Note: If the macro DEBUG is defined, the -d command-line switch for
   debugging is available in the resultant executable.
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <strings.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "parser.h"
#include "parser.g"

#include "box-type.h"
#include "top.h"
#include "extract-elts.h"
#include "sysname-types.h"
#include "elts.h"
#include "ambig.h"
#include "pict.h"

/* GLOBAL VARIABLES ======================================================= */

String Argv0;

/* LOCAL FUNCTIONS ======================================================== */

static void InitModules()
{
    InitParser();
    ConfigureHashTable();
}

static ShutDownModules()
{
    ShutDownParser();
}

static Boolean ProcessIffFile(iff_fp,filename,pict,build_tree)
  FILE *iff_fp;
  String filename;
  INOUT Pict *pict;
  Boolean build_tree;
/* Parse and process the file 'iff_fp' named 'filename'. Store the resulting
   picture elements in 'pict'. 'Progname' is the name of the program for
   error-reporting purposes. If 'build_tree' is True, call the routine
   BuildTypeTree() to construct the BoxType tree.

   Returns True iff errors were encountered parsing the file.
*/
{
    Boolean result = False;
    Entry *p_tree;		/* parse tree returned from parse */

    /* process the input file; parse and extract from the parse tree */
    if (((p_tree=ParseFile(iff_fp))==NULL           /* build parse tree */
	 || Extract(p_tree,pict)                    /* extract data */
	 || (build_tree && BuildTypeTree(pict)))) { /* build type tree */
	fprintf(stderr,"%s: %d error(s) encountered parsing file '%s'.\n",
		Argv0,parse_error_cnt,filename);
	result = True;
    }

    /* wrap up this parsing */
    FreeParseTree(p_tree);
    return(result);
}

static int ProcessInput(argc,argv)
  int argc;
  char *argv[];
/* Return an error code; 0 = no errors. */
{
    int option;			/* command-line flag */
    Boolean err_flag = False;	/* error encountered processing command-line */
    Boolean parse_err = False;	/* error encountered parsing IFF file */
    String boxtype_name;	/* name of input boxtype file */
    String input_name;		/* name of input file */
    String output_name;		/* name of output file (or NULL if none) */
    FILE *input_fp;		/* file to read IFF file from */
    FILE *boxtype_fp;		/* file to read IFF boxtype info from */
    int ambig_cnt;		/* number of ambiguities found */
    PermList *perm;		/* name of access permission in '-p'*/
    Pict *pict;			/* structure for the instance picture */
    Options *options;		/* command-line options */

    /* globals needed to use getopt(3) */
    int getopt();
    extern char *optarg;
    extern int optind;

    /* process command-line arguments */
    options = NewOptions();
    boxtype_name = BOXTYPE_FILE_NAME;
    output_name = (String)NULL;
    while ((option=getopt(argc,argv,"mMlap:t:o:")) != EOF) {
	switch(option) {
	  case 'm':
	    if (options->full_matrix) {
		fprintf(stderr,"%s: -m and -M are mutually exclusive\n",Argv0);
		err_flag = True;
		break;
	    }
	    options->matrix = True; break;
	  case 'M':
	    if (options->matrix) {
		fprintf(stderr,"%s: -m and -M are mutually exclusive\n",Argv0);
		err_flag = True;
		break;
	    }
	    options->full_matrix = True; break;
	  case 'l':
	    options->lisp_output = True; break;
	  case 'a':
	    options->atom_output = True; break;
	  case 'p':
	    perm = AllocOne(PermList);
	    perm->perm_name = optarg;
	    SpliceIntoList(options->perm_list,perm);
	    break;
	  case 't':
	    boxtype_name = optarg; break;
	  case 'o':
	    output_name = optarg; break;
	  default:
	    err_flag = True; break;
	}
    }

    /* check for command-line errors */
    if (err_flag == True || (argc-optind) > 1) {
	fprintf(stderr,SYNTAX_STRING,Argv0);
	return(COMMAND_EXIT);
    }

    /* create new picture stucture */
    pict = NewPict();

    /* open and process a 'boxtype_fp' file */
    if ((boxtype_fp=fopen(boxtype_name,"r")) == NULL) {
	/* boxtype file not found in current directory */
	if (*boxtype_name != '/') {
	    /* if relative path, try looking in BOXTYPE_FILE_DIR */
	    String full_name;
	    full_name =
	      AllocArray(char,strlen(BOXTYPE_FILE_DIR)+strlen(boxtype_name)+1);
	    strcpy(full_name,BOXTYPE_FILE_DIR);
	    strcat(full_name,boxtype_name);
	    boxtype_fp = fopen(full_name,"r");
	    Dealloc(full_name);
	}
    }
    if (boxtype_fp == NULL) {
	fprintf(stderr,"%s: Boxtype file '%s' could not be opened.\n",
		Argv0,boxtype_name);
	return(ERROR_EXIT);
    }
    if (ProcessIffFile(boxtype_fp,boxtype_name,pict,True)) parse_err = True;
    fclose(boxtype_fp);

    /* open & process an 'input_fp' file */
    if (argc > optind) {
	input_name = argv[optind];
	if ((input_fp=fopen(input_name,"r")) == NULL) {
	    fprintf(stderr,"%s: File '%s' could not be opened for reading.\n",
		    Argv0,argv[optind]);
	    return(ERROR_EXIT);
	}
    } else {
	input_fp = stdin;
	input_name = "stdin";
    }
    if (ProcessIffFile(input_fp,input_name,pict,False)) parse_err = True;
    if (input_fp != stdin) fclose(input_fp);

    /* return immediately if parse errors encountered */
    if (parse_err) return(ERROR_EXIT);
#ifdef OLD_DEBUG
    DisplayBSysnameTable(pict->b_sysnames);
    DisplayASysnameTable(pict->a_sysnames);
#endif OLD_DEBUG

    /* build array of arrow permissions in 'pict->perm_names' */
    BuildPermissionArray(pict);

    /* open output_fp file if necessary */
    if (output_name != (String)NULL) {
	if ((options->fp=fopen(output_name,"w")) == NULL) {
	    fprintf(stderr,"%s: File '%s' could not be opened for writing.\n",
		    Argv0,output_name);
	    return(ERROR_EXIT);
	}
    }

    /* check the picture for ambiguity, sending any output to 'output_fp' */
    if ((ambig_cnt=CheckAmbig(options,pict)) > 0) {
	/* ambiguities found */
	fprintf(stderr,"%s: found %d ambiguit%s.\n",Argv0,ambig_cnt,
		((ambig_cnt == 1) ? "y" : "ies"));
    }

    /* close output_fp if necessary */
    if (options->fp != stdout) fclose(options->fp);
    return((ambig_cnt > 0) ? AMBIG_EXIT : OKAY_EXIT);
}

/* GLOBAL FUNCTIONS ======================================================= */

int main(argc,argv)
  int argc;
  char *argv[];
{
    int exit_code;

    /* set program_name */
    Argv0 = (String)rindex(argv[0],'/');
    if (Argv0) Argv0++;
    else Argv0 = (String)argv[0];

    /* process input files */
    InitModules();
    exit_code = ProcessInput(argc,argv);
    ShutDownModules();
    return(exit_code);
}
