/* COMPILE.C

   Module for compiling a Pict representation of a constraint picture.

   $Header: compile.c,v 1.6 91/11/13 03:12:13 heydon Locked $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


/* Used for MAXPATHLEN definition */
#include <sys/param.h>

/* Used to get time of day this program was run */
#include <sys/types.h>
#include <sys/time.h>

#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "error.h"

#include "rank.h"
#include "compile.h"
#include "top.g"
#include "parse-pred.h"
#include "box-type.h"
#include "interval.h"
#include "var.h"
#include "id-table.h"
#include "objs.h"

/* MACRO DEFINITIONS ======================================================= */

/* for use by DateString() */
#define DATE_LEN 28
#define DATE_STR "%hour:%min [%noon|%am], %3Wday, %3Month %day %year"

/* LOCAL DEBUGGING FUNCTIONS =============================================== */

#ifdef OLD_DEBUG

static void DisplayBoxTypeRanksHelper(bt,level)
  BoxType *bt;
  int level;
{
    int i;
    float children_sum;
    BoxTypeList *curr;

    for (i=level; i>0; i--) fputs("  ",stderr);
    fprintf(stderr,"%s: %.6f, %.6f\n",bt->name,bt->sum_rank,bt->rank);
    if (bt->children) {
	children_sum = 0.0;
	StepLinkedList(curr,bt->children) {
	    children_sum += curr->bt->sum_rank;
	    DisplayBoxTypeRanksHelper(curr->bt,level+1);
	}
	for (i=level+1; i>0; i--) fputs("  ",stderr);
	fprintf(stderr,"Total: %.6f\n",children_sum);
    }
}

static void DisplayBoxTypeRanks(bt)
  BoxType *bt;
{
    fputs("\nBoxType Ranks:\n",stderr);
    DisplayBoxTypeRanksHelper(bt,1);
}
#endif OLD_DEBUG

/* LOCAL FUNCTIONS ========================================================= */

static void AddAdj(e1,e2)
  Elt *e1;
  Elt *e2;
/* Add 'e1' to the adjency list of 'e2'.
*/
{
    EltList *cell;

    cell = AllocOne(EltList);
    cell->elt = e1;
    cell->next = e2->adj_elts;
    e2->adj_elts = cell;
}

static void BuildAdjacencies(pict)
  INOUT Pict *pict;
/* Build the list of arrows adjacent to each box and boxes adjacent to each
   arrow by traversing the list of arrows.
*/
{
    ArrowList *curr;
    Arrow *a;
    StepLinkedList(curr,pict->arrows) {
	a = curr->elt->u.a;
	AddAdj(curr->elt,a->from);
	AddAdj(curr->elt,a->to);
	AddAdj(a->from,curr->elt);
	AddAdj(a->to,curr->elt);
    }
#ifdef OLD_DEBUG
    DisplayAllAdjacencies(pict);
#endif OLD_DEBUG
}

static Boolean StarredBoxNotContained(pict)
  Pict *pict;
/* Verifies that every starred box is contained in some other box. The
   'u2.contained' field should be true for every starred box. This field has
   been set by the routine ExtractInsideEntries(), where box containment is
   detected. After this routine, that field may be freely overwritten.
*/
{
    BoxList *bl;
    Box *b;
    Boolean result = False;
    StepLinkedList(bl,pict->boxes) {
	b = bl->elt->u.b;
	if (b->starred && !(b->u2.contained)) {
	    fprintf(stderr,
		    "%s: starred box %d not contained in any box\n",
		    argv0,bl->elt->sysname);
	    result = True;
	}
    }
    return(result);
}

static Boolean IsContainmentRoot(b)
  BoxElt *b;
/* Returns True iff 'b' has no containment edges pointing at it (i.e., it has
   in-degree 0, and is thus a potential root).
*/
{
    ArrowList *curr;
    Arrow *a;
    StepLinkedList(curr,b->adj_elts) {
	a = curr->elt->u.a;
	if (a->kind == Containment && a->to == b) return(False);
    }
    return(True);
}

static Boolean ContainmentSearch(b_elt)
  BoxElt *b_elt;
/* Does a DFS of the node 'b_elt'. First 'b_elt' is designated as "open".
   Then, all of its unmarked children are searched. If any of them are
   currently open, a cycle has occurred. In this case, the cycle is reported
   to stderr, and True is returned. Otherwise, each of the unmarked children
   of 'b_elt' is searched recursively. When those searches have finished,
   'b_elt' is "closed" and marked. In this case, False is returned.
*/
{
    Box *b = b_elt->u.b;
    ArrowList *al;
    Arrow *a;

    /* use a static to indicate state as we pop nodes when a cycle is found */
    static Boolean print_cycle = False;	/* indicates whether to print or not */

    if (b->u2.open) {		/* cycle detected; print an error message */
	fprintf(stderr,"%s: containment cycle found among boxes:",argv0);
	b_elt->marked = True;	/* mark so we know where to stop printing */
	print_cycle = True;	/* indicate that we should start printing */
	return(True);
    }
    b->u2.open = True;
    StepLinkedList(al,b_elt->adj_elts) {
	a = al->elt->u.a;
	if (a->kind == Containment && a->from == b_elt) {
	    /* child of this box is 'a->to' */
	    if (ContainmentSearch(a->to) == True) {
		if (print_cycle) {
		    /* cycle already found; print sysname of this box */
		    fprintf(stderr," %d",b_elt->sysname);
		}
		if (b_elt->marked) {
		    /* we found the start of the cycle, so stop printing */
		    print_cycle = False;
		    fputc('\n',stderr);
		}
		return(True);
	    }
	}
    }
    /* wrap up and return success */
    b->u2.open = False;
    b_elt->marked = True;
    return(False);
}

static Boolean ContainmentCycle(pict)
  Pict *pict;
/* Returns True iff a cycle exists in the graph in which nodes are the box
   patterns of 'pict', and edges are the containment arrows of 'pict'. The
   graph is represented by adjacency lists, where the adjacency information
   associated with a node is stored in the 'adj_elts' field of each BoxElt.
   If there is a cycle, an error message is printed to stderr describing the
   cycle (namely, the list of sysnames of boxes participating in the cycle).
   If there is more than one cycle, only one is printed; which one gets found
   first is implementation dependent.

   NOTE: The "open" and "marked" Box fields contain garbage at the end of this
   function. They should not be read, and they may be freely overwritten.

   IMPLEMENTATION: This routine uses depth-first search to find a cycle in
   O(m+n) time. We associate an "open" bit with each node. A node is "open"
   (and its open bit is turned on) if we have visited the node, and are still
   exploring the subtree of one of its children. If an edge goes to an "open"
   node, then that edge is a back edge, and so a cycle exists. We also use a
   bit to "mark" each node we visit, so we can tell which nodes are candidate
   roots.

   We start by considering those nodes with in-degree 0. We "search" any such
   node. If no cycles are found, we pick the next un-marked node with
   in-degree 0, and search it. We continue in this way until there are no more
   unmarked nodes with in-degree 0. At the end of this process, if there are
   *any* unmarked nodes, then we have a cycle. We can pick any one of them,
   search it, and a cycle will be produced.
*/
{
    BoxElt *b;
    BoxList *curr;

    /* initialize bits associated with each box */
    StepLinkedList(curr,pict->boxes) {
	b = curr->elt;
	b->marked = b->u.b->u2.open = False;
    }

    /* loop over boxes with in-degree 0 */
    StepLinkedList(curr,pict->boxes) {
	b = curr->elt;
	if (b->marked == False && IsContainmentRoot(b)) {
	    if (ContainmentSearch(b) == True) return(True);
	}
    }

    /* loop over any remaining un-marked boxes */
    StepLinkedList(curr,pict->boxes) {
	b = curr->elt;
	if (b->marked == False) {
	    Assert(ContainmentSearch(b) == True);
	    return(True);
	}
    }
    return(False);		/* no cycles found */
}

static VarType PrimValueTypeToVarType(val_type)
  PrimValueType val_type;
{
    VarType result;
    switch (val_type) {
      case IntVal:                result = IntVarType; break;
      case IdVal: case StringVal: result = StringVarType; break;
      case BoolVal:               result = BoolVarType; break;
      case BoxTypeVal:            result = BoxTypeVarType; break;
    }
    return(result);
}

static int SearchSimplePred(sp,b,pict)
  SimplePred *sp;
  BoxElt *b;
  Pict *pict;
/* Performs two checks: 1) that the attribute named in 'sp' is a known
   attribute, and 2) that the value to which that attribute is compared has a
   type consistent with the type of the attribute. If either check fails, an
   error is reported as occurring in box 'b->sysname'.

   If 'sp' is a variable comparison, an attempt is made to find the variable
   in the ID Hash Table. If there is already a variable installed there, the
   routine verifies that each use of the variable has the same type. If not,
   an error is reported. If the variable is *not* found in the ID Hash Table,
   this routine has the side-effect of installing a new variable in the ID
   Hash Table *and* in the list of variables 'pict->var_head'.

   Returns number of errors found processing the SimplePred 'sp' (the result
   is always either 0 or 1).
*/
{
    TableEntry *t;

    /* find the attribute in the ID Hash Table */
    if ((t=FindTableId(pict->table,AttrNameId,sp->attr_name)) == NULL) {
	fprintf(stderr,
		"%s: unknown attribute named '%s' in box %d\n",
		argv0,sp->attr_name,b->sysname);
	return(1);
    }

    /* verify that 'kind' is consistent with 'type' */
    if (t->u.attr->type.list_nesting != 0) {
	fprintf(stderr,"%s: attribute named '%s' in box %d is a list type\n",
		argv0,sp->attr_name,b->sysname);
	return(1);
    }
    switch (t->u.attr->type.prim) {
      case IntVal:
	if (!(sp->kind == IntPredKind || sp->kind == VarPredKind)) {
	    goto incomparable_error;
	}
	break;
      case IdVal:
      case StringVal:
      case BoxTypeVal:
	if (!(sp->kind == StringPredKind || sp->kind == VarPredKind)) {
	    goto incomparable_error;
	}
	break;
      case BoolVal:
	if (sp->kind != BoolPredKind) {
	    goto incomparable_error;
	}
	break;
    }

    /* set 'type' field of 'sp' from attribute type */
    sp->type = t->u.attr->type.prim;

    /* install variable if necessary */
    if (sp->kind == VarPredKind) {
	BoxList *bl;
	Var *var;
	VarExpr *v = sp->u.var_expr; /* variable expression in 'sp' */
	/*
	 * variable comparisons cannot be anything but equality (yet) */
	if (sp->rel != Eq) {
	    fprintf(stderr,
		    "%s: variable '$%s' not compared for equality in box %d\n",
		    argv0,v->name,b->sysname);
	    return(1);
	}
	/*
	 * variables not allowed to have a suffix or prefix (yet) */
	if (v->prefix != NULL || v->suffix != NULL) {
	    fprintf(stderr,
		    "%s: variable '$%s' has a prefix or suffix in box %d\n",
		    argv0,v->name,b->sysname);
	    return(1);
	}
	/*
	 * see if this variable has been encountered */
	if ((t=FindTableId(pict->table,VarNameId,v->name))
	    == (TableEntry *)NULL) {
	    VarList *var_list;	/* variable list cell */
	    VarCnt *cnt;
	    /*
	     * variable not installed, so install it in the IdHashTable */
	    var = AllocOne(Var);
	    var->name = sp->u.var_expr->name;
	    var->type = PrimValueTypeToVarType(sp->type);
	    var->first = NOT_BOUND;
	    var->cnt = cnt = AllocOne(VarCnt);
	    cnt->eq_cnt = cnt->low_cnt = cnt->high_cnt = 0;
	    var->boxes = (BoxList *)NULL;
	    (void)AddTableId(pict->table,VarNameId,var->name,(Generic *)var);
	    /*
	     * install it in 'pict->var_head' */
	    var_list = AllocOne(VarList);
	    var_list->var = var;
	    SpliceIntoList(pict->var_head,var_list);
	} else {
	    /* variable already installed, verify this one is correct type */
	    var = t->u.var;
	    if (var->type != PrimValueTypeToVarType(sp->type)) {
		fprintf(stderr,
			"%s: variable '$%s' has multiple type associations\n",
			argv0,t->name);
		return(1);
	    }
	}
	/*
	 * add this box to the list for this variable if not already there */
	StepLinkedList(bl,var->boxes) {
	    if (bl->elt == b) break;
	}
	if (bl == (BoxList *)NULL) {
	    /* the box is not already in the list, so add it */
	    bl = AllocOne(BoxList);
	    bl->elt = b;
	    SpliceIntoList(var->boxes,bl);
	}
    }
    return(0);

incomparable_error:
    fprintf(stderr,
	    "%s: attribute '%s' compared to wrong type in box %d\n",
	    argv0,sp->attr_name,b->sysname);
    return(1);
}

static int SearchPred(pred,b,pict)
  Predicate *pred;
  BoxElt *b;
  Pict *pict;
{
    int error_cnt = 0;

    if (pred) {
	error_cnt = SearchSimplePred(pred->simple_pred,b,pict);
	error_cnt += SearchPred(pred->pred,b,pict);
    }
    return(error_cnt);
}

static Boolean CheckTypes(pict)
  INOUT Pict *pict;
/* Verifies that the type of each attribute named in a box predicate is
   comparable to the type of the value to which it is compared. If not, an
   error is reported.

   This routine also has the side effect of setting the 'type' field in each
   SimplePred and of installing variables in 'pict->table'. An error is
   reported if a variable is installed under more than one type.

   RETURNS True iff an error is reported. All errors are reported to stderr.
*/
{
    BoxList *bl;
    int error_cnt = 0;

    StepLinkedList(bl,pict->boxes) {
	error_cnt += SearchPred(bl->elt->u.b->u1.pred,bl->elt,pict);
    }
    return(MakeBoolean(error_cnt > 0));
}

static BoxSide SideFromType(b_elt,pict)
  BoxElt *b_elt;
  Pict *pict;
/* Search the predicate of 'b_elt' for a "type" comparison. If possible, use
   that predicate to determine which "side" 'b_elt' must fall on, and return
   it. If there is no "type" predicate, or if there are only type comparisons
   with variables, then return 'NeitherSide'.
*/
{
    Box *b = b_elt->u.b;
    Predicate *pred;
    SimplePred *sp;
    TableEntry *t;

    Assert(b->kind == PredKind);
    for (pred=b->u1.pred; pred; pred = pred->pred) {
	sp = pred->simple_pred;
	if (SameString(sp->attr_name,"type") && sp->kind != VarPredKind) {
	    Assert(sp->type == BoxTypeVal);
	    if ((t=FindTableId(pict->table,BoxTypeId,sp->u.string)) == NULL) {
		continue;	/* skip this error; it will be caught later */
	    }
	    switch (t->u.box_type->class) {
	      case Subject: return(LeftSide);
	      case Object: return(RightSide);
	      case Entity: return(NeitherSide);
	    }
	}
    }
    return(NeitherSide);
}

static BoxSide GetSideUnion(b_elt)
  BoxElt *b_elt;
/* Returns the bitwise OR of the "side" fields of all the boxes in the
   connected component (according to containment edges) rooted at 'b', with
   the exception that the "Orig" bit of the returned result is always 0.

   IMPLEMENTATION: The 'u2.visited' Box field is used to mark nodes visited by
   this routine so they are not visited more than once. When this routine
   completes, they will all be True. Since this field is not used later, it
   may be freely overwritten.
*/
{
    Box *b = b_elt->u.b;
    ArrowList *curr;
    Arrow *a;
    BoxElt *other;		/* box at other end of containment arrow */
    BoxSide result;

    b->u2.visited = True;
    result = b->side;
    StepLinkedList(curr,b_elt->adj_elts) {
	a = curr->elt->u.a;
	if (a->kind == Containment) {
	    other = (a->from == b_elt) ? a->to : a->from;
	    if (!(other->u.b->u2.visited)) {
		result = (BoxSide)((Uint)result | (Uint)GetSideUnion(other));
	    }
	}
    }
    return(result);
}

static void PropagateSide(b_elt,side)
  BoxElt *b_elt;
  BoxSide side;
/* For each box in the connected component (according to containment edges)
   of 'b_elt', OR the 'side' field of that box with the bits 'side'. Also, set
   the 'marked' field of all such boxes.
*/
{
    Box *b = b_elt->u.b;
    ArrowList *curr;
    Arrow *a;
    BoxElt *other;		/* box at other end of containment arrow */

    b_elt->marked = True;
    b->side = side;
    StepLinkedList(curr,b_elt->adj_elts) {
	a = curr->elt->u.a;
	if (a->kind == Containment) {
	    other = (a->from == b_elt) ? a->to : a->from;
	    if (!(other->marked)) {
		PropagateSide(other,side);
	    }
	}
    }
}

static Boolean MarkLeftRight(pict)
  INOUT Pict *pict;
/* Marks each box in 'pict' as being either a "left" or a "right" box, by
   setting the 'u.b->side' field of each BoxElt. The rules for determining the
   type of a box are as follows. If the box is at the tail of a syntax or
   semantics arrow, it is a "left" box; if it is at the head of such an arrow,
   it is a "right" box. If there is a "type" simple predicate on the box, then
   that information is also used to set the "side" of the box.

   If it is attached by a containment arrow to a box marked T, then it gets
   marked T as well. A constraint is ill-formed if any box would be marked
   *both* "left" and "right" by this procedure.

   Returns True iff the constraint is ill-formed. In this case, an error
   message is printed to stderr for each problematic box.
*/
{
    BoxList *bl;
    ArrowList *al;
    int error_cnt;

    /* initialize box markings -- these may be redundant */
    StepLinkedList(bl,pict->boxes) {
	BoxElt *b = bl->elt;
	b->marked = b->u.b->u2.visited = False;
	b->u.b->side = NeitherSide;
    }

    /* mark definite left/right boxes from syntax/semantics arrows */
    StepLinkedList(al,pict->arrows) {
	Arrow *a = al->elt->u.a;
	if (a->kind == Syntax || a->kind == Semantics) {
	    a->from->u.b->side =
		(BoxSide)((Uint)a->from->u.b->side|(Uint)LeftSide);
	    a->to->u.b->side =
		(BoxSide)((Uint)a->to->u.b->side|(Uint)RightSide);
	}
    }

    /* search for "type" simple predicate to determine the box side */
    StepLinkedList(bl,pict->boxes) {
	Box *b = bl->elt->u.b;
	if (b->side == NeitherSide) {
	    b->side = SideFromType(bl->elt,pict);
	}
    }

    /* close left/right markings over containment arrows */
    StepLinkedList(bl,pict->boxes) {
	BoxElt *b = bl->elt;
	if (b->marked == False) {
	    PropagateSide(b,GetSideUnion(b));
	}
    }

    /* look for conflicts (boxes marked both left and right) */
    error_cnt = 0;
    StepLinkedList(bl,pict->boxes) {
	BoxElt *b = bl->elt;
	if (b->u.b->side == BothSides) {
	    fprintf(stderr,"%s: box %d is both a 'left' and 'right' box\n",
		    argv0,b->sysname);
	    error_cnt++;
	}
    }
    return(error_cnt > 0 ? True : False);
}

static void BoxTypeSumRanks(bt,val)
  BoxType *bt;
  float val;
/* Given that 'val' is the newly-computed 'sum_rank' of the parent of 'bt',
   set the 'sum_rank' of 'bt' to the product of 'val' and the current
   'sum_rank' of 'bt', and continue this process recursively for all
   descendants of 'bt' in the type-tree.
*/
{
    BoxTypeList *curr;

    bt->sum_rank *= val;
    StepLinkedList(curr,bt->children) {
	BoxTypeSumRanks(curr->bt,bt->sum_rank);
    }
}

static void BoxTypeRanks(bt)
  BoxType *bt;
/* Recursively compute the 'rank' field of each BoxType in the subtree rooted
   at 'bt' as the 'sum_rank' of 'bt' minus the 'sum_rank' values of each of
   bt's children.
*/
{
    BoxTypeList *curr;

    bt->rank = bt->sum_rank;
    StepLinkedList(curr,bt->children) {
	bt->rank -= curr->bt->sum_rank;
	BoxTypeRanks(curr->bt);
    }
}

static int BoxTypePreorderTraverse(bt,val)
  BoxType *bt;
  int val;
/* Do a DFS pre-order traversal of the subtree rooted at 'bt', assigning 'val'
   to the 'num' field of 'bt'. The 'top_range' field is also set to the
   maximum 'num' value of its descendents. In this way, BoxType 'a' is an
   ancestor of BoxType 'b' iff a->num <= b->num <= a->top_range.
*/
{
    BoxTypeList *curr;

    bt->num = val;
    StepLinkedList(curr,bt->children) {
	val = BoxTypePreorderTraverse(curr->bt,val+1);
    }
    return(bt->top_range=val);
}

static Boolean InitBoxTypeValues(pict)
  Pict *pict;
/* Sets the 'sum_rank' field of each BoxType in the list 'pict->box_types' to
   the product of its current 'sum_rank' and those of its ancestors all the
   way up to the root. Using these new 'sum_rank' values, each 'rank' is then
   computed as the node's 'sum_rank' minus the 'sum_rank' values of its
   children. See notes in II:14-15.

   This routine also sets the 'num' and 'top_range' fields of each BoxType so
   that ancestorship queries can be answered in O(1) time.

   Returns False for purposes of its use in CompilePict().
*/
{
    BoxType *root;		/* unique 'entity' type having no parents */
    BoxTypeList *curr;

    /* find the 'root' type */
    StepLinkedList(curr,pict->box_types) {
	if (curr->bt->u.parent == (BoxType *)NULL) break;
    }
    Assert(curr != (BoxTypeList *)NULL);
    root = curr->bt;

    /* compute the sum_rank's recursively */
    BoxTypeSumRanks(root,1.0);

    /* compute the rank's recursively */
    BoxTypeRanks(root);

    /* compute 'num' and 'top_range' values */
    (void)BoxTypePreorderTraverse(root,1);
#ifdef OLD_DEBUG
    DisplayBoxTypeRanks(root);
#endif OLD_DEBUG
    return(False);
}

static String DateString()
/* RETURNS a String containing the current date and time in the form:
   "12:52 pm, Mon, Sep 17 1990" or "12:00 noon, Tue, Oct 3 1990".
*/
{
    static char buff[DATE_LEN+1];
    char *fdate();
    struct tm *tm;
    struct timeval tp;
    struct timezone tzp;

    if (gettimeofday(&tp,&tzp)) {
	perror(argv0);
	exit(-1);
    }
    tm = localtime((time_t *)&tp.tv_sec);
    return(fdate(buff,DATE_STR,tm));
}

static void GenerateHeader(fp,input_name)
  FILE *fp;
  String input_name;
{
    int i;
    int num_pkgs = 6;		/* # of elements in pkg[] */
    static String pkg[] =
	{ "objs", "boxtype", "func-ops", "hds", "iter", "ipql" };
    extern char *getwd();

    /* generate header comments */
    fputs(";;; IPQL program created by cpc(1)\n;;;\n",fp);
    fputs(";;; File: ",fp);
    if (input_name) {
	if (*input_name != '/') {
	    /* relative path; print current working directory first */
	    char path[MAXPATHLEN];
	    fprintf(fp,"%s/",getwd(path));
	}
	fprintf(fp,"%s\n",input_name);
    } else {
	/* no input name, so input is coming from stdin */
	fputs("stdin\n",fp);
    }
    fprintf(fp,";;; Date: %s\n\n",DateString());
    fprintf(fp,"(in-package 'constraints)\n\n");

    /* generate "require" and "use-package" commands */
    StepIndex(i,0,num_pkgs) { fprintf(fp,"(require '%s)\n",pkg[i]); }
    StepIndex(i,0,num_pkgs) { fprintf(fp,"(use-package '%s)\n",pkg[i]); }
}

static void SearchType(fp,bt)
  FILE *fp;
  BoxType *bt;
{
    BoxTypeList *curr;
    AttrList *attr;

    /* print a (new-boxtype...) command for this type */
    fprintf(fp,"(new-boxtype %s ",bt->name);
    switch (bt->class) {
      case Entity:  fputs(FALSE_KEY,fp); break;
      case Subject: fputs(SUBJ_KEY,fp);  break;
      case Object:  fputs(OBJ_KEY,fp);   break;
    }
    fprintf(fp," %s",bt->u.parent ? bt->u.parent->name : "NIL");
    /* print names of non-list attributes */
    StepLinkedList(attr,bt->attr_head) {
	if (attr->attr->type.list_nesting == 0) {
	    fprintf(fp," %s",attr->attr->name);
	}
    }
    fputs(")\n",fp);

    /* recursively search children of this type */
    StepLinkedList(curr,bt->children) {	SearchType(fp,curr->bt); }
}

static void GenerateTypeTree(fp,pict)
  FILE *fp;
  Pict *pict;
{
    fputs("\n;;; define boxtype tree\n(start-boxtypes)\n",fp);
    SearchType(fp,FindTableId(pict->table,BoxTypeId,"entity")->u.box_type);
    fputs("(end-boxtypes)\n",fp);
}

static void GenerateObjectCode(fp,pict)
  FILE *fp;
  Pict *pict;
/* Generate the "object" code corresponding to the picture 'pict' to the FILE
   'fp'. The output consists of 5 parts:

   1) A "header", which contains comments describing the name of the input
      file and the time/date the object file is produced, plus some headers
      necessary for every IPQL program,
   2) IPQL commands to build the box-type tree defined by the box-types file,
   3) IPQL commands to construct the HDS structures for this constraint,
   4) IPQL commands to query the user for an instance picture and load it, and
   5) IPQL commands to perform the actual query.
*/
{
    String fname,ext;

    /* (1) generate the "header" */
    GenerateHeader(fp,pict->input);

    /* (2) generate the box-type tree constructors */
    GenerateTypeTree(fp,pict);

    /* wrap code inside a defun */
    if ((fname=rindex(pict->input,'/')) != NULL) { fname++; }
    else { fname = pict->input; }
    if ((ext=index(fname,'.')) != NULL) { *ext = '\0'; }
    fprintf(fp,"\n(defun %s (&optional instance-file)\n",fname);
    if (ext != NULL) { *ext = '.'; }
    fputs("\n;;; initialize constraint\n(new-constraint)\n",fp);

    /* (3) generate the HDS constructors */
    GenerateHdsConstructors(fp,pict); /* (in hds.c module) */

    /* (4) generate commands to read an instance picture */
    fputs("\n;;; load the instance picture\n",fp);
    fputs("(load-instance instance-file)\n",fp);

    /* (5) generate the IPQL query */
    GenerateQuery(fp,pict);	      /* (in hds.c module) */

    /* close defun */
    fputs(")\n",fp);
}

/* GLOBAL FUNCTIONS ======================================================== */

Boolean CompilePict(fp,pict)
  FILE *fp;
  INOUT Pict *pict;

{
    /* build full adjency matrix among objects */
    BuildAdjacencies(pict);

    /* check well-formedness conditions on constraint pictures */
    if (StarredBoxNotContained(pict) /* check for non-contained starred box */
	|| ContainmentCycle(pict)    /* check for cycle of containments */
	|| CheckTypes(pict)	     /* check all attrs of correct type */
	|| MarkLeftRight(pict)	     /* use containments to mark box sides */
	|| InitBoxTypeValues(pict)   /* fill in values of each BoxType */
	|| FormAllIntrvls(pict)) {   /* check all intervals non-empty */
	return(True);
    }

    /* order objects using ranks and thicknesses */
    OrderObjs(pict);

    /* construct the HDS structures for this picture */
    ConstructHdsStructures(pict);

    /* generate the "object" code file */
    GenerateObjectCode(fp,pict);

    /* return success */
    return(False);
}
