/* EXTRACT-OBJS.C

   Module for extracting information from a cpc(1) parse tree.

   $Header: extract-objs.c,v 1.5 91/11/12 16:24:52 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "error.h"
#include "id-hash.h"
#include "parser.h"
#include "parser.g"
#include "extract.h"

#include "extract-objs.h"
#include "parse-pred.h"
#include "attr.h"
#include "box-type.h"
#include "id-table.h"
#include "objs.h"

/* ----------------------------------------------------------------------------
 *
 * NOTE: When data is extracted from the parse tree, it must always be
 * *copied*, since the parse tree (and the ID Hash Table built into the parse
 * library) are deallocated when extraction is complete. This copying is
 * almost always performed in the process of adding the name to the ID Hash
 * Table (the routine AddTableId() makes a copy of the name automatically).
 *
 * ----------------------------------------------------------------------------
*/

/* MACRO DEFINITIONS ======================================================= */

#define MAX_PNAMES 8		/* maximum # of prop names in any ENTRY */

/* DEBUGGING FUNCTIONS ===================================================== */

#ifdef OLD_DEBUG

static void DisplayBoxType(bt)
  BoxType *bt;
{
    AttrList *curr;
    Attr *attr;
    int nesting;

    fprintf(stderr,"\nExtracted BOXTYPE entry\n");
    fprintf(stderr,"  name = '%s', ",bt->name);
    fprintf(stderr,"parent = '%s'\n",
	    bt->u.supertype==(String)NULL ? "NONE" : bt->u.supertype);
    StepLinkedList(curr,bt->attr_head) {
	attr = curr->attr;
	fprintf(stderr,"  < %s, ",attr->name);
	switch(attr->type.prim) {
	  case IntVal:     fputs("integer",stderr); break;
	  case IdVal:      fputs("identifier",stderr); break;
	  case StringVal:  fputs("string",stderr); break;
	  case BoolVal:    fputs("boolean",stderr); break;
	  case BoxTypeVal: fputs("box-type",stderr); break;
	}
	StepIndex(nesting,0,attr->type.list_nesting) {
	    fputs("-list",stderr);
	}
	fprintf(stderr,", %s >\n",attr->required ? "mandatory" : "optional");
    }
}
#endif OLD_DEBUG

/* LOCAL FUNCTIONS ========================================================= */

/* Conversion Functions ---------------------------------------------------- */

static void ConvertThickness(p_val_ptr,val_ptr,line_no)
  PropVal *p_val_ptr;
  Generic *val_ptr;
  int line_no;
{
    Thickness *result_ptr = (Thickness *)val_ptr;

#ifdef OLD_DEBUG
    if (PropValTypeOf(p_val_ptr) != IdPValType) {
	ProgrammerErrorI("extract.c",
			 "prop_val_type should be IdPValType instead of 0x%x",
			 PropValTypeOf(p_val_ptr));
    }
#endif OLD_DEBUG
    if (SameString("thick",IdValOf(p_val_ptr))) {
	*result_ptr = Thick;
    } else if (SameString("thin",IdValOf(p_val_ptr))) {
	*result_ptr = Thin;
    } else {
	ParseErrorS(line_no,
		    "thickness '%s' should be either 'thick' or 'thin'",
		    IdValOf(p_val_ptr));
    }
}

static void ConvertArrowKind(p_val_ptr,val_ptr,line_no)
  PropVal *p_val_ptr;
  Generic *val_ptr;
  int line_no;
{
    ArrowKind *result_ptr = (ArrowKind *)val_ptr;

#ifdef OLD_DEBUG
    if (PropValTypeOf(p_val_ptr) != IdPValType) {
	ProgrammerErrorI("extract.c",
			 "prop_val_type should be IdPValType instead of 0x%x",
			 PropValTypeOf(p_val_ptr));
    }
#endif OLD_DEBUG
    if (SameString("sem",IdValOf(p_val_ptr))) {
	*result_ptr = Semantics;
    } else if (SameString("syn",IdValOf(p_val_ptr))) {
	*result_ptr = Syntax;
    } else if (SameString("con",IdValOf(p_val_ptr))) {
	*result_ptr = Containment;
    } else {
	ParseErrorS(line_no,
		    "arrow kind '%s' should be 'sem', 'syn', or 'con'",
		    IdValOf(p_val_ptr));
    }
}

static void ConvertParity(p_val_ptr,val_ptr,line_no)
  PropVal *p_val_ptr;
  Generic *val_ptr;
  int line_no;
{
    ArrowParity *result_ptr = (ArrowParity *)val_ptr;

#ifdef OLD_DEBUG
    if (PropValTypeOf(p_val_ptr) != IdPValType) {
	ProgrammerErrorI("extract.c",
			 "prop_val_type should be IdPValType instead of 0x%x",
			 PropValTypeOf(p_val_ptr));
    }
#endif OLD_DEBUG
    if (SameString("pos",IdValOf(p_val_ptr))) {
	*result_ptr = Pos;
    } else if (SameString("neg",IdValOf(p_val_ptr))) {
	*result_ptr = Neg;
    } else {
	ParseErrorS(line_no,
		    "arrow parity '%s' should be either 'pos' or 'neg'",
		    IdValOf(p_val_ptr));
    }
}

static void ConvertAttribute(p_val_ptr,val_ptr,line_no)
  PropVal *p_val_ptr;
  Generic *val_ptr;
  int line_no;
/* This routine converts the attribute value in 'p_val_ptr', storing the
   result in 'val_ptr', which is really of type (Attr *). The value
   represented by 'p_val_ptr' has the following form:

        { <attr-name>, <type>, "mandatory" | "optional" [, <default>] }.

   The <default> should only be supplied if the third item is "optional"; it
   is the default value to use in the case where none is supplied. This
   routine fills in the 'name', 'type', and 'required' fields of 'val_ptr'.
*/
{
    int member_cnt;		/* number of current element (starting @ 1) */
    ListEntry *curr;		/* current element of 'p_val_ptr' */
    String id;			/* current member string (into table) */
    String end;			/* end of 'id' in the case of member 2 */
    Attr *result_ptr = (Attr *)val_ptr;

    if (p_val_ptr->prop_val_type != ListPValType) {
	ParseError(line_no,"attributes must be a list of lists");
	return;
    }
    member_cnt = 0;
    StepLinkedList(curr,p_val_ptr->val.list_head) {
	if (++member_cnt < 4) {
	    if (curr->list_prop_val_ptr->prop_val_type != IdPValType) {
		ParseErrorI(line_no,
		"attribute descriptor contains a non-identifier in field %d",
			    member_cnt);
		return;
	    }
	    id = curr->list_prop_val_ptr->val.id_val;
	}
	switch(member_cnt) {
	  case 1:		/* attribute name */
	    result_ptr->name = id; /* copy made when installed in table */
	    break;
	  case 2:		/* attribute type */
	    /* NOTE: We can't change 'id' permanently because it points into
	       the ID hash table. We change it temporarily to make the string
	       comparisons, and then change it back afterwards */
	    /*
	     * compute proper list nesting value */
	    result_ptr->type.list_nesting = 0;
	    for (end = id+strlen(id)-5;
		 end>id && !strncmp(end,"-list",5); end-=5) {
		(result_ptr->type.list_nesting)++;
	    }
	    /* temporarily change '-' of *first* "-list" to '\0' */
	    for (end=id; *end; end++) {
		if (!strncmp(end,"-list",5)) { break; }
	    }
	    if (*end == '-') {
		*end = '\0';	/* temporarily change to string terminator */
	    } else {
		end = (String)NULL; /* set to NULL to indicate no change */
	    }
	    /* match the proper base type */
	    if (SameString(id,"integer")) {
		result_ptr->type.prim = IntVal;
	    } else if (SameString(id,"identifier")) {
		result_ptr->type.prim = IdVal;
	    } else if (SameString(id,"string")) {
		result_ptr->type.prim = StringVal;
	    } else if (SameString(id,"boolean")) {
		result_ptr->type.prim = BoolVal;
	    } else if (SameString(id,"box-type")) {
		result_ptr->type.prim = BoxTypeVal;
	    } else {
		ParseErrorS(line_no,"'%s' not a legal attribute type",
			    result_ptr->name);
	    }
	    /* change '\0' back to '-' if necessary */
	    if (end) { *end = '-'; }
	    break;
	  case 3:		/* mandatory or optional? */
	    if (SameString(id,"mandatory")) {
		result_ptr->required = True;
	    } else if (SameString(id,"optional")) {
		result_ptr->required = False;
	    } else {
		ParseErrorS(line_no,
		"'required' field for '%s' must be 'mandatory' or 'optional'",
			    result_ptr->name);
	    }
	    break;
	  case 4:		/* default value for optional attribute*/
	    if (result_ptr->required) {
		ParseErrorS(line_no,
	        "default value for '%s' not allowed with mandatory attribute",
			    result_ptr->name);
	    }
	    /* ignore default value */
	    break;
	  default:
	    ParseErrorS(line_no,
		   "attribute descriptor for '%s' contains too many elements",
			result_ptr->name);
	    return;
	}
    }
    if (member_cnt < 3) {
	ParseErrorS(line_no,
		    "attribute descriptor for '%s' contains too few elements",
		    result_ptr->name);
	return;
    }
}

/* Box Functions ----------------------------------------------------------- */

static BoxElt *FindBox(sysname,pict,line_no)
  int sysname;
  Pict *pict;
  int line_no;
/* Return a pointer to the BoxElt in 'pict' with sysname of 'sysname'. If no
   such box is found, report an error using line number 'line_no'.

   IMPLEMENTATION NOTE: This function uses an inefficient linear-time search.
   For more efficiency, we could build a hash table of boxes by sysname, and
   then find each box in constant time by consulting that hash table. Since
   the number of boxes in a picture is likely to be small, that approach was
   deemed unnecessary, at least in the prototype.
*/
{
    BoxList *curr;
    StepLinkedList(curr,pict->boxes) {
	if (curr->elt->sysname == sysname) return(curr->elt);
    }
    ParseErrorI(line_no,"no box with sysname %d",sysname);
    return((BoxElt *)NULL);
}

static BoxElt *ExtractBox(box_entry)
  Entry *box_entry;
/* Return a pointer to a new Box containing attributes according to the Entry
   'box_entry', or NULL if errors were found in the Entry.
*/
{
    BoxElt *result;
    String name;

    /* allocate a box and initialize optional attributes */
    result = NewBox();
    StartNewPNameList();
    AddPNameDesignator("sysname",True,ConvertInt,
		       (Generic *)&(result->sysname),NULL_LIST);
    AddPNameDesignator("name",False,ConvertString,
		       (Generic *)&name,NULL_LIST);
    AddPNameDesignator("thickness",True,ConvertThickness,
		       (Generic *)&(result->thickness),NULL_LIST);
    AddPNameDesignator("starred?",False,ConvertBoolean,
		       (Generic *)&(result->u.b->starred),NULL_LIST);
    if (MatchPNames(box_entry)) {
	/* errors found; deallocate space and return error result */
	FreeBoxElt(result);
	result = (BoxElt *)NULL;
    } else {
	/* no errors found; parse the predicate in 'name' */
	result->u.b->u1.pred =
	    (ValFoundP("name") && *name != '\0')
		? ParsePred(name,EntryLineNumberOf(box_entry))
		: (Predicate *)NULL;
    }
#ifdef OLD_DEBUG
    fprintf(stderr,"\nExtracted BOX, sysname = %d\nParse Tree:\n",
	    result->sysname);
    DisplayPred(result->u.b->u1.pred);
#endif OLD_DEBUG
    return(result);
}

static void InsertBox(b,pict)
  BoxElt *b;
  Pict *pict;
{
    BoxList *b_cell;

    if (b != (BoxElt *)NULL) {
	/* allocate a BoxList cell and make it point to 'b' */
	b_cell = AllocOne(BoxList);
	b_cell->elt = (Elt *)b;
	/* insert the box in the list */
	pict->elt_cnt++;
	SpliceIntoList(pict->boxes,b_cell);
    }
}

static void ExtractBoxEntries(p_tree,pict)
  Entry *p_tree;
  INOUT Pict *pict;
/* Traverse the list of entries in 'p_tree', creating Boxes for each BOX entry
   in the tree. Add these Boxes to the list 'pict->boxes'.
*/
{
    Entry *curr_ent;

    StepLinkedList (curr_ent,p_tree) {
	if (EntryTypeOf(curr_ent) == BoxEntry) {
	    InsertBox(ExtractBox(curr_ent),pict);
	}
    }
}

/* Arrow Functions --------------------------------------------------------- */

static ArrowElt *ExtractArrow(arrow_entry,pict)
  Entry *arrow_entry;
  Pict *pict;
{
    ArrowElt *result;		/* resulting arrow */
    Arrow *a;			/* result->u.a */
    int to,from;		/* sysnames of from and to boxes */
    ListDesc perm_list,*perms;	/* list of permissions */
    PermList *perm_val;		/* new cell for permission name */
    String perm_name;		/* name of current permission */
    TableEntry *t;		/* entry for permission 'perm_name' */
    int line_no = EntryLineNumberOf(arrow_entry);

    /* initialize so we don't have to dereference 'perm_list' everywhere */
    perms = &perm_list;

    /* create new arrow and install attribute designators */
    result = NewArrow();
    a = result->u.a;
    StartNewPNameList();
    AddPNameDesignator("sysname",True,ConvertInt,
		       (Generic *)&(result->sysname),NULL_LIST);
    AddPNameDesignator("kind",True,ConvertArrowKind,
		       (Generic *)&(a->kind),NULL_LIST);
    AddPNameDesignator("from",True,ConvertInt,(Generic *)&from,NULL_LIST);
    AddPNameDesignator("to",True,ConvertInt,(Generic *)&to,NULL_LIST);
    AddPNameDesignator("thickness",True,ConvertThickness,
		       (Generic *)&(result->thickness),NULL_LIST);
    AddPNameDesignator("parity",True,ConvertParity,
		       (Generic *)&(a->parity),NULL_LIST);
    AddPNameDesignator("permissions",False,ConvertId,
		       (Generic *)NULL,perms);
    AddPNameDesignator("starred?",False,ConvertBoolean,
		       (Generic *)&(a->u.starred),NULL_LIST);

    /* extract values from the entry */
    if (MatchPNames(arrow_entry)) { goto arrow_error; }

    /* parse arrow permission(s), if applicable */
    if (a->kind != Containment) { /* syntax or semantics arrow */
	a->u.perm_list = (PermList *)NULL;
	if (ValFoundP("permissions")) {
	    /* proccess all "permissions" values, creating a PermList */
	    while (NextListEntryPtrOf(perms) != NULL) {
		/*
		 * parse the permission name */
		if (MatchNextListElement(perms,(Generic *)&(perm_name))) {
		    goto arrow_error;
		}
		/*
		 * verify that this permission name is valid */
		if ((t=FindTableId(pict->table,PermNameId,perm_name))==NULL) {
		    ParseErrorS(line_no,"unknown arrow permission '%s'",
				perm_name);
		    goto arrow_error;
		}
		/* allocate a new PermList cell and fill it in */
		perm_val = AllocOne(PermList);
		SpliceIntoList(a->u.perm_list,perm_val);
		perm_val->perm = t->name;
	    }
	}
    } else {			/* containment arrow */
	/* make sure the starred attribute was given */
	if (!ValFoundP("starred?")) {
	    ParseError(line_no,"missing required 'starred?' attribute");
	    goto arrow_error;
	}
    }
    /* convert box sysnames to Box pointers */
    a->from = FindBox(from,pict,EntryLineNumberOf(arrow_entry));
    a->to = FindBox(to,pict,EntryLineNumberOf(arrow_entry));
#ifdef OLD_DEBUG
    fprintf(stderr,"\nExtracted ARROW, sysname = %d\n",result->sysname);
#endif OLD_DEBUG
    return(result);

arrow_error:
    /* errors found; deallocate space and return error result */
    FreeArrowElt(result);
    return((ArrowElt *)NULL);
}

static void InsertArrow(a,pict)
  ArrowElt *a;
  Pict *pict;
/* If 'a' is non-NULL, insert the ArrowElt 'a' into the 'arrows' field of
   'pict'.
*/
{  
    ArrowList *a_cell;

    if (a != (ArrowElt *)NULL) {
	/*
	 * allocate an ArrowList cell and make it point to 'a' */
	a_cell = AllocOne(ArrowList);
	a_cell->elt = (Elt *)a;
	/*
	 * insert the arrow in the list */
	pict->elt_cnt++;
	SpliceIntoList(pict->arrows,a_cell);
    }
}

/* Inside Functions -------------------------------------------------------- */

#define DefaultThickness(_b1,_b2)\
 (((((_b1)->thickness)==Thick) && (((_b2)->thickness)==Thick)) ? Thick : Thin)

static ArrowList *ExtractInside(inside_entry,pict)
  Entry *inside_entry;
  Pict *pict;
{
    ArrowElt *a_elt;		/* new ArrowElt added to result list */
    Arrow *a;			/* a = a_elt->u.a */
    int from,to;		/* sysnames of child and parent boxes*/
    BoxElt *to_box;		/* BoxElt with sysname 'to' */
    ArrowList *temp;
    ArrowList *result = (ArrowList *)NULL;
    ListDesc children_list,*children;

    static int next_sysname = 0; /* virtual sysname of containment arrow */

    /* initialize so we don't have to dereference children_list everywhere */
    children = &children_list;

    /* install attribute designators */
    StartNewPNameList();
    AddPNameDesignator("parent",True,ConvertInt,(Generic *)&to,NULL_LIST);
    AddPNameDesignator("children",True,ConvertInt,(Generic *)NULL,children);

    /* extract data from the entry */
    if (MatchPNames(inside_entry)) { goto inside_error; }

    /* set 'to_box' to point to parent */
    to_box = FindBox(to,pict,EntryLineNumberOf(inside_entry));

    /* create a new arrow for each child box */
    while (NextListEntryPtrOf(children) != NULL) {
	/*
	 * insert new ArrowList at head of list */
	temp = AllocOne(ArrowList);
	a_elt = NewArrow();
	temp->elt = (Elt *)a_elt;
	SpliceIntoList(result,temp);
	/*
	 * fill in new Arrow */
	a_elt->sysname = --next_sysname;
	a = a_elt->u.a;
	a->kind = Containment;
	a->to = to_box;
	a->parity = Pos;
	/*
	 * get next child sysname */
	if (MatchNextListElement(children,(Generic *)&from)) {
	    goto inside_error;
	}
	/*
	 * fill in remainder of Arrow (parts depending on 'from' box */
	a->from = FindBox(from,pict,EntryLineNumberOf(inside_entry));
	a_elt->thickness = DefaultThickness(a->from,a->to);
	a->u.starred = a->from->u.b->starred;
	/*
	 * mark 'from' box as being contained */
	a->from->u.b->u2.contained = True;
#ifdef OLD_DEBUG
	fprintf(stderr,"\nExtracted INSIDE containment ARROW\n");
	fprintf(stderr,"  from = %d, to = %d\n",from,to);
#endif OLD_DEBUG
    }
    return(result);

inside_error:
    FreeArrowList(result);
    return((ArrowList *)NULL);
}

static void InsertArrowList(l,pict)
  ArrowList *l;
  Pict *pict;
/* Prepend the ArrowList 'l' to the 'arrows' field of 'pict'.
*/
{
    if (l != (ArrowList *)NULL) {
	ArrowList **a_ptr = &l;
	/* find end of list */
	while (*a_ptr) {
	    a_ptr = &((*a_ptr)->next);
	    pict->elt_cnt++;
	}
	/* prepend 'l' */
	*a_ptr = pict->arrows;
	pict->arrows = l;
    }
}

/* Editor Functions -------------------------------------------------------- */

static void ExtractEditor(ed_entry,pict)
  Entry *ed_entry;
  INOUT Pict *pict;
/* Process the 'range' and 'parity' property values of the EDITOR Entry
   'ed_entry'. This routine sets the 'range' field of 'pict' according to the
   'count-range' value. If 'count-range' is a single-valued list, then the
   upper bound gets the designator for infinity.
*/
{
    ListDesc range_list_struct,*range_list;

    /* initialize so we don't have to take address of range_list everywhere */
    range_list = &range_list_struct;

    /* initialize list of PNameDesignators */
    StartNewPNameList();
    AddPNameDesignator("count-range",False,ConvertInt,
		       (Generic *)NULL,range_list);

    /* extract property values */
    if (MatchPNames(ed_entry)) { return; }

    /* extract 'range' values */
    if (ValFoundPOf(DesigPtrOf(range_list))) {
	/* a 'range' was specified */
	if (NextListEntryPtrOf(range_list) == NULL) {
	    ParseError(EntryLineNumberOf(ed_entry),"empty 'count-range'");
	    return;
	}
	/* match low range value */
	if (MatchNextListElement(range_list,(Generic *)&(pict->range.low))) {
	    return;
	}
	if (NextListEntryPtrOf(range_list) == NULL) {
	    /* default if no upper limit given */
	    pict->range.high = INFINITY;
	} else {
	    /* match upper limit value */
	    if (MatchNextListElement(range_list,
				     (Generic *)&(pict->range.high))) {
		return;
	    }
	    /* make sure at most 2 elements in list */
	    if (NextListEntryPtrOf(range_list) != NULL) {
		ParseError(EntryLineNumberOf(ed_entry),
		  "'count-range' list must contain 1 or 2 integers only");
		return;
	    }
	}
	/* make sure low <= high */
	if (pict->range.low > pict->range.high) {
	    ParseError(EntryLineNumberOf(ed_entry),
		       "'count-range' is [low,high], where low > high");
	    return;
	}
    }
#ifdef OLD_DEBUG
    fprintf(stderr,"\nExtracted EDITOR entry\n  Count-Range = (%d,%d)\n",
	    pict->range.low,pict->range.high);
#endif OLD_DEBUG
}

/* Box Type Functions ------------------------------------------------------ */

static BoxType *ExtractBoxType(type_entry)
  Entry *type_entry;
{
    BoxType *result = NewBoxType(EntryLineNumberOf(type_entry));
    ListDesc attr_list,*attr_list_ptr;
    Attr *attr;			/* next converted attribute */
    AttrList *attr_cell;	/* list cell for converted attribute */
    String supertype;		/* name of supertype */

    /* initialize so we don't have to dereference 'attr_list' everywhere */
    attr_list_ptr = &attr_list;

    /* initialize list of PNameDesignators */
    StartNewPNameList();
    AddPNameDesignator("type-name",True,ConvertId,
		       (Generic *)&(result->name),NULL_LIST);
    AddPNameDesignator("supertype",False,ConvertId,
		       (Generic *)&supertype,NULL_LIST);
    AddPNameDesignator("attributes",False,ConvertAttribute,
		       (Generic *)NULL,attr_list_ptr);

    /* extract the data from the Entry */
    if (MatchPNames(type_entry)) { goto type_error; }

    /* copy 'supertype' string if supplied */
    if (ValFoundP("supertype")) { CopyString(result->u.supertype,supertype); }

    /* extract attribute information */
    if (ValFoundP("attributes")) {
	while (NextListEntryPtrOf(attr_list_ptr) != NULL) {
	    attr = AllocOne(Attr);
	    if (MatchNextListElement(attr_list_ptr,(Generic *)attr)) {
		Dealloc(attr);
	    } else {
		/*
		 * bind to the BoxType name */
		attr->boxtype_name = result->name;
		/*
		 * insert the new attribute at the head of the attr list */
		attr_cell = AllocOne(AttrList);
		attr_cell->attr = attr;
		SpliceIntoList(result->attr_head,attr_cell);
	    }
	}
    }
#ifdef OLD_DEBUG
    DisplayBoxType(result);
#endif OLD_DEBUG
    return(result);

type_error:
    Dealloc(result);
    return((BoxType *)NULL);
}

static void InsertAttributes(a_list,line_no,pict)
  AttrList *a_list;		/* list of attributes to install */
  int line_no;			/* error message line number */
  Pict *pict;
{
    Attr *attr;

    StepInitializedLinkedList(a_list) {
	attr = a_list->attr;
	if (FindTableId(pict->table,AttrNameId,attr->name) != NULL) {
	    ParseErrorS(line_no,"attribute name '%s' already used",attr->name);
	} else {
	    /* install the attribute in the hash table */
	    (void)AddTableId(pict->table,AttrNameId,attr->name,(Generic*)attr);
	}
    }
}

static void InsertType(bt,pict)
  BoxType *bt;
  INOUT Pict *pict;
{
    BoxTypeList *tl;

    if (bt != (BoxType *)NULL) {
	if (FindTableId(pict->table,BoxTypeId,bt->name) != NULL) {
	    ParseErrorS(bt->line_no,"box type '%s' already exists",bt->name);
	    Dealloc(bt);
	} else {
	    /*
	     * add the type name to the IdHashTable */
	    (void)AddTableId(pict->table,BoxTypeId,bt->name,(Generic *)bt);
	    /*
	     * add the BoxType to the front of the pict->box_types list */
	    tl = AllocOne(BoxTypeList);
	    tl->bt = bt;
	    SpliceIntoList(pict->box_types,tl);
	    /*
	     * add the attribute names assoc. w/ 'bt' to the IdHashTable */
	    InsertAttributes(bt->attr_head,bt->line_no,pict);
	}
    }
}

/* Permission Functions ---------------------------------------------------- */

static String ExtractPerm(perm_entry)
  Entry *perm_entry;
{
    String result;

    /* initialize list of PNameDesignators */
    StartNewPNameList();
    AddPNameDesignator("perm-name",True,ConvertId,
		       (Generic *)(&result),NULL_LIST);

    /* extract the data from the Entry */
    if (MatchPNames(perm_entry)) { result = (String)NULL; }
    return(result);
}

static void InsertPerm(name,line_no,pict)
  String name;
  int line_no;
  INOUT Pict *pict;
{
    static int x = -1;		/* dummy index value */

    if (FindTableId(pict->table,PermNameId,name) != NULL) {
	ParseErrorS(line_no,"permission '%s' already exists",name);
	Dealloc(name);
    } else {
	/* add the permission 'name' to the IdHashTable */
	(void)AddTableId(pict->table,PermNameId,name,(Generic *)(&x));
    }
}

/* Extract Other Entries --------------------------------------------------- */

static void ExtractOtherEntries(p_tree,pict)
  Entry *p_tree;
  INOUT Pict *pict;
/* Traverse the list of entries in 'p_tree', processing ARROW, INSIDE, EDITOR,
   and TYPE entries in the tree. Create Arrows, Ranges, and BoxType's as
   necessary, and update the 'arrows', 'range', and 'box_types' fields of
   'pict' as necessary.
*/
{
    Entry *curr_ent;

    StepLinkedList (curr_ent,p_tree) {
	switch (EntryTypeOf(curr_ent)) {
	  case ArrowEntry:
	    InsertArrow(ExtractArrow(curr_ent,pict),pict);
	    break;
	  case InsideEntry:
	    InsertArrowList(ExtractInside(curr_ent,pict),pict);
	    break;
	  case EditorEntry:
	    ExtractEditor(curr_ent,pict);
	    break;
	  case TypeEntry:
	    InsertType(ExtractBoxType(curr_ent),pict);
	    break;
	  case PermEntry:
	    InsertPerm(ExtractPerm(curr_ent),EntryLineNumberOf(curr_ent),pict);
	    break;
	}
    }
}

/* GLOBAL FUNCTIONS ======================================================== */

void ConfigureHashTable()
{
    /* install entries */
    AddEntryName("BOX",        BoxEntry);
    AddEntryName("ARROW",      ArrowEntry);
    AddEntryName("INSIDE",     InsideEntry);
    AddEntryName("EDITOR",     EditorEntry);
    AddEntryName("BOXTYPE",    TypeEntry);
    AddEntryName("PERMISSION", PermEntry);

    /* install BOX/ARROW property names */
    AddPropName("sysname",   ObjectPNames,  IntPValType);
    AddPropName("thickness", ObjectPNames,  IdPValType);
    AddPropName("starred?",  ObjectPNames,  IdPValType);

    /* install BOX property names */
    AddPropName("name",      BoxPName,      StringPValType);
    AddPropName("type",      BoxPName,      IdPValType);

    /* install ARROW property names */
    AddPropName("kind",      ArrowPName,    IdPValType);
    AddPropName("from",      ArrowPName,    IntPValType);
    AddPropName("to",        ArrowPName,    IntPValType);
    AddPropName("parity",    ArrowPName,    IdPValType);
    AddPropName("permissions",ArrowPName,   IdListPValType);

    /* install INSIDE property names */
    AddPropName("children",  InsidePName,   IntListPValType);
    AddPropName("parent",    InsidePName,   IntPValType);

    /* install EDITOR property names */
    AddPropName("count-range",EditorPName,  IntListPValType);

    /* install TYPE property names */
    AddPropName("type-name", TypePName,     IdPValType);
    AddPropName("supertype", TypePName,     IdPValType);
    AddPropName("attributes",TypePName,     AllPValTypes);

    /* install PERMISSION property names */
    AddPropName("perm-name", PermPName,     IdPValType);
}

Boolean Extract(p_tree,pict)
  Entry *p_tree;
  INOUT Pict *pict;
{
    InitExtract(MAX_PNAMES);
    ExtractBoxEntries(p_tree,pict);
    ExtractOtherEntries(p_tree,pict);
    return(MakeBoolean(parse_error_cnt > 0));
}
