/* HINTS.C

   Module for reading and storing cpc(1) hint files.

   $Header: hints.c,v 1.5 91/12/26 22:11:11 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "error.h"

#include "top.g"
#include "rank.h"
#include "attr.h"
#include "hints.h"
#include "id-table.h"
#include "objs.h"

/* MACRO DEFINITIONS ======================================================= */

/* maximum length of lines in the hint file + 1 */
#define MAX_LINE_LEN 101

/* Pos and Neg parity indices */
#define POS ((int)Pos)
#define NEG ((int)Neg)

/* LOCAL FUNCTIONS ========================================================= */

#define WhiteSpace(_c)    ((_c)==' ' || (_c)=='\t')
#define NonWhiteSpace(_c) ((_c)!=' ' && (_c)!='\t' && (_c)!='\0')
#define NonQuote(_c)      ((_c)!='"' && (_c)!='\0')

static Boolean GetNextLine(fp,keyword,args,line_no)
  FILE *fp;
  OUT String *keyword;
  OUT String *args;
  INOUT int *line_no;
/* Read the next line of the file 'fp', skipping blank lines and comment lines
   (lines beginning with '#'). 'keyword' is set to point to the first "word"
   on the line; it is terminated. 'args' is set to point two characters after
   the last character of 'keyword'. Both of these strings are simply pointers
   into a static buffer, so they are temporary, and should be copied if they
   will be needed after the next call to this function. The 'line_no' value is
   incremented for each line read (including skipped blank and comment lines).

   Returns True iff there are no more lines to read.
*/
{
    static char buff[MAX_LINE_LEN];
    String temp;
    char terminator;
    int last_index;		/* index of last char in buff[] */

    /* read next line into buffer, skipping blank and comment lines */
    while (True) {
	(*line_no)++;
	if (fgets(buff,MAX_LINE_LEN,fp) == NULL) return(True);
	if (*buff != '#' && *buff != '\n') break;
    }

    /* change trailing '\n' preserved by fgets() to '\0' */
    last_index = strlen(buff) - 1;
    if (buff[last_index] == '\n') {
	buff[last_index] = '\0';
    } else {
	fprintf(stderr,"%s: FATAL ERROR -- hint file line %d too long\n",
		argv0,*line_no);
	exit(-1);
    }

    /* initialize '*keyword' */
    temp = buff;
    while (WhiteSpace(*temp)) temp++;    /* find start of keyword */
    *keyword = temp;
    while (NonWhiteSpace(*temp)) temp++; /* find end of keyword */
    terminator = *temp;		         /* save original char at end */
    *temp = '\0';		         /* terminate keyword */

    /* initialize '*args' */
    if (terminator != '\0') temp++;
    *args = temp;
    return(False);
}

static Boolean ParseArgs(args,max_words,words,word_cnt)
  String args;			/* argument string to parse */
  int max_words;		/* maximum # of words to parse */
  OUT String words[];		/* words resulting from parse */
  OUT int *word_cnt;		/* number of words parsed */
/* Parses the arguments in 'args', setting 'word_cnt' to the number of
   arguments found, and setting 'words[]' to point to these arguments.
   'words[]' is assumed to be an array containing space for at least
   'max_words' entries. The 'words[]' pointers point into the 'args' array
   only; copies of the strings are not made. An "argument" is a consecutive
   block of non-whitespace characters OR the set of characters inside a pair
   of double quotes.

   Returns True iff there are more than 'max_words' arguments in 'args'.
*/
{
    int terminator;

    *word_cnt = 0;
    while (*args) {
	/*
	 * skip whitespace to next arg */
	while (WhiteSpace(*args)) args++;
	/*
	 * check that we are not already at end or about to overrun limit*/
	if (!(*args)) break;	/* stop if end of string */
	if (*word_cnt >= max_words) return(True);
	/*
	 * set pointer to this word */
	if (*args == '"') {
	    /* string */
	    args++;		/* skip over initial double-quote */
	    words[(*word_cnt)++] = args;
	    while (NonQuote(*args)) args++;
	} else {
	    /* non-string */
	    words[(*word_cnt)++] = args;
	    while (NonWhiteSpace(*args)) args++;
	}
	/*
	 * terminate word, advancing if not already end of line */
	terminator = *args;
	*args = '\0';
	if (terminator) args++;
    }
    return(False);
}

static Boolean ParsePerms(args,line_no,pict,perm_cnt)
  String args;			/* argument string */
  int line_no;			/* line number to report error on */
  Pict *pict;
  OUT int *perm_cnt;		/* number of permission names set */
/* Parses the string 'args' into names of permissions, setting 'perm_cnt' to
   the number of permissions parsed. Each permission is looked up in the Id
   Hash Table 'pict->table', and it's 'u.index' field is set to the index of
   the permission name (starting with 0). If the permission is not found in
   the table, an error is reported as occurring on line number 'line_no'.

   Returns True iff 'args' names some permission not found in the Id Hash
   Table 'pict->table', i.e., an error was reported.
*/
{
    int i;
    int error_cnt = 0;
    TableEntry *tbl;
    String perms[MAX_LINE_LEN/2];

    /* parse arguments and set '*perm_cnt' to the number found */
    if (ParseArgs(args,MAX_LINE_LEN/2,perms,perm_cnt)) {
	ParseError(line_no,"too many PERM arguments");
	return(True);
    }

    /* set the index of each permission */
    StepIndex(i,0,*perm_cnt) {
	if ((tbl=FindTableId(pict->table,PermNameId,perms[i])) == NULL) {
	    ParseErrorS(line_no,"no permission named '%s'",perms[i]);
	    error_cnt++;
	} else {
	    tbl->u.index = i;	/* set index for this permission */
	}
    }

    /* create and install a dummy permission named "AllPerms" for totals */
    (void)AddTableId(pict->table,PermNameId,"AllPerms",(Generic *)perm_cnt);
    return(MakeBoolean(error_cnt > 0));
}

static Boolean ParseArrow(args,line_no,perm_cnt,pict)
  String args;
  int line_no;
  int perm_cnt;
  Pict *pict;
/* Parses the ARROW arguments 'args'. The first two arguments tell the type of
   arrow and the anchor index. There should be 'perm_cnt' remaining floating
   point arguments. This routine allocates an array of 'perm_cnt' floats,
   fills that array in with the floating arguments, and assigns the array to
   the appropriate 'pict->ranks[][]' entry. If the entry is non-NULL, or if
   there are other errors parsing the arguments, and error is reported as
   occurring on line number 'line_no'.

   Returns True iff an error was encountered.
*/
{
    int type_index;		/* kind of arrow (0, 1, 2, or 3) */
    int anchor_cnt;		/* number of anchors (0, 1, 2, or 3 (both)) */
    int parity;			/* POS or NEG */
    int arg_cnt;		/* number of actual arguments */
    int i;
    float *values,value_total=0.0;
    String *words;

    /* parse line */
    words = AllocArray(String,perm_cnt+3);
    if (ParseArgs(args,perm_cnt+3,words,&arg_cnt)) {
	ParseError(line_no,"too many permission values");
	goto arrow_error;
    }

    /* determine type_index */
    if (SameString(words[0],"syn")) type_index = 0;
    else if (SameString(words[0],"sem")) type_index = 1;
    else if (SameString(words[0],"con")) { type_index = 2; perm_cnt = 2; }
    else if (SameString(words[0],"con*")) { type_index = 3; perm_cnt = 2; }
    else {
	ParseErrorS(line_no,"unknown arrow type '%s'",words[0]);
	goto arrow_error;
    }
    if (arg_cnt-3 != perm_cnt) {
	ParseError(line_no,"incorrect number of permission values");
	goto arrow_error;
    }

    /* determine anchor_cnt: 'T'=>1, 'H'=>2, 0=>0, 2=>3 */
    switch (*words[1]) {
      case '0': anchor_cnt = 0; break;
      case 'T': anchor_cnt = 1; break;
      case 'H': anchor_cnt = 2; break;
      case '2': anchor_cnt = 3; break;
      default:
	ParseError(line_no,"anchor count is not 'T', 'H', 0, or 2");
	goto arrow_error;
    }

    /* determine parity: '+'=>POS, '-'=>NEG */
    switch (*words[2]) {
      case '+': parity = POS; break;
      case '-': parity = NEG; break;
      default:
	ParseError(line_no,"parity is not '+' or '-'");
	goto arrow_error;
    }

    /* parse floating point values */
    values = AllocArray(float,perm_cnt+1);
    StepIndex(i,0,perm_cnt) {
	if (sscanf(words[i+3],"%f",values+i) != 1) {
	    ParseErrorI(line_no,"value %d is not a floating point number",i);
	    goto arrow_error;
	}
	value_total += values[i];
    }
    values[perm_cnt] = value_total;

    /* assign floating point numbers to appropriate array entry */
    if (pict->ranks[type_index][anchor_cnt][parity] != NULL) {
	ParseError(line_no,"repeated arrow-type/anchor-cnt/parity index");
	goto arrow_error;
    }
    pict->ranks[type_index][anchor_cnt][parity] = values;

    /* clean up and return okay */
    Dealloc(words);
    return(False);

arrow_error:
    Dealloc(words);
    return(True);
}

static Boolean ParseBoxType(args,line_no,pict)
  String args;
  int line_no;
  Pict *pict;
{
    String *words;
    int arg_cnt;
    TableEntry *tbl;
    float value;

    /* parse line */
    words = AllocArray(String,2);
    if (ParseArgs(args,2,words,&arg_cnt) || arg_cnt != 2) {
	ParseError(line_no,"incorrect number of BOXTYPE arguments");
	goto box_type_error;
    }

    /* find BoxType named 'words[0]' */
    if ((tbl=FindTableId(pict->table,BoxTypeId,words[0])) == NULL) {
	ParseErrorS(line_no,"no box type named '%s' exists",words[0]);
	goto box_type_error;
    }
    if (tbl->u.box_type->sum_rank >= 0.0) {
	ParseErrorS(line_no,"box type '%s' has already been set",words[0]);
	goto box_type_error;
    }
    if (sscanf(words[1],"%f",&value) != 1 || value < 0.0 || value > 1.0) {
	ParseErrorS(line_no,"illegal floating point value '%s'",words[1]);
	goto box_type_error;
    }
    tbl->u.box_type->sum_rank = value;
    Dealloc(words);
    return(False);

box_type_error:
    Dealloc(words);
    return(True);	
}

static Boolean ParseAttribute(args,line_no,pict)
  String args;
  int line_no;
  Pict *pict;
{
    String *words;
    int arg_cnt;
    TableEntry *tbl;
    Attr *attr;

    /* parse line */
    words = AllocArray(String,4);
    if (ParseArgs(args,4,words,&arg_cnt) || arg_cnt < 3) {
	goto attr_arg_cnt_error;
    }

    /* find Attr named 'words[0]' */
    if ((tbl=FindTableId(pict->table,AttrNameId,words[0])) == NULL) {
	ParseErrorS(line_no,"no attribute named '%s' exists",words[0]);
	goto attr_error;
    }

    /* process the other arguments based on the type of the attribute */
    attr = tbl->u.attr;
    switch (attr->type.prim) {
      case IntVal:
	if (arg_cnt != 4) goto attr_arg_cnt_error;
	if (sscanf(words[1],"%d",&(attr->min.i_val)) != 1
	    || sscanf(words[2],"%d",&(attr->max.i_val)) != 1
	    || sscanf(words[3],"%d",&(attr->range_size)) != 1) {
	    ParseError(line_no,"illegal non-integer argument");
	    goto attr_error;
	}
	attr->dist = AttrValDistance(attr->min,attr->max,IntKind);
	break;
      case IdVal:
      case StringVal:
	if (arg_cnt != 4) goto attr_arg_cnt_error;
	CopyString(attr->min.s_val,words[1]);
	CopyString(attr->max.s_val,words[2]);
	if (sscanf(words[3],"%d",&(attr->range_size)) != 1) {
	    ParseErrorS(line_no,"illegal 'range_size' argument '%s'",words[3]);
	    goto attr_error;
	}
	attr->dist = AttrValDistance(attr->min,attr->max,StringKind);
	break;
      case BoolVal:
	if (arg_cnt != 3) goto attr_arg_cnt_error;
	if (sscanf(words[1],"%f",&(attr->min.f_val)) != 1
	    || sscanf(words[2],"%f",&(attr->max.f_val)) != 1) {
	    ParseError(line_no,"illegal non-floating-point argument");
	    goto attr_error;
	}
	break;
      case BoxTypeVal:
	ParseError(line_no,"ATTR not applicable to box-type attributes");
	goto attr_error;
    }
    Dealloc(words);
    return(False);

attr_arg_cnt_error:
    ParseError(line_no,"incorrect number of ATTR arguments");
attr_error:
    Dealloc(words);
    return(True);
}

static Boolean ParseHds(args,line_no,pict)
  String args;
  int line_no;
  Pict *pict;
/* Parses the string 'args' to produce an HdsHint structure. The new structure
   is stored in the appropriate 'pict->attrs[]' list of HdsHints. An HDS hint
   line takes the form:

       HDS element attr-name hds-type "arg-string" ["bst-arg-string"]

   where 'element' is "box", "syn", "subj", "obj", or "con", 'attr-name' is
   the name of the attribute (which is looked up in the ID Hash Table
   'pict->table'), 'hds-type' is the name of the hds to construct,
   'arg-string' is the argument to pass in order to construct that HDS, and
   'bst-arg-string' is the argument to pass in the event that the HDS is
   upgraded to a BST-HDS by the compiler.

   Returns True iff errors were encountered processing this hint line.
*/
{
    int arg_cnt;		/* number of arguments */
    int element;		/* index for this element type */
    TableEntry *tbl;		/* hash table entry for attribute */
    HdsKind kind;		/* kind of HDS */
    HdsHint *hds_hint;		/* newly created HdsHint */
    HdsHintList *cell,**curr;
    String hds_args[MAX_LINE_LEN/2];

    /* parse arguments and set 'arg_cnt' to the number found */
    if (ParseArgs(args,MAX_LINE_LEN/2,hds_args,&arg_cnt)) {
	ParseError(line_no,"too many HDS arguments");
	return(True);
    }

    /* make sure we have a valid number of arguments */
    if (arg_cnt < 3 || arg_cnt > 5) {
	ParseError(line_no,"incorrect number of HDS arguments");
	return(True);
    }

    /* determine 'element' value */
         if (SameString(hds_args[0],"box"))  { element = BOX; }
    else if (SameString(hds_args[0],"syn"))  { element = SYN; }
    else if (SameString(hds_args[0],"con"))  { element = CON; }
    else if (SameString(hds_args[0],"subj")) { element = SUBJ; }
    else if (SameString(hds_args[0],"obj"))  { element = OBJ; }
    else {
	ParseErrorS(line_no,"illegal element type '%s'",hds_args[0]);
	return(True);
    }

    /* locate corresponding attribute or install a new one if necessary */
    if ((tbl=FindTableId(pict->table,AttrNameId,hds_args[1])) == NULL) {
	Attr *attr = AllocOne(Attr);
	attr->type.prim = IdVal;
	attr->required = True;
	tbl = AddTableId(pict->table,AttrNameId,hds_args[1],(Generic *)attr);
    }

    /* determine kind of hds */
    if (SameString(hds_args[2],"bst"))      { kind = BstHds; }
    else if (SameString(hds_args[2],"xht")) { kind = XhtHds; }
    else if (SameString(hds_args[2],"ht"))  { kind = HtHds; }
    else {
	ParseErrorS(line_no,"unknown discriminator type '%s'",hds_args[2]);
	return(True);
    }

    /* create new HdsHint and cell */
    cell = AllocOne(HdsHintList);
    cell->next = NULL;
    cell->hds_hint = hds_hint = AllocOne(HdsHint);
    hds_hint->name = tbl->u.attr->name;
    hds_hint->type = tbl->u.attr->type.prim;
    hds_hint->kind = kind;
    if (arg_cnt >= 4) {	CopyString(hds_hint->arg,hds_args[3]); }
    else { hds_hint->arg = ""; }
    if (arg_cnt == 5) {	CopyString(hds_hint->bst_arg,hds_args[4]); }
    else { hds_hint->bst_arg = ""; }

    /* install the new 'cell' at the *end* of the proper list */
    StepPtrLinkedList(curr,pict->attrs[element]) /* empty loop */ ;
    *curr = cell;
    return(False);
}

/* GLOBAL FUNCTIONS ======================================================== */

Boolean ProcessHintFile(fp,name,pict)
  FILE *fp;
  String name;
  Pict *pict;
{
    int error_cnt = 0;
    int line_no = 0;
    String keyword,args;
    int perm_cnt = 0;

    while (!GetNextLine(fp,&keyword,&args,&line_no)) {
	if (SameString(keyword,"PERMS")) {
	    if (ParsePerms(args,line_no,pict,&perm_cnt)) error_cnt++;
	} else if (SameString(keyword,"ARROW")) {
	    if (perm_cnt <= 0) {
	       ParseError(line_no,"a PERMS line must proceed each ARROW line");
	       error_cnt++;
	    } else if (ParseArrow(args,line_no,perm_cnt,pict)) error_cnt++;
	} else if (SameString(keyword,"BOXTYPE")) {
	    if (ParseBoxType(args,line_no,pict)) error_cnt++;
	} else if (SameString(keyword,"ATTR")) {
	    if (ParseAttribute(args,line_no,pict)) error_cnt++;
	} else if (SameString(keyword,"DT")) {
	    if (ParseHds(args,line_no,pict)) error_cnt++;
	} else {
	    ParseErrorS(line_no,"unknown hint file command '%s'",keyword);
	    error_cnt++;
	}
    }

    if (error_cnt) {
	fprintf(stderr,"%s: %d error(s) processing hint file '%s'\n",
		argv0,error_cnt,name);
	return(True);
    }
    return(False);
}
