/* ID-TABLE.H

   Type definitions and function declarations for the id-table.c module, which
   provides facilities for creating and managing a hash table of identifiers.

   $Header: id-table.h,v 1.1 91/08/25 16:42:13 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#ifndef ID_TABLE_H
#define ID_TABLE_H

#include <my-types.h>

/* Attr in TableEntry type */
#include "attr.h"

/* BoxType in TableEntry type */
#include "box-type.h"

/* Var in TableEntry type */
#include "var.h"

/* TYPE DEFINITIONS ======================================================== */

typedef enum {
    BoxTypeId, AttrNameId, PermNameId, VarNameId
} IdKind;

typedef struct table_entry {
    String name;		/* ID that is the key for this entry */
    IdKind kind;		/* kind of identifier in the hash table */
    union {
	BoxType *box_type;	/* kind == BoxTypeId */
	Attr *attr;		/* kind == AttrNameId */
	int index;		/* kind == PermNameId -- index into hints */
	Var *var;		/* kind == VarNameId */
    } u;
} TableEntry;

typedef struct table_entry_list {
    struct table_entry_list *next;
    TableEntry *entry;
} TableEntryList;

typedef TableEntryList **IdHashTable;

/* DEBUGGING FUNCTION DECLARATIONS ========================================= */

#ifdef DEBUG

void DisplayIdHashTable( /* IdHashTable t */ );
/* Print the contents of the non-empty buckets of 't' to stderr. For each
   non-empty bucket, the index of the bucket followed by the names of the
   types contained in that bucket are printed.
*/

#endif DEBUG

/* FUNCTION DECLARATIONS =================================================== */

IdHashTable NewIdHashTable( /* void */ );
/* Create and return a new, initialized IdHashTable.
*/

TableEntry *FindTableId( /* IdHashTable t, IdKind kind, String name */ );
/* Return a pointer to the TableEntry in the hash table 't' under the key
   'name' with of type 'kind', or NULL if there is no such TableEntry.
*/

TableEntry *AddTableId( /* INOUT IdHashTable t, IdKind kind, String name,
			   Generic *val */ );
/* Add the value 'val' to the IdHashTable 't' under the identifier 'name' as
   having IdKind 'kind'. 'Val' will be caste according to 'kind'. It should be
   a (BoxType *) if kind == BoxTypeId, a (Attr *) if kind == AttrNameID, or a
   (Var *) if kind == VarNameId.

   NOTE: A *copy* of 'name' is made and stored in the TableEntry. Moreover, if
   'kind' is BoxTypeId, AttrNameId, or VarNameId, then the 'name' field of the
   'val' structure is made to point to the new copy.

   If a TableEntry with the same name already exists in the hash table, this
   one will still be inserted anyhow, and results of future FindTableId()
   operations on that name are undefined. The new TableEntry is returned.
*/

#endif ID_TABLE_H
