/* INTERVAL.C

   Module for computing and manipulating value intervals.

   $Header: interval.c,v 1.5 91/11/13 03:12:58 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "top.g"
#include "parse-pred.h"
#include "interval.h"
#include "id-table.h"
#include "objs.h"

/* LOCAL VARIABLES ========================================================= */

/* Bogus BoxType to indicate failure in box type interval intersection */
static BoxType NoType;

/* LOCAL DEBUGGING FUNCTIONS =============================================== */

#ifdef DEBUG

void DisplayBoxIntrvls(b_elt)
  BoxElt *b_elt;
/* Display each of the intervals for the box 'b_elt' to stderr.
*/
{
    IntrvlList *curr;

    fprintf(stderr,"\nIntervals for BOX %d:\n",b_elt->sysname);
    StepLinkedList(curr,b_elt->u.b->u1.intvls) {
	fprintf(stderr,"  %10s: ",curr->i->name);
	DisplayIntrvl(curr->i);
	fputc('\n',stderr);
    }
}
#endif DEBUG

/* LOCAL FUNCTIONS ========================================================= */

typedef enum { MinSelect=0x0, MaxSelect=0x1 } MaxMinSelect;

static void FreeValue(v,kind)
  Value *v;
  IntrvlKind kind;
/* Deallocates the space used by 'v' appearing in an interval with kind
   'kind'. The 'kind' value is used to free the actual space used by the value
   if necessary.
*/
{
    Value *curr;

    while (v != (Value *)NULL) {
	curr = v;
	v = v->next;
	switch (kind) {
	  case StringKind:
	    Dealloc(curr->u.string);
	    break;
	  case VarKind:
	    if (curr->u.v_bnd->prefix) Dealloc(curr->u.v_bnd->prefix);
	    if (curr->u.v_bnd->suffix) Dealloc(curr->u.v_bnd->suffix);
	    Dealloc(curr->u.v_bnd);
	    break;
	} /* switch() */
	Dealloc(curr);
    } /* while() */
}

/* ----------------------------------------------------------------------------
 *
 * NOTE: All of the GreaterEqType(v1,v2) functions are designed to return True
 * iff 'v1' >= 'v2', according to some definition of ">=". The MaxMin() routine
 * uses the result to decide which value is the "winner" according to the
 * selection criterion (max or min) by XOR'ing the selection criterion with
 * the GreaterType() result.
 *
 * ----------------------------------------------------------------------------
*/

#ifdef MACRO_FUNCTION

static Boolean GreaterEqInt(v1,v2)
  Value *v1,*v2;
/* Returns True iff 'v1' >= 'v2'. This uses the invariant that integer ranges
   are always in the canonical form having kind == EqComp.
*/
#endif MACRO_FUNCTION

#define GreaterEqInt(_v1,_v2) \
  (MakeBoolean((_v1)->u.int_val >= (_v2)->u.int_val))

static Boolean GreaterEqString(v1,v2,select)
  Value *v1,*v2;
  MaxMinSelect select;
/* Returns True iff 'v1->u.string' >= 'v2->u.string'. In the case where the
   strings are the same, we need the selection criteria 'select' to decide the
   result. For example, if the comparison of 'v1' is strict but that of 'v2'
   is not, then if 'select' is MinSelect, we return True, but if 'select' is
   MaxSelect, we return False. In the case where the strings are equal and
   both comparisons are the same (either both strict or neither strict), then
   the result returned is arbitrary.
*/
{
    int comp;
    Boolean result;
    if ((comp=strcmp(v1->u.string,v2->u.string)) < 0) {
	return(False);
    } else if (comp > 0) {
	return(True);
    } else {
	switch (select) {
	  case MinSelect: result = MakeBoolean(v2->kind == EqComp); break;
	  case MaxSelect: result = MakeBoolean(v1->kind == EqComp); break;
	}
	return(result);
    }
}

#ifdef MACRO_FUNCTION

static Boolean GreaterEqBool(v1,v2)
  Value *v1,*v2;
/* Returns True iff 'v1->u.bool' is True, i.e., if 'v1->u.bool' >=
   'v2->u.bool', where True > False.
*/
#endif MACRO_FUNCTION

#define GreaterEqBool(_v1,_v2)\
    (MakeBoolean((_v1)->u.bool || !(_v2)->u.bool))

static Boolean Ancestor(t1,t2)
  BoxType *t1,*t2;
/* Returns True iff 't1' is a (not necessarily proper) ancestor of 't2' in the
   type-tree.

   Pre-Condition: the 'num' and 'top_range' field of all BoxType's must have
   been filled in by a call to the routine compile.InitBoxTypeValues().

   IMPLEMENTATION NOTE: This algorithm implements a O(1)-time ancestorship
   test using the invariant established by the call to the routine
   compile.BoxTypePreorderTraverse().
*/
{
    return(MakeBoolean((t1->num <= t2->num) && (t2->num <= t1->top_range)));
}

static BoxType *LeastCommonAncestor(t1,t2)
  BoxType *t1,*t2;
/* Returns a pointer to the BoxType representing the least common ancestor of
   't1' and 't2' in the type tree. This value is always well-defined, since
   every type is a subtype of the 'entity' built-in root type.

   IMPLEMENTATION: We use an O(d * Anc)-time search, where 'd' is the max of
   the depths of 't1' and 't2' and 'Anc' is the time it takes to perform the
   ancestor test. The algorithm works by starting at 't1', working up to the
   root from 't1'. At each step, we ask if the current node is an ancestor of
   't2'. The first such node is the LCA.
*/
{
    for (; t1 != (BoxType *)NULL; t1 = t1->u.parent) {
	if (Ancestor(t1,t2)) break;
    }
    return(t1);
}

static Boolean GreaterEqBoxType(v1,v2,select)
  INOUT Value *v1;
  Value *v2;
  MaxMinSelect select;
/* If 'v1' and 'v2' are the same type, this routine uses the strictness of
   'v1' and 'v2' and the selection criterion 'select' to decide the winner,
   just like GreaterEqString().

   Otherwise, depending on whether 'select' is MaxSelect or MinSelect, this
   routine does one of two different things. If it is MaxSelect, then the
   "winner" should be the least common ancestor (in the box-type tree) of 'v1'
   and 'v2'. It therefore computes the LCA, stores it in 'v1', and returns a
   value such that 'v1' is guaranteed to be the winner in the routine MaxMin().

   If 'select' is MinSelect, the "winner" should be the descendant in the
   box-type tree. For example, if 'v1' is an ancestor of 'v2', then False is
   returned (since v1 > v2). However, if neither value is the ancestor of the
   other in the type tree, then the bogus Value 'NoType' is stored in 'v1',
   and a value is returned such that 'v1' is guaranteed to be the winner in
   the routine MaxMin().
*/
{
    Boolean result;
    if (v1->u.box_type == v2->u.box_type) {
	switch (select) {
	  case MinSelect: return(MakeBoolean(v2->kind == EqComp));
	  case MaxSelect: return(MakeBoolean(v1->kind == EqComp));
	}
    }
    switch (select) {
      case MaxSelect:
	v1->u.box_type = LeastCommonAncestor(v1->u.box_type,v2->u.box_type);
	result = MakeBoolean(select); /* guarantee 'v1' will be the winner */
	break;
      case MinSelect:
	if (Ancestor(v1->u.box_type,v2->u.box_type)) {
	    result = True;	/* 'v1' > 'v2' */
	} else if (Ancestor(v2->u.box_type,v1->u.box_type)) {
	    result = False;	/* 'v1' < 'v2' */
	} else {
	    v1->kind = EqComp;
	    v1->u.box_type = &NoType;
	    result = MakeBoolean(select); /* guarantee 'v1' the winner */
	}
	break;
    }
    return(result);
}

static Boolean GreaterEqVar(v1,v2,select)
  Value *v1,**v2;
  MaxMinSelect select;
/* The variables '*v1' and '*v2' are assumed to be non-NULL linked lists of
   Value's. The "union" of the lists is formed in *v1, and a result is returned
   so 'v1' is always considered to be the winner. '*v2' is set to NULL.
*/
{
    Value *end1;
    for (end1=v1; end1->next != NULL; end1 = end1->next) /* empty */ ;
    end1->next = *v2;
    *v2 = (Value *)NULL;
    return(MakeBoolean(select)); /* return value so 'v1' is the winner */
}

static Value *MaxMin(select,kind,v1,v2)
  MaxMinSelect select;
  IntrvlKind kind;
  Value *v1,*v2;
/* Returns either 'v1' or 'v2', depending on which one is maximal (if 'select'
   == MaxSelect) or minimal (if 'select' == MinSelect). Both 'v1' and 'v2' are
   Value's of IntrvlKind 'kind'. Either value may be NULL, but not both. In
   this case, the non-NULL value is always returned. If both values are
   non-NULL, then whichever value is *not* returned is deallocated.

   Note: Usually, MaxMin() is called to compare the low (or high) values of
   two different IntrvlRange structures. This is done twice in
   IntersectIntrvls() below to find the intersection of two intervals.
*/
{
    Boolean v1_bigger;

    if (v1 == (Value *)NULL) return(v2);
    if (v2 == (Value *)NULL) return(v1);
    switch (kind) {
      case IntKind:     v1_bigger = GreaterEqInt(v1,v2); break;
      case StringKind:  v1_bigger = GreaterEqString(v1,v2,select); break;
      case BoolKind:    v1_bigger = GreaterEqBool(v1,v2); break;
      case BoxTypeKind: v1_bigger = GreaterEqBoxType(v1,v2,select); break;
      case VarKind:     v1_bigger = GreaterEqVar(v1,&v2,select); break;
    }
    if ((int)select ^ (int)v1_bigger) {
	FreeValue(v1,kind); return(v2);
    } else {
	FreeValue(v2,kind); return(v1);
    }
}

/* ----------------------------------------------------------------------------
 *
 * NOTE: In general, the EmptyTypeRange(r) functions return True iff the range
 * 'r' denotes an empty range for the type of the values. There are different
 * notions of an empty range for each type.
 *
 * ----------------------------------------------------------------------------
*/

#ifdef MACRO_FUNCTION

static Boolean EmptyIntRange(r)
  IntrvlRange *r;
/* Returns True iff the integer range 'r' is empty. This routine assumes the
   range is in integer canonical form.
*/
#endif MACRO_FUNCTION

#define EmptyIntRange(_r) \
    MakeBoolean((_r)->low->u.int_val > (_r)->high->u.int_val)

static Boolean EmptyStringRange(r)
  IntrvlRange *r;
{
    int comp = strcmp(r->low->u.string,r->high->u.string);
    if (comp < 0) return(False);
    else if (comp > 0) return(True);
    else return(MakeBoolean(r->low->kind==StrictComp ||
			    r->high->kind==StrictComp));
}

#ifdef MACRO_FUNCTION

static Boolean EmptyBoolRange(r)
  IntrvlRange *r;
/* Returns True iff the Boolean range 'r' is empty, i.e., if 'r->low->u.bool'
   is True and 'r->high->u.bool' is False.
*/
#endif MACRO_FUNCTION

#define EmptyBoolRange(_r) \
   MakeBoolean((_r)->low->u.bool && !((_r)->high->u.bool))

static Boolean EmptyBoxTypeRange(r)
  IntrvlRange *r;
/* This routine assumes 'r' has a lower value in canonical form, and makes the
   upper value have canonical form as well.

   Returns True if the "high" Value of 'r' is the error-indicating value
   'NoType' or if the "high" value is not an ancestor of the "low" value (or
   if they are equal and the upper comparison is strict); False otherwise.

   Pre-condition: r->low != NULL && r->high != NULL.
*/
{
    BoxType *curr;

    /* immediately eliminate case where 'high' type is undefined */
    if (r->high->u.box_type == &NoType) return(True);

    /* put type in canonical form */
    if (r->high->kind == StrictComp) {
	for (curr=r->low->u.box_type; curr; curr = curr->u.parent) {
	    if (curr->u.parent == r->high->u.box_type) break;
	}
	if (curr != (BoxType *)NULL) {
	    r->high->u.box_type = curr;
	    r->high->kind = EqComp;
	}
    }

    /* decide if interval is empty or not */
    if (r->low->u.box_type == r->high->u.box_type) {
	/* equal types; make decision based on strictness of upper value */
	return(MakeBoolean(r->high->kind == StrictComp));
    } else {
	return(NotOf(Ancestor(r->high->u.box_type,r->low->u.box_type)));
    }
}

static Boolean EmptyVarRange(r)
  IntrvlRange *r;
/* Returns True iff r's 'low' and 'high' Value lists contain a duplicate
   element and the comparison is strict in at least one of them. That is the
   only way we can tell at compile time that a variable interval is empty.

   NOTE: This function uses that fact that attribute names in variable Values
   are pointers into the ID Hash Table, so pointer equality works as name
   equality.

   PRE-CONDITION: This routine assumes that no attribute is duplicated in
   either list (of course, there may be an attribute in both lists).
*/
{
    Value *low,*high;

    /* loop over all (low,high) pairs */
    StepLinkedList(low,r->low) {
	StepLinkedList(high,r->high) {
	    if (low->u.v_bnd->attr_name == high->u.v_bnd->attr_name &&
		(low->kind == StrictComp || high->kind == StrictComp)) {
		return(True);
	    }
	}
    }
    return(False);
}

static void IntersectIntrvls(i1,i2)
  Intrvl *i1,*i2;
/* Compute the intersection of 'i1' and 'i2', storing the result in 'i1'.
   Also, the space occupied by 'i2' is freed.
*/
{
    IntrvlRange *r1=i1->range, *r2=i2->range;

    r1->low = MaxMin(MaxSelect,i1->kind,r1->low,r2->low);
    r1->high = MaxMin(MinSelect,i1->kind,r1->high,r2->high);
    FreeIntrvl(i2,False);
}

static IntrvlKind PrimValueTypeToIntrvlKind(val_type)
  PrimValueType val_type;
{
    IntrvlKind result;
    switch (val_type) {
      case IntVal:                result = IntKind; break;
      case IdVal: case StringVal: result = StringKind; break;
      case BoolVal:               result = BoolKind; break;
      case BoxTypeVal:            result = BoxTypeKind; break;
    }
    return(result);
}

static Boolean CopyValue(kind,val,pred,sysname,pict)
  IntrvlKind kind;
  Value *val;
  SimplePred *pred;
  int sysname;
  Pict *pict;
/* Given that 'kind' != VarKind, assign the value in 'pred' to 'val',
   performing a conversion in the BoxTypeKind case as necessary. New space is
   allocated for StringKind intervals only (see FreeValue() above). These
   routines also convert each type to a canonical form if possible.

   Returns True iff an error is reported. Errors occur if conversions fail.
*/
{
    TableEntry *t;
    Relation rel = pred->rel;

    switch (kind) {
      case IntKind:
	val->u.int_val = pred->u.int_val;
	/* convert to integer canonical form in which kind == EqComp */
	if (val->kind == StrictComp) {
	    val->kind = EqComp;
	    switch (rel) {
	      case Less: val->u.int_val--; break;
	      case Greater: val->u.int_val++; break;
	    }
	}
	break;
      case StringKind:
	CopyString(val->u.string,pred->u.string);
	break;
      case BoolKind:
	val->u.bool = pred->u.bool;
	break;
      case BoxTypeKind:
	if ((t=FindTableId(pict->table,BoxTypeId,pred->u.string))
	    == (TableEntry *)NULL) {
	    fprintf(stderr,"%s: unknown box type name '%s' in box %d\n",
		    argv0,pred->u.string,sysname);
	    return(True);
	}
	val->u.box_type = t->u.box_type;
	/* convert to box-type canonical form */
	if (val->kind == StrictComp) {
	    if (rel == Greater) {
		val->kind = EqComp;
		val->u.box_type = val->u.box_type->u.parent;
	    } else if (rel == Less && (val->u.box_type->children == NULL)) {
		fprintf(stderr,"%s: no possible strict subtypes of '%s' ",
			argv0,val->u.box_type->name);
		fprintf(stderr,"in box %d",sysname);
		return(True);
	    }
	}
	break;
    }
    return(False);
}

static void SafeCopyString(a,b)
  String *a;
  String b;
{
    if (b != NULL) { CopyString(*a,b); }
    else *a = NULL;
}

static Intrvl *NewNoneqIntrvl(pred,sysname,pict,intvl)
  SimplePred *pred;
  int sysname;
  Pict *pict;
  INOUT Intrvl *intvl;
/* Like NewIntrvl() below, only 'pred->rel' is guaranteed not to be 'Eq'. If
   intvl is NULL, then a new Intrvl is created and returned. Otherwise,
   'intvl' is assumed to be a valid interval with only its 'low' or 'high'
   value filled in (but not both); the empty value is filled in by this
   routine.

   PRE-CONDITION: Any variable or attribute named by 'pred' must already exist
   in the Id Hash Table 'pict->table'.
*/
{
    Intrvl *result;		/* return interval */
    IntrvlRange *range;		/* the range of the interval */
    Value *val;			/* new value in (taken from 'pred') */
    String name = pred->attr_name;

    if (intvl == (Intrvl *)NULL) {
	result = AllocOne(Intrvl);
	result->range = range = AllocOne(IntrvlRange);
	range->low = range->high = (Value *)NULL;
    } else {
	result = intvl;
	range = result->range;
    }
    val = AllocOne(Value);
    switch (pred->rel) {
      case Less:
	val->kind = StrictComp;	SpliceIntoList(range->high,val); break;
      case LessEq:
	val->kind = EqComp; SpliceIntoList(range->high,val); break;
      case Greater:
	val->kind = StrictComp; SpliceIntoList(range->low,val); break;
      case GreaterEq:
	val->kind = EqComp; SpliceIntoList(range->low,val); break;
      case Eq:
	fprintf(stderr,"%s: rel==Eq passed to NewNoneqIntrvl()\n",argv0);
	exit(-1);
      case Neq:
	fprintf(stderr,"%s: # in box %d not implemented in predicates\n",
		argv0,sysname);
	exit(-1);
      default:
	fprintf(stderr,"%s: unknown relation in box %d\n",argv0,sysname);
	exit(-1);
    } /* switch */
    if (pred->kind == VarPredKind) {
	String var = pred->u.var_expr->name; /* temp var name of 'pred' */
	VarBound *vb;
	/*
	 * fill in 'name' and 'kind' fields of 'result' */
	result->name = FindTableId(pict->table,VarNameId,var)->u.var->name;
	result->kind = VarKind;
	/*
	 * copy bounding information into val */
	vb = val->u.v_bnd = AllocOne(VarBound);
	vb->attr_name = FindTableId(pict->table,AttrNameId,name)->u.attr->name;
	SafeCopyString(&(vb->prefix),pred->u.var_expr->prefix);
	SafeCopyString(&(vb->suffix),pred->u.var_expr->suffix);
    } else {			/* kind != VarPredKind */
	/*
	 * fill in 'name' and 'kind' fields of 'result' */
	result->name = FindTableId(pict->table,AttrNameId,name)->u.attr->name;
	result->kind = PrimValueTypeToIntrvlKind(pred->type);
	/*
	 * copy the appropriate kind of value into 'val' */
	if (CopyValue(result->kind,val,pred,sysname,pict)) {
	    return((Intrvl *)NULL);
	}
    }
    return(result);
}

static Intrvl *NewIntrvl(pred,sysname,pict)
  SimplePred *pred;
  int sysname;
  Pict *pict;
/* Create a new Intrvl corresponding to the SimplePred 'pred'. If any errors
   occur while forming the interval (e.g., if 'pred' is a BoxTypeVal and names
   a non-existent BoxType), NULL is returned. 'Sysname' is the sysname of the
   box containing 'pred'; it is used in error reporting.
*/
{
    Intrvl *result;

    if (pred->rel == Eq) {
	/* simulate equality by forming two intervals */
	pred->rel = LessEq;
	if ((result=NewNoneqIntrvl(pred,sysname,pict,(Intrvl *)NULL))==NULL) {
	    return((Intrvl *)NULL);
	}
	pred->rel = GreaterEq;
	result = NewNoneqIntrvl(pred,sysname,pict,result);
    } else {
	result = NewNoneqIntrvl(pred,sysname,pict,(Intrvl *)NULL);
    }
    if (pred->kind == VarPredKind) {
	Value *temp;		             /* temp for swapping */
	IntrvlRange *range = result->range;  /* temp range */
	/*
	 * swap low and high in the variable case */
	temp = range->low; range->low = range->high; range->high = temp;
    }
    return(result);
}

static void MergeIntrvl(intvl,b)
  Intrvl *intvl;
  Box *b;
/* Incorporate the Intrvl 'intvl' into the 'u1.intvls' field of the BoxElt
   'b_elt'. If there is no interval for the attribute or variable named by
   'intvl', then 'intvl' is simply installed in the list. If there already is
   an attribute or variable named by 'intvl' in the list, then the
   intersection of the intervals is computed and stored in the original
   interval; the interal 'intvl' is freed in that case.

   NOTE: An interval has the same name as that of 'intvl' if their names *and*
   their kinds agree. We require the latter so that a variable may have the
   same name as an attribute without conflict (there will be a separate copy
   of each string in the IdHashTable, since they have different IdKind's in
   the table).
*/
{
    IntrvlList *curr;

    /* do linear search for matching interval */
    StepLinkedList(curr,b->u1.intvls) {
	if (intvl->kind == curr->i->kind
	    && SameString(intvl->name,curr->i->name)) {
	    break;
	}
    }
    if (curr == (IntrvlList *)NULL) {
	/* no match found; simply insert at front of list */
	curr = AllocOne(IntrvlList);
	curr->i = intvl;
	curr->next = b->u1.intvls;
	b->u1.intvls = curr;
    } else {
	/* match found, so merge the two intervals */
	IntersectIntrvls(curr->i,intvl);
    }
}

static Boolean EmptyIntrvl(i,sysname)
  Intrvl *i;
  int sysname;
/* Returns True iff 'i' represents an empty interval. In this case and if
   'sysname' is non-negative, an error is reported to stderr naming the
   problem attribute, interval, and the sysname 'sysname' of the box with the
   problem. In the case where two variable intervals are intersected, we can
   detect a limited number of ways for the resulting interval to be empty, but
   only when both the 'low' and 'high' values name the same attribute.
*/
{
    Boolean empty;
    IntrvlRange *r = i->range;

    if ((empty=NotOf(r->low == NULL || r->high == NULL))) {
	switch (i->kind) {
	  case IntKind:     empty = EmptyIntRange(r); break;
	  case StringKind:  empty = EmptyStringRange(r); break;
	  case BoolKind:    empty = EmptyBoolRange(r); break;
	  case BoxTypeKind: empty = EmptyBoxTypeRange(r); break;
	  case VarKind:     empty = EmptyVarRange(r); break;
	}
	if (sysname >= 0 && empty) {
	    fprintf(stderr,"%s: empty '",argv0);
	    if (i->kind == VarKind) fputc('$',stderr);
	    fprintf(stderr,"%s' interval ",i->name);
	    DisplayIntrvl(i);
	    fprintf(stderr," in box %d\n",sysname);
	}
    }
    return(empty);
}

static Boolean FormIntrvls(b_elt,pict)
  BoxElt *b_elt;
  Pict *pict;
/* Forms the intervals for BoxElt 'b'. This routine frees the space occupied
   by 'b->u.b->u1.pred' and sets 'b->u.b->kind' to IntvlKind. Reports errors
   as in FormAllIntrvls() below.

   Returns True iff any errors are reported.
*/
{
    Boolean result = False;
    Box *b = b_elt->u.b;
    Predicate *head,*curr;
    Intrvl *i;
    IntrvlList *il;
    EltList *adj;
    Boolean atomic,non_atomic;
    Arrow *a;

    static SimplePred type_pred =
	{ BoxTypeVal, "type", Less, StringPredKind };
    static SimplePred atomic_pred =
	{ BoolVal, "atomic", Eq, BoolPredKind };

    /* initialize the Box so it contains an IntrvlList instead of a Pred */
    b->kind = IntvlKind;
    head = b->u1.pred;
    b->u1.intvls = (IntrvlList *)NULL;

    /* install a "type" upper-bound depending on the "side" of the box */
    type_pred.rel = Less;
    switch (b->side) {
      case LeftSide:
	type_pred.u.string = SUBJ_NAME;
	MergeIntrvl(NewIntrvl(&type_pred,b_elt->sysname,pict),b);
	break;
      case RightSide:
	type_pred.u.string = OBJ_NAME;
	MergeIntrvl(NewIntrvl(&type_pred,b_elt->sysname,pict),b);
	break;
    } /* switch() */

    /* determine if this box should be atomic or non-atomic (or both) */
    atomic = non_atomic = False;
    StepLinkedList(adj,b_elt->adj_elts) {
	if (adj->elt->kind == ArrowEltKind) {
	    a = adj->elt->u.a;
	    switch (a->kind) {
	      case Semantics:
		/* this box is incident on a semantics arrow */
		if (AtomicOnly) { atomic = True; }
		break;
	      case Containment:
		/* this box is incident on some form of containment arrow */
		if (a->to == b_elt) { /* the arrow points at this box */
		    non_atomic = True;
		}
		break;
	    } /* switch() */
	} /* if() */
    } /* StepLinkedList() */

    /* install an "atomic" interval if incident on a semantics arrow */
    if (AtomicOnly && atomic) {
	atomic_pred.rel = Eq;
	atomic_pred.u.bool = True;
	MergeIntrvl(NewIntrvl(&atomic_pred,b_elt->sysname,pict),b);
    }

    /* install an "atomic" interval if pointed to by containment arrow */
    if (non_atomic) {
	atomic_pred.rel = Eq;
	atomic_pred.u.bool = False;
	MergeIntrvl(NewIntrvl(&atomic_pred,b_elt->sysname,pict),b);
    }

    /* install an Intrvl for each SimplePred */
    type_pred.rel = LessEq;
    for (curr=head; curr; curr=curr->pred) {
	if ((i=NewIntrvl(curr->simple_pred,b_elt->sysname,pict)) == NULL) {
	    result = True;
	} else {
	    String attr_name = curr->simple_pred->attr_name;
	    TableEntry *tbl;
	    /*
	     * install the simple predicate as an interval */
	    MergeIntrvl(i,b);
	    /*
	     * install a type restriction if necessary */
	    tbl = FindTableId(pict->table,AttrNameId,attr_name);
	    type_pred.u.string = tbl->u.attr->boxtype_name;
	    if (!SameString(type_pred.u.string,"entity")) {
		MergeIntrvl(NewIntrvl(&type_pred,b_elt->sysname,pict),b);
	    }
	}
    }

    /* free space used by predicate parse tree */
    FreePred(head);

    /* check that all computed intervals are non-empty */
    StepLinkedList(il,b->u1.intvls) {
	if (EmptyIntrvl(il->i,b_elt->sysname)) result = True;
    }
#ifdef DEBUG
    DisplayBoxIntrvls(b_elt);
#endif DEBUG
    return(result);
}

static void DisplayValue(v,kind)
  Value *v;
  IntrvlKind kind;
{
    VarBound *vb;

    if (v == (Value *)NULL) {
	fputs("NONE",stderr);
    } else {
	switch (kind) {
	  case IntKind:
	    fprintf(stderr,"%d",v->u.int_val);
	    break;
	  case StringKind:
	    fprintf(stderr,"\"%s\"",v->u.string);
	    break;
	  case BoolKind:
	    fprintf(stderr,"%s",v->u.bool ? "True" : "False");
	    break;
	  case BoxTypeKind:
	    fprintf(stderr,"%s",v->u.box_type->name);
	    break;
	  case VarKind:
	    vb = v->u.v_bnd;
	    if (vb->prefix) fprintf(stderr,"\"%s\" ",vb->prefix);
	    fprintf(stderr,"%s",vb->attr_name);
	    if (vb->suffix) fprintf(stderr," \"%s\"",vb->suffix);
	    break;
	}
    }
}

static void DisplayValueList(v,kind)
  Value *v;
  IntrvlKind kind;
/* Print a (potential list) of Value's 'v' of kind 'kind' to stderr.
*/
{
    if (v != NULL && v->next != NULL) {
	/* more than one item in list */
	fputc('{',stderr);
	StepInitializedLinkedList(v) {
	    DisplayValue(v,kind);
	    if (v->next != NULL) fputc(',',stderr);
	}
	fputc('}',stderr);
    } else {
	/* 0 or 1 item in list only */
	DisplayValue(v,kind);
    }
}

#ifdef UNUSED

static Boolean SameStringPtrs(s1,s2)
  String s1,s2;
/* Return True iff both 's1' and 's2' are NULL *or* both 's1' and 's2' are
   non-NULL are contain the same string.
*/
{
    if (s1==NULL && s2==NULL) { return(True); }
    if (s1==NULL || s2==NULL) { return(False); }
    return(MakeBoolean(SameString(s1,s2)));
}

static Boolean SameVarBound(vb1,vb2)
  VarBound *vb1,*vb2;
/* Return True iff 'vb1' and 'vb2' are "equal".
*/
{
    if ((vb1->attr_name != vb2->attr_name)
	|| !SameStringPtrs(vb1->prefix,vb2->prefix)
	|| !SameStringPtrs(vb1->suffix,vb2->suffix)) {
	return(False);
    }
    return(True);
}
#endif UNUSED

static Boolean GreaterEqValue(kind,v1,v2,select)
  IntrvlKind kind;
  Value *v1,*v2;
  MaxMinSelect select;
/* Returns True iff 'v1' >= 'v2', where 'v1' and 'v2' are both of kind
   'kind', and the '>=' relation is the "greater than" relation on this kind
   of value.
*/
{
    Boolean v1_bigger;
    int result;
    Value temp_v1;

    switch (kind) {
      case IntKind:     return(GreaterEqInt(v1,v2));
      case StringKind:  return(GreaterEqString(v1,v2,select));
      case BoolKind:    return(GreaterEqBool(v1,v2));
      case BoxTypeKind:
	temp_v1.kind = v1->kind;
	temp_v1.u.box_type = v1->u.box_type;
	v1_bigger = GreaterEqBoxType(&temp_v1,v2,select);
	switch (select) {
	  case MinSelect:
	    return(v1->u.box_type == &NoType ? False : v1_bigger);
	  case MaxSelect:
	    result = v1_bigger && temp_v1.u.box_type==v1->u.box_type;
	    return(MakeBoolean(result));
	}
	break;
      case VarKind:
	fprintf(stderr,"VarKind breaks interval.GreaterEqValue()\n");
	break;
    }
    return(MakeBoolean(select));
}

static Boolean Above(kind,r1,r2)
  IntrvlKind kind;
  IntrvlRange *r1,*r2;
{
    Intrvl i;
    IntrvlRange rng;

    i.kind = kind;
    i.range = &rng;
    rng.low = r1->low;
    rng.high = r2->high;
    return(EmptyIntrvl(&i,-1));
}

/* GLOBAL FUNCTIONS ======================================================== */

void FreeIntrvl(i,free_values)
  Intrvl *i;
  Boolean free_values;
{
    if (free_values) {
	if (i->range->low != i->range->high) {
	    /* don't free both if they are identical! */
	    FreeValue(i->range->low,i->kind);
	}
	FreeValue(i->range->high,i->kind);
    }
    Dealloc(i->range);
    Dealloc(i);
}

Boolean FormAllIntrvls(pict)
  Pict *pict;
{
    Boolean result = False;
    BoxList *curr;

    /* initialize bogus "NoType" */
    NoType.name = "NoType";

    /* Form the intervals for each Box in 'pict' */
    StepLinkedList(curr,pict->boxes) {
	if (FormIntrvls((BoxElt *)curr->elt,pict)) result = True;
    }
    return(result);
}

void DisplayIntrvl(i)
  Intrvl *i;
{
    if (i->range->low != (Value *)NULL) {
	fprintf(stderr,"%c ",(i->range->low->kind == EqComp) ? '[' : '(');
    } else {
	fputs("[ ",stderr);
    }
    DisplayValueList(i->range->low,i->kind);
    fputs(" , ",stderr);
    DisplayValueList(i->range->high,i->kind);
    if (i->range->high != (Value *)NULL) {
	fprintf(stderr," %c",(i->range->high->kind == EqComp) ? ']' : ')');
    } else {
	fputs(" ]",stderr);
    }
}

Boolean EqIntrvl(intrvl)
  Intrvl *intrvl;
/* IMPLEMENTATION NOTE: We do not need to check that the upper and lower
   bounds are not strict, since if the low and high values are the same, then
   the bounds must not be strict, or else the interval is empty, and would
   have been detected as such already.
*/
{
    Value *l,*low = intrvl->range->low;
    Value *h,*high = intrvl->range->high;

    if ((low != (Value *)NULL) && (high != (Value *)NULL)) {
	switch (intrvl->kind) {
	  case IntKind:
	    return(MakeBoolean(low->u.int_val == high->u.int_val));
	  case StringKind:
	    return(MakeBoolean(SameString(low->u.string,high->u.string)));
	  case BoolKind:
	    return(MakeBoolean(low->u.bool == high->u.bool));
	  case BoxTypeKind:
	    return(MakeBoolean(low->u.box_type == high->u.box_type));
	  case VarKind:
	    StepLinkedList(l,low) {
		StepLinkedList(h,high) {
		    if (l->u.v_bnd->attr_name == h->u.v_bnd->attr_name) {
			return(True);
		    }
		}
	    }
	}
    }
    return(False);
}

Boolean SameVal(kind,v1,v2)
  IntrvlKind kind;
  Val v1,v2;
{
    switch (kind) {
      case IntKind:     return(MakeBoolean(v1.int_val==v2.int_val));
      case StringKind:  return(MakeBoolean(SameString(v1.string,v2.string)));
      case BoolKind:    return(MakeBoolean(v1.bool==v2.bool));
      case BoxTypeKind: return(MakeBoolean(v1.box_type==v2.box_type));
      case VarKind:
	fprintf(stderr,"VarKind not processed by interval.SameVal()\n");
	break;
    }
    return(False);
}

void CopyVal(kind,v1,v2)
  IntrvlKind kind;
  Val *v1,v2;
{
    if (kind == StringKind) {
	CopyString(v1->string,v2.string);
    } else {
	*v1 = v2;
    }
}

String ValName(kind,type,val)
  IntrvlKind kind;
  PrimValueType type;
  Val val;
{
    String result;
    static char buff[80];

    switch (kind) {
      case IntKind:
	sprintf(buff,"%d",val.int_val);
	result = buff;
	break;
      case StringKind:
	buff[0] = '\0';
	if (type==StringVal) { strcat(buff,"\""); }
	strcat(buff,val.string);
	if (type==StringVal) { strcat(buff,"\""); }
	result = buff;
	break;
      case BoolKind:
	result = val.bool ? TRUE_KEY : FALSE_KEY;
	break;
      case BoxTypeKind:
	result = val.box_type->name;
	break;
      case VarKind:
	fprintf(stderr,"VarKind breaks interval.ValName()\n");
	exit(-1);
    }
    return(result);
}

Boolean InsideIntrvl(slot,intrvl)
  Slot *slot;
  Intrvl *intrvl;
{
    if (intrvl==(Intrvl *)NULL || (slot->key==Others && intrvl==NoIntrvl)) {
	return(True);
    }
    if (intrvl->kind == slot->kind) {
	IntrvlRange *rng = intrvl->range;
	Boolean above_low = True;
	Boolean below_high = True;
	Value value;
	value.kind = EqComp;
	value.u = *(slot->key);
	if (rng->low) {
	    above_low = GreaterEqValue(slot->kind,&value,rng->low,MinSelect);
	}
	if (rng->high) {
	    below_high = GreaterEqValue(slot->kind,rng->high,&value,MaxSelect);
	}
	return(MakeBoolean(above_low && below_high));
    }
    return(False);
}

Boolean DisjointIntrvls(i1,i2)
  Intrvl *i1,*i2;
{
    Assert(i1->kind != VarKind && i1->kind == i2->kind);
    return(MakeBoolean(Above(i1->kind,i1->range,i2->range)
		       || Above(i1->kind,i2->range,i1->range)));
}
