/* OBJS.H

   Type definitions and function declarations for the objs.c module, which
   provides facilities for creating and storing objects in a constraint
   picture.

   $Header: objs.h,v 1.5 91/11/13 03:13:04 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#ifndef OBJS_H
#define OBJS_H

#include <my-types.h>

/* Predicate in Box type */
#include "parse-pred.h"

/* BoxTypeList in Pict type */
#include "box-type.h"

/* ArrowRanks in Pict type */
#include "hints.h"

/* Hds in Pict type */
#include "hds.h"

/* IntrvlList in Box type */
#include "interval.h"

/* VarList in Box and Pict types */
#include "var.h"

/* IdHashTable in Pict type */
#include "id-table.h"

/* MACRO DEFINITIONS ======================================================= */

/* lower Range limit */
#define NONE (0)

/* unbounded upper Range limit */
#define INFINITY (-2)

/* Unknown entries in Box and Arrow objects */
#define NO_SYSNAME (-1)
#define NO_ORDER (-1)
#define NO_RANK (-1.0)

/* indices for each graphical element */
#define NUM_ELT_KINDS 4
#define BOX 0
#define SYN 1
#define CON 2
#define SEM 3

/* add additional indices for the DT's */
#define NUM_HDS_KINDS 5
#define OBJ  SEM		/* must be same as SEM above */
#define SUBJ 4

/* TYPE DEFINITIONS ======================================================== */

/* Box-Related Types ------------------------------------------------------- */

typedef enum {
    PredKind, IntvlKind
} LimitKind;

typedef enum {
    NeitherSide=0x0, LeftSide=0x1, RightSide=0x2, BothSides=0x3
} BoxSide;

typedef struct box {
    LimitKind kind;		/* determines which 'u1' field in use */
    union {
	Predicate *pred;	/* box predicate parse tree */
	IntrvlList *intvls;	/* list of intervals assoc. w/ 'pred' */
    } u1;
    Boolean starred;		/* True iff this box is starred */
    BoxSide side;		/* "type" of box pattern, left or right */
    union {
	Boolean contained;	/* True iff this box is directly contained */
	Boolean open;		/* True iff we are still searching this node */
	Boolean visited;	/* has node been visited in GetSideUnion()? */
    } u2;
    VarList *first_vars;	/* variables first bound at this box */
} Box;

/* Arrow-Related Types ----------------------------------------------------- */

typedef struct perm_list {
    struct perm_list *next;
    String perm;		/* name of access permission (in table) */
} PermList;

typedef enum {
    Pos=0, Neg=1
} ArrowParity;

typedef enum {
    Syntax=0, Semantics=1, Containment=2
} ArrowKind;

typedef struct arrow {
    ArrowKind kind;		/* kind of arrow */
    struct elt *from,*to;	/* tail and head boxes */
    ArrowParity parity;		/* arrow parity */
    union {
	Boolean starred;	/* if kind == Containment */
	PermList *perm_list;	/* if kind != Containment */
    } u;
} Arrow;

/* Element-Related Types --------------------------------------------------- */

#define THICKNESSES 2

typedef enum {
    Thick=0, Thin=1, UnknownThickness
} Thickness;

typedef enum {
    BoxEltKind, ArrowEltKind
} EltKind;

typedef struct elt_list {
    struct elt_list *next;
    struct elt *elt;
} EltList, BoxList, ArrowList;

typedef struct elt {
    int sysname;		/* sysname of element */
    Thickness thickness;	/* thickness of element */
    EltList *adj_elts;		/* list of adjacent elements */
    Boolean frontier;		/* ordering algorithm */
    Boolean marked;		/* used by various searching algs */
    int order;			/* order number */
    float rank;			/* heuristic ranking */
    EltKind kind;		/* box (BoxEltKind) or arrow (ArrowEltKind)? */
    union {
	Box *b;			/* when kind == BoxElt */
	Arrow *a;		/* when kind == ArrowElt */
    } u;
} Elt, BoxElt, ArrowElt;

/* Pict-Related Types ------------------------------------------------------ */

typedef struct range {
    int low,high;		/* low, high range for thin (blue) part */
} Range;

typedef struct pict {
    String input;		       /* name of input FILE */
    int elt_cnt;		       /* total # of boxes and arrows */
    BoxList *boxes;		       /* list of boxes in the constraint */
    ArrowList *arrows;		       /* list of arrows in the constraint */
    Range range;		       /* constraint range */
    Elt **elt;			       /* array of elements in order */
    BoxTypeList *box_types;	       /* list of box types from types file */
    VarList *var_head;		       /* list of variables in constraint */
    IdHashTable table;		       /* table of type, attr, var names */
    ArrowRanks ranks;		       /* ranking value for each arrow kind */
    EltList *elts[NUM_ELT_KINDS];      /* lists of each kind of element */
    HdsHintList *attrs[NUM_HDS_KINDS]; /* lists of attr/HDS associations */
    Hds *hds[NUM_HDS_KINDS];	       /* root HDS for each kind of element */
} Pict;

/* DEBUGGING FUNCTION DECLARATIONS ========================================= */

#ifdef DEBUG

void DisplayAllAdjacencies( /* Pict *pict */ );
/* For each BoxElt and ArrowElt in 'pict' (stored in the 'boxes' and 'arrows'
   fields, respectively), display the sysname of the element and the sysnames
   of all Elt's adjacent on it (in the 'adj_elts' field).
*/

#endif DEBUG

/* FUNCTION DECLARATIONS =================================================== */

BoxElt *NewBox( /* void */ );
/* Return a pointer to a new, initialized Box object.
*/

#ifdef MACRO_FUNCTION
void FreeBoxElt( /* BoxElt *b */ );
/* Deallocate the memory associated with the BoxElt 'b'.
*/
#endif MACRO_FUNCTION

#define FreeBoxElt(_elt) FreeElt((Elt *)(_elt))

ArrowElt *NewArrow( /* void */ );
/* Return a pointer to a new, initialized Arrow object.
*/

#ifdef MACRO_FUNCTION
void FreeArrowElt( /* ArrowElt *a */ );
/* Deallocate the memory associated with the ArrowElt 'a'.
*/
#endif MACRO_FUNCTION

#define FreeArrowElt(_elt) FreeElt((Elt *)(_elt))

void FreeElt( /* Elt *elt */ );
/* Deallocate the memory associated with the Elt 'elt'.
*/

void FreeEltList( /* EltList *l */ );
/* Deallocate the memory associated with each of the elements on the EltList
   'l'.
*/

#ifdef MACRO_FUNCTION
void FreeBoxList( /* BoxList *l */ );
/* Deallocate the memory associated with each of the box elements on the
   BoxList 'l'.
*/
#endif MACRO_FUNCTION

#define FreeBoxList(_l) FreeEltList((EltList *)(_l))

#ifdef MACRO_FUNCTION
void FreeArrowList( /* ArrowList *l */ );
/* Deallocate the memory associated with each of the Arrow elements on the
   ArrowList 'l'.
*/
#endif MACRO_FUNCTION

#define FreeArrowList(_l) FreeEltList((EltList *)(_l))

Pict *NewPict( /* void */ );
/* Return a pointer to a new, initialized Pict object.
*/

Elt *MinBoxAdjOrder( /* BoxElt *b_elt */ );
/* Returns the element that has the minimal order over the set of elements
   consisting of the box 'b_elt' and all arrows adjacent on it.
*/

void InitRelevantIntrvl( /* Pict *pict */ );
/* Initialization function that must be called once before the first call to
   RelevantIntrvl().
*/

IntrvlList *RelevantIntrvl( /* Elt *elt, String name, OUT Boolean *rng,
			       IntrvlList *one_item */ );
/* If the element 'elt' has direct or indirect bearing on the attribute named
   'name', return a pointer to an IntrvlList indicating this fact; otherwise,
   return NULL. By "direct" bearing, we mean that 'elt' is a box containing an
   explicit interval for the attribute named 'name'. By "indirect" bearing, we
   mean that the properties of 'elt' have bearing on the attribute named
   'name', but not due to an explicit interval. For example, if 'name' is
   "class" and 'elt' is a box with a "type" interval of [NONE,subject), then
   we return the interval [:SUBJ,:SUBJ].

   In the case where we cannot return an explicit interval (as in the case
   where 'elt' is an arrow whose tail box is anchored and 'name' is "from"
   -- in this case, we know the tail box is fixed, but we don't have an exact
   value to fix it to), return a one-item list containing the special value
   result == NoIntrvl instead.

   The value '*rng' is set to True in those cases where the HDS for this
   attribute must be promoted to a BST-HDS to implement a range; False
   otherwise.

   Any one-item lists are stored in the buffer provided by 'one_item' (which
   should be a pointer to space allocated for an IntrvlList).

   NOTE: It is neither the case that the result == NoIntrvl implies '*rng' is
   True, nor the converse. We could have result == NoIntrvl but '*rng' False
   (as in the case where the "from" box of an arrow is a unique, unknown
   value, but no range is necessary); we could have '*rng' True but result !=
   NoIntrvl (as in the case where an explict range such as [5,12) is
   specified). If we have *either* result == NoIntrvl OR '*rng' True, then the
   ":OTHERS" slot of the Hds will be used for this interval.

   IMPLEMENTATION NOTE: The returned interval may be short-lived, i.e., it may
   only be good until the next call to this function.
*/

void FreeIntrvlList( /* IntrvlList *i_list, *one_item */ );
/* Deallocate memory associated with 'i_list', so long as 'i_list' is not the
   special value NoIntrvls and is not the same as the buffer 'one_item'.
*/

Var *SearchVarList( /* VarList *v_list, String var_name */ );
/* Search the list of variables 'v_list' for a variable named 'var_name' which
   is assumed to exist in the IdHashTable, and return that Var if it is found;
   NULL otherwise.
*/

BoxList *PotentialSamePrecursors( /* BoxElt *b_elt, Pict *pict */ );
/* Return a list of boxes in 'pict' that appear before 'b_elt' in the ordering
   and that could potentially be the same box element as 'b_elt'.
*/

void FreeBoxListSkeleton( /* BoxList *bl */ );
/* Deallocate the structures comprising the *skeleton* only of the BoxList
   'bl'.
*/

double CountVarRange( /* IntrvlRange *r, OUT VarCnt *cnt, IdHashTable table,
			 Boolean eql_yet */ );
/* Set the values of 'cnt' corresponding to the IntrvlRange 'r'. In
   particular, 'cnt->eq_cnt' is set to the size of the intersection set of the
   lists of attribute names 'r->low' and 'r->high'. Let the intersection set
   be S. Then 'cnt->low_cnt' and 'cnt->high_cnt' are set as follows:

     'cnt->low_cnt'  = |'r->low' \setminus S|,
     'cnt->high_cnt' = |'r->high' \setminus S|,

   where |set| denotes the size of the set. Thus, the counts are the number of
   attributes in each list *unique* to that list (i.e., not also occurring in
   the other list).

   If any equalities are found, the sizes of the ranges associated with these
   equal attributes are looked up in 'table'. Say these sizes are
   size_1,...,size_n. If 'eql_yet' is True, then the product of the *inverses*
   of these sizes is returned; otherwise, the product of the *inverses* of
   size_2 through size_n is returned. If there are no equalities, then 1.0 is
   returned.
*/

#endif OBJS_H
